#' Generate all crosses from given parent set.
#' @param n Number of parents
#' @param line_id Character vector of parent names
#' @references
#' K.P. Schliep (2011) phangorn: phylogenetic analysis in R
#'
#' @returns
#' List of crossing procedure.
#' The output value is defined by "multiPhylo" class.
#'
#' @examples
#' n <- 3
#' line_id <- c("x1", "x2", "x7")
#' rslt <- allCrosses(n, line_id)
#' plot(rslt[[1]])
#' plot(rslt[[2]])
#' plot(rslt[[3]])
#'
#' @note
#' This function is adapted and modified from `allTrees`, `dfactorial`, and `ldfactorial` functions in the R package `phangorn` (GPL-2 | GPL-3).
#' See: https://cran.r-project.org/web/packages/phangorn/index.html
#'
#' @export


allCrosses <- function (n, line_id) {

  rooted = TRUE # Modified from allTrees function in phangorn package

  n <- as.integer(n)
  nt <- as.integer(round(dfactorial(2 * (n + rooted) - 5)))
  if ((n + rooted) > 10) {
    stop(gettextf("That would generate %d trees, and take up more than %d MB of memory!",
                  nt, as.integer(round(nt/1000)), domain = "R-phangorn"))
  }
  if (n < 2) {
    stop("A tree must have at least two genotypes.")
  }
  if (!rooted && n == 2) {
    stop("An unrooted tree must have at least three taxa.")
  }
  if (rooted) {
    edge <- matrix(NA, 2 * n - 2, 2)
    edge[1:2, ] <- c(n + 1L, n + 1L, 1L, 2L)
  }
  else {
    edge <- matrix(NA, 2 * n - 3, 2)
    edge[1:3, ] <- c(n + 1L, n + 1L, n + 1L, 1L, 2L, 3L)
  }
  edges <- list()
  edges[[1]] <- edge
  m <- 1
  nedge <- 1
  trees <- vector("list", nt)
  if ((n + rooted) > 3) {
    i <- 3L + (!rooted)
    pa <- n + 2L
    nr <- 2L + (!rooted)
    while (i < (n + 1L)) {
      nedge <- nedge + 2
      m2 <- m * nedge
      newedges <- vector("list", m2)
      for (j in 1:m) {
        edge <- edges[[j]]
        l <- nr
        edgeA <- edge
        edgeB <- edge
        for (k in 1L:l) {
          edge <- edgeA
          node <- edge[k, 1]
          edge[k, 1] <- pa
          edge[l + 1, ] <- c(pa, i)
          edge[l + 2, ] <- c(node, pa)
          newedges[[(j - 1) * (l + rooted) + k]] <- edge
        }
        if (rooted) {
          edgeB[] <- as.integer(sub(n + 1L, pa, edgeB))
          edge <- edgeB
          edge[l + 1, ] <- c(n + 1L, i)
          edge[l + 2, ] <- c(n + 1L, pa)
          newedges[[j * (l + 1)]] <- edge
        }
      }
      edges <- newedges
      m <- m2
      i <- i + 1L
      pa <- pa + 1L
      nr <- nr + 2L
    }
  }
  for (x in 1:m) {
    edge <- edges[[x]]
    edge <- edge[ape::reorderRcpp(edge, n, n + 1L, 2L), ]
    tree <- list(edge = edge)
    tree$Nnode <- as.integer(n - 2L + rooted)
    attr(tree, "order") <- "postorder"
    class(tree) <- "phylo"
    trees[[x]] <- tree
  }
  attr(trees, "TipLabel") <- if (is.null(line_id))
    paste("t", 1:n, sep = "")
  else line_id
  class(trees) <- "multiPhylo"
  trees
}

# This function is adapted from `dfactorial` function in the R package `phangorn`.

dfactorial <- function (x)
  exp(ldfactorial(x))

# This function is adapted from `ldfactorial` functions in the R package `phangorn`.

ldfactorial <- function (x)
{
  x <- (x + 1)/2
  res <- lgamma(2 * x) - (lgamma(x) + (x - 1) * log(2))
  res
}
