//! Constants needed to parse markdown.
//!
//! Most of these constants are magic numbers, such as the number of markers
//! needed to parse [code (fenced)][raw_flow]
//! ([`CODE_FENCED_SEQUENCE_SIZE_MIN`][]) or the max number of allowed markers
//! in a [heading (atx)][heading_atx]
//! ([`HEADING_ATX_OPENING_FENCE_SIZE_MAX`][]).
//!
//! Some constants are instead lists of things, such as the list of tag names
//! considered in the **raw** production of [HTML (flow)][html_flow]
//! ([`HTML_RAW_NAMES`][]), or the list of named character references
//! ([`CHARACTER_REFERENCES`][]).
//!
//! [raw_flow]: crate::construct::raw_flow
//! [heading_atx]: crate::construct::heading_atx
//! [html_flow]: crate::construct::html_flow

/// The number of characters allowed in a protocol of an [autolink][].
///
/// The protocol part is the `xxx` in `<xxx://example.com>`.
/// 32 characters is fine, 33 is too many.
///
/// [autolink]: crate::construct::autolink
pub const AUTOLINK_SCHEME_SIZE_MAX: usize = 32;

/// The number of characters allowed in a domain of an email [autolink][].
///
/// There can be multiple “domains”.
/// A domain part is each `xxx` in `<example@xxx.xxx.xxx>`.
/// 63 characters is fine, 64 is too many.
///
/// [autolink]: crate::construct::autolink
pub const AUTOLINK_DOMAIN_SIZE_MAX: usize = 63;

/// The max number of characters in a decimal numeric
/// [character reference][character_reference].
///
/// To illustrate, this allows `&#9999999;` and disallows `&#99999990;`.
/// This limit is imposed because all bigger numbers are invalid.
///
/// [character_reference]: crate::construct::character_reference
pub const CHARACTER_REFERENCE_DECIMAL_SIZE_MAX: usize = 7;

/// The max number of characters in a hexadecimal numeric
/// [character reference][character_reference].
///
/// To illustrate, this allows `&#xff9999;` and disallows `&#xff99990;`.
/// This limit is imposed because all bigger numbers are invalid.
///
/// [character_reference]: crate::construct::character_reference
pub const CHARACTER_REFERENCE_HEXADECIMAL_SIZE_MAX: usize = 6;

/// The max number of characters in a named
/// [character reference][character_reference].
///
/// This is the number of the longest name in [`CHARACTER_REFERENCES`][].
/// It allows `&CounterClockwiseContourIntegral;` and prevents the parser from
/// continuing for eons.
///
/// [character_reference]: crate::construct::character_reference
pub const CHARACTER_REFERENCE_NAMED_SIZE_MAX: usize = 31;

/// The number of markers needed for [code (fenced)][raw_flow] to form.
///
/// Like many things in markdown, the number is `3`.
///
/// [raw_flow]: crate::construct::raw_flow
pub const CODE_FENCED_SEQUENCE_SIZE_MIN: usize = 3;

/// The number of markers needed for [frontmatter][] to form.
///
/// Like many things in markdown, the number is `3`.
///
/// [frontmatter]: crate::construct::frontmatter
pub const FRONTMATTER_SEQUENCE_SIZE: usize = 3;

/// The number of the longest tag name in [`GFM_HTML_TAGFILTER_NAMES`][].
///
/// This is currently the size of `plaintext`.
pub const GFM_HTML_TAGFILTER_SIZE_MAX: usize = 9;

/// List of HTML tag names that are escaped by GFMs tag filter.
///
/// Tag name matching must be performed insensitive to case, and thus this list
/// includes lowercase tag names.
///
/// ## References
///
/// * [*§ 6.1 Disallowed Raw HTML (extension)* in GFM](https://github.github.com/gfm/#disallowed-raw-html-extension-)
pub const GFM_HTML_TAGFILTER_NAMES: [&str; 9] = [
    "iframe",
    "noembed",
    "noframes",
    "plaintext",
    "script",
    "style",
    "textarea",
    "title",
    "xmp",
];

/// The number of preceding spaces needed for a [hard break
/// (trailing)][whitespace] to form.
///
/// [whitespace]: crate::construct::partial_whitespace
pub const HARD_BREAK_PREFIX_SIZE_MIN: usize = 2;

/// The max number of markers allowed to form a [heading (atx)][heading_atx].
///
/// This limitation is imposed by HTML, which imposes a max heading rank of
/// `6`.
///
/// [heading_atx]: crate::construct::heading_atx
pub const HEADING_ATX_OPENING_FENCE_SIZE_MAX: usize = 6;

/// List of HTML tag names that form the **basic** production of
/// [HTML (flow)][html_flow].
///
/// The **basic** production allows interleaving HTML and markdown with blank
/// lines and allows flow (block) elements to interrupt definitions, paragraphs,
/// and heading (setext).
/// Tag name matching must be performed insensitive to case, and thus this list
/// includes lowercase tag names.
///
/// Tag names not on this list result in the **complete** production.
///
/// > 👉 **Note**: `source` was removed on `main` of the `CommonMark` spec and
/// > is slated to be released in `CommonMark@0.31`.
///
/// ## References
///
/// * [*§ 4.6 HTML blocks* in `CommonMark`](https://spec.commonmark.org/0.31/#html-blocks)
/// * [*Remove source element as HTML block start condition* as `commonmark/commonmark-spec#710`](https://github.com/commonmark/commonmark-spec/pull/710)
///
/// [html_flow]: crate::construct::html_flow
pub const HTML_BLOCK_NAMES: [&str; 62] = [
    "address",
    "article",
    "aside",
    "base",
    "basefont",
    "blockquote",
    "body",
    "caption",
    "center",
    "col",
    "colgroup",
    "dd",
    "details",
    "dialog",
    "dir",
    "div",
    "dl",
    "dt",
    "fieldset",
    "figcaption",
    "figure",
    "footer",
    "form",
    "frame",
    "frameset",
    "h1",
    "h2",
    "h3",
    "h4",
    "h5",
    "h6",
    "head",
    "header",
    "hr",
    "html",
    "iframe",
    "legend",
    "li",
    "link",
    "main",
    "menu",
    "menuitem",
    "nav",
    "noframes",
    "ol",
    "optgroup",
    "option",
    "p",
    "param",
    "search",
    "section",
    "summary",
    "table",
    "tbody",
    "td",
    "tfoot",
    "th",
    "thead",
    "title",
    "tr",
    "track",
    "ul",
];

/// Magic string of CDATA (after `<![`).
///
/// Used in the **cdata** production of [HTML (flow)][html_flow] and
/// [HTML (text)][html_text].
///
/// [html_flow]: crate::construct::html_flow
/// [html_text]: crate::construct::html_text
pub const HTML_CDATA_PREFIX: [u8; 6] = [b'C', b'D', b'A', b'T', b'A', b'['];

/// List of HTML tag names that form the **raw** production of
/// [HTML (flow)][html_flow].
///
/// The **raw** production allows blank lines and thus no interleaving with
/// markdown.
/// Tag name matching must be performed insensitive to case, and thus this list
/// includes lowercase tag names.
///
/// The number of the longest tag name is also stored as a constant in
/// [`HTML_RAW_SIZE_MAX`][].
///
/// > 👉 **Note**: `textarea` was added in `CommonMark@0.30`.
///
/// ## References
///
/// * [*§ 4.6 HTML blocks* in `CommonMark`](https://spec.commonmark.org/0.31/#html-blocks)
///
/// [html_flow]: crate::construct::html_flow
pub const HTML_RAW_NAMES: [&str; 4] = ["pre", "script", "style", "textarea"];

/// The number of the longest tag name in [`HTML_RAW_NAMES`][].
///
/// This is currently the size of `textarea`.
pub const HTML_RAW_SIZE_MAX: usize = 8;

/// To safeguard performance, labels are capped at a large number: `999`.
pub const LINK_REFERENCE_SIZE_MAX: usize = 999;

/// The max number of decimals allowed to form an (ordered)
/// [list item][list-item].
///
/// `CommonMark` caps this at 10 digits (9 is fine, 10 not).
/// This limit is imposed because bigger numbers result in integer overflows
/// in some browsers.
///
/// ## References
///
/// * [*§ 5.2 List items* in `CommonMark`](https://spec.commonmark.org/0.31/#ordered-list-marker)
///
/// [list-item]: crate::construct::list_item
pub const LIST_ITEM_VALUE_SIZE_MAX: usize = 10;

/// The number of markers needed for [math (flow)][raw_flow] to form.
///
/// Unlike code (fenced), this number is `2`.
///
/// [raw_flow]: crate::construct::raw_flow
pub const MATH_FLOW_SEQUENCE_SIZE_MIN: usize = 2;

/// Maximum allowed unbalanced parens in destination.
///
/// There can be many balanced parens, but if there are 33 opens that were not
/// yet closed, the destination does not parse.
/// `CommonMark` requires that at least 3 opening parens are allowed.
/// See: <https://spec.commonmark.org/0.31/#link-destination>,
/// In practice, this is quite low, and several places instead cap it at 32.
/// See: <https://github.com/remarkjs/react-markdown/issues/658#issuecomment-984345577>.
pub const RESOURCE_DESTINATION_BALANCE_MAX: usize = 32;

/// List of protocols allowed, when operating safely, as `href` on `a`.
///
/// This list is based on what is allowed by GitHub.
pub const SAFE_PROTOCOL_HREF: [&str; 6] = ["http", "https", "irc", "ircs", "mailto", "xmpp"];

/// List of protocols allowed, when operating safely, as `src` on `img`.
///
/// This list is based on what is allowed by GitHub.
pub const SAFE_PROTOCOL_SRC: [&str; 2] = ["http", "https"];

/// The number of characters that form a tab stop.
///
/// This relates to the number of whitespace characters needed to form certain
/// constructs in markdown, most notable the whitespace required to form
/// [code (indented)][code_indented].
///
/// [code_indented]: crate::construct::code_indented
pub const TAB_SIZE: usize = 4;

/// The number of markers needed for a [thematic break][thematic_break] to form.
///
/// Like many things in markdown, the number is `3`.
///
/// [thematic_break]: crate::construct::thematic_break
pub const THEMATIC_BREAK_MARKER_COUNT_MIN: usize = 3;

// Important: please touch the below lists as few times as possible to keep Git small.

/// List of names and values that form named [character reference][character_reference]s.
///
/// This list is sensitive to casing.
///
/// The number of the longest name (`CounterClockwiseContourIntegral`) is also
/// stored as a constant in [`CHARACTER_REFERENCE_NAMED_SIZE_MAX`][].
///
/// ## References
///
/// * [*§ 2.5 Entity and numeric character references* in `CommonMark`](https://spec.commonmark.org/0.31/#entity-and-numeric-character-references)
///
/// [character_reference]: crate::construct::character_reference
pub static CHARACTER_REFERENCES: [(&str, &str); 2125] = [
    ("AElig", "Æ"),
    ("AMP", "&"),
    ("Aacute", "Á"),
    ("Abreve", "Ă"),
    ("Acirc", "Â"),
    ("Acy", "А"),
    ("Afr", "𝔄"),
    ("Agrave", "À"),
    ("Alpha", "Α"),
    ("Amacr", "Ā"),
    ("And", "⩓"),
    ("Aogon", "Ą"),
    ("Aopf", "𝔸"),
    ("ApplyFunction", "⁡"),
    ("Aring", "Å"),
    ("Ascr", "𝒜"),
    ("Assign", "≔"),
    ("Atilde", "Ã"),
    ("Auml", "Ä"),
    ("Backslash", "∖"),
    ("Barv", "⫧"),
    ("Barwed", "⌆"),
    ("Bcy", "Б"),
    ("Because", "∵"),
    ("Bernoullis", "ℬ"),
    ("Beta", "Β"),
    ("Bfr", "𝔅"),
    ("Bopf", "𝔹"),
    ("Breve", "˘"),
    ("Bscr", "ℬ"),
    ("Bumpeq", "≎"),
    ("CHcy", "Ч"),
    ("COPY", "©"),
    ("Cacute", "Ć"),
    ("Cap", "⋒"),
    ("CapitalDifferentialD", "ⅅ"),
    ("Cayleys", "ℭ"),
    ("Ccaron", "Č"),
    ("Ccedil", "Ç"),
    ("Ccirc", "Ĉ"),
    ("Cconint", "∰"),
    ("Cdot", "Ċ"),
    ("Cedilla", "¸"),
    ("CenterDot", "·"),
    ("Cfr", "ℭ"),
    ("Chi", "Χ"),
    ("CircleDot", "⊙"),
    ("CircleMinus", "⊖"),
    ("CirclePlus", "⊕"),
    ("CircleTimes", "⊗"),
    ("ClockwiseContourIntegral", "∲"),
    ("CloseCurlyDoubleQuote", "”"),
    ("CloseCurlyQuote", "’"),
    ("Colon", "∷"),
    ("Colone", "⩴"),
    ("Congruent", "≡"),
    ("Conint", "∯"),
    ("ContourIntegral", "∮"),
    ("Copf", "ℂ"),
    ("Coproduct", "∐"),
    ("CounterClockwiseContourIntegral", "∳"),
    ("Cross", "⨯"),
    ("Cscr", "𝒞"),
    ("Cup", "⋓"),
    ("CupCap", "≍"),
    ("DD", "ⅅ"),
    ("DDotrahd", "⤑"),
    ("DJcy", "Ђ"),
    ("DScy", "Ѕ"),
    ("DZcy", "Џ"),
    ("Dagger", "‡"),
    ("Darr", "↡"),
    ("Dashv", "⫤"),
    ("Dcaron", "Ď"),
    ("Dcy", "Д"),
    ("Del", "∇"),
    ("Delta", "Δ"),
    ("Dfr", "𝔇"),
    ("DiacriticalAcute", "´"),
    ("DiacriticalDot", "˙"),
    ("DiacriticalDoubleAcute", "˝"),
    ("DiacriticalGrave", "`"),
    ("DiacriticalTilde", "˜"),
    ("Diamond", "⋄"),
    ("DifferentialD", "ⅆ"),
    ("Dopf", "𝔻"),
    ("Dot", "¨"),
    ("DotDot", "⃜"),
    ("DotEqual", "≐"),
    ("DoubleContourIntegral", "∯"),
    ("DoubleDot", "¨"),
    ("DoubleDownArrow", "⇓"),
    ("DoubleLeftArrow", "⇐"),
    ("DoubleLeftRightArrow", "⇔"),
    ("DoubleLeftTee", "⫤"),
    ("DoubleLongLeftArrow", "⟸"),
    ("DoubleLongLeftRightArrow", "⟺"),
    ("DoubleLongRightArrow", "⟹"),
    ("DoubleRightArrow", "⇒"),
    ("DoubleRightTee", "⊨"),
    ("DoubleUpArrow", "⇑"),
    ("DoubleUpDownArrow", "⇕"),
    ("DoubleVerticalBar", "∥"),
    ("DownArrow", "↓"),
    ("DownArrowBar", "⤓"),
    ("DownArrowUpArrow", "⇵"),
    ("DownBreve", "̑"),
    ("DownLeftRightVector", "⥐"),
    ("DownLeftTeeVector", "⥞"),
    ("DownLeftVector", "↽"),
    ("DownLeftVectorBar", "⥖"),
    ("DownRightTeeVector", "⥟"),
    ("DownRightVector", "⇁"),
    ("DownRightVectorBar", "⥗"),
    ("DownTee", "⊤"),
    ("DownTeeArrow", "↧"),
    ("Downarrow", "⇓"),
    ("Dscr", "𝒟"),
    ("Dstrok", "Đ"),
    ("ENG", "Ŋ"),
    ("ETH", "Ð"),
    ("Eacute", "É"),
    ("Ecaron", "Ě"),
    ("Ecirc", "Ê"),
    ("Ecy", "Э"),
    ("Edot", "Ė"),
    ("Efr", "𝔈"),
    ("Egrave", "È"),
    ("Element", "∈"),
    ("Emacr", "Ē"),
    ("EmptySmallSquare", "◻"),
    ("EmptyVerySmallSquare", "▫"),
    ("Eogon", "Ę"),
    ("Eopf", "𝔼"),
    ("Epsilon", "Ε"),
    ("Equal", "⩵"),
    ("EqualTilde", "≂"),
    ("Equilibrium", "⇌"),
    ("Escr", "ℰ"),
    ("Esim", "⩳"),
    ("Eta", "Η"),
    ("Euml", "Ë"),
    ("Exists", "∃"),
    ("ExponentialE", "ⅇ"),
    ("Fcy", "Ф"),
    ("Ffr", "𝔉"),
    ("FilledSmallSquare", "◼"),
    ("FilledVerySmallSquare", "▪"),
    ("Fopf", "𝔽"),
    ("ForAll", "∀"),
    ("Fouriertrf", "ℱ"),
    ("Fscr", "ℱ"),
    ("GJcy", "Ѓ"),
    ("GT", ">"),
    ("Gamma", "Γ"),
    ("Gammad", "Ϝ"),
    ("Gbreve", "Ğ"),
    ("Gcedil", "Ģ"),
    ("Gcirc", "Ĝ"),
    ("Gcy", "Г"),
    ("Gdot", "Ġ"),
    ("Gfr", "𝔊"),
    ("Gg", "⋙"),
    ("Gopf", "𝔾"),
    ("GreaterEqual", "≥"),
    ("GreaterEqualLess", "⋛"),
    ("GreaterFullEqual", "≧"),
    ("GreaterGreater", "⪢"),
    ("GreaterLess", "≷"),
    ("GreaterSlantEqual", "⩾"),
    ("GreaterTilde", "≳"),
    ("Gscr", "𝒢"),
    ("Gt", "≫"),
    ("HARDcy", "Ъ"),
    ("Hacek", "ˇ"),
    ("Hat", "^"),
    ("Hcirc", "Ĥ"),
    ("Hfr", "ℌ"),
    ("HilbertSpace", "ℋ"),
    ("Hopf", "ℍ"),
    ("HorizontalLine", "─"),
    ("Hscr", "ℋ"),
    ("Hstrok", "Ħ"),
    ("HumpDownHump", "≎"),
    ("HumpEqual", "≏"),
    ("IEcy", "Е"),
    ("IJlig", "Ĳ"),
    ("IOcy", "Ё"),
    ("Iacute", "Í"),
    ("Icirc", "Î"),
    ("Icy", "И"),
    ("Idot", "İ"),
    ("Ifr", "ℑ"),
    ("Igrave", "Ì"),
    ("Im", "ℑ"),
    ("Imacr", "Ī"),
    ("ImaginaryI", "ⅈ"),
    ("Implies", "⇒"),
    ("Int", "∬"),
    ("Integral", "∫"),
    ("Intersection", "⋂"),
    ("InvisibleComma", "⁣"),
    ("InvisibleTimes", "⁢"),
    ("Iogon", "Į"),
    ("Iopf", "𝕀"),
    ("Iota", "Ι"),
    ("Iscr", "ℐ"),
    ("Itilde", "Ĩ"),
    ("Iukcy", "І"),
    ("Iuml", "Ï"),
    ("Jcirc", "Ĵ"),
    ("Jcy", "Й"),
    ("Jfr", "𝔍"),
    ("Jopf", "𝕁"),
    ("Jscr", "𝒥"),
    ("Jsercy", "Ј"),
    ("Jukcy", "Є"),
    ("KHcy", "Х"),
    ("KJcy", "Ќ"),
    ("Kappa", "Κ"),
    ("Kcedil", "Ķ"),
    ("Kcy", "К"),
    ("Kfr", "𝔎"),
    ("Kopf", "𝕂"),
    ("Kscr", "𝒦"),
    ("LJcy", "Љ"),
    ("LT", "<"),
    ("Lacute", "Ĺ"),
    ("Lambda", "Λ"),
    ("Lang", "⟪"),
    ("Laplacetrf", "ℒ"),
    ("Larr", "↞"),
    ("Lcaron", "Ľ"),
    ("Lcedil", "Ļ"),
    ("Lcy", "Л"),
    ("LeftAngleBracket", "⟨"),
    ("LeftArrow", "←"),
    ("LeftArrowBar", "⇤"),
    ("LeftArrowRightArrow", "⇆"),
    ("LeftCeiling", "⌈"),
    ("LeftDoubleBracket", "⟦"),
    ("LeftDownTeeVector", "⥡"),
    ("LeftDownVector", "⇃"),
    ("LeftDownVectorBar", "⥙"),
    ("LeftFloor", "⌊"),
    ("LeftRightArrow", "↔"),
    ("LeftRightVector", "⥎"),
    ("LeftTee", "⊣"),
    ("LeftTeeArrow", "↤"),
    ("LeftTeeVector", "⥚"),
    ("LeftTriangle", "⊲"),
    ("LeftTriangleBar", "⧏"),
    ("LeftTriangleEqual", "⊴"),
    ("LeftUpDownVector", "⥑"),
    ("LeftUpTeeVector", "⥠"),
    ("LeftUpVector", "↿"),
    ("LeftUpVectorBar", "⥘"),
    ("LeftVector", "↼"),
    ("LeftVectorBar", "⥒"),
    ("Leftarrow", "⇐"),
    ("Leftrightarrow", "⇔"),
    ("LessEqualGreater", "⋚"),
    ("LessFullEqual", "≦"),
    ("LessGreater", "≶"),
    ("LessLess", "⪡"),
    ("LessSlantEqual", "⩽"),
    ("LessTilde", "≲"),
    ("Lfr", "𝔏"),
    ("Ll", "⋘"),
    ("Lleftarrow", "⇚"),
    ("Lmidot", "Ŀ"),
    ("LongLeftArrow", "⟵"),
    ("LongLeftRightArrow", "⟷"),
    ("LongRightArrow", "⟶"),
    ("Longleftarrow", "⟸"),
    ("Longleftrightarrow", "⟺"),
    ("Longrightarrow", "⟹"),
    ("Lopf", "𝕃"),
    ("LowerLeftArrow", "↙"),
    ("LowerRightArrow", "↘"),
    ("Lscr", "ℒ"),
    ("Lsh", "↰"),
    ("Lstrok", "Ł"),
    ("Lt", "≪"),
    ("Map", "⤅"),
    ("Mcy", "М"),
    ("MediumSpace", " "),
    ("Mellintrf", "ℳ"),
    ("Mfr", "𝔐"),
    ("MinusPlus", "∓"),
    ("Mopf", "𝕄"),
    ("Mscr", "ℳ"),
    ("Mu", "Μ"),
    ("NJcy", "Њ"),
    ("Nacute", "Ń"),
    ("Ncaron", "Ň"),
    ("Ncedil", "Ņ"),
    ("Ncy", "Н"),
    ("NegativeMediumSpace", "\u{200B}"),
    ("NegativeThickSpace", "\u{200B}"),
    ("NegativeThinSpace", "\u{200B}"),
    ("NegativeVeryThinSpace", "\u{200B}"),
    ("NestedGreaterGreater", "≫"),
    ("NestedLessLess", "≪"),
    ("NewLine", "\n"),
    ("Nfr", "𝔑"),
    ("NoBreak", "\u{2060}"),
    ("NonBreakingSpace", " "),
    ("Nopf", "ℕ"),
    ("Not", "⫬"),
    ("NotCongruent", "≢"),
    ("NotCupCap", "≭"),
    ("NotDoubleVerticalBar", "∦"),
    ("NotElement", "∉"),
    ("NotEqual", "≠"),
    ("NotEqualTilde", "≂̸"),
    ("NotExists", "∄"),
    ("NotGreater", "≯"),
    ("NotGreaterEqual", "≱"),
    ("NotGreaterFullEqual", "≧̸"),
    ("NotGreaterGreater", "≫̸"),
    ("NotGreaterLess", "≹"),
    ("NotGreaterSlantEqual", "⩾̸"),
    ("NotGreaterTilde", "≵"),
    ("NotHumpDownHump", "≎̸"),
    ("NotHumpEqual", "≏̸"),
    ("NotLeftTriangle", "⋪"),
    ("NotLeftTriangleBar", "⧏̸"),
    ("NotLeftTriangleEqual", "⋬"),
    ("NotLess", "≮"),
    ("NotLessEqual", "≰"),
    ("NotLessGreater", "≸"),
    ("NotLessLess", "≪̸"),
    ("NotLessSlantEqual", "⩽̸"),
    ("NotLessTilde", "≴"),
    ("NotNestedGreaterGreater", "⪢̸"),
    ("NotNestedLessLess", "⪡̸"),
    ("NotPrecedes", "⊀"),
    ("NotPrecedesEqual", "⪯̸"),
    ("NotPrecedesSlantEqual", "⋠"),
    ("NotReverseElement", "∌"),
    ("NotRightTriangle", "⋫"),
    ("NotRightTriangleBar", "⧐̸"),
    ("NotRightTriangleEqual", "⋭"),
    ("NotSquareSubset", "⊏̸"),
    ("NotSquareSubsetEqual", "⋢"),
    ("NotSquareSuperset", "⊐̸"),
    ("NotSquareSupersetEqual", "⋣"),
    ("NotSubset", "⊂⃒"),
    ("NotSubsetEqual", "⊈"),
    ("NotSucceeds", "⊁"),
    ("NotSucceedsEqual", "⪰̸"),
    ("NotSucceedsSlantEqual", "⋡"),
    ("NotSucceedsTilde", "≿̸"),
    ("NotSuperset", "⊃⃒"),
    ("NotSupersetEqual", "⊉"),
    ("NotTilde", "≁"),
    ("NotTildeEqual", "≄"),
    ("NotTildeFullEqual", "≇"),
    ("NotTildeTilde", "≉"),
    ("NotVerticalBar", "∤"),
    ("Nscr", "𝒩"),
    ("Ntilde", "Ñ"),
    ("Nu", "Ν"),
    ("OElig", "Œ"),
    ("Oacute", "Ó"),
    ("Ocirc", "Ô"),
    ("Ocy", "О"),
    ("Odblac", "Ő"),
    ("Ofr", "𝔒"),
    ("Ograve", "Ò"),
    ("Omacr", "Ō"),
    ("Omega", "Ω"),
    ("Omicron", "Ο"),
    ("Oopf", "𝕆"),
    ("OpenCurlyDoubleQuote", "“"),
    ("OpenCurlyQuote", "‘"),
    ("Or", "⩔"),
    ("Oscr", "𝒪"),
    ("Oslash", "Ø"),
    ("Otilde", "Õ"),
    ("Otimes", "⨷"),
    ("Ouml", "Ö"),
    ("OverBar", "‾"),
    ("OverBrace", "⏞"),
    ("OverBracket", "⎴"),
    ("OverParenthesis", "⏜"),
    ("PartialD", "∂"),
    ("Pcy", "П"),
    ("Pfr", "𝔓"),
    ("Phi", "Φ"),
    ("Pi", "Π"),
    ("PlusMinus", "±"),
    ("Poincareplane", "ℌ"),
    ("Popf", "ℙ"),
    ("Pr", "⪻"),
    ("Precedes", "≺"),
    ("PrecedesEqual", "⪯"),
    ("PrecedesSlantEqual", "≼"),
    ("PrecedesTilde", "≾"),
    ("Prime", "″"),
    ("Product", "∏"),
    ("Proportion", "∷"),
    ("Proportional", "∝"),
    ("Pscr", "𝒫"),
    ("Psi", "Ψ"),
    ("QUOT", "\""),
    ("Qfr", "𝔔"),
    ("Qopf", "ℚ"),
    ("Qscr", "𝒬"),
    ("RBarr", "⤐"),
    ("REG", "®"),
    ("Racute", "Ŕ"),
    ("Rang", "⟫"),
    ("Rarr", "↠"),
    ("Rarrtl", "⤖"),
    ("Rcaron", "Ř"),
    ("Rcedil", "Ŗ"),
    ("Rcy", "Р"),
    ("Re", "ℜ"),
    ("ReverseElement", "∋"),
    ("ReverseEquilibrium", "⇋"),
    ("ReverseUpEquilibrium", "⥯"),
    ("Rfr", "ℜ"),
    ("Rho", "Ρ"),
    ("RightAngleBracket", "⟩"),
    ("RightArrow", "→"),
    ("RightArrowBar", "⇥"),
    ("RightArrowLeftArrow", "⇄"),
    ("RightCeiling", "⌉"),
    ("RightDoubleBracket", "⟧"),
    ("RightDownTeeVector", "⥝"),
    ("RightDownVector", "⇂"),
    ("RightDownVectorBar", "⥕"),
    ("RightFloor", "⌋"),
    ("RightTee", "⊢"),
    ("RightTeeArrow", "↦"),
    ("RightTeeVector", "⥛"),
    ("RightTriangle", "⊳"),
    ("RightTriangleBar", "⧐"),
    ("RightTriangleEqual", "⊵"),
    ("RightUpDownVector", "⥏"),
    ("RightUpTeeVector", "⥜"),
    ("RightUpVector", "↾"),
    ("RightUpVectorBar", "⥔"),
    ("RightVector", "⇀"),
    ("RightVectorBar", "⥓"),
    ("Rightarrow", "⇒"),
    ("Ropf", "ℝ"),
    ("RoundImplies", "⥰"),
    ("Rrightarrow", "⇛"),
    ("Rscr", "ℛ"),
    ("Rsh", "↱"),
    ("RuleDelayed", "⧴"),
    ("SHCHcy", "Щ"),
    ("SHcy", "Ш"),
    ("SOFTcy", "Ь"),
    ("Sacute", "Ś"),
    ("Sc", "⪼"),
    ("Scaron", "Š"),
    ("Scedil", "Ş"),
    ("Scirc", "Ŝ"),
    ("Scy", "С"),
    ("Sfr", "𝔖"),
    ("ShortDownArrow", "↓"),
    ("ShortLeftArrow", "←"),
    ("ShortRightArrow", "→"),
    ("ShortUpArrow", "↑"),
    ("Sigma", "Σ"),
    ("SmallCircle", "∘"),
    ("Sopf", "𝕊"),
    ("Sqrt", "√"),
    ("Square", "□"),
    ("SquareIntersection", "⊓"),
    ("SquareSubset", "⊏"),
    ("SquareSubsetEqual", "⊑"),
    ("SquareSuperset", "⊐"),
    ("SquareSupersetEqual", "⊒"),
    ("SquareUnion", "⊔"),
    ("Sscr", "𝒮"),
    ("Star", "⋆"),
    ("Sub", "⋐"),
    ("Subset", "⋐"),
    ("SubsetEqual", "⊆"),
    ("Succeeds", "≻"),
    ("SucceedsEqual", "⪰"),
    ("SucceedsSlantEqual", "≽"),
    ("SucceedsTilde", "≿"),
    ("SuchThat", "∋"),
    ("Sum", "∑"),
    ("Sup", "⋑"),
    ("Superset", "⊃"),
    ("SupersetEqual", "⊇"),
    ("Supset", "⋑"),
    ("THORN", "Þ"),
    ("TRADE", "™"),
    ("TSHcy", "Ћ"),
    ("TScy", "Ц"),
    ("Tab", "\t"),
    ("Tau", "Τ"),
    ("Tcaron", "Ť"),
    ("Tcedil", "Ţ"),
    ("Tcy", "Т"),
    ("Tfr", "𝔗"),
    ("Therefore", "∴"),
    ("Theta", "Θ"),
    ("ThickSpace", "  "),
    ("ThinSpace", " "),
    ("Tilde", "∼"),
    ("TildeEqual", "≃"),
    ("TildeFullEqual", "≅"),
    ("TildeTilde", "≈"),
    ("Topf", "𝕋"),
    ("TripleDot", "⃛"),
    ("Tscr", "𝒯"),
    ("Tstrok", "Ŧ"),
    ("Uacute", "Ú"),
    ("Uarr", "↟"),
    ("Uarrocir", "⥉"),
    ("Ubrcy", "Ў"),
    ("Ubreve", "Ŭ"),
    ("Ucirc", "Û"),
    ("Ucy", "У"),
    ("Udblac", "Ű"),
    ("Ufr", "𝔘"),
    ("Ugrave", "Ù"),
    ("Umacr", "Ū"),
    ("UnderBar", "_"),
    ("UnderBrace", "⏟"),
    ("UnderBracket", "⎵"),
    ("UnderParenthesis", "⏝"),
    ("Union", "⋃"),
    ("UnionPlus", "⊎"),
    ("Uogon", "Ų"),
    ("Uopf", "𝕌"),
    ("UpArrow", "↑"),
    ("UpArrowBar", "⤒"),
    ("UpArrowDownArrow", "⇅"),
    ("UpDownArrow", "↕"),
    ("UpEquilibrium", "⥮"),
    ("UpTee", "⊥"),
    ("UpTeeArrow", "↥"),
    ("Uparrow", "⇑"),
    ("Updownarrow", "⇕"),
    ("UpperLeftArrow", "↖"),
    ("UpperRightArrow", "↗"),
    ("Upsi", "ϒ"),
    ("Upsilon", "Υ"),
    ("Uring", "Ů"),
    ("Uscr", "𝒰"),
    ("Utilde", "Ũ"),
    ("Uuml", "Ü"),
    ("VDash", "⊫"),
    ("Vbar", "⫫"),
    ("Vcy", "В"),
    ("Vdash", "⊩"),
    ("Vdashl", "⫦"),
    ("Vee", "⋁"),
    ("Verbar", "‖"),
    ("Vert", "‖"),
    ("VerticalBar", "∣"),
    ("VerticalLine", "|"),
    ("VerticalSeparator", "❘"),
    ("VerticalTilde", "≀"),
    ("VeryThinSpace", " "),
    ("Vfr", "𝔙"),
    ("Vopf", "𝕍"),
    ("Vscr", "𝒱"),
    ("Vvdash", "⊪"),
    ("Wcirc", "Ŵ"),
    ("Wedge", "⋀"),
    ("Wfr", "𝔚"),
    ("Wopf", "𝕎"),
    ("Wscr", "𝒲"),
    ("Xfr", "𝔛"),
    ("Xi", "Ξ"),
    ("Xopf", "𝕏"),
    ("Xscr", "𝒳"),
    ("YAcy", "Я"),
    ("YIcy", "Ї"),
    ("YUcy", "Ю"),
    ("Yacute", "Ý"),
    ("Ycirc", "Ŷ"),
    ("Ycy", "Ы"),
    ("Yfr", "𝔜"),
    ("Yopf", "𝕐"),
    ("Yscr", "𝒴"),
    ("Yuml", "Ÿ"),
    ("ZHcy", "Ж"),
    ("Zacute", "Ź"),
    ("Zcaron", "Ž"),
    ("Zcy", "З"),
    ("Zdot", "Ż"),
    ("ZeroWidthSpace", "\u{200B}"),
    ("Zeta", "Ζ"),
    ("Zfr", "ℨ"),
    ("Zopf", "ℤ"),
    ("Zscr", "𝒵"),
    ("aacute", "á"),
    ("abreve", "ă"),
    ("ac", "∾"),
    ("acE", "∾̳"),
    ("acd", "∿"),
    ("acirc", "â"),
    ("acute", "´"),
    ("acy", "а"),
    ("aelig", "æ"),
    ("af", "⁡"),
    ("afr", "𝔞"),
    ("agrave", "à"),
    ("alefsym", "ℵ"),
    ("aleph", "ℵ"),
    ("alpha", "α"),
    ("amacr", "ā"),
    ("amalg", "⨿"),
    ("amp", "&"),
    ("and", "∧"),
    ("andand", "⩕"),
    ("andd", "⩜"),
    ("andslope", "⩘"),
    ("andv", "⩚"),
    ("ang", "∠"),
    ("ange", "⦤"),
    ("angle", "∠"),
    ("angmsd", "∡"),
    ("angmsdaa", "⦨"),
    ("angmsdab", "⦩"),
    ("angmsdac", "⦪"),
    ("angmsdad", "⦫"),
    ("angmsdae", "⦬"),
    ("angmsdaf", "⦭"),
    ("angmsdag", "⦮"),
    ("angmsdah", "⦯"),
    ("angrt", "∟"),
    ("angrtvb", "⊾"),
    ("angrtvbd", "⦝"),
    ("angsph", "∢"),
    ("angst", "Å"),
    ("angzarr", "⍼"),
    ("aogon", "ą"),
    ("aopf", "𝕒"),
    ("ap", "≈"),
    ("apE", "⩰"),
    ("apacir", "⩯"),
    ("ape", "≊"),
    ("apid", "≋"),
    ("apos", "'"),
    ("approx", "≈"),
    ("approxeq", "≊"),
    ("aring", "å"),
    ("ascr", "𝒶"),
    ("ast", "*"),
    ("asymp", "≈"),
    ("asympeq", "≍"),
    ("atilde", "ã"),
    ("auml", "ä"),
    ("awconint", "∳"),
    ("awint", "⨑"),
    ("bNot", "⫭"),
    ("backcong", "≌"),
    ("backepsilon", "϶"),
    ("backprime", "‵"),
    ("backsim", "∽"),
    ("backsimeq", "⋍"),
    ("barvee", "⊽"),
    ("barwed", "⌅"),
    ("barwedge", "⌅"),
    ("bbrk", "⎵"),
    ("bbrktbrk", "⎶"),
    ("bcong", "≌"),
    ("bcy", "б"),
    ("bdquo", "„"),
    ("becaus", "∵"),
    ("because", "∵"),
    ("bemptyv", "⦰"),
    ("bepsi", "϶"),
    ("bernou", "ℬ"),
    ("beta", "β"),
    ("beth", "ℶ"),
    ("between", "≬"),
    ("bfr", "𝔟"),
    ("bigcap", "⋂"),
    ("bigcirc", "◯"),
    ("bigcup", "⋃"),
    ("bigodot", "⨀"),
    ("bigoplus", "⨁"),
    ("bigotimes", "⨂"),
    ("bigsqcup", "⨆"),
    ("bigstar", "★"),
    ("bigtriangledown", "▽"),
    ("bigtriangleup", "△"),
    ("biguplus", "⨄"),
    ("bigvee", "⋁"),
    ("bigwedge", "⋀"),
    ("bkarow", "⤍"),
    ("blacklozenge", "⧫"),
    ("blacksquare", "▪"),
    ("blacktriangle", "▴"),
    ("blacktriangledown", "▾"),
    ("blacktriangleleft", "◂"),
    ("blacktriangleright", "▸"),
    ("blank", "␣"),
    ("blk12", "▒"),
    ("blk14", "░"),
    ("blk34", "▓"),
    ("block", "█"),
    ("bne", "=⃥"),
    ("bnequiv", "≡⃥"),
    ("bnot", "⌐"),
    ("bopf", "𝕓"),
    ("bot", "⊥"),
    ("bottom", "⊥"),
    ("bowtie", "⋈"),
    ("boxDL", "╗"),
    ("boxDR", "╔"),
    ("boxDl", "╖"),
    ("boxDr", "╓"),
    ("boxH", "═"),
    ("boxHD", "╦"),
    ("boxHU", "╩"),
    ("boxHd", "╤"),
    ("boxHu", "╧"),
    ("boxUL", "╝"),
    ("boxUR", "╚"),
    ("boxUl", "╜"),
    ("boxUr", "╙"),
    ("boxV", "║"),
    ("boxVH", "╬"),
    ("boxVL", "╣"),
    ("boxVR", "╠"),
    ("boxVh", "╫"),
    ("boxVl", "╢"),
    ("boxVr", "╟"),
    ("boxbox", "⧉"),
    ("boxdL", "╕"),
    ("boxdR", "╒"),
    ("boxdl", "┐"),
    ("boxdr", "┌"),
    ("boxh", "─"),
    ("boxhD", "╥"),
    ("boxhU", "╨"),
    ("boxhd", "┬"),
    ("boxhu", "┴"),
    ("boxminus", "⊟"),
    ("boxplus", "⊞"),
    ("boxtimes", "⊠"),
    ("boxuL", "╛"),
    ("boxuR", "╘"),
    ("boxul", "┘"),
    ("boxur", "└"),
    ("boxv", "│"),
    ("boxvH", "╪"),
    ("boxvL", "╡"),
    ("boxvR", "╞"),
    ("boxvh", "┼"),
    ("boxvl", "┤"),
    ("boxvr", "├"),
    ("bprime", "‵"),
    ("breve", "˘"),
    ("brvbar", "¦"),
    ("bscr", "𝒷"),
    ("bsemi", "⁏"),
    ("bsim", "∽"),
    ("bsime", "⋍"),
    ("bsol", "\\"),
    ("bsolb", "⧅"),
    ("bsolhsub", "⟈"),
    ("bull", "•"),
    ("bullet", "•"),
    ("bump", "≎"),
    ("bumpE", "⪮"),
    ("bumpe", "≏"),
    ("bumpeq", "≏"),
    ("cacute", "ć"),
    ("cap", "∩"),
    ("capand", "⩄"),
    ("capbrcup", "⩉"),
    ("capcap", "⩋"),
    ("capcup", "⩇"),
    ("capdot", "⩀"),
    ("caps", "∩︀"),
    ("caret", "⁁"),
    ("caron", "ˇ"),
    ("ccaps", "⩍"),
    ("ccaron", "č"),
    ("ccedil", "ç"),
    ("ccirc", "ĉ"),
    ("ccups", "⩌"),
    ("ccupssm", "⩐"),
    ("cdot", "ċ"),
    ("cedil", "¸"),
    ("cemptyv", "⦲"),
    ("cent", "¢"),
    ("centerdot", "·"),
    ("cfr", "𝔠"),
    ("chcy", "ч"),
    ("check", "✓"),
    ("checkmark", "✓"),
    ("chi", "χ"),
    ("cir", "○"),
    ("cirE", "⧃"),
    ("circ", "ˆ"),
    ("circeq", "≗"),
    ("circlearrowleft", "↺"),
    ("circlearrowright", "↻"),
    ("circledR", "®"),
    ("circledS", "Ⓢ"),
    ("circledast", "⊛"),
    ("circledcirc", "⊚"),
    ("circleddash", "⊝"),
    ("cire", "≗"),
    ("cirfnint", "⨐"),
    ("cirmid", "⫯"),
    ("cirscir", "⧂"),
    ("clubs", "♣"),
    ("clubsuit", "♣"),
    ("colon", ":"),
    ("colone", "≔"),
    ("coloneq", "≔"),
    ("comma", ","),
    ("commat", "@"),
    ("comp", "∁"),
    ("compfn", "∘"),
    ("complement", "∁"),
    ("complexes", "ℂ"),
    ("cong", "≅"),
    ("congdot", "⩭"),
    ("conint", "∮"),
    ("copf", "𝕔"),
    ("coprod", "∐"),
    ("copy", "©"),
    ("copysr", "℗"),
    ("crarr", "↵"),
    ("cross", "✗"),
    ("cscr", "𝒸"),
    ("csub", "⫏"),
    ("csube", "⫑"),
    ("csup", "⫐"),
    ("csupe", "⫒"),
    ("ctdot", "⋯"),
    ("cudarrl", "⤸"),
    ("cudarrr", "⤵"),
    ("cuepr", "⋞"),
    ("cuesc", "⋟"),
    ("cularr", "↶"),
    ("cularrp", "⤽"),
    ("cup", "∪"),
    ("cupbrcap", "⩈"),
    ("cupcap", "⩆"),
    ("cupcup", "⩊"),
    ("cupdot", "⊍"),
    ("cupor", "⩅"),
    ("cups", "∪︀"),
    ("curarr", "↷"),
    ("curarrm", "⤼"),
    ("curlyeqprec", "⋞"),
    ("curlyeqsucc", "⋟"),
    ("curlyvee", "⋎"),
    ("curlywedge", "⋏"),
    ("curren", "¤"),
    ("curvearrowleft", "↶"),
    ("curvearrowright", "↷"),
    ("cuvee", "⋎"),
    ("cuwed", "⋏"),
    ("cwconint", "∲"),
    ("cwint", "∱"),
    ("cylcty", "⌭"),
    ("dArr", "⇓"),
    ("dHar", "⥥"),
    ("dagger", "†"),
    ("daleth", "ℸ"),
    ("darr", "↓"),
    ("dash", "‐"),
    ("dashv", "⊣"),
    ("dbkarow", "⤏"),
    ("dblac", "˝"),
    ("dcaron", "ď"),
    ("dcy", "д"),
    ("dd", "ⅆ"),
    ("ddagger", "‡"),
    ("ddarr", "⇊"),
    ("ddotseq", "⩷"),
    ("deg", "°"),
    ("delta", "δ"),
    ("demptyv", "⦱"),
    ("dfisht", "⥿"),
    ("dfr", "𝔡"),
    ("dharl", "⇃"),
    ("dharr", "⇂"),
    ("diam", "⋄"),
    ("diamond", "⋄"),
    ("diamondsuit", "♦"),
    ("diams", "♦"),
    ("die", "¨"),
    ("digamma", "ϝ"),
    ("disin", "⋲"),
    ("div", "÷"),
    ("divide", "÷"),
    ("divideontimes", "⋇"),
    ("divonx", "⋇"),
    ("djcy", "ђ"),
    ("dlcorn", "⌞"),
    ("dlcrop", "⌍"),
    ("dollar", "$"),
    ("dopf", "𝕕"),
    ("dot", "˙"),
    ("doteq", "≐"),
    ("doteqdot", "≑"),
    ("dotminus", "∸"),
    ("dotplus", "∔"),
    ("dotsquare", "⊡"),
    ("doublebarwedge", "⌆"),
    ("downarrow", "↓"),
    ("downdownarrows", "⇊"),
    ("downharpoonleft", "⇃"),
    ("downharpoonright", "⇂"),
    ("drbkarow", "⤐"),
    ("drcorn", "⌟"),
    ("drcrop", "⌌"),
    ("dscr", "𝒹"),
    ("dscy", "ѕ"),
    ("dsol", "⧶"),
    ("dstrok", "đ"),
    ("dtdot", "⋱"),
    ("dtri", "▿"),
    ("dtrif", "▾"),
    ("duarr", "⇵"),
    ("duhar", "⥯"),
    ("dwangle", "⦦"),
    ("dzcy", "џ"),
    ("dzigrarr", "⟿"),
    ("eDDot", "⩷"),
    ("eDot", "≑"),
    ("eacute", "é"),
    ("easter", "⩮"),
    ("ecaron", "ě"),
    ("ecir", "≖"),
    ("ecirc", "ê"),
    ("ecolon", "≕"),
    ("ecy", "э"),
    ("edot", "ė"),
    ("ee", "ⅇ"),
    ("efDot", "≒"),
    ("efr", "𝔢"),
    ("eg", "⪚"),
    ("egrave", "è"),
    ("egs", "⪖"),
    ("egsdot", "⪘"),
    ("el", "⪙"),
    ("elinters", "⏧"),
    ("ell", "ℓ"),
    ("els", "⪕"),
    ("elsdot", "⪗"),
    ("emacr", "ē"),
    ("empty", "∅"),
    ("emptyset", "∅"),
    ("emptyv", "∅"),
    ("emsp13", " "),
    ("emsp14", " "),
    ("emsp", " "),
    ("eng", "ŋ"),
    ("ensp", " "),
    ("eogon", "ę"),
    ("eopf", "𝕖"),
    ("epar", "⋕"),
    ("eparsl", "⧣"),
    ("eplus", "⩱"),
    ("epsi", "ε"),
    ("epsilon", "ε"),
    ("epsiv", "ϵ"),
    ("eqcirc", "≖"),
    ("eqcolon", "≕"),
    ("eqsim", "≂"),
    ("eqslantgtr", "⪖"),
    ("eqslantless", "⪕"),
    ("equals", "="),
    ("equest", "≟"),
    ("equiv", "≡"),
    ("equivDD", "⩸"),
    ("eqvparsl", "⧥"),
    ("erDot", "≓"),
    ("erarr", "⥱"),
    ("escr", "ℯ"),
    ("esdot", "≐"),
    ("esim", "≂"),
    ("eta", "η"),
    ("eth", "ð"),
    ("euml", "ë"),
    ("euro", "€"),
    ("excl", "!"),
    ("exist", "∃"),
    ("expectation", "ℰ"),
    ("exponentiale", "ⅇ"),
    ("fallingdotseq", "≒"),
    ("fcy", "ф"),
    ("female", "♀"),
    ("ffilig", "ﬃ"),
    ("fflig", "ﬀ"),
    ("ffllig", "ﬄ"),
    ("ffr", "𝔣"),
    ("filig", "ﬁ"),
    ("fjlig", "fj"),
    ("flat", "♭"),
    ("fllig", "ﬂ"),
    ("fltns", "▱"),
    ("fnof", "ƒ"),
    ("fopf", "𝕗"),
    ("forall", "∀"),
    ("fork", "⋔"),
    ("forkv", "⫙"),
    ("fpartint", "⨍"),
    ("frac12", "½"),
    ("frac13", "⅓"),
    ("frac14", "¼"),
    ("frac15", "⅕"),
    ("frac16", "⅙"),
    ("frac18", "⅛"),
    ("frac23", "⅔"),
    ("frac25", "⅖"),
    ("frac34", "¾"),
    ("frac35", "⅗"),
    ("frac38", "⅜"),
    ("frac45", "⅘"),
    ("frac56", "⅚"),
    ("frac58", "⅝"),
    ("frac78", "⅞"),
    ("frasl", "⁄"),
    ("frown", "⌢"),
    ("fscr", "𝒻"),
    ("gE", "≧"),
    ("gEl", "⪌"),
    ("gacute", "ǵ"),
    ("gamma", "γ"),
    ("gammad", "ϝ"),
    ("gap", "⪆"),
    ("gbreve", "ğ"),
    ("gcirc", "ĝ"),
    ("gcy", "г"),
    ("gdot", "ġ"),
    ("ge", "≥"),
    ("gel", "⋛"),
    ("geq", "≥"),
    ("geqq", "≧"),
    ("geqslant", "⩾"),
    ("ges", "⩾"),
    ("gescc", "⪩"),
    ("gesdot", "⪀"),
    ("gesdoto", "⪂"),
    ("gesdotol", "⪄"),
    ("gesl", "⋛︀"),
    ("gesles", "⪔"),
    ("gfr", "𝔤"),
    ("gg", "≫"),
    ("ggg", "⋙"),
    ("gimel", "ℷ"),
    ("gjcy", "ѓ"),
    ("gl", "≷"),
    ("glE", "⪒"),
    ("gla", "⪥"),
    ("glj", "⪤"),
    ("gnE", "≩"),
    ("gnap", "⪊"),
    ("gnapprox", "⪊"),
    ("gne", "⪈"),
    ("gneq", "⪈"),
    ("gneqq", "≩"),
    ("gnsim", "⋧"),
    ("gopf", "𝕘"),
    ("grave", "`"),
    ("gscr", "ℊ"),
    ("gsim", "≳"),
    ("gsime", "⪎"),
    ("gsiml", "⪐"),
    ("gt", ">"),
    ("gtcc", "⪧"),
    ("gtcir", "⩺"),
    ("gtdot", "⋗"),
    ("gtlPar", "⦕"),
    ("gtquest", "⩼"),
    ("gtrapprox", "⪆"),
    ("gtrarr", "⥸"),
    ("gtrdot", "⋗"),
    ("gtreqless", "⋛"),
    ("gtreqqless", "⪌"),
    ("gtrless", "≷"),
    ("gtrsim", "≳"),
    ("gvertneqq", "≩︀"),
    ("gvnE", "≩︀"),
    ("hArr", "⇔"),
    ("hairsp", " "),
    ("half", "½"),
    ("hamilt", "ℋ"),
    ("hardcy", "ъ"),
    ("harr", "↔"),
    ("harrcir", "⥈"),
    ("harrw", "↭"),
    ("hbar", "ℏ"),
    ("hcirc", "ĥ"),
    ("hearts", "♥"),
    ("heartsuit", "♥"),
    ("hellip", "…"),
    ("hercon", "⊹"),
    ("hfr", "𝔥"),
    ("hksearow", "⤥"),
    ("hkswarow", "⤦"),
    ("hoarr", "⇿"),
    ("homtht", "∻"),
    ("hookleftarrow", "↩"),
    ("hookrightarrow", "↪"),
    ("hopf", "𝕙"),
    ("horbar", "―"),
    ("hscr", "𝒽"),
    ("hslash", "ℏ"),
    ("hstrok", "ħ"),
    ("hybull", "⁃"),
    ("hyphen", "‐"),
    ("iacute", "í"),
    ("ic", "⁣"),
    ("icirc", "î"),
    ("icy", "и"),
    ("iecy", "е"),
    ("iexcl", "¡"),
    ("iff", "⇔"),
    ("ifr", "𝔦"),
    ("igrave", "ì"),
    ("ii", "ⅈ"),
    ("iiiint", "⨌"),
    ("iiint", "∭"),
    ("iinfin", "⧜"),
    ("iiota", "℩"),
    ("ijlig", "ĳ"),
    ("imacr", "ī"),
    ("image", "ℑ"),
    ("imagline", "ℐ"),
    ("imagpart", "ℑ"),
    ("imath", "ı"),
    ("imof", "⊷"),
    ("imped", "Ƶ"),
    ("in", "∈"),
    ("incare", "℅"),
    ("infin", "∞"),
    ("infintie", "⧝"),
    ("inodot", "ı"),
    ("int", "∫"),
    ("intcal", "⊺"),
    ("integers", "ℤ"),
    ("intercal", "⊺"),
    ("intlarhk", "⨗"),
    ("intprod", "⨼"),
    ("iocy", "ё"),
    ("iogon", "į"),
    ("iopf", "𝕚"),
    ("iota", "ι"),
    ("iprod", "⨼"),
    ("iquest", "¿"),
    ("iscr", "𝒾"),
    ("isin", "∈"),
    ("isinE", "⋹"),
    ("isindot", "⋵"),
    ("isins", "⋴"),
    ("isinsv", "⋳"),
    ("isinv", "∈"),
    ("it", "⁢"),
    ("itilde", "ĩ"),
    ("iukcy", "і"),
    ("iuml", "ï"),
    ("jcirc", "ĵ"),
    ("jcy", "й"),
    ("jfr", "𝔧"),
    ("jmath", "ȷ"),
    ("jopf", "𝕛"),
    ("jscr", "𝒿"),
    ("jsercy", "ј"),
    ("jukcy", "є"),
    ("kappa", "κ"),
    ("kappav", "ϰ"),
    ("kcedil", "ķ"),
    ("kcy", "к"),
    ("kfr", "𝔨"),
    ("kgreen", "ĸ"),
    ("khcy", "х"),
    ("kjcy", "ќ"),
    ("kopf", "𝕜"),
    ("kscr", "𝓀"),
    ("lAarr", "⇚"),
    ("lArr", "⇐"),
    ("lAtail", "⤛"),
    ("lBarr", "⤎"),
    ("lE", "≦"),
    ("lEg", "⪋"),
    ("lHar", "⥢"),
    ("lacute", "ĺ"),
    ("laemptyv", "⦴"),
    ("lagran", "ℒ"),
    ("lambda", "λ"),
    ("lang", "⟨"),
    ("langd", "⦑"),
    ("langle", "⟨"),
    ("lap", "⪅"),
    ("laquo", "«"),
    ("larr", "←"),
    ("larrb", "⇤"),
    ("larrbfs", "⤟"),
    ("larrfs", "⤝"),
    ("larrhk", "↩"),
    ("larrlp", "↫"),
    ("larrpl", "⤹"),
    ("larrsim", "⥳"),
    ("larrtl", "↢"),
    ("lat", "⪫"),
    ("latail", "⤙"),
    ("late", "⪭"),
    ("lates", "⪭︀"),
    ("lbarr", "⤌"),
    ("lbbrk", "❲"),
    ("lbrace", "{"),
    ("lbrack", "["),
    ("lbrke", "⦋"),
    ("lbrksld", "⦏"),
    ("lbrkslu", "⦍"),
    ("lcaron", "ľ"),
    ("lcedil", "ļ"),
    ("lceil", "⌈"),
    ("lcub", "{"),
    ("lcy", "л"),
    ("ldca", "⤶"),
    ("ldquo", "“"),
    ("ldquor", "„"),
    ("ldrdhar", "⥧"),
    ("ldrushar", "⥋"),
    ("ldsh", "↲"),
    ("le", "≤"),
    ("leftarrow", "←"),
    ("leftarrowtail", "↢"),
    ("leftharpoondown", "↽"),
    ("leftharpoonup", "↼"),
    ("leftleftarrows", "⇇"),
    ("leftrightarrow", "↔"),
    ("leftrightarrows", "⇆"),
    ("leftrightharpoons", "⇋"),
    ("leftrightsquigarrow", "↭"),
    ("leftthreetimes", "⋋"),
    ("leg", "⋚"),
    ("leq", "≤"),
    ("leqq", "≦"),
    ("leqslant", "⩽"),
    ("les", "⩽"),
    ("lescc", "⪨"),
    ("lesdot", "⩿"),
    ("lesdoto", "⪁"),
    ("lesdotor", "⪃"),
    ("lesg", "⋚︀"),
    ("lesges", "⪓"),
    ("lessapprox", "⪅"),
    ("lessdot", "⋖"),
    ("lesseqgtr", "⋚"),
    ("lesseqqgtr", "⪋"),
    ("lessgtr", "≶"),
    ("lesssim", "≲"),
    ("lfisht", "⥼"),
    ("lfloor", "⌊"),
    ("lfr", "𝔩"),
    ("lg", "≶"),
    ("lgE", "⪑"),
    ("lhard", "↽"),
    ("lharu", "↼"),
    ("lharul", "⥪"),
    ("lhblk", "▄"),
    ("ljcy", "љ"),
    ("ll", "≪"),
    ("llarr", "⇇"),
    ("llcorner", "⌞"),
    ("llhard", "⥫"),
    ("lltri", "◺"),
    ("lmidot", "ŀ"),
    ("lmoust", "⎰"),
    ("lmoustache", "⎰"),
    ("lnE", "≨"),
    ("lnap", "⪉"),
    ("lnapprox", "⪉"),
    ("lne", "⪇"),
    ("lneq", "⪇"),
    ("lneqq", "≨"),
    ("lnsim", "⋦"),
    ("loang", "⟬"),
    ("loarr", "⇽"),
    ("lobrk", "⟦"),
    ("longleftarrow", "⟵"),
    ("longleftrightarrow", "⟷"),
    ("longmapsto", "⟼"),
    ("longrightarrow", "⟶"),
    ("looparrowleft", "↫"),
    ("looparrowright", "↬"),
    ("lopar", "⦅"),
    ("lopf", "𝕝"),
    ("loplus", "⨭"),
    ("lotimes", "⨴"),
    ("lowast", "∗"),
    ("lowbar", "_"),
    ("loz", "◊"),
    ("lozenge", "◊"),
    ("lozf", "⧫"),
    ("lpar", "("),
    ("lparlt", "⦓"),
    ("lrarr", "⇆"),
    ("lrcorner", "⌟"),
    ("lrhar", "⇋"),
    ("lrhard", "⥭"),
    ("lrm", "‎"),
    ("lrtri", "⊿"),
    ("lsaquo", "‹"),
    ("lscr", "𝓁"),
    ("lsh", "↰"),
    ("lsim", "≲"),
    ("lsime", "⪍"),
    ("lsimg", "⪏"),
    ("lsqb", "["),
    ("lsquo", "‘"),
    ("lsquor", "‚"),
    ("lstrok", "ł"),
    ("lt", "<"),
    ("ltcc", "⪦"),
    ("ltcir", "⩹"),
    ("ltdot", "⋖"),
    ("lthree", "⋋"),
    ("ltimes", "⋉"),
    ("ltlarr", "⥶"),
    ("ltquest", "⩻"),
    ("ltrPar", "⦖"),
    ("ltri", "◃"),
    ("ltrie", "⊴"),
    ("ltrif", "◂"),
    ("lurdshar", "⥊"),
    ("luruhar", "⥦"),
    ("lvertneqq", "≨︀"),
    ("lvnE", "≨︀"),
    ("mDDot", "∺"),
    ("macr", "¯"),
    ("male", "♂"),
    ("malt", "✠"),
    ("maltese", "✠"),
    ("map", "↦"),
    ("mapsto", "↦"),
    ("mapstodown", "↧"),
    ("mapstoleft", "↤"),
    ("mapstoup", "↥"),
    ("marker", "▮"),
    ("mcomma", "⨩"),
    ("mcy", "м"),
    ("mdash", "—"),
    ("measuredangle", "∡"),
    ("mfr", "𝔪"),
    ("mho", "℧"),
    ("micro", "µ"),
    ("mid", "∣"),
    ("midast", "*"),
    ("midcir", "⫰"),
    ("middot", "·"),
    ("minus", "−"),
    ("minusb", "⊟"),
    ("minusd", "∸"),
    ("minusdu", "⨪"),
    ("mlcp", "⫛"),
    ("mldr", "…"),
    ("mnplus", "∓"),
    ("models", "⊧"),
    ("mopf", "𝕞"),
    ("mp", "∓"),
    ("mscr", "𝓂"),
    ("mstpos", "∾"),
    ("mu", "μ"),
    ("multimap", "⊸"),
    ("mumap", "⊸"),
    ("nGg", "⋙̸"),
    ("nGt", "≫⃒"),
    ("nGtv", "≫̸"),
    ("nLeftarrow", "⇍"),
    ("nLeftrightarrow", "⇎"),
    ("nLl", "⋘̸"),
    ("nLt", "≪⃒"),
    ("nLtv", "≪̸"),
    ("nRightarrow", "⇏"),
    ("nVDash", "⊯"),
    ("nVdash", "⊮"),
    ("nabla", "∇"),
    ("nacute", "ń"),
    ("nang", "∠⃒"),
    ("nap", "≉"),
    ("napE", "⩰̸"),
    ("napid", "≋̸"),
    ("napos", "ŉ"),
    ("napprox", "≉"),
    ("natur", "♮"),
    ("natural", "♮"),
    ("naturals", "ℕ"),
    ("nbsp", " "),
    ("nbump", "≎̸"),
    ("nbumpe", "≏̸"),
    ("ncap", "⩃"),
    ("ncaron", "ň"),
    ("ncedil", "ņ"),
    ("ncong", "≇"),
    ("ncongdot", "⩭̸"),
    ("ncup", "⩂"),
    ("ncy", "н"),
    ("ndash", "–"),
    ("ne", "≠"),
    ("neArr", "⇗"),
    ("nearhk", "⤤"),
    ("nearr", "↗"),
    ("nearrow", "↗"),
    ("nedot", "≐̸"),
    ("nequiv", "≢"),
    ("nesear", "⤨"),
    ("nesim", "≂̸"),
    ("nexist", "∄"),
    ("nexists", "∄"),
    ("nfr", "𝔫"),
    ("ngE", "≧̸"),
    ("nge", "≱"),
    ("ngeq", "≱"),
    ("ngeqq", "≧̸"),
    ("ngeqslant", "⩾̸"),
    ("nges", "⩾̸"),
    ("ngsim", "≵"),
    ("ngt", "≯"),
    ("ngtr", "≯"),
    ("nhArr", "⇎"),
    ("nharr", "↮"),
    ("nhpar", "⫲"),
    ("ni", "∋"),
    ("nis", "⋼"),
    ("nisd", "⋺"),
    ("niv", "∋"),
    ("njcy", "њ"),
    ("nlArr", "⇍"),
    ("nlE", "≦̸"),
    ("nlarr", "↚"),
    ("nldr", "‥"),
    ("nle", "≰"),
    ("nleftarrow", "↚"),
    ("nleftrightarrow", "↮"),
    ("nleq", "≰"),
    ("nleqq", "≦̸"),
    ("nleqslant", "⩽̸"),
    ("nles", "⩽̸"),
    ("nless", "≮"),
    ("nlsim", "≴"),
    ("nlt", "≮"),
    ("nltri", "⋪"),
    ("nltrie", "⋬"),
    ("nmid", "∤"),
    ("nopf", "𝕟"),
    ("not", "¬"),
    ("notin", "∉"),
    ("notinE", "⋹̸"),
    ("notindot", "⋵̸"),
    ("notinva", "∉"),
    ("notinvb", "⋷"),
    ("notinvc", "⋶"),
    ("notni", "∌"),
    ("notniva", "∌"),
    ("notnivb", "⋾"),
    ("notnivc", "⋽"),
    ("npar", "∦"),
    ("nparallel", "∦"),
    ("nparsl", "⫽⃥"),
    ("npart", "∂̸"),
    ("npolint", "⨔"),
    ("npr", "⊀"),
    ("nprcue", "⋠"),
    ("npre", "⪯̸"),
    ("nprec", "⊀"),
    ("npreceq", "⪯̸"),
    ("nrArr", "⇏"),
    ("nrarr", "↛"),
    ("nrarrc", "⤳̸"),
    ("nrarrw", "↝̸"),
    ("nrightarrow", "↛"),
    ("nrtri", "⋫"),
    ("nrtrie", "⋭"),
    ("nsc", "⊁"),
    ("nsccue", "⋡"),
    ("nsce", "⪰̸"),
    ("nscr", "𝓃"),
    ("nshortmid", "∤"),
    ("nshortparallel", "∦"),
    ("nsim", "≁"),
    ("nsime", "≄"),
    ("nsimeq", "≄"),
    ("nsmid", "∤"),
    ("nspar", "∦"),
    ("nsqsube", "⋢"),
    ("nsqsupe", "⋣"),
    ("nsub", "⊄"),
    ("nsubE", "⫅̸"),
    ("nsube", "⊈"),
    ("nsubset", "⊂⃒"),
    ("nsubseteq", "⊈"),
    ("nsubseteqq", "⫅̸"),
    ("nsucc", "⊁"),
    ("nsucceq", "⪰̸"),
    ("nsup", "⊅"),
    ("nsupE", "⫆̸"),
    ("nsupe", "⊉"),
    ("nsupset", "⊃⃒"),
    ("nsupseteq", "⊉"),
    ("nsupseteqq", "⫆̸"),
    ("ntgl", "≹"),
    ("ntilde", "ñ"),
    ("ntlg", "≸"),
    ("ntriangleleft", "⋪"),
    ("ntrianglelefteq", "⋬"),
    ("ntriangleright", "⋫"),
    ("ntrianglerighteq", "⋭"),
    ("nu", "ν"),
    ("num", "#"),
    ("numero", "№"),
    ("numsp", " "),
    ("nvDash", "⊭"),
    ("nvHarr", "⤄"),
    ("nvap", "≍⃒"),
    ("nvdash", "⊬"),
    ("nvge", "≥⃒"),
    ("nvgt", ">⃒"),
    ("nvinfin", "⧞"),
    ("nvlArr", "⤂"),
    ("nvle", "≤⃒"),
    ("nvlt", "<⃒"),
    ("nvltrie", "⊴⃒"),
    ("nvrArr", "⤃"),
    ("nvrtrie", "⊵⃒"),
    ("nvsim", "∼⃒"),
    ("nwArr", "⇖"),
    ("nwarhk", "⤣"),
    ("nwarr", "↖"),
    ("nwarrow", "↖"),
    ("nwnear", "⤧"),
    ("oS", "Ⓢ"),
    ("oacute", "ó"),
    ("oast", "⊛"),
    ("ocir", "⊚"),
    ("ocirc", "ô"),
    ("ocy", "о"),
    ("odash", "⊝"),
    ("odblac", "ő"),
    ("odiv", "⨸"),
    ("odot", "⊙"),
    ("odsold", "⦼"),
    ("oelig", "œ"),
    ("ofcir", "⦿"),
    ("ofr", "𝔬"),
    ("ogon", "˛"),
    ("ograve", "ò"),
    ("ogt", "⧁"),
    ("ohbar", "⦵"),
    ("ohm", "Ω"),
    ("oint", "∮"),
    ("olarr", "↺"),
    ("olcir", "⦾"),
    ("olcross", "⦻"),
    ("oline", "‾"),
    ("olt", "⧀"),
    ("omacr", "ō"),
    ("omega", "ω"),
    ("omicron", "ο"),
    ("omid", "⦶"),
    ("ominus", "⊖"),
    ("oopf", "𝕠"),
    ("opar", "⦷"),
    ("operp", "⦹"),
    ("oplus", "⊕"),
    ("or", "∨"),
    ("orarr", "↻"),
    ("ord", "⩝"),
    ("order", "ℴ"),
    ("orderof", "ℴ"),
    ("ordf", "ª"),
    ("ordm", "º"),
    ("origof", "⊶"),
    ("oror", "⩖"),
    ("orslope", "⩗"),
    ("orv", "⩛"),
    ("oscr", "ℴ"),
    ("oslash", "ø"),
    ("osol", "⊘"),
    ("otilde", "õ"),
    ("otimes", "⊗"),
    ("otimesas", "⨶"),
    ("ouml", "ö"),
    ("ovbar", "⌽"),
    ("par", "∥"),
    ("para", "¶"),
    ("parallel", "∥"),
    ("parsim", "⫳"),
    ("parsl", "⫽"),
    ("part", "∂"),
    ("pcy", "п"),
    ("percnt", "%"),
    ("period", "."),
    ("permil", "‰"),
    ("perp", "⊥"),
    ("pertenk", "‱"),
    ("pfr", "𝔭"),
    ("phi", "φ"),
    ("phiv", "ϕ"),
    ("phmmat", "ℳ"),
    ("phone", "☎"),
    ("pi", "π"),
    ("pitchfork", "⋔"),
    ("piv", "ϖ"),
    ("planck", "ℏ"),
    ("planckh", "ℎ"),
    ("plankv", "ℏ"),
    ("plus", "+"),
    ("plusacir", "⨣"),
    ("plusb", "⊞"),
    ("pluscir", "⨢"),
    ("plusdo", "∔"),
    ("plusdu", "⨥"),
    ("pluse", "⩲"),
    ("plusmn", "±"),
    ("plussim", "⨦"),
    ("plustwo", "⨧"),
    ("pm", "±"),
    ("pointint", "⨕"),
    ("popf", "𝕡"),
    ("pound", "£"),
    ("pr", "≺"),
    ("prE", "⪳"),
    ("prap", "⪷"),
    ("prcue", "≼"),
    ("pre", "⪯"),
    ("prec", "≺"),
    ("precapprox", "⪷"),
    ("preccurlyeq", "≼"),
    ("preceq", "⪯"),
    ("precnapprox", "⪹"),
    ("precneqq", "⪵"),
    ("precnsim", "⋨"),
    ("precsim", "≾"),
    ("prime", "′"),
    ("primes", "ℙ"),
    ("prnE", "⪵"),
    ("prnap", "⪹"),
    ("prnsim", "⋨"),
    ("prod", "∏"),
    ("profalar", "⌮"),
    ("profline", "⌒"),
    ("profsurf", "⌓"),
    ("prop", "∝"),
    ("propto", "∝"),
    ("prsim", "≾"),
    ("prurel", "⊰"),
    ("pscr", "𝓅"),
    ("psi", "ψ"),
    ("puncsp", " "),
    ("qfr", "𝔮"),
    ("qint", "⨌"),
    ("qopf", "𝕢"),
    ("qprime", "⁗"),
    ("qscr", "𝓆"),
    ("quaternions", "ℍ"),
    ("quatint", "⨖"),
    ("quest", "?"),
    ("questeq", "≟"),
    ("quot", "\""),
    ("rAarr", "⇛"),
    ("rArr", "⇒"),
    ("rAtail", "⤜"),
    ("rBarr", "⤏"),
    ("rHar", "⥤"),
    ("race", "∽̱"),
    ("racute", "ŕ"),
    ("radic", "√"),
    ("raemptyv", "⦳"),
    ("rang", "⟩"),
    ("rangd", "⦒"),
    ("range", "⦥"),
    ("rangle", "⟩"),
    ("raquo", "»"),
    ("rarr", "→"),
    ("rarrap", "⥵"),
    ("rarrb", "⇥"),
    ("rarrbfs", "⤠"),
    ("rarrc", "⤳"),
    ("rarrfs", "⤞"),
    ("rarrhk", "↪"),
    ("rarrlp", "↬"),
    ("rarrpl", "⥅"),
    ("rarrsim", "⥴"),
    ("rarrtl", "↣"),
    ("rarrw", "↝"),
    ("ratail", "⤚"),
    ("ratio", "∶"),
    ("rationals", "ℚ"),
    ("rbarr", "⤍"),
    ("rbbrk", "❳"),
    ("rbrace", "}"),
    ("rbrack", "]"),
    ("rbrke", "⦌"),
    ("rbrksld", "⦎"),
    ("rbrkslu", "⦐"),
    ("rcaron", "ř"),
    ("rcedil", "ŗ"),
    ("rceil", "⌉"),
    ("rcub", "}"),
    ("rcy", "р"),
    ("rdca", "⤷"),
    ("rdldhar", "⥩"),
    ("rdquo", "”"),
    ("rdquor", "”"),
    ("rdsh", "↳"),
    ("real", "ℜ"),
    ("realine", "ℛ"),
    ("realpart", "ℜ"),
    ("reals", "ℝ"),
    ("rect", "▭"),
    ("reg", "®"),
    ("rfisht", "⥽"),
    ("rfloor", "⌋"),
    ("rfr", "𝔯"),
    ("rhard", "⇁"),
    ("rharu", "⇀"),
    ("rharul", "⥬"),
    ("rho", "ρ"),
    ("rhov", "ϱ"),
    ("rightarrow", "→"),
    ("rightarrowtail", "↣"),
    ("rightharpoondown", "⇁"),
    ("rightharpoonup", "⇀"),
    ("rightleftarrows", "⇄"),
    ("rightleftharpoons", "⇌"),
    ("rightrightarrows", "⇉"),
    ("rightsquigarrow", "↝"),
    ("rightthreetimes", "⋌"),
    ("ring", "˚"),
    ("risingdotseq", "≓"),
    ("rlarr", "⇄"),
    ("rlhar", "⇌"),
    ("rlm", "‏"),
    ("rmoust", "⎱"),
    ("rmoustache", "⎱"),
    ("rnmid", "⫮"),
    ("roang", "⟭"),
    ("roarr", "⇾"),
    ("robrk", "⟧"),
    ("ropar", "⦆"),
    ("ropf", "𝕣"),
    ("roplus", "⨮"),
    ("rotimes", "⨵"),
    ("rpar", ")"),
    ("rpargt", "⦔"),
    ("rppolint", "⨒"),
    ("rrarr", "⇉"),
    ("rsaquo", "›"),
    ("rscr", "𝓇"),
    ("rsh", "↱"),
    ("rsqb", "]"),
    ("rsquo", "’"),
    ("rsquor", "’"),
    ("rthree", "⋌"),
    ("rtimes", "⋊"),
    ("rtri", "▹"),
    ("rtrie", "⊵"),
    ("rtrif", "▸"),
    ("rtriltri", "⧎"),
    ("ruluhar", "⥨"),
    ("rx", "℞"),
    ("sacute", "ś"),
    ("sbquo", "‚"),
    ("sc", "≻"),
    ("scE", "⪴"),
    ("scap", "⪸"),
    ("scaron", "š"),
    ("sccue", "≽"),
    ("sce", "⪰"),
    ("scedil", "ş"),
    ("scirc", "ŝ"),
    ("scnE", "⪶"),
    ("scnap", "⪺"),
    ("scnsim", "⋩"),
    ("scpolint", "⨓"),
    ("scsim", "≿"),
    ("scy", "с"),
    ("sdot", "⋅"),
    ("sdotb", "⊡"),
    ("sdote", "⩦"),
    ("seArr", "⇘"),
    ("searhk", "⤥"),
    ("searr", "↘"),
    ("searrow", "↘"),
    ("sect", "§"),
    ("semi", ";"),
    ("seswar", "⤩"),
    ("setminus", "∖"),
    ("setmn", "∖"),
    ("sext", "✶"),
    ("sfr", "𝔰"),
    ("sfrown", "⌢"),
    ("sharp", "♯"),
    ("shchcy", "щ"),
    ("shcy", "ш"),
    ("shortmid", "∣"),
    ("shortparallel", "∥"),
    ("shy", "\u{AD}"),
    ("sigma", "σ"),
    ("sigmaf", "ς"),
    ("sigmav", "ς"),
    ("sim", "∼"),
    ("simdot", "⩪"),
    ("sime", "≃"),
    ("simeq", "≃"),
    ("simg", "⪞"),
    ("simgE", "⪠"),
    ("siml", "⪝"),
    ("simlE", "⪟"),
    ("simne", "≆"),
    ("simplus", "⨤"),
    ("simrarr", "⥲"),
    ("slarr", "←"),
    ("smallsetminus", "∖"),
    ("smashp", "⨳"),
    ("smeparsl", "⧤"),
    ("smid", "∣"),
    ("smile", "⌣"),
    ("smt", "⪪"),
    ("smte", "⪬"),
    ("smtes", "⪬︀"),
    ("softcy", "ь"),
    ("sol", "/"),
    ("solb", "⧄"),
    ("solbar", "⌿"),
    ("sopf", "𝕤"),
    ("spades", "♠"),
    ("spadesuit", "♠"),
    ("spar", "∥"),
    ("sqcap", "⊓"),
    ("sqcaps", "⊓︀"),
    ("sqcup", "⊔"),
    ("sqcups", "⊔︀"),
    ("sqsub", "⊏"),
    ("sqsube", "⊑"),
    ("sqsubset", "⊏"),
    ("sqsubseteq", "⊑"),
    ("sqsup", "⊐"),
    ("sqsupe", "⊒"),
    ("sqsupset", "⊐"),
    ("sqsupseteq", "⊒"),
    ("squ", "□"),
    ("square", "□"),
    ("squarf", "▪"),
    ("squf", "▪"),
    ("srarr", "→"),
    ("sscr", "𝓈"),
    ("ssetmn", "∖"),
    ("ssmile", "⌣"),
    ("sstarf", "⋆"),
    ("star", "☆"),
    ("starf", "★"),
    ("straightepsilon", "ϵ"),
    ("straightphi", "ϕ"),
    ("strns", "¯"),
    ("sub", "⊂"),
    ("subE", "⫅"),
    ("subdot", "⪽"),
    ("sube", "⊆"),
    ("subedot", "⫃"),
    ("submult", "⫁"),
    ("subnE", "⫋"),
    ("subne", "⊊"),
    ("subplus", "⪿"),
    ("subrarr", "⥹"),
    ("subset", "⊂"),
    ("subseteq", "⊆"),
    ("subseteqq", "⫅"),
    ("subsetneq", "⊊"),
    ("subsetneqq", "⫋"),
    ("subsim", "⫇"),
    ("subsub", "⫕"),
    ("subsup", "⫓"),
    ("succ", "≻"),
    ("succapprox", "⪸"),
    ("succcurlyeq", "≽"),
    ("succeq", "⪰"),
    ("succnapprox", "⪺"),
    ("succneqq", "⪶"),
    ("succnsim", "⋩"),
    ("succsim", "≿"),
    ("sum", "∑"),
    ("sung", "♪"),
    ("sup1", "¹"),
    ("sup2", "²"),
    ("sup3", "³"),
    ("sup", "⊃"),
    ("supE", "⫆"),
    ("supdot", "⪾"),
    ("supdsub", "⫘"),
    ("supe", "⊇"),
    ("supedot", "⫄"),
    ("suphsol", "⟉"),
    ("suphsub", "⫗"),
    ("suplarr", "⥻"),
    ("supmult", "⫂"),
    ("supnE", "⫌"),
    ("supne", "⊋"),
    ("supplus", "⫀"),
    ("supset", "⊃"),
    ("supseteq", "⊇"),
    ("supseteqq", "⫆"),
    ("supsetneq", "⊋"),
    ("supsetneqq", "⫌"),
    ("supsim", "⫈"),
    ("supsub", "⫔"),
    ("supsup", "⫖"),
    ("swArr", "⇙"),
    ("swarhk", "⤦"),
    ("swarr", "↙"),
    ("swarrow", "↙"),
    ("swnwar", "⤪"),
    ("szlig", "ß"),
    ("target", "⌖"),
    ("tau", "τ"),
    ("tbrk", "⎴"),
    ("tcaron", "ť"),
    ("tcedil", "ţ"),
    ("tcy", "т"),
    ("tdot", "⃛"),
    ("telrec", "⌕"),
    ("tfr", "𝔱"),
    ("there4", "∴"),
    ("therefore", "∴"),
    ("theta", "θ"),
    ("thetasym", "ϑ"),
    ("thetav", "ϑ"),
    ("thickapprox", "≈"),
    ("thicksim", "∼"),
    ("thinsp", " "),
    ("thkap", "≈"),
    ("thksim", "∼"),
    ("thorn", "þ"),
    ("tilde", "˜"),
    ("times", "×"),
    ("timesb", "⊠"),
    ("timesbar", "⨱"),
    ("timesd", "⨰"),
    ("tint", "∭"),
    ("toea", "⤨"),
    ("top", "⊤"),
    ("topbot", "⌶"),
    ("topcir", "⫱"),
    ("topf", "𝕥"),
    ("topfork", "⫚"),
    ("tosa", "⤩"),
    ("tprime", "‴"),
    ("trade", "™"),
    ("triangle", "▵"),
    ("triangledown", "▿"),
    ("triangleleft", "◃"),
    ("trianglelefteq", "⊴"),
    ("triangleq", "≜"),
    ("triangleright", "▹"),
    ("trianglerighteq", "⊵"),
    ("tridot", "◬"),
    ("trie", "≜"),
    ("triminus", "⨺"),
    ("triplus", "⨹"),
    ("trisb", "⧍"),
    ("tritime", "⨻"),
    ("trpezium", "⏢"),
    ("tscr", "𝓉"),
    ("tscy", "ц"),
    ("tshcy", "ћ"),
    ("tstrok", "ŧ"),
    ("twixt", "≬"),
    ("twoheadleftarrow", "↞"),
    ("twoheadrightarrow", "↠"),
    ("uArr", "⇑"),
    ("uHar", "⥣"),
    ("uacute", "ú"),
    ("uarr", "↑"),
    ("ubrcy", "ў"),
    ("ubreve", "ŭ"),
    ("ucirc", "û"),
    ("ucy", "у"),
    ("udarr", "⇅"),
    ("udblac", "ű"),
    ("udhar", "⥮"),
    ("ufisht", "⥾"),
    ("ufr", "𝔲"),
    ("ugrave", "ù"),
    ("uharl", "↿"),
    ("uharr", "↾"),
    ("uhblk", "▀"),
    ("ulcorn", "⌜"),
    ("ulcorner", "⌜"),
    ("ulcrop", "⌏"),
    ("ultri", "◸"),
    ("umacr", "ū"),
    ("uml", "¨"),
    ("uogon", "ų"),
    ("uopf", "𝕦"),
    ("uparrow", "↑"),
    ("updownarrow", "↕"),
    ("upharpoonleft", "↿"),
    ("upharpoonright", "↾"),
    ("uplus", "⊎"),
    ("upsi", "υ"),
    ("upsih", "ϒ"),
    ("upsilon", "υ"),
    ("upuparrows", "⇈"),
    ("urcorn", "⌝"),
    ("urcorner", "⌝"),
    ("urcrop", "⌎"),
    ("uring", "ů"),
    ("urtri", "◹"),
    ("uscr", "𝓊"),
    ("utdot", "⋰"),
    ("utilde", "ũ"),
    ("utri", "▵"),
    ("utrif", "▴"),
    ("uuarr", "⇈"),
    ("uuml", "ü"),
    ("uwangle", "⦧"),
    ("vArr", "⇕"),
    ("vBar", "⫨"),
    ("vBarv", "⫩"),
    ("vDash", "⊨"),
    ("vangrt", "⦜"),
    ("varepsilon", "ϵ"),
    ("varkappa", "ϰ"),
    ("varnothing", "∅"),
    ("varphi", "ϕ"),
    ("varpi", "ϖ"),
    ("varpropto", "∝"),
    ("varr", "↕"),
    ("varrho", "ϱ"),
    ("varsigma", "ς"),
    ("varsubsetneq", "⊊︀"),
    ("varsubsetneqq", "⫋︀"),
    ("varsupsetneq", "⊋︀"),
    ("varsupsetneqq", "⫌︀"),
    ("vartheta", "ϑ"),
    ("vartriangleleft", "⊲"),
    ("vartriangleright", "⊳"),
    ("vcy", "в"),
    ("vdash", "⊢"),
    ("vee", "∨"),
    ("veebar", "⊻"),
    ("veeeq", "≚"),
    ("vellip", "⋮"),
    ("verbar", "|"),
    ("vert", "|"),
    ("vfr", "𝔳"),
    ("vltri", "⊲"),
    ("vnsub", "⊂⃒"),
    ("vnsup", "⊃⃒"),
    ("vopf", "𝕧"),
    ("vprop", "∝"),
    ("vrtri", "⊳"),
    ("vscr", "𝓋"),
    ("vsubnE", "⫋︀"),
    ("vsubne", "⊊︀"),
    ("vsupnE", "⫌︀"),
    ("vsupne", "⊋︀"),
    ("vzigzag", "⦚"),
    ("wcirc", "ŵ"),
    ("wedbar", "⩟"),
    ("wedge", "∧"),
    ("wedgeq", "≙"),
    ("weierp", "℘"),
    ("wfr", "𝔴"),
    ("wopf", "𝕨"),
    ("wp", "℘"),
    ("wr", "≀"),
    ("wreath", "≀"),
    ("wscr", "𝓌"),
    ("xcap", "⋂"),
    ("xcirc", "◯"),
    ("xcup", "⋃"),
    ("xdtri", "▽"),
    ("xfr", "𝔵"),
    ("xhArr", "⟺"),
    ("xharr", "⟷"),
    ("xi", "ξ"),
    ("xlArr", "⟸"),
    ("xlarr", "⟵"),
    ("xmap", "⟼"),
    ("xnis", "⋻"),
    ("xodot", "⨀"),
    ("xopf", "𝕩"),
    ("xoplus", "⨁"),
    ("xotime", "⨂"),
    ("xrArr", "⟹"),
    ("xrarr", "⟶"),
    ("xscr", "𝓍"),
    ("xsqcup", "⨆"),
    ("xuplus", "⨄"),
    ("xutri", "△"),
    ("xvee", "⋁"),
    ("xwedge", "⋀"),
    ("yacute", "ý"),
    ("yacy", "я"),
    ("ycirc", "ŷ"),
    ("ycy", "ы"),
    ("yen", "¥"),
    ("yfr", "𝔶"),
    ("yicy", "ї"),
    ("yopf", "𝕪"),
    ("yscr", "𝓎"),
    ("yucy", "ю"),
    ("yuml", "ÿ"),
    ("zacute", "ź"),
    ("zcaron", "ž"),
    ("zcy", "з"),
    ("zdot", "ż"),
    ("zeetrf", "ℨ"),
    ("zeta", "ζ"),
    ("zfr", "𝔷"),
    ("zhcy", "ж"),
    ("zigrarr", "⇝"),
    ("zopf", "𝕫"),
    ("zscr", "𝓏"),
    ("zwj", "‍"),
    ("zwnj", "‌"),
];

// Important: please touch the below lists as few times as possible to keep Git small.

/// List of names and values that form named character reference in HTML 4.
///
/// This list is normally not used in markdown, but it is used in MDX, because
/// in JSX attribute values, only the old HTML 4 character references are
/// supported.
///
/// This list is sensitive to casing.
///
/// ## References
///
/// * [*§ 1.5.2 HTML Character References* in `JSX`](https://facebook.github.io/jsx/#sec-HTMLCharacterReference)
pub const CHARACTER_REFERENCES_HTML_4: [(&str, &str); 252] = [
    ("AElig", "Æ"),
    ("Aacute", "Á"),
    ("Acirc", "Â"),
    ("Agrave", "À"),
    ("Alpha", "Α"),
    ("Aring", "Å"),
    ("Atilde", "Ã"),
    ("Auml", "Ä"),
    ("Beta", "Β"),
    ("Ccedil", "Ç"),
    ("Chi", "Χ"),
    ("Dagger", "‡"),
    ("Delta", "Δ"),
    ("ETH", "Ð"),
    ("Eacute", "É"),
    ("Ecirc", "Ê"),
    ("Egrave", "È"),
    ("Epsilon", "Ε"),
    ("Eta", "Η"),
    ("Euml", "Ë"),
    ("Gamma", "Γ"),
    ("Iacute", "Í"),
    ("Icirc", "Î"),
    ("Igrave", "Ì"),
    ("Iota", "Ι"),
    ("Iuml", "Ï"),
    ("Kappa", "Κ"),
    ("Lambda", "Λ"),
    ("Mu", "Μ"),
    ("Ntilde", "Ñ"),
    ("Nu", "Ν"),
    ("OElig", "Œ"),
    ("Oacute", "Ó"),
    ("Ocirc", "Ô"),
    ("Ograve", "Ò"),
    ("Omega", "Ω"),
    ("Omicron", "Ο"),
    ("Oslash", "Ø"),
    ("Otilde", "Õ"),
    ("Ouml", "Ö"),
    ("Phi", "Φ"),
    ("Pi", "Π"),
    ("Prime", "″"),
    ("Psi", "Ψ"),
    ("Rho", "Ρ"),
    ("Scaron", "Š"),
    ("Sigma", "Σ"),
    ("THORN", "Þ"),
    ("Tau", "Τ"),
    ("Theta", "Θ"),
    ("Uacute", "Ú"),
    ("Ucirc", "Û"),
    ("Ugrave", "Ù"),
    ("Upsilon", "Υ"),
    ("Uuml", "Ü"),
    ("Xi", "Ξ"),
    ("Yacute", "Ý"),
    ("Yuml", "Ÿ"),
    ("Zeta", "Ζ"),
    ("aacute", "á"),
    ("acirc", "â"),
    ("acute", "´"),
    ("aelig", "æ"),
    ("agrave", "à"),
    ("alefsym", "ℵ"),
    ("alpha", "α"),
    ("amp", "&"),
    ("and", "∧"),
    ("ang", "∠"),
    ("aring", "å"),
    ("asymp", "≈"),
    ("atilde", "ã"),
    ("auml", "ä"),
    ("bdquo", "„"),
    ("beta", "β"),
    ("brvbar", "¦"),
    ("bull", "•"),
    ("cap", "∩"),
    ("ccedil", "ç"),
    ("cedil", "¸"),
    ("cent", "¢"),
    ("chi", "χ"),
    ("circ", "ˆ"),
    ("clubs", "♣"),
    ("cong", "≅"),
    ("copy", "©"),
    ("crarr", "↵"),
    ("cup", "∪"),
    ("curren", "¤"),
    ("dArr", "⇓"),
    ("dagger", "†"),
    ("darr", "↓"),
    ("deg", "°"),
    ("delta", "δ"),
    ("diams", "♦"),
    ("divide", "÷"),
    ("eacute", "é"),
    ("ecirc", "ê"),
    ("egrave", "è"),
    ("empty", "∅"),
    ("emsp", " "),
    ("ensp", " "),
    ("epsilon", "ε"),
    ("equiv", "≡"),
    ("eta", "η"),
    ("eth", "ð"),
    ("euml", "ë"),
    ("euro", "€"),
    ("exist", "∃"),
    ("fnof", "ƒ"),
    ("forall", "∀"),
    ("frac12", "½"),
    ("frac14", "¼"),
    ("frac34", "¾"),
    ("frasl", "⁄"),
    ("gamma", "γ"),
    ("ge", "≥"),
    ("gt", ">"),
    ("hArr", "⇔"),
    ("harr", "↔"),
    ("hearts", "♥"),
    ("hellip", "…"),
    ("iacute", "í"),
    ("icirc", "î"),
    ("iexcl", "¡"),
    ("igrave", "ì"),
    ("image", "ℑ"),
    ("infin", "∞"),
    ("int", "∫"),
    ("iota", "ι"),
    ("iquest", "¿"),
    ("isin", "∈"),
    ("iuml", "ï"),
    ("kappa", "κ"),
    ("lArr", "⇐"),
    ("lambda", "λ"),
    ("lang", "〈"),
    ("laquo", "«"),
    ("larr", "←"),
    ("lceil", "⌈"),
    ("ldquo", "“"),
    ("le", "≤"),
    ("lfloor", "⌊"),
    ("lowast", "∗"),
    ("loz", "◊"),
    ("lrm", "‎"),
    ("lsaquo", "‹"),
    ("lsquo", "‘"),
    ("lt", "<"),
    ("macr", "¯"),
    ("mdash", "—"),
    ("micro", "µ"),
    ("middot", "·"),
    ("minus", "−"),
    ("mu", "μ"),
    ("nabla", "∇"),
    ("nbsp", " "),
    ("ndash", "–"),
    ("ne", "≠"),
    ("ni", "∋"),
    ("not", "¬"),
    ("notin", "∉"),
    ("nsub", "⊄"),
    ("ntilde", "ñ"),
    ("nu", "ν"),
    ("oacute", "ó"),
    ("ocirc", "ô"),
    ("oelig", "œ"),
    ("ograve", "ò"),
    ("oline", "‾"),
    ("omega", "ω"),
    ("omicron", "ο"),
    ("oplus", "⊕"),
    ("or", "∨"),
    ("ordf", "ª"),
    ("ordm", "º"),
    ("oslash", "ø"),
    ("otilde", "õ"),
    ("otimes", "⊗"),
    ("ouml", "ö"),
    ("para", "¶"),
    ("part", "∂"),
    ("permil", "‰"),
    ("perp", "⊥"),
    ("phi", "φ"),
    ("pi", "π"),
    ("piv", "ϖ"),
    ("plusmn", "±"),
    ("pound", "£"),
    ("prime", "′"),
    ("prod", "∏"),
    ("prop", "∝"),
    ("psi", "ψ"),
    ("quot", "\""),
    ("rArr", "⇒"),
    ("radic", "√"),
    ("rang", "〉"),
    ("raquo", "»"),
    ("rarr", "→"),
    ("rceil", "⌉"),
    ("rdquo", "”"),
    ("real", "ℜ"),
    ("reg", "®"),
    ("rfloor", "⌋"),
    ("rho", "ρ"),
    ("rlm", "‏"),
    ("rsaquo", "›"),
    ("rsquo", "’"),
    ("sbquo", "‚"),
    ("scaron", "š"),
    ("sdot", "⋅"),
    ("sect", "§"),
    ("shy", "\u{AD}"),
    ("sigma", "σ"),
    ("sigmaf", "ς"),
    ("sim", "∼"),
    ("spades", "♠"),
    ("sub", "⊂"),
    ("sube", "⊆"),
    ("sum", "∑"),
    ("sup", "⊃"),
    ("sup1", "¹"),
    ("sup2", "²"),
    ("sup3", "³"),
    ("supe", "⊇"),
    ("szlig", "ß"),
    ("tau", "τ"),
    ("there4", "∴"),
    ("theta", "θ"),
    ("thetasym", "ϑ"),
    ("thinsp", " "),
    ("thorn", "þ"),
    ("tilde", "˜"),
    ("times", "×"),
    ("trade", "™"),
    ("uArr", "⇑"),
    ("uacute", "ú"),
    ("uarr", "↑"),
    ("ucirc", "û"),
    ("ugrave", "ù"),
    ("uml", "¨"),
    ("upsih", "ϒ"),
    ("upsilon", "υ"),
    ("uuml", "ü"),
    ("weierp", "℘"),
    ("xi", "ξ"),
    ("yacute", "ý"),
    ("yen", "¥"),
    ("yuml", "ÿ"),
    ("zeta", "ζ"),
    ("zwj", "‍"),
    ("zwnj", "‌"),
];

#[cfg(test)]
mod tests {
    use super::*;
    use alloc::format;

    #[test]
    fn constants() {
        assert_eq!(
            CHARACTER_REFERENCE_DECIMAL_SIZE_MAX,
            format!("{}", 0x0010_ffff).len(),
            "`CHARACTER_REFERENCE_DECIMAL_SIZE_MAX`"
        );

        assert_eq!(
            CHARACTER_REFERENCE_HEXADECIMAL_SIZE_MAX,
            format!("{:x}", 0x0010_ffff).len(),
            "`CHARACTER_REFERENCE_HEXADECIMAL_SIZE_MAX`"
        );

        assert_eq!(
            CHARACTER_REFERENCE_NAMED_SIZE_MAX,
            longest(&CHARACTER_REFERENCES.map(|d| d.0)).unwrap().len(),
            "`CHARACTER_REFERENCE_NAMED_SIZE_MAX`"
        );

        assert_eq!(
            GFM_HTML_TAGFILTER_SIZE_MAX,
            longest(&GFM_HTML_TAGFILTER_NAMES).unwrap().len(),
            "`GFM_HTML_TAGFILTER_SIZE_MAX`"
        );

        assert_eq!(
            HTML_RAW_SIZE_MAX,
            longest(&HTML_RAW_NAMES).unwrap().len(),
            "`HTML_RAW_SIZE_MAX`"
        );
    }

    fn longest<'a>(list: &[&'a str]) -> Option<&'a str> {
        let mut max = 0;
        let mut result = None;
        for name in list {
            let len = name.len();
            if len > max {
                max = len;
                result = Some(*name);
            }
        }

        result
    }
}
