# Copyright (c) 2009-2012 testtools developers. See LICENSE for details.

"""ContentType - a MIME Content Type."""


class ContentType:
    """A content type from http://www.iana.org/assignments/media-types/

    :ivar type: The primary type, e.g. "text" or "application"
    :ivar subtype: The subtype, e.g. "plain" or "octet-stream"
    :ivar parameters: A dict of additional parameters specific to the
        content type.
    """

    def __init__(
        self, primary_type: str, sub_type: str, parameters: dict[str, str] | None = None
    ) -> None:
        """Create a ContentType."""
        if None in (primary_type, sub_type):
            raise ValueError(f"None not permitted in {primary_type!r}, {sub_type!r}")
        self.type = primary_type
        self.subtype = sub_type
        self.parameters = parameters or {}

    def __eq__(self, other: object) -> bool:
        if type(other) is not ContentType:
            return False
        return self.__dict__ == other.__dict__

    def __repr__(self) -> str:
        if self.parameters:
            params = "; "
            params += "; ".join(
                sorted(f'{k}="{v}"' for k, v in self.parameters.items())
            )
        else:
            params = ""
        return f"{self.type}/{self.subtype}{params}"


JSON = ContentType("application", "json")

UTF8_TEXT = ContentType("text", "plain", {"charset": "utf8"})
