;; Pipeline descriptions of Andes NDS32 cpu for GNU compiler
;; Copyright (C) 2012-2026 Free Software Foundation, Inc.
;; Contributed by Andes Technology Corporation.
;;
;; This file is part of GCC.
;;
;; GCC is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published
;; by the Free Software Foundation; either version 3, or (at your
;; option) any later version.
;;
;; GCC is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
;; or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
;; License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GCC; see the file COPYING3.  If not see
;; <http://www.gnu.org/licenses/>.


;; ------------------------------------------------------------------------
;; Define N13 pipeline settings.
;; ------------------------------------------------------------------------

(define_automaton "nds32_n13_machine")

;; ------------------------------------------------------------------------
;; Pipeline Stages
;; ------------------------------------------------------------------------
;; F1 - Instruction Fetch First
;;   Instruction Tag/Data Arrays
;;   ITLB Address Translation
;;   Branch Target Buffer Prediction
;; F2 - Instruction Fetch Second
;;   Instruction Cache Hit Detection
;;   Cache Way Selection
;;   Inustruction Alignment
;; I1 - Instruction Issue First / Instruction Decode
;;   Instruction Cache Replay Triggering
;;   32/16-Bit Instruction Decode
;;   Return Address Stack Prediction
;; I2 - Instruction Issue Second / Register File Access
;;   Instruction Issue Logic
;;   Register File Access
;; E1 - Instruction Execute First / Address Generation / MAC First
;;   Data Access Address generation
;;   Multiply Operation
;; E2 - Instruction Execute Second / Data Access First / MAC Second /
;;      ALU Execute
;;   Skewed ALU
;;   Branch/Jump/Return Resolution
;;   Data Tag/Data arrays
;;   DTLB address translation
;;   Accumulation Operation
;; E3 - Instruction Execute Third / Data Access Second
;;   Data Cache Hit Detection
;;   Cache Way Selection
;;   Data Alignment
;; E4 - Instruction Execute Fourth / Write Back
;;   Interruption Resolution
;;   Instruction Retire
;;   Register File Write Back

(define_cpu_unit "n13_i1" "nds32_n13_machine")
(define_cpu_unit "n13_i2" "nds32_n13_machine")
(define_cpu_unit "n13_e1" "nds32_n13_machine")
(define_cpu_unit "n13_e2" "nds32_n13_machine")
(define_cpu_unit "n13_e3" "nds32_n13_machine")
(define_cpu_unit "n13_e4" "nds32_n13_machine")

(define_insn_reservation "nds_n13_unknown" 1
  (and (eq_attr "type" "unknown")
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1, n13_e2, n13_e3, n13_e4")

(define_insn_reservation "nds_n13_misc" 1
  (and (eq_attr "type" "misc")
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1, n13_e2, n13_e3, n13_e4")

(define_insn_reservation "nds_n13_mmu" 1
  (and (eq_attr "type" "mmu")
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1, n13_e2, n13_e3, n13_e4")

(define_insn_reservation "nds_n13_alu" 1
  (and (eq_attr "type" "alu")
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1, n13_e2, n13_e3, n13_e4")

(define_insn_reservation "nds_n13_alu_shift" 1
  (and (eq_attr "type" "alu_shift")
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i2+n13_e1, n13_e1+n13_e2, n13_e2+n13_e3, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_pbsad" 1
  (and (eq_attr "type" "pbsad")
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1, n13_e2*2, n13_e3, n13_e4")

(define_insn_reservation "nds_n13_pbsada" 1
  (and (eq_attr "type" "pbsada")
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1, n13_e2*3, n13_e3, n13_e4")

(define_insn_reservation "nds_n13_load" 1
  (and (match_test "nds32::load_single_p (insn)")
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1, n13_e2, n13_e3, n13_e4")

(define_insn_reservation "nds_n13_store" 1
  (and (match_test "nds32::store_single_p (insn)")
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1, n13_e2, n13_e3, n13_e4")

(define_insn_reservation "nds_n13_load_multiple_1" 1
  (and (and (eq_attr "type" "load_multiple")
	    (eq_attr "combo" "1"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1, n13_e2, n13_e3, n13_e4")

(define_insn_reservation "nds_n13_load_multiple_2" 1
  (and (ior (and (eq_attr "type" "load_multiple")
		 (eq_attr "combo" "2"))
	    (match_test "nds32::load_double_p (insn)"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i2+n13_e1, n13_e1+n13_e2, n13_e2+n13_e3, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_load_multiple_3" 1
  (and (and (eq_attr "type" "load_multiple")
	    (eq_attr "combo" "3"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2+n13_i2, n13_i1+n13_i2+n13_e1, n13_i2+n13_e1+n13_e2, n13_e1+n13_e2+n13_e3, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_load_multiple_4" 1
  (and (and (eq_attr "type" "load_multiple")
	    (eq_attr "combo" "4"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i1+n13_i2+n13_e1, n13_i1+n13_i2+n13_e1+n13_e2, n13_i2+n13_e1+n13_e2+n13_e3, n13_e1+n13_e2+n13_e3+n13_e4, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_load_multiple_5" 1
  (and (and (eq_attr "type" "load_multiple")
	    (eq_attr "combo" "5"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i1+n13_i2+n13_e1, n13_i1+n13_i2+n13_e1+n13_e2, n13_i1+n13_i2+n13_e1+n13_e2+n13_e3, n13_i2+n13_e1+n13_e2+n13_e3+n13_e4, n13_e1+n13_e2+n13_e3+n13_e4, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_load_multiple_6" 1
  (and (and (eq_attr "type" "load_multiple")
	    (eq_attr "combo" "6"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i1+n13_i2+n13_e1, n13_i1+n13_i2+n13_e1+n13_e2, n13_i1+n13_i2+n13_e1+n13_e2+n13_e3, n13_i1+n13_i2+n13_e1+n13_e2+n13_e3+n13_e4, n13_i2+n13_e1+n13_e2+n13_e3+n13_e4, n13_e1+n13_e2+n13_e3+n13_e4, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_load_multiple_7" 1
  (and (and (eq_attr "type" "load_multiple")
	    (eq_attr "combo" "7"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i1+n13_i2+n13_e1, n13_i1+n13_i2+n13_e1+n13_e2, n13_i1+n13_i2+n13_e1+n13_e2+n13_e3, (n13_i1+n13_i2+n13_e1+n13_e2+n13_e3+n13_e4)*2, n13_i2+n13_e1+n13_e2+n13_e3+n13_e4, n13_e1+n13_e2+n13_e3+n13_e4, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_load_multiple_8" 1
  (and (and (eq_attr "type" "load_multiple")
	    (eq_attr "combo" "8"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i1+n13_i2+n13_e1, n13_i1+n13_i2+n13_e1+n13_e2, n13_i1+n13_i2+n13_e1+n13_e2+n13_e3, (n13_i1+n13_i2+n13_e1+n13_e2+n13_e3+n13_e4)*3, n13_i2+n13_e1+n13_e2+n13_e3+n13_e4, n13_e1+n13_e2+n13_e3+n13_e4, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_load_multiple_12" 1
  (and (and (eq_attr "type" "load_multiple")
	    (eq_attr "combo" "12"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i1+n13_i2+n13_e1, n13_i1+n13_i2+n13_e1+n13_e2, n13_i1+n13_i2+n13_e1+n13_e2+n13_e3, (n13_i1+n13_i2+n13_e1+n13_e2+n13_e3+n13_e4)*7, n13_i2+n13_e1+n13_e2+n13_e3+n13_e4, n13_e1+n13_e2+n13_e3+n13_e4, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_store_multiple_1" 1
  (and (and (eq_attr "type" "store_multiple")
	    (eq_attr "combo" "1"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1, n13_e2, n13_e3, n13_e4")

(define_insn_reservation "nds_n13_store_multiple_2" 1
  (and (ior (and (eq_attr "type" "store_multiple")
		 (eq_attr "combo" "2"))
	    (match_test "nds32::store_double_p (insn)"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i2+n13_e1, n13_e1+n13_e2, n13_e2+n13_e3, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_store_multiple_3" 1
  (and (and (eq_attr "type" "store_multiple")
	    (eq_attr "combo" "3"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2+n13_i2, n13_i1+n13_i2+n13_e1, n13_i2+n13_e1+n13_e2, n13_e1+n13_e2+n13_e3, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_store_multiple_4" 1
  (and (and (eq_attr "type" "store_multiple")
	    (eq_attr "combo" "4"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i1+n13_i2+n13_e1, n13_i1+n13_i2+n13_e1+n13_e2, n13_i2+n13_e1+n13_e2+n13_e3, n13_e1+n13_e2+n13_e3+n13_e4, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_store_multiple_5" 1
  (and (and (eq_attr "type" "store_multiple")
	    (eq_attr "combo" "5"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i1+n13_i2+n13_e1, n13_i1+n13_i2+n13_e1+n13_e2, n13_i1+n13_i2+n13_e1+n13_e2+n13_e3, n13_i2+n13_e1+n13_e2+n13_e3+n13_e4, n13_e1+n13_e2+n13_e3+n13_e4, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_store_multiple_6" 1
  (and (and (eq_attr "type" "store_multiple")
	    (eq_attr "combo" "6"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i1+n13_i2+n13_e1, n13_i1+n13_i2+n13_e1+n13_e2, n13_i1+n13_i2+n13_e1+n13_e2+n13_e3, n13_i1+n13_i2+n13_e1+n13_e2+n13_e3+n13_e4, n13_i2+n13_e1+n13_e2+n13_e3+n13_e4, n13_e1+n13_e2+n13_e3+n13_e4, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_store_multiple_7" 1
  (and (and (eq_attr "type" "store_multiple")
	    (eq_attr "combo" "7"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i1+n13_i2+n13_e1, n13_i1+n13_i2+n13_e1+n13_e2, n13_i1+n13_i2+n13_e1+n13_e2+n13_e3, (n13_i1+n13_i2+n13_e1+n13_e2+n13_e3+n13_e4)*2, n13_i2+n13_e1+n13_e2+n13_e3+n13_e4, n13_e1+n13_e2+n13_e3+n13_e4, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_store_multiple_8" 1
  (and (and (eq_attr "type" "store_multiple")
	    (eq_attr "combo" "8"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i1+n13_i2+n13_e1, n13_i1+n13_i2+n13_e1+n13_e2, n13_i1+n13_i2+n13_e1+n13_e2+n13_e3, (n13_i1+n13_i2+n13_e1+n13_e2+n13_e3+n13_e4)*3, n13_i2+n13_e1+n13_e2+n13_e3+n13_e4, n13_e1+n13_e2+n13_e3+n13_e4, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

(define_insn_reservation "nds_n13_store_multiple_12" 1
  (and (and (eq_attr "type" "store_multiple")
	    (eq_attr "combo" "12"))
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i1+n13_i2, n13_i1+n13_i2+n13_e1, n13_i1+n13_i2+n13_e1+n13_e2, n13_i1+n13_i2+n13_e1+n13_e2+n13_e3, (n13_i1+n13_i2+n13_e1+n13_e2+n13_e3+n13_e4)*7, n13_i2+n13_e1+n13_e2+n13_e3+n13_e4, n13_e1+n13_e2+n13_e3+n13_e4, n13_e2+n13_e3+n13_e4, n13_e3+n13_e4, n13_e4")

;; The multiplier at E1 takes two cycles.
(define_insn_reservation "nds_n13_mul" 1
  (and (eq_attr "type" "mul")
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1*2, n13_e2, n13_e3, n13_e4")

(define_insn_reservation "nds_n13_mac" 1
  (and (eq_attr "type" "mac")
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1*2, n13_e2, n13_e3, n13_e4")

;; The cycles consumed at E2 are 32 - CLZ(abs(Ra)) + 2,
;; so the worst case is 34.
(define_insn_reservation "nds_n13_div" 1
  (and (eq_attr "type" "div")
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1, n13_e2*34, n13_e3, n13_e4")

(define_insn_reservation "nds_n13_branch" 1
  (and (eq_attr "type" "branch")
       (eq_attr "pipeline_model" "n13"))
  "n13_i1, n13_i2, n13_e1, n13_e2, n13_e3, n13_e4")

;; ------------------------------------------------------------------------
;; Comment Notations and Bypass Rules
;; ------------------------------------------------------------------------
;; Producers (LHS)
;;   LD
;;     Load data from the memory and produce the loaded data. The result is
;;     ready at E3.
;;   LMW(N, M)
;;     There are N micro-operations within an instruction that loads multiple
;;     words. The result produced by the M-th micro-operation is sent to
;;     consumers. The result is ready at E3.
;;   ADDR_OUT
;;     Most load/store instructions can produce an address output if updating
;;     the base register is required. The result is ready at E2, which is
;;     produced by ALU.
;;   ALU, ALU_SHIFT, SIMD
;;     Compute data in ALU and produce the data. The result is ready at E2.
;;   MUL, MAC
;;     Compute data in the multiply-adder and produce the data. The result
;;     is ready at E2.
;;   DIV
;;     Compute data in the divider and produce the data. The result is ready
;;     at E2.
;;   BR
;;     Branch-with-link instructions produces a result containing the return
;;     address. The result is ready at E2.
;;
;; Consumers (RHS)
;;   ALU
;;     General ALU instructions require operands at E2.
;;   ALU_E1
;;     Some special ALU instructions, such as BSE, BSP and MOVD44, require
;;     operand at E1.
;;   MUL, DIV, PBSAD, MMU
;;     Operands are required at E1.
;;   PBSADA_Rt, PBSADA_RaRb
;;     Operands Ra and Rb are required at E1, and the operand Rt is required
;;     at E2.
;;   ALU_SHIFT_Rb
;;     An ALU-SHIFT instruction consists of a shift micro-operation followed
;;     by an arithmetic micro-operation. The operand Rb is used by the first
;;     micro-operation, and there are some latencies if data dependency occurs.
;;   MAC_RaRb
;;     A MAC instruction does multiplication at E1 and does accumulation at E2,
;;     so the operand Rt is required at E2, and operands Ra and Rb are required
;;     at E1.
;;   ADDR_IN
;;     If an instruction requires an address as its input operand, the address
;;     is required at E1.
;;   ST
;;     A store instruction requires its data at E2.
;;   SMW(N, M)
;;     There are N micro-operations within an instruction that stores multiple
;;     words. Each M-th micro-operation requires its data at E2.
;;   BR
;;     If a branch instruction is conditional, its input data is required at E2.

;; LD -> ALU_E1, PBSAD, PBSADA_RaRb, MUL, MAC_RaRb, DIV, MMU, ADDR_IN
(define_bypass 3
  "nds_n13_load"
  "nds_n13_alu, nds_n13_pbsad, nds_n13_pbsada,\
   nds_n13_mul, nds_n13_mac, nds_n13_div,\
   nds_n13_mmu,\
   nds_n13_load, nds_n13_store,\
   nds_n13_load_multiple_1,nds_n13_load_multiple_2, nds_n13_load_multiple_3,\
   nds_n13_load_multiple_4,nds_n13_load_multiple_5, nds_n13_load_multiple_6,\
   nds_n13_load_multiple_7,nds_n13_load_multiple_8, nds_n13_load_multiple_12,\
   nds_n13_store_multiple_1,nds_n13_store_multiple_2, nds_n13_store_multiple_3,\
   nds_n13_store_multiple_4,nds_n13_store_multiple_5, nds_n13_store_multiple_6,\
   nds_n13_store_multiple_7,nds_n13_store_multiple_8, nds_n13_store_multiple_12"
  "nds32_n13_load_to_e1_p"
)

;; LD -> ALU, ALU_SHIFT_Rb, PBSADA_Rt, BR, ST, SMW(N, 1)
(define_bypass 2
  "nds_n13_load"
  "nds_n13_alu, nds_n13_alu_shift, nds_n13_pbsada, nds_n13_branch, nds_n13_store,\
   nds_n13_store_multiple_1,nds_n13_store_multiple_2, nds_n13_store_multiple_3,\
   nds_n13_store_multiple_4,nds_n13_store_multiple_5, nds_n13_store_multiple_6,\
   nds_n13_store_multiple_7,nds_n13_store_multiple_8, nds_n13_store_multiple_12"
  "nds32_n13_load_to_e2_p"
)

;; LMW(N, N) -> ALU_E1, PBSAD, PBSADA_RaRb, MUL, MAC_RaRb, DIV, MMU, ADDR_IN
(define_bypass 3
  "nds_n13_load_multiple_1,nds_n13_load_multiple_2, nds_n13_load_multiple_3,\
   nds_n13_load_multiple_4,nds_n13_load_multiple_5, nds_n13_load_multiple_6,\
   nds_n13_load_multiple_7,nds_n13_load_multiple_8, nds_n13_load_multiple_12"
  "nds_n13_alu, nds_n13_pbsad, nds_n13_pbsada,\
   nds_n13_mul, nds_n13_mac, nds_n13_div,\
   nds_n13_mmu,\
   nds_n13_load, nds_n13_store,\
   nds_n13_load_multiple_1,nds_n13_load_multiple_2, nds_n13_load_multiple_3,\
   nds_n13_load_multiple_4,nds_n13_load_multiple_5, nds_n13_load_multiple_6,\
   nds_n13_load_multiple_7,nds_n13_load_multiple_8, nds_n13_load_multiple_12,\
   nds_n13_store_multiple_1,nds_n13_store_multiple_2, nds_n13_store_multiple_3,\
   nds_n13_store_multiple_4,nds_n13_store_multiple_5, nds_n13_store_multiple_6,\
   nds_n13_store_multiple_7,nds_n13_store_multiple_8, nds_n13_store_multiple_12"
  "nds32_n13_last_load_to_e1_p")

;; LMW(N, N) -> ALU, ALU_SHIFT_Rb, PBSADA_Rt, BR, ST, SMW(N, 1)
(define_bypass 2
  "nds_n13_load_multiple_1,nds_n13_load_multiple_2, nds_n13_load_multiple_3,\
   nds_n13_load_multiple_4,nds_n13_load_multiple_5, nds_n13_load_multiple_6,\
   nds_n13_load_multiple_7,nds_n13_load_multiple_8, nds_n13_load_multiple_12"
  "nds_n13_alu, nds_n13_alu_shift, nds_n13_pbsada, nds_n13_branch, nds_n13_store,\
   nds_n13_store_multiple_1,nds_n13_store_multiple_2, nds_n13_store_multiple_3,\
   nds_n13_store_multiple_4,nds_n13_store_multiple_5, nds_n13_store_multiple_6,\
   nds_n13_store_multiple_7,nds_n13_store_multiple_8, nds_n13_store_multiple_12"
  "nds32_n13_last_load_to_e2_p"
)

;; LMW(N, N - 1) -> ALU_E1, PBSAD, PBSADA_RaRb, MUL, MAC_RaRb, DIV, MMU, ADDR_IN
(define_bypass 2
  "nds_n13_load_multiple_1,nds_n13_load_multiple_2, nds_n13_load_multiple_3,\
   nds_n13_load_multiple_4,nds_n13_load_multiple_5, nds_n13_load_multiple_6,\
   nds_n13_load_multiple_7,nds_n13_load_multiple_8, nds_n13_load_multiple_12"
  "nds_n13_alu, nds_n13_pbsad, nds_n13_pbsada,\
   nds_n13_mul, nds_n13_mac, nds_n13_div,\
   nds_n13_mmu,\
   nds_n13_load, nds_n13_store,\
   nds_n13_load_multiple_1,nds_n13_load_multiple_2, nds_n13_load_multiple_3,\
   nds_n13_load_multiple_4,nds_n13_load_multiple_5, nds_n13_load_multiple_6,\
   nds_n13_load_multiple_7,nds_n13_load_multiple_8, nds_n13_load_multiple_12,\
   nds_n13_store_multiple_1,nds_n13_store_multiple_2, nds_n13_store_multiple_3,\
   nds_n13_store_multiple_4,nds_n13_store_multiple_5, nds_n13_store_multiple_6,\
   nds_n13_store_multiple_7,nds_n13_store_multiple_8, nds_n13_store_multiple_12"
  "nds32_n13_last_two_load_to_e1_p")

;; ALU, ALU_SHIFT, SIMD, BR, MUL, MAC, DIV, ADDR_OUT
;;   ->  ALU_E1, PBSAD, PBSADA_RaRb, MUL, MAC_RaRb, DIV, MMU, ADDR_IN
(define_bypass 2
  "nds_n13_alu, nds_n13_alu_shift, nds_n13_pbsad, nds_n13_pbsada, nds_n13_branch,\
   nds_n13_mul, nds_n13_mac, nds_n13_div,\
   nds_n13_load, nds_n13_store,\
   nds_n13_load_multiple_1,nds_n13_load_multiple_2, nds_n13_load_multiple_3,\
   nds_n13_load_multiple_4,nds_n13_load_multiple_5, nds_n13_load_multiple_6,\
   nds_n13_load_multiple_7,nds_n13_load_multiple_8, nds_n13_load_multiple_12,\
   nds_n13_store_multiple_1,nds_n13_store_multiple_2, nds_n13_store_multiple_3,\
   nds_n13_store_multiple_4,nds_n13_store_multiple_5, nds_n13_store_multiple_6,\
   nds_n13_store_multiple_7,nds_n13_store_multiple_8, nds_n13_store_multiple_12"
  "nds_n13_alu, nds_n13_pbsad, nds_n13_pbsada,\
   nds_n13_mul, nds_n13_mac, nds_n13_div,\
   nds_n13_mmu,\
   nds_n13_load, nds_n13_store,\
   nds_n13_load_multiple_1,nds_n13_load_multiple_2, nds_n13_load_multiple_3,\
   nds_n13_load_multiple_4,nds_n13_load_multiple_5, nds_n13_load_multiple_6,\
   nds_n13_load_multiple_7,nds_n13_load_multiple_8, nds_n13_load_multiple_12,\
   nds_n13_store_multiple_1,nds_n13_store_multiple_2, nds_n13_store_multiple_3,\
   nds_n13_store_multiple_4,nds_n13_store_multiple_5, nds_n13_store_multiple_6,\
   nds_n13_store_multiple_7,nds_n13_store_multiple_8, nds_n13_store_multiple_12"
  "nds32_n13_e2_to_e1_p")
