/*
 * SPDX-License-Identifier: GPL-3.0-only
 * MuseScore-Studio-CLA-applies
 *
 * MuseScore Studio
 * Music Composition & Notation
 *
 * Copyright (C) 2021 MuseScore Limited
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#pragma once

#include <QObject>

#include "engraving/dom/accidental.h"
#include "engraving/dom/chord.h"
#include "engraving/dom/guitarbend.h"
#include "engraving/dom/hairpin.h"
#include "engraving/dom/harmony.h"
#include "engraving/dom/mmrestrange.h"
#include "engraving/dom/ottava.h"
#include "engraving/dom/part.h"
#include "engraving/dom/spacer.h"
#include "engraving/dom/staff.h"
#include "engraving/dom/textbase.h"
#include "engraving/dom/timesig.h"
#include "engraving/dom/tremolobar.h"
#include "engraving/dom/tripletfeel.h"
#include "engraving/types/types.h"
#include "engraving/rendering/layoutoptions.h"

namespace mu::engraving::apiv1::enums {
Q_NAMESPACE;

enum class AccidentalType {
    ///.\{
    NONE          = int(mu::engraving::AccidentalType::NONE),
    FLAT          = int(mu::engraving::AccidentalType::FLAT),
    NATURAL       = int(mu::engraving::AccidentalType::NATURAL),
    SHARP         = int(mu::engraving::AccidentalType::SHARP),
    SHARP2        = int(mu::engraving::AccidentalType::SHARP2),
    FLAT2         = int(mu::engraving::AccidentalType::FLAT2),
    SHARP3        = int(mu::engraving::AccidentalType::SHARP3),
    FLAT3         = int(mu::engraving::AccidentalType::FLAT3),
    NATURAL_FLAT  = int(mu::engraving::AccidentalType::NATURAL_FLAT),
    NATURAL_SHARP = int(mu::engraving::AccidentalType::NATURAL_SHARP),
    SHARP_SHARP   = int(mu::engraving::AccidentalType::SHARP_SHARP),

    // Gould arrow quartertone
    FLAT_ARROW_UP      = int(mu::engraving::AccidentalType::FLAT_ARROW_UP),
    FLAT_ARROW_DOWN    = int(mu::engraving::AccidentalType::FLAT_ARROW_DOWN),
    NATURAL_ARROW_UP   = int(mu::engraving::AccidentalType::NATURAL_ARROW_UP),
    NATURAL_ARROW_DOWN = int(mu::engraving::AccidentalType::NATURAL_ARROW_DOWN),
    SHARP_ARROW_UP     = int(mu::engraving::AccidentalType::SHARP_ARROW_UP),
    SHARP_ARROW_DOWN   = int(mu::engraving::AccidentalType::SHARP_ARROW_DOWN),
    SHARP2_ARROW_UP    = int(mu::engraving::AccidentalType::SHARP2_ARROW_UP),
    SHARP2_ARROW_DOWN  = int(mu::engraving::AccidentalType::SHARP2_ARROW_DOWN),
    FLAT2_ARROW_UP     = int(mu::engraving::AccidentalType::FLAT2_ARROW_UP),
    FLAT2_ARROW_DOWN   = int(mu::engraving::AccidentalType::FLAT2_ARROW_DOWN),
    ARROW_DOWN         = int(mu::engraving::AccidentalType::ARROW_DOWN),
    ARROW_UP           = int(mu::engraving::AccidentalType::ARROW_UP),

    // Stein-Zimmermann
    MIRRORED_FLAT  = int(mu::engraving::AccidentalType::MIRRORED_FLAT),
    MIRRORED_FLAT2 = int(mu::engraving::AccidentalType::MIRRORED_FLAT2),
    SHARP_SLASH    = int(mu::engraving::AccidentalType::SHARP_SLASH),
    SHARP_SLASH4   = int(mu::engraving::AccidentalType::SHARP_SLASH4),

    // Arel-Ezgi-Uzdilek (AEU)
    FLAT_SLASH2  = int(mu::engraving::AccidentalType::FLAT_SLASH2),
    FLAT_SLASH   = int(mu::engraving::AccidentalType::FLAT_SLASH),
    SHARP_SLASH3 = int(mu::engraving::AccidentalType::SHARP_SLASH3),
    SHARP_SLASH2 = int(mu::engraving::AccidentalType::SHARP_SLASH2),

    // Extended Helmholtz-Ellis accidentals (just intonation)
    DOUBLE_FLAT_ONE_ARROW_DOWN  = int(mu::engraving::AccidentalType::DOUBLE_FLAT_ONE_ARROW_DOWN),
    FLAT_ONE_ARROW_DOWN         = int(mu::engraving::AccidentalType::FLAT_ONE_ARROW_DOWN),
    NATURAL_ONE_ARROW_DOWN      = int(mu::engraving::AccidentalType::NATURAL_ONE_ARROW_DOWN),
    SHARP_ONE_ARROW_DOWN        = int(mu::engraving::AccidentalType::SHARP_ONE_ARROW_DOWN),
    DOUBLE_SHARP_ONE_ARROW_DOWN = int(mu::engraving::AccidentalType::DOUBLE_SHARP_ONE_ARROW_DOWN),
    DOUBLE_FLAT_ONE_ARROW_UP    = int(mu::engraving::AccidentalType::DOUBLE_FLAT_ONE_ARROW_UP),

    FLAT_ONE_ARROW_UP           = int(mu::engraving::AccidentalType::FLAT_ONE_ARROW_UP),
    NATURAL_ONE_ARROW_UP        = int(mu::engraving::AccidentalType::NATURAL_ONE_ARROW_UP),
    SHARP_ONE_ARROW_UP          = int(mu::engraving::AccidentalType::SHARP_ONE_ARROW_UP),
    DOUBLE_SHARP_ONE_ARROW_UP   = int(mu::engraving::AccidentalType::DOUBLE_SHARP_ONE_ARROW_UP),
    DOUBLE_FLAT_TWO_ARROWS_DOWN = int(mu::engraving::AccidentalType::DOUBLE_FLAT_TWO_ARROWS_DOWN),
    FLAT_TWO_ARROWS_DOWN        = int(mu::engraving::AccidentalType::FLAT_TWO_ARROWS_DOWN),

    NATURAL_TWO_ARROWS_DOWN      = int(mu::engraving::AccidentalType::NATURAL_TWO_ARROWS_DOWN),
    SHARP_TWO_ARROWS_DOWN        = int(mu::engraving::AccidentalType::SHARP_TWO_ARROWS_DOWN),
    DOUBLE_SHARP_TWO_ARROWS_DOWN = int(mu::engraving::AccidentalType::DOUBLE_SHARP_TWO_ARROWS_DOWN),
    DOUBLE_FLAT_TWO_ARROWS_UP    = int(mu::engraving::AccidentalType::DOUBLE_FLAT_TWO_ARROWS_UP),
    FLAT_TWO_ARROWS_UP           = int(mu::engraving::AccidentalType::FLAT_TWO_ARROWS_UP),
    NATURAL_TWO_ARROWS_UP        = int(mu::engraving::AccidentalType::NATURAL_TWO_ARROWS_UP),

    SHARP_TWO_ARROWS_UP           = int(mu::engraving::AccidentalType::SHARP_TWO_ARROWS_UP),
    DOUBLE_SHARP_TWO_ARROWS_UP    = int(mu::engraving::AccidentalType::DOUBLE_SHARP_TWO_ARROWS_UP),
    DOUBLE_FLAT_THREE_ARROWS_DOWN = int(mu::engraving::AccidentalType::DOUBLE_FLAT_THREE_ARROWS_DOWN),
    FLAT_THREE_ARROWS_DOWN        = int(mu::engraving::AccidentalType::FLAT_THREE_ARROWS_DOWN),
    NATURAL_THREE_ARROWS_DOWN     = int(mu::engraving::AccidentalType::NATURAL_THREE_ARROWS_DOWN),
    SHARP_THREE_ARROWS_DOWN       = int(mu::engraving::AccidentalType::SHARP_THREE_ARROWS_DOWN),

    DOUBLE_SHARP_THREE_ARROWS_DOWN = int(mu::engraving::AccidentalType::DOUBLE_SHARP_THREE_ARROWS_DOWN),
    DOUBLE_FLAT_THREE_ARROWS_UP    = int(mu::engraving::AccidentalType::DOUBLE_FLAT_THREE_ARROWS_UP),
    FLAT_THREE_ARROWS_UP           = int(mu::engraving::AccidentalType::FLAT_THREE_ARROWS_UP),
    NATURAL_THREE_ARROWS_UP        = int(mu::engraving::AccidentalType::NATURAL_THREE_ARROWS_UP),
    SHARP_THREE_ARROWS_UP          = int(mu::engraving::AccidentalType::SHARP_THREE_ARROWS_UP),
    DOUBLE_SHARP_THREE_ARROWS_UP   = int(mu::engraving::AccidentalType::DOUBLE_SHARP_THREE_ARROWS_UP),

    LOWER_ONE_SEPTIMAL_COMMA        = int(mu::engraving::AccidentalType::LOWER_ONE_SEPTIMAL_COMMA),
    RAISE_ONE_SEPTIMAL_COMMA        = int(mu::engraving::AccidentalType::RAISE_ONE_SEPTIMAL_COMMA),
    LOWER_TWO_SEPTIMAL_COMMAS       = int(mu::engraving::AccidentalType::LOWER_TWO_SEPTIMAL_COMMAS),
    RAISE_TWO_SEPTIMAL_COMMAS       = int(mu::engraving::AccidentalType::RAISE_TWO_SEPTIMAL_COMMAS),
    LOWER_ONE_UNDECIMAL_QUARTERTONE = int(mu::engraving::AccidentalType::LOWER_ONE_UNDECIMAL_QUARTERTONE),
    RAISE_ONE_UNDECIMAL_QUARTERTONE = int(mu::engraving::AccidentalType::RAISE_ONE_UNDECIMAL_QUARTERTONE),

    LOWER_ONE_TRIDECIMAL_QUARTERTONE = int(mu::engraving::AccidentalType::LOWER_ONE_TRIDECIMAL_QUARTERTONE),
    RAISE_ONE_TRIDECIMAL_QUARTERTONE = int(mu::engraving::AccidentalType::RAISE_ONE_TRIDECIMAL_QUARTERTONE),

    DOUBLE_FLAT_EQUAL_TEMPERED   = int(mu::engraving::AccidentalType::DOUBLE_FLAT_EQUAL_TEMPERED),
    FLAT_EQUAL_TEMPERED          = int(mu::engraving::AccidentalType::FLAT_EQUAL_TEMPERED),
    NATURAL_EQUAL_TEMPERED       = int(mu::engraving::AccidentalType::NATURAL_EQUAL_TEMPERED),
    SHARP_EQUAL_TEMPERED         = int(mu::engraving::AccidentalType::SHARP_EQUAL_TEMPERED),
    DOUBLE_SHARP_EQUAL_TEMPERED  = int(mu::engraving::AccidentalType::DOUBLE_SHARP_EQUAL_TEMPERED),
    QUARTER_FLAT_EQUAL_TEMPERED  = int(mu::engraving::AccidentalType::QUARTER_FLAT_EQUAL_TEMPERED),
    QUARTER_SHARP_EQUAL_TEMPERED = int(mu::engraving::AccidentalType::QUARTER_SHARP_EQUAL_TEMPERED),

    FLAT_17       = int(mu::engraving::AccidentalType::FLAT_17),
    SHARP_17      = int(mu::engraving::AccidentalType::SHARP_17),
    FLAT_19       = int(mu::engraving::AccidentalType::FLAT_19),
    SHARP_19      = int(mu::engraving::AccidentalType::SHARP_19),
    FLAT_23       = int(mu::engraving::AccidentalType::FLAT_23),
    SHARP_23      = int(mu::engraving::AccidentalType::SHARP_23),
    FLAT_31       = int(mu::engraving::AccidentalType::FLAT_31),
    SHARP_31      = int(mu::engraving::AccidentalType::SHARP_31),
    FLAT_53       = int(mu::engraving::AccidentalType::FLAT_53),
    SHARP_53      = int(mu::engraving::AccidentalType::SHARP_53),
    EQUALS_ALMOST = int(mu::engraving::AccidentalType::EQUALS_ALMOST),
    EQUALS        = int(mu::engraving::AccidentalType::EQUALS),
    TILDE         = int(mu::engraving::AccidentalType::TILDE),

    // Persian
    SORI  = int(mu::engraving::AccidentalType::SORI),
    KORON = int(mu::engraving::AccidentalType::KORON),

    // Wyschnegradsky
    TEN_TWELFTH_FLAT     = int(mu::engraving::AccidentalType::TEN_TWELFTH_FLAT),
    TEN_TWELFTH_SHARP    = int(mu::engraving::AccidentalType::TEN_TWELFTH_SHARP),
    ELEVEN_TWELFTH_FLAT  = int(mu::engraving::AccidentalType::ELEVEN_TWELFTH_FLAT),
    ELEVEN_TWELFTH_SHARP = int(mu::engraving::AccidentalType::ELEVEN_TWELFTH_SHARP),
    ONE_TWELFTH_FLAT     = int(mu::engraving::AccidentalType::ONE_TWELFTH_FLAT),
    ONE_TWELFTH_SHARP    = int(mu::engraving::AccidentalType::ONE_TWELFTH_SHARP),
    TWO_TWELFTH_FLAT     = int(mu::engraving::AccidentalType::TWO_TWELFTH_FLAT),
    TWO_TWELFTH_SHARP    = int(mu::engraving::AccidentalType::TWO_TWELFTH_SHARP),
    THREE_TWELFTH_FLAT   = int(mu::engraving::AccidentalType::THREE_TWELFTH_FLAT),
    THREE_TWELFTH_SHARP  = int(mu::engraving::AccidentalType::THREE_TWELFTH_SHARP),
    FOUR_TWELFTH_FLAT    = int(mu::engraving::AccidentalType::FOUR_TWELFTH_FLAT),
    FOUR_TWELFTH_SHARP   = int(mu::engraving::AccidentalType::FOUR_TWELFTH_SHARP),
    FIVE_TWELFTH_FLAT    = int(mu::engraving::AccidentalType::FIVE_TWELFTH_FLAT),
    FIVE_TWELFTH_SHARP   = int(mu::engraving::AccidentalType::FIVE_TWELFTH_SHARP),
    SIX_TWELFTH_FLAT     = int(mu::engraving::AccidentalType::SIX_TWELFTH_FLAT),
    SIX_TWELFTH_SHARP    = int(mu::engraving::AccidentalType::SIX_TWELFTH_SHARP),
    SEVEN_TWELFTH_FLAT   = int(mu::engraving::AccidentalType::SEVEN_TWELFTH_FLAT),
    SEVEN_TWELFTH_SHARP  = int(mu::engraving::AccidentalType::SEVEN_TWELFTH_SHARP),
    EIGHT_TWELFTH_FLAT   = int(mu::engraving::AccidentalType::EIGHT_TWELFTH_FLAT),
    EIGHT_TWELFTH_SHARP  = int(mu::engraving::AccidentalType::EIGHT_TWELFTH_SHARP),
    NINE_TWELFTH_FLAT    = int(mu::engraving::AccidentalType::NINE_TWELFTH_FLAT),
    NINE_TWELFTH_SHARP   = int(mu::engraving::AccidentalType::NINE_TWELFTH_SHARP),

    // (Spartan) Sagittal
    SAGITTAL_5V7KD      = int(mu::engraving::AccidentalType::SAGITTAL_5V7KD),
    SAGITTAL_5V7KU      = int(mu::engraving::AccidentalType::SAGITTAL_5V7KU),
    SAGITTAL_5CD        = int(mu::engraving::AccidentalType::SAGITTAL_5CD),
    SAGITTAL_5CU        = int(mu::engraving::AccidentalType::SAGITTAL_5CU),
    SAGITTAL_7CD        = int(mu::engraving::AccidentalType::SAGITTAL_7CD),
    SAGITTAL_7CU        = int(mu::engraving::AccidentalType::SAGITTAL_7CU),
    SAGITTAL_25SDD      = int(mu::engraving::AccidentalType::SAGITTAL_25SDD),
    SAGITTAL_25SDU      = int(mu::engraving::AccidentalType::SAGITTAL_25SDU),
    SAGITTAL_35MDD      = int(mu::engraving::AccidentalType::SAGITTAL_35MDD),
    SAGITTAL_35MDU      = int(mu::engraving::AccidentalType::SAGITTAL_35MDU),
    SAGITTAL_11MDD      = int(mu::engraving::AccidentalType::SAGITTAL_11MDD),
    SAGITTAL_11MDU      = int(mu::engraving::AccidentalType::SAGITTAL_11MDU),
    SAGITTAL_11LDD      = int(mu::engraving::AccidentalType::SAGITTAL_11LDD),
    SAGITTAL_11LDU      = int(mu::engraving::AccidentalType::SAGITTAL_11LDU),
    SAGITTAL_35LDD      = int(mu::engraving::AccidentalType::SAGITTAL_35LDD),
    SAGITTAL_35LDU      = int(mu::engraving::AccidentalType::SAGITTAL_35LDU),
    SAGITTAL_FLAT25SU   = int(mu::engraving::AccidentalType::SAGITTAL_FLAT25SU),
    SAGITTAL_SHARP25SD  = int(mu::engraving::AccidentalType::SAGITTAL_SHARP25SD),
    SAGITTAL_FLAT7CU    = int(mu::engraving::AccidentalType::SAGITTAL_FLAT7CU),
    SAGITTAL_SHARP7CD   = int(mu::engraving::AccidentalType::SAGITTAL_SHARP7CD),
    SAGITTAL_FLAT5CU    = int(mu::engraving::AccidentalType::SAGITTAL_FLAT5CU),
    SAGITTAL_SHARP5CD   = int(mu::engraving::AccidentalType::SAGITTAL_SHARP5CD),
    SAGITTAL_FLAT5V7KU  = int(mu::engraving::AccidentalType::SAGITTAL_FLAT5V7KU),
    SAGITTAL_SHARP5V7KD = int(mu::engraving::AccidentalType::SAGITTAL_SHARP5V7KD),
    SAGITTAL_FLAT       = int(mu::engraving::AccidentalType::SAGITTAL_FLAT),
    SAGITTAL_SHARP      = int(mu::engraving::AccidentalType::SAGITTAL_SHARP),

    // Turkish folk music
    ONE_COMMA_FLAT    = int(mu::engraving::AccidentalType::ONE_COMMA_FLAT),
    ONE_COMMA_SHARP   = int(mu::engraving::AccidentalType::ONE_COMMA_SHARP),
    TWO_COMMA_FLAT    = int(mu::engraving::AccidentalType::TWO_COMMA_FLAT),
    TWO_COMMA_SHARP   = int(mu::engraving::AccidentalType::TWO_COMMA_SHARP),
    THREE_COMMA_FLAT  = int(mu::engraving::AccidentalType::THREE_COMMA_FLAT),
    THREE_COMMA_SHARP = int(mu::engraving::AccidentalType::THREE_COMMA_SHARP),
    FOUR_COMMA_FLAT   = int(mu::engraving::AccidentalType::FOUR_COMMA_FLAT),
    //FOUR_COMMA_SHARP = int(mu::engraving::AccidentalType::FOUR_COMMA_SHARP),
    FIVE_COMMA_SHARP  = int(mu::engraving::AccidentalType::FIVE_COMMA_SHARP),

    END = int(mu::engraving::AccidentalType::END)
///\}
};
Q_ENUM_NS(AccidentalType);

enum class AccidentalBracket {
    NONE        = int(mu::engraving::AccidentalBracket::NONE),
    PARENTHESIS = int(mu::engraving::AccidentalBracket::PARENTHESIS),
    BRACKET     = int(mu::engraving::AccidentalBracket::BRACKET),
    BRACE       = int(mu::engraving::AccidentalBracket::BRACE),
};
Q_ENUM_NS(AccidentalBracket);

enum class ElementType {
    ///.\{
    INVALID               = int(mu::engraving::ElementType::INVALID),

    BRACKET_ITEM          = int(mu::engraving::ElementType::BRACKET_ITEM),
    PART                  = int(mu::engraving::ElementType::PART),
    STAFF                 = int(mu::engraving::ElementType::STAFF),
    SCORE                 = int(mu::engraving::ElementType::SCORE),
    TEXT                  = int(mu::engraving::ElementType::TEXT),
    LAYOUT_BREAK          = int(mu::engraving::ElementType::LAYOUT_BREAK),
    MEASURE_NUMBER        = int(mu::engraving::ElementType::MEASURE_NUMBER),
    MMREST_RANGE          = int(mu::engraving::ElementType::MMREST_RANGE),
    INSTRUMENT_NAME       = int(mu::engraving::ElementType::INSTRUMENT_NAME),
    SLUR_SEGMENT          = int(mu::engraving::ElementType::SLUR_SEGMENT),
    TIE_SEGMENT           = int(mu::engraving::ElementType::TIE_SEGMENT),
    LAISSEZ_VIB_SEGMENT   = int(mu::engraving::ElementType::LAISSEZ_VIB_SEGMENT),
    PARTIAL_TIE_SEGMENT   = int(mu::engraving::ElementType::PARTIAL_TIE_SEGMENT),
    BAR_LINE              = int(mu::engraving::ElementType::BAR_LINE),
    STAFF_LINES           = int(mu::engraving::ElementType::STAFF_LINES),
    SYSTEM_DIVIDER        = int(mu::engraving::ElementType::SYSTEM_DIVIDER),
    STEM_SLASH            = int(mu::engraving::ElementType::STEM_SLASH),
    ARPEGGIO              = int(mu::engraving::ElementType::ARPEGGIO),
    ACCIDENTAL            = int(mu::engraving::ElementType::ACCIDENTAL),
    LEDGER_LINE           = int(mu::engraving::ElementType::LEDGER_LINE),
    STEM                  = int(mu::engraving::ElementType::STEM),
    HOOK                  = int(mu::engraving::ElementType::HOOK),
    NOTE                  = int(mu::engraving::ElementType::NOTE),
    SYMBOL                = int(mu::engraving::ElementType::SYMBOL),
    CLEF                  = int(mu::engraving::ElementType::CLEF),
    KEYSIG                = int(mu::engraving::ElementType::KEYSIG),
    AMBITUS               = int(mu::engraving::ElementType::AMBITUS),
    TIMESIG               = int(mu::engraving::ElementType::TIMESIG),
    REST                  = int(mu::engraving::ElementType::REST),
    MMREST                = int(mu::engraving::ElementType::MMREST),
    DEAD_SLAPPED          = int(mu::engraving::ElementType::DEAD_SLAPPED),
    BREATH                = int(mu::engraving::ElementType::BREATH),
    MEASURE_REPEAT        = int(mu::engraving::ElementType::MEASURE_REPEAT),
    TIE                   = int(mu::engraving::ElementType::TIE),
    LAISSEZ_VIB           = int(mu::engraving::ElementType::LAISSEZ_VIB),
    PARTIAL_TIE           = int(mu::engraving::ElementType::PARTIAL_TIE),
    ARTICULATION          = int(mu::engraving::ElementType::ARTICULATION),
    ORNAMENT              = int(mu::engraving::ElementType::ORNAMENT),
    FERMATA               = int(mu::engraving::ElementType::FERMATA),
    CHORDLINE             = int(mu::engraving::ElementType::CHORDLINE),
    DYNAMIC               = int(mu::engraving::ElementType::DYNAMIC),
    EXPRESSION            = int(mu::engraving::ElementType::EXPRESSION),
    BEAM                  = int(mu::engraving::ElementType::BEAM),
    LYRICS                = int(mu::engraving::ElementType::LYRICS),
    FIGURED_BASS          = int(mu::engraving::ElementType::FIGURED_BASS),
    FIGURED_BASS_ITEM     = int(mu::engraving::ElementType::FIGURED_BASS_ITEM),
    MARKER                = int(mu::engraving::ElementType::MARKER),
    JUMP                  = int(mu::engraving::ElementType::JUMP),
    FINGERING             = int(mu::engraving::ElementType::FINGERING),
    TUPLET                = int(mu::engraving::ElementType::TUPLET),
    TEMPO_TEXT            = int(mu::engraving::ElementType::TEMPO_TEXT),
    STAFF_TEXT            = int(mu::engraving::ElementType::STAFF_TEXT),
    SYSTEM_TEXT           = int(mu::engraving::ElementType::SYSTEM_TEXT),
    SOUND_FLAG            = int(mu::engraving::ElementType::SOUND_FLAG),
    PLAY_COUNT_TEXT       = int(mu::engraving::ElementType::PLAY_COUNT_TEXT),
    PLAYTECH_ANNOTATION   = int(mu::engraving::ElementType::PLAYTECH_ANNOTATION),
    CAPO                  = int(mu::engraving::ElementType::CAPO),
    STRING_TUNINGS        = int(mu::engraving::ElementType::STRING_TUNINGS),
    TRIPLET_FEEL          = int(mu::engraving::ElementType::TRIPLET_FEEL),
    REHEARSAL_MARK        = int(mu::engraving::ElementType::REHEARSAL_MARK),
    INSTRUMENT_CHANGE     = int(mu::engraving::ElementType::INSTRUMENT_CHANGE),
    STAFFTYPE_CHANGE      = int(mu::engraving::ElementType::STAFFTYPE_CHANGE),
    HARMONY               = int(mu::engraving::ElementType::HARMONY),
    FRET_DIAGRAM          = int(mu::engraving::ElementType::FRET_DIAGRAM),
    HARP_DIAGRAM          = int(mu::engraving::ElementType::HARP_DIAGRAM),
    BEND                  = int(mu::engraving::ElementType::BEND),
    TREMOLOBAR            = int(mu::engraving::ElementType::TREMOLOBAR),
    VOLTA                 = int(mu::engraving::ElementType::VOLTA),
    HAIRPIN_SEGMENT       = int(mu::engraving::ElementType::HAIRPIN_SEGMENT),
    OTTAVA_SEGMENT        = int(mu::engraving::ElementType::OTTAVA_SEGMENT),
    TRILL_SEGMENT         = int(mu::engraving::ElementType::TRILL_SEGMENT),
    LET_RING_SEGMENT      = int(mu::engraving::ElementType::LET_RING_SEGMENT),
    GRADUAL_TEMPO_CHANGE_SEGMENT = int(mu::engraving::ElementType::GRADUAL_TEMPO_CHANGE_SEGMENT),
    VIBRATO_SEGMENT       = int(mu::engraving::ElementType::VIBRATO_SEGMENT),
    PALM_MUTE_SEGMENT     = int(mu::engraving::ElementType::PALM_MUTE_SEGMENT),
    WHAMMY_BAR_SEGMENT    = int(mu::engraving::ElementType::WHAMMY_BAR_SEGMENT),
    RASGUEADO_SEGMENT     = int(mu::engraving::ElementType::RASGUEADO_SEGMENT),
    HARMONIC_MARK_SEGMENT = int(mu::engraving::ElementType::HARMONIC_MARK_SEGMENT),
    PICK_SCRAPE_SEGMENT   = int(mu::engraving::ElementType::PICK_SCRAPE_SEGMENT),
    TEXTLINE_SEGMENT      = int(mu::engraving::ElementType::TEXTLINE_SEGMENT),
    VOLTA_SEGMENT         = int(mu::engraving::ElementType::VOLTA_SEGMENT),
    PEDAL_SEGMENT         = int(mu::engraving::ElementType::PEDAL_SEGMENT),
    LYRICSLINE_SEGMENT    = int(mu::engraving::ElementType::LYRICSLINE_SEGMENT),
    PARTIAL_LYRICSLINE_SEGMENT = int(mu::engraving::ElementType::PARTIAL_LYRICSLINE_SEGMENT),
    GLISSANDO_SEGMENT     = int(mu::engraving::ElementType::GLISSANDO_SEGMENT),
    NOTELINE_SEGMENT      = int(mu::engraving::ElementType::NOTELINE_SEGMENT),
    STAFF_VISIBILITY_INDICATOR = int(mu::engraving::ElementType::STAFF_VISIBILITY_INDICATOR),
    SYSTEM_LOCK_INDICATOR = int(mu::engraving::ElementType::SYSTEM_LOCK_INDICATOR),
    SPACER                = int(mu::engraving::ElementType::SPACER),
    STAFF_STATE           = int(mu::engraving::ElementType::STAFF_STATE),
    NOTEHEAD              = int(mu::engraving::ElementType::NOTEHEAD),
    NOTEDOT               = int(mu::engraving::ElementType::NOTEDOT),
    IMAGE                 = int(mu::engraving::ElementType::IMAGE),
    MEASURE               = int(mu::engraving::ElementType::MEASURE),
    SELECTION             = int(mu::engraving::ElementType::SELECTION),
    LASSO                 = int(mu::engraving::ElementType::LASSO),
    SHADOW_NOTE           = int(mu::engraving::ElementType::SHADOW_NOTE),
    TAB_DURATION_SYMBOL   = int(mu::engraving::ElementType::TAB_DURATION_SYMBOL),
    FSYMBOL               = int(mu::engraving::ElementType::FSYMBOL),
    PAGE                  = int(mu::engraving::ElementType::PAGE),
    HAIRPIN               = int(mu::engraving::ElementType::HAIRPIN),
    OTTAVA                = int(mu::engraving::ElementType::OTTAVA),
    PEDAL                 = int(mu::engraving::ElementType::PEDAL),
    TRILL                 = int(mu::engraving::ElementType::TRILL),
    LET_RING              = int(mu::engraving::ElementType::LET_RING),
    GRADUAL_TEMPO_CHANGE  = int(mu::engraving::ElementType::GRADUAL_TEMPO_CHANGE),
    VIBRATO               = int(mu::engraving::ElementType::VIBRATO),
    PALM_MUTE             = int(mu::engraving::ElementType::PALM_MUTE),
    WHAMMY_BAR            = int(mu::engraving::ElementType::WHAMMY_BAR),
    RASGUEADO             = int(mu::engraving::ElementType::RASGUEADO),
    HARMONIC_MARK         = int(mu::engraving::ElementType::HARMONIC_MARK),
    TEXTLINE              = int(mu::engraving::ElementType::TEXTLINE),
    TEXTLINE_BASE         = int(mu::engraving::ElementType::TEXTLINE_BASE),
    NOTELINE              = int(mu::engraving::ElementType::NOTELINE),
    LYRICSLINE            = int(mu::engraving::ElementType::LYRICSLINE),
    PARTIAL_LYRICSLINE    = int(mu::engraving::ElementType::PARTIAL_LYRICSLINE),
    GLISSANDO             = int(mu::engraving::ElementType::GLISSANDO),
    BRACKET               = int(mu::engraving::ElementType::BRACKET),
    SEGMENT               = int(mu::engraving::ElementType::SEGMENT),
    SYSTEM                = int(mu::engraving::ElementType::SYSTEM),
    CHORD                 = int(mu::engraving::ElementType::CHORD),
    SLUR                  = int(mu::engraving::ElementType::SLUR),
    HBOX                  = int(mu::engraving::ElementType::HBOX),
    VBOX                  = int(mu::engraving::ElementType::VBOX),
    TBOX                  = int(mu::engraving::ElementType::TBOX),
    FBOX                  = int(mu::engraving::ElementType::FBOX),
    ACTION_ICON           = int(mu::engraving::ElementType::ACTION_ICON),
    BAGPIPE_EMBELLISHMENT = int(mu::engraving::ElementType::BAGPIPE_EMBELLISHMENT),
    STICKING              = int(mu::engraving::ElementType::STICKING),
    GRACE_NOTES_GROUP     = int(mu::engraving::ElementType::GRACE_NOTES_GROUP),
    GUITAR_BEND           = int(mu::engraving::ElementType::GUITAR_BEND),
    GUITAR_BEND_SEGMENT   = int(mu::engraving::ElementType::GUITAR_BEND_SEGMENT),
    GUITAR_BEND_HOLD      = int(mu::engraving::ElementType::GUITAR_BEND_HOLD),
    GUITAR_BEND_HOLD_SEGMENT = int(mu::engraving::ElementType::GUITAR_BEND_HOLD_SEGMENT),
    GUITAR_BEND_TEXT      = int(mu::engraving::ElementType::GUITAR_BEND_TEXT),
    TREMOLO_TWOCHORD      = int(mu::engraving::ElementType::TREMOLO_TWOCHORD),
    TREMOLO_SINGLECHORD   = int(mu::engraving::ElementType::TREMOLO_SINGLECHORD),
    TIME_TICK_ANCHOR      = int(mu::engraving::ElementType::TIME_TICK_ANCHOR),
    PARENTHESIS           = int(mu::engraving::ElementType::PARENTHESIS),
    HAMMER_ON_PULL_OFF    = int(mu::engraving::ElementType::HAMMER_ON_PULL_OFF),
    HAMMER_ON_PULL_OFF_SEGMENT = int(mu::engraving::ElementType::HAMMER_ON_PULL_OFF_SEGMENT),
    HAMMER_ON_PULL_OFF_TEXT = int(mu::engraving::ElementType::HAMMER_ON_PULL_OFF_TEXT),
    TAPPING               = int(mu::engraving::ElementType::TAPPING),
    TAPPING_HALF_SLUR     = int(mu::engraving::ElementType::TAPPING_HALF_SLUR),
    TAPPING_HALF_SLUR_SEGMENT = int(mu::engraving::ElementType::TAPPING_HALF_SLUR_SEGMENT),
    TAPPING_TEXT          = int(mu::engraving::ElementType::TAPPING_TEXT),

    ROOT_ITEM             = int(mu::engraving::ElementType::ROOT_ITEM),
    DUMMY                 = int(mu::engraving::ElementType::DUMMY),

    TREMOLO               = int(mu::engraving::ElementType::INVALID), // deprecated
    COMPOUND              = int(mu::engraving::ElementType::INVALID), // deprecated
    ELEMENT               = int(mu::engraving::ElementType::INVALID), // deprecated
    ELEMENT_LIST          = int(mu::engraving::ElementType::INVALID), // deprecated
    STAFF_LIST            = int(mu::engraving::ElementType::INVALID), // deprecated
    MEASURE_LIST          = int(mu::engraving::ElementType::INVALID), // deprecated
    OSSIA                 = int(mu::engraving::ElementType::INVALID), // deprecated

    MAXTYPE               = int(mu::engraving::ElementType::MAXTYPE)
                            ///\}
};
Q_ENUM_NS(ElementType);

enum class OrnamentStyle {
    DEFAULT = int(mu::engraving::OrnamentStyle::DEFAULT),
    BAROQUE = int(mu::engraving::OrnamentStyle::BAROQUE),
};
Q_ENUM_NS(OrnamentStyle);

enum class Align : char {
    LEFT     = 0,
    RIGHT    = 1,
    HCENTER  = 2,
    TOP      = 0,
    BOTTOM   = 4,
    VCENTER  = 8,
    BASELINE = 16,
    CENTER   = Align::HCENTER | Align::VCENTER,
    HMASK    = Align::LEFT | Align::RIGHT | Align::HCENTER,
    VMASK    = Align::TOP | Align::BOTTOM | Align::VCENTER | Align::BASELINE
};
Q_ENUM_NS(Align);

constexpr Align operator &(Align a1, Align a2)
{
    return static_cast<Align>(static_cast<char>(a1) & static_cast<char>(a2));
}

//! NOTE just Placement for compatibility
enum class Placement {
    ABOVE = int(mu::engraving::PlacementV::ABOVE),
    BELOW = int(mu::engraving::PlacementV::BELOW),
};
Q_ENUM_NS(Placement);

enum class PlacementH {
    LEFT   = int(mu::engraving::PlacementH::LEFT),
    CENTER = int(mu::engraving::PlacementH::CENTER),
    RIGHT  = int(mu::engraving::PlacementH::RIGHT),
};
Q_ENUM_NS(PlacementH);

enum class TextPlace {
    AUTO  = int(mu::engraving::TextPlace::AUTO),
    ABOVE = int(mu::engraving::TextPlace::ABOVE),
    BELOW = int(mu::engraving::TextPlace::BELOW),
    LEFT  = int(mu::engraving::TextPlace::LEFT),
};
Q_ENUM_NS(TextPlace);

enum class Direction {
    AUTO = int(mu::engraving::DirectionV::AUTO),
    UP   = int(mu::engraving::DirectionV::UP),
    DOWN = int(mu::engraving::DirectionV::DOWN),
};
Q_ENUM_NS(Direction);

enum class DirectionH {
    AUTO  = int(mu::engraving::DirectionH::AUTO),
    LEFT  = int(mu::engraving::DirectionH::LEFT),
    RIGHT = int(mu::engraving::DirectionH::RIGHT),
};
Q_ENUM_NS(DirectionH);

enum class Orientation {
    VERTICAL   = int(mu::engraving::Orientation::VERTICAL),
    HORIZONTAL = int(mu::engraving::Orientation::HORIZONTAL),
};
Q_ENUM_NS(Orientation);

enum class AutoOnOff {
    AUTO = int(mu::engraving::AutoOnOff::AUTO),
    ON   = int(mu::engraving::AutoOnOff::ON),
    OFF  = int(mu::engraving::AutoOnOff::OFF),
};
Q_ENUM_NS(AutoOnOff);

enum class AutoCustomHide {
    AUTO   = int(mu::engraving::AutoCustomHide::AUTO),
    CUSTOM = int(mu::engraving::AutoCustomHide::CUSTOM),
    HIDE   = int(mu::engraving::AutoCustomHide::HIDE),
};
Q_ENUM_NS(AutoCustomHide);

enum class VoiceAssignment {
    ALL_VOICE_IN_INSTRUMENT = int(mu::engraving::VoiceAssignment::ALL_VOICE_IN_INSTRUMENT),
    ALL_VOICE_IN_STAFF      = int(mu::engraving::VoiceAssignment::ALL_VOICE_IN_STAFF),
    CURRENT_VOICE_ONLY      = int(mu::engraving::VoiceAssignment::CURRENT_VOICE_ONLY),
};
Q_ENUM_NS(VoiceAssignment);

enum class SpacerType {
    UP    = int(mu::engraving::SpacerType::UP),
    DOWN  = int(mu::engraving::SpacerType::DOWN),
    FIXED = int(mu::engraving::SpacerType::FIXED),
};
Q_ENUM_NS(SpacerType);

enum class LayoutBreakType {
    PAGE    = int(mu::engraving::LayoutBreakType::PAGE),
    LINE    = int(mu::engraving::LayoutBreakType::LINE),
    SECTION = int(mu::engraving::LayoutBreakType::SECTION),
    NOBREAK = int(mu::engraving::LayoutBreakType::NOBREAK),
};
Q_ENUM_NS(LayoutBreakType);

enum class DurationType {
    V_LONG    = int(mu::engraving::DurationType::V_LONG),
    V_BREVE   = int(mu::engraving::DurationType::V_BREVE),
    V_WHOLE   = int(mu::engraving::DurationType::V_WHOLE),
    V_HALF    = int(mu::engraving::DurationType::V_HALF),
    V_QUARTER = int(mu::engraving::DurationType::V_QUARTER),
    V_EIGHTH  = int(mu::engraving::DurationType::V_EIGHTH),
    V_16TH    = int(mu::engraving::DurationType::V_16TH),
    V_32ND    = int(mu::engraving::DurationType::V_32ND),
    V_64TH    = int(mu::engraving::DurationType::V_64TH),
    V_128TH   = int(mu::engraving::DurationType::V_128TH),
    V_256TH   = int(mu::engraving::DurationType::V_256TH),
    V_512TH   = int(mu::engraving::DurationType::V_512TH),
    V_1024TH  = int(mu::engraving::DurationType::V_1024TH),
    V_ZERO    = int(mu::engraving::DurationType::V_ZERO),
    V_MEASURE = int(mu::engraving::DurationType::V_MEASURE),
    V_INVALID = int(mu::engraving::DurationType::V_INVALID),
};
Q_ENUM_NS(DurationType);

enum class VeloType {
    OFFSET_VAL = int(mu::engraving::VeloType::OFFSET_VAL),
    USER_VAL   = int(mu::engraving::VeloType::USER_VAL),
};
Q_ENUM_NS(VeloType);

enum class BeamMode {
    AUTO    = int(mu::engraving::BeamMode::AUTO),
    BEGIN   = int(mu::engraving::BeamMode::BEGIN),
    MID     = int(mu::engraving::BeamMode::MID),
    END     = int(mu::engraving::BeamMode::END),
    NONE    = int(mu::engraving::BeamMode::NONE),
    BEGIN32 = int(mu::engraving::BeamMode::BEGIN16), // these names for the beam modes are deprecated! this beam mode begins a 16th beam
    BEGIN64 = int(mu::engraving::BeamMode::BEGIN32), // this one begins a 32nd beam
    INVALID = int(mu::engraving::BeamMode::INVALID),
};
Q_ENUM_NS(BeamMode);

enum class GlissandoType {
    STRAIGHT = int(mu::engraving::GlissandoType::STRAIGHT),
    WAVY     = int(mu::engraving::GlissandoType::WAVY)
};
Q_ENUM_NS(GlissandoType);

enum class GlissandoStyle {
    CHROMATIC  = int(mu::engraving::GlissandoStyle::CHROMATIC),
    WHITE_KEYS = int(mu::engraving::GlissandoStyle::WHITE_KEYS),
    BLACK_KEYS = int(mu::engraving::GlissandoStyle::BLACK_KEYS),
    DIATONIC   = int(mu::engraving::GlissandoStyle::DIATONIC),
    PORTAMENTO = int(mu::engraving::GlissandoStyle::PORTAMENTO)
};
Q_ENUM_NS(GlissandoStyle);

enum class HarmonyType {
    ///.\{
    STANDARD  = int(mu::engraving::HarmonyType::STANDARD),
    ROMAN     = int(mu::engraving::HarmonyType::ROMAN),
    NASHVILLE = int(mu::engraving::HarmonyType::NASHVILLE)
                ///\}
};
Q_ENUM_NS(HarmonyType);

enum class HarmonyVoicing {
    INVALID    = int(mu::engraving::Voicing::INVALID),
    AUTO       = int(mu::engraving::Voicing::AUTO),
    ROOT_ONLY  = int(mu::engraving::Voicing::ROOT_ONLY),
    CLOSE      = int(mu::engraving::Voicing::CLOSE),
    DROP_2     = int(mu::engraving::Voicing::DROP_2),
    SIX_NOTE   = int(mu::engraving::Voicing::SIX_NOTE),
    FOUR_NOTE  = int(mu::engraving::Voicing::FOUR_NOTE),
    THREE_NOTE = int(mu::engraving::Voicing::THREE_NOTE),
};
Q_ENUM_NS(HarmonyVoicing);

enum class HDuration {
    INVALID                 = int(mu::engraving::HDuration::INVALID),
    UNTIL_NEXT_CHORD_SYMBOL = int(mu::engraving::HDuration::UNTIL_NEXT_CHORD_SYMBOL),
    STOP_AT_MEASURE_END     = int(mu::engraving::HDuration::STOP_AT_MEASURE_END),
    SEGMENT_DURATION        = int(mu::engraving::HDuration::SEGMENT_DURATION),
};
Q_ENUM_NS(HDuration);

enum class FrameType {
    NO_FRAME = int(mu::engraving::FrameType::NO_FRAME),
    SQUARE   = int(mu::engraving::FrameType::SQUARE),
    CIRCLE   = int(mu::engraving::FrameType::CIRCLE),
};
Q_ENUM_NS(FrameType);

enum class VerticalAlignment {
    AlignUndefined   = int(mu::engraving::VerticalAlignment::AlignUndefined),
    AlignNormal       = int(mu::engraving::VerticalAlignment::AlignNormal),
    AlignSuperScript = int(mu::engraving::VerticalAlignment::AlignSuperScript),
    AlignSubScript   = int(mu::engraving::VerticalAlignment::AlignSubScript),
};
Q_ENUM_NS(VerticalAlignment);

enum class TremoloBarType {
    DIP          = int(mu::engraving::TremoloBarType::DIP),
    DIVE         = int(mu::engraving::TremoloBarType::DIVE),
    RELEASE_UP   = int(mu::engraving::TremoloBarType::RELEASE_UP),
    INVERTED_DIP = int(mu::engraving::TremoloBarType::INVERTED_DIP),
    RETURN       = int(mu::engraving::TremoloBarType::RETURN),
    RELEASE_DOWN = int(mu::engraving::TremoloBarType::RELEASE_DOWN),
    CUSTOM       = int(mu::engraving::TremoloBarType::CUSTOM),
};
Q_ENUM_NS(TremoloBarType);

enum class PreferSharpFlat {
    NONE   = int(mu::engraving::PreferSharpFlat::NONE),
    SHARPS = int(mu::engraving::PreferSharpFlat::SHARPS),
    FLATS  = int(mu::engraving::PreferSharpFlat::FLATS),
    AUTO   = int(mu::engraving::PreferSharpFlat::AUTO),
};
Q_ENUM_NS(PreferSharpFlat);

enum class NoteHeadType {
    HEAD_AUTO    = int(mu::engraving::NoteHeadType::HEAD_AUTO),
    HEAD_WHOLE   = int(mu::engraving::NoteHeadType::HEAD_WHOLE),
    HEAD_HALF    = int(mu::engraving::NoteHeadType::HEAD_HALF),
    HEAD_QUARTER = int(mu::engraving::NoteHeadType::HEAD_QUARTER),
    HEAD_BREVIS  = int(mu::engraving::NoteHeadType::HEAD_BREVIS),
    HEAD_TYPES   = int(mu::engraving::NoteHeadType::HEAD_TYPES),
};
Q_ENUM_NS(NoteHeadType);

enum class NoteHeadScheme {
    HEAD_AUTO                = int(mu::engraving::NoteHeadScheme::HEAD_AUTO),
    HEAD_NORMAL              = int(mu::engraving::NoteHeadScheme::HEAD_NORMAL),
    HEAD_PITCHNAME           = int(mu::engraving::NoteHeadScheme::HEAD_PITCHNAME),
    HEAD_PITCHNAME_GERMAN    = int(mu::engraving::NoteHeadScheme::HEAD_PITCHNAME_GERMAN),
    HEAD_SOLFEGE             = int(mu::engraving::NoteHeadScheme::HEAD_SOLFEGE),
    HEAD_SOLFEGE_FIXED       = int(mu::engraving::NoteHeadScheme::HEAD_SOLFEGE_FIXED),
    HEAD_SHAPE_NOTE_4        = int(mu::engraving::NoteHeadScheme::HEAD_SHAPE_NOTE_4),
    HEAD_SHAPE_NOTE_7_AIKIN  = int(mu::engraving::NoteHeadScheme::HEAD_SHAPE_NOTE_7_AIKIN),
    HEAD_SHAPE_NOTE_7_FUNK   = int(mu::engraving::NoteHeadScheme::HEAD_SHAPE_NOTE_7_FUNK),
    HEAD_SHAPE_NOTE_7_WALKER = int(mu::engraving::NoteHeadScheme::HEAD_SHAPE_NOTE_7_WALKER),
    HEAD_SCHEMES             = int(mu::engraving::NoteHeadScheme::HEAD_SCHEMES),
};
Q_ENUM_NS(NoteHeadScheme);

enum class NoteHeadGroup {
    HEAD_NORMAL          = int(mu::engraving::NoteHeadGroup::HEAD_NORMAL),
    HEAD_CROSS           = int(mu::engraving::NoteHeadGroup::HEAD_CROSS),
    HEAD_PLUS            = int(mu::engraving::NoteHeadGroup::HEAD_PLUS),
    HEAD_XCIRCLE         = int(mu::engraving::NoteHeadGroup::HEAD_XCIRCLE),
    HEAD_WITHX           = int(mu::engraving::NoteHeadGroup::HEAD_WITHX),
    HEAD_TRIANGLE_UP     = int(mu::engraving::NoteHeadGroup::HEAD_TRIANGLE_UP),
    HEAD_TRIANGLE_DOWN   = int(mu::engraving::NoteHeadGroup::HEAD_TRIANGLE_DOWN),
    HEAD_SLASHED1        = int(mu::engraving::NoteHeadGroup::HEAD_SLASHED1),
    HEAD_SLASHED2        = int(mu::engraving::NoteHeadGroup::HEAD_SLASHED2),
    HEAD_DIAMOND         = int(mu::engraving::NoteHeadGroup::HEAD_DIAMOND),
    HEAD_DIAMOND_OLD     = int(mu::engraving::NoteHeadGroup::HEAD_DIAMOND_OLD),
    HEAD_CIRCLED         = int(mu::engraving::NoteHeadGroup::HEAD_CIRCLED),
    HEAD_CIRCLED_LARGE   = int(mu::engraving::NoteHeadGroup::HEAD_CIRCLED_LARGE),
    HEAD_LARGE_ARROW     = int(mu::engraving::NoteHeadGroup::HEAD_LARGE_ARROW),
    HEAD_BREVIS_ALT      = int(mu::engraving::NoteHeadGroup::HEAD_BREVIS_ALT),

    HEAD_SLASH           = int(mu::engraving::NoteHeadGroup::HEAD_SLASH),
    HEAD_LARGE_DIAMOND   = int(mu::engraving::NoteHeadGroup::HEAD_LARGE_DIAMOND),

    HEAD_SOL             = int(mu::engraving::NoteHeadGroup::HEAD_SOL),
    HEAD_LA              = int(mu::engraving::NoteHeadGroup::HEAD_LA),
    HEAD_FA              = int(mu::engraving::NoteHeadGroup::HEAD_FA),
    HEAD_MI              = int(mu::engraving::NoteHeadGroup::HEAD_MI),
    HEAD_DO              = int(mu::engraving::NoteHeadGroup::HEAD_DO),
    HEAD_RE              = int(mu::engraving::NoteHeadGroup::HEAD_RE),
    HEAD_TI              = int(mu::engraving::NoteHeadGroup::HEAD_TI),

    HEAD_HEAVY_CROSS     = int(mu::engraving::NoteHeadGroup::HEAD_HEAVY_CROSS),
    HEAD_HEAVY_CROSS_HAT = int(mu::engraving::NoteHeadGroup::HEAD_HEAVY_CROSS_HAT)
};
Q_ENUM_NS(NoteHeadGroup);

enum class NoteType {
    ///.\{
    NORMAL        = (int)(mu::engraving::NoteType::NORMAL),
    ACCIACCATURA  = (int)(mu::engraving::NoteType::ACCIACCATURA),
    APPOGGIATURA  = (int)(mu::engraving::NoteType::APPOGGIATURA),
    GRACE4        = (int)(mu::engraving::NoteType::GRACE4),
    GRACE16       = (int)(mu::engraving::NoteType::GRACE16),
    GRACE32       = (int)(mu::engraving::NoteType::GRACE32),
    GRACE8_AFTER  = (int)(mu::engraving::NoteType::GRACE8_AFTER),
    GRACE16_AFTER = (int)(mu::engraving::NoteType::GRACE16_AFTER),
    GRACE32_AFTER = (int)(mu::engraving::NoteType::GRACE32_AFTER),
    INVALID       = (int)(mu::engraving::NoteType::INVALID)
                    ///\}
};
Q_ENUM_NS(NoteType);

//---------------------------------------------------------
//   PlayEventType
/// Determines whether ornaments are automatically generated
/// when playing a score and whether the PlayEvents are saved
/// in the score file.
//---------------------------------------------------------

enum class PlayEventType {
    ///.\{
    Auto = int(mu::engraving::PlayEventType::Auto), ///< Play events for all notes are calculated by MuseScore.
    User = int(mu::engraving::PlayEventType::User), ///< Some play events are modified by user. Those events are written into the mscx file.
    ///.\}
};
Q_ENUM_NS(PlayEventType);

enum class SegmentType {
    ///.\{
    Invalid                    = int(mu::engraving::SegmentType::Invalid),
    BeginBarLine               = int(mu::engraving::SegmentType::BeginBarLine),
    HeaderClef                 = int(mu::engraving::SegmentType::HeaderClef),
    KeySig                     = int(mu::engraving::SegmentType::KeySig),
    Ambitus                    = int(mu::engraving::SegmentType::Ambitus),
    TimeSig                    = int(mu::engraving::SegmentType::TimeSig),
    StartRepeatBarLine         = int(mu::engraving::SegmentType::StartRepeatBarLine),
    ClefStartRepeatAnnounce    = int(mu::engraving::SegmentType::ClefStartRepeatAnnounce),
    KeySigStartRepeatAnnounce  = int(mu::engraving::SegmentType::KeySigStartRepeatAnnounce),
    TimeSigStartRepeatAnnounce = int(mu::engraving::SegmentType::TimeSigStartRepeatAnnounce),
    Clef                       = int(mu::engraving::SegmentType::Clef),
    BarLine                    = int(mu::engraving::SegmentType::BarLine),
    Breath                     = int(mu::engraving::SegmentType::Breath),
    //--
    TimeTick                   = int(mu::engraving::SegmentType::TimeTick),
    ChordRest                  = int(mu::engraving::SegmentType::ChordRest),
    //--
    ClefRepeatAnnounce         = int(mu::engraving::SegmentType::ClefRepeatAnnounce),
    KeySigRepeatAnnounce       = int(mu::engraving::SegmentType::KeySigRepeatAnnounce),
    TimeSigRepeatAnnounce      = int(mu::engraving::SegmentType::TimeSigRepeatAnnounce),
    //--
    EndBarLine                 = int(mu::engraving::SegmentType::EndBarLine),
    KeySigAnnounce             = int(mu::engraving::SegmentType::KeySigAnnounce),
    TimeSigAnnounce            = int(mu::engraving::SegmentType::TimeSigAnnounce),
    //--
    All                        = int(mu::engraving::SegmentType::All),
    /// Alias for `BeginBarLine | StartRepeatBarLine | BarLine | EndBarLine`
    BarLineType                = int(mu::engraving::SegmentType::BarLineType),
    CourtesyTimeSigType        = int(mu::engraving::SegmentType::CourtesyTimeSigType),
    CourtesyKeySigType         = int(mu::engraving::SegmentType::CourtesyKeySigType),
    CourtesyClefType           = int(mu::engraving::SegmentType::CourtesyClefType),
    TimeSigType                = int(mu::engraving::SegmentType::TimeSigType),
    KeySigType                 = int(mu::engraving::SegmentType::KeySigType),
    ClefType                   = int(mu::engraving::SegmentType::ClefType),
    ///\}
};
Q_ENUM_NS(SegmentType);

enum class BarLineType {
    NORMAL            = int(mu::engraving::BarLineType::NORMAL),
    SINGLE            = int(mu::engraving::BarLineType::SINGLE),
    DOUBLE            = int(mu::engraving::BarLineType::DOUBLE),
    START_REPEAT      = int(mu::engraving::BarLineType::START_REPEAT),
    LEFT_REPEAT       = int(mu::engraving::BarLineType::LEFT_REPEAT),
    END_REPEAT        = int(mu::engraving::BarLineType::END_REPEAT),
    RIGHT_REPEAT      = int(mu::engraving::BarLineType::RIGHT_REPEAT),
    BROKEN            = int(mu::engraving::BarLineType::BROKEN),
    DASHED            = int(mu::engraving::BarLineType::DASHED),
    END               = int(mu::engraving::BarLineType::END),
    FINAL             = int(mu::engraving::BarLineType::FINAL),
    END_START_REPEAT  = int(mu::engraving::BarLineType::END_START_REPEAT),
    LEFT_RIGHT_REPEAT = int(mu::engraving::BarLineType::LEFT_RIGHT_REPEAT),
    DOTTED            = int(mu::engraving::BarLineType::DOTTED),
    REVERSE_END       = int(mu::engraving::BarLineType::REVERSE_END),
    REVERSE_FINALE    = int(mu::engraving::BarLineType::REVERSE_FINALE),
    HEAVY             = int(mu::engraving::BarLineType::HEAVY),
    DOUBLE_HEAVY      = int(mu::engraving::BarLineType::DOUBLE_HEAVY),
};
Q_ENUM_NS(BarLineType);

enum class Tid {
    DEFAULT            = int(mu::engraving::TextStyleType::DEFAULT),

    /// Page-oriented styles
    TITLE              = int(mu::engraving::TextStyleType::TITLE),
    SUBTITLE           = int(mu::engraving::TextStyleType::SUBTITLE),
    COMPOSER           = int(mu::engraving::TextStyleType::COMPOSER),
    POET               = int(mu::engraving::TextStyleType::LYRICIST),
    TRANSLATOR         = int(mu::engraving::TextStyleType::TRANSLATOR),
    FRAME              = int(mu::engraving::TextStyleType::FRAME),
    INSTRUMENT_EXCERPT = int(mu::engraving::TextStyleType::INSTRUMENT_EXCERPT),
    INSTRUMENT_LONG    = int(mu::engraving::TextStyleType::INSTRUMENT_LONG),
    INSTRUMENT_SHORT   = int(mu::engraving::TextStyleType::INSTRUMENT_SHORT),
    INSTRUMENT_CHANGE  = int(mu::engraving::TextStyleType::INSTRUMENT_CHANGE),
    HEADER             = int(mu::engraving::TextStyleType::HEADER),
    FOOTER             = int(mu::engraving::TextStyleType::FOOTER),
    COPYRIGHT          = int(mu::engraving::TextStyleType::COPYRIGHT),
    PAGE_NUMBER        = int(mu::engraving::TextStyleType::PAGE_NUMBER),

    /// Measure-oriented styles
    MEASURE_NUMBER           = int(mu::engraving::TextStyleType::MEASURE_NUMBER),
    MEASURE_NUMBER_ALTERNATE = int(mu::engraving::TextStyleType::MEASURE_NUMBER_ALTERNATE),
    MMREST_RANGE             = int(mu::engraving::TextStyleType::MMREST_RANGE),

    /// System-level styles
    TEMPO             = int(mu::engraving::TextStyleType::TEMPO),
    TEMPO_CHANGE      = int(mu::engraving::TextStyleType::TEMPO_CHANGE),
    METRONOME         = int(mu::engraving::TextStyleType::METRONOME),
    REPEAT_PLAY_COUNT = int(mu::engraving::TextStyleType::REPEAT_PLAY_COUNT),
    REPEAT_LEFT       = int(mu::engraving::TextStyleType::REPEAT_LEFT),       // align to start of measure
    REPEAT_RIGHT      = int(mu::engraving::TextStyleType::REPEAT_RIGHT),      // align to end of measure
    REHEARSAL_MARK    = int(mu::engraving::TextStyleType::REHEARSAL_MARK),
    SYSTEM            = int(mu::engraving::TextStyleType::SYSTEM),

    /// Staff oriented styles
    STAFF             = int(mu::engraving::TextStyleType::STAFF),
    EXPRESSION        = int(mu::engraving::TextStyleType::EXPRESSION),
    DYNAMICS          = int(mu::engraving::TextStyleType::DYNAMICS),
    HAIRPIN           = int(mu::engraving::TextStyleType::HAIRPIN),
    LYRICS_ODD        = int(mu::engraving::TextStyleType::LYRICS_ODD),
    LYRICS_EVEN       = int(mu::engraving::TextStyleType::LYRICS_EVEN),
    HARMONY_A         = int(mu::engraving::TextStyleType::HARMONY_A),
    HARMONY_B         = int(mu::engraving::TextStyleType::HARMONY_B),
    HARMONY_ROMAN     = int(mu::engraving::TextStyleType::HARMONY_ROMAN),
    HARMONY_NASHVILLE = int(mu::engraving::TextStyleType::HARMONY_NASHVILLE),

    /// Note oriented styles
    TUPLET                   = int(mu::engraving::TextStyleType::TUPLET),
    ARTICULATION             = int(mu::engraving::TextStyleType::ARTICULATION),
    STICKING                 = int(mu::engraving::TextStyleType::STICKING),
    FINGERING                = int(mu::engraving::TextStyleType::FINGERING),
    LH_GUITAR_FINGERING      = int(mu::engraving::TextStyleType::LH_GUITAR_FINGERING),
    RH_GUITAR_FINGERING      = int(mu::engraving::TextStyleType::RH_GUITAR_FINGERING),
    HAMMER_ON_PULL_OFF       = int(mu::engraving::TextStyleType::HAMMER_ON_PULL_OFF),
    STRING_NUMBER            = int(mu::engraving::TextStyleType::STRING_NUMBER),
    STRING_TUNINGS           = int(mu::engraving::TextStyleType::STRING_TUNINGS),
    FRET_DIAGRAM_FINGERING   = int(mu::engraving::TextStyleType::FRET_DIAGRAM_FINGERING),
    FRET_DIAGRAM_FRET_NUMBER = int(mu::engraving::TextStyleType::FRET_DIAGRAM_FRET_NUMBER),
    HARP_PEDAL_DIAGRAM       = int(mu::engraving::TextStyleType::HARP_PEDAL_DIAGRAM),
    HARP_PEDAL_TEXT_DIAGRAM  = int(mu::engraving::TextStyleType::HARP_PEDAL_TEXT_DIAGRAM),

    /// Line-oriented styles
    TEXTLINE  = int(mu::engraving::TextStyleType::TEXTLINE),
    NOTELINE  = int(mu::engraving::TextStyleType::NOTELINE),
    VOLTA     = int(mu::engraving::TextStyleType::VOLTA),
    OTTAVA    = int(mu::engraving::TextStyleType::OTTAVA),
    GLISSANDO = int(mu::engraving::TextStyleType::GLISSANDO),
    PEDAL     = int(mu::engraving::TextStyleType::PEDAL),
    BEND      = int(mu::engraving::TextStyleType::BEND),
    LET_RING  = int(mu::engraving::TextStyleType::LET_RING),
    PALM_MUTE = int(mu::engraving::TextStyleType::PALM_MUTE),

    /// User styles
    USER1  = int(mu::engraving::TextStyleType::USER1),
    USER2  = int(mu::engraving::TextStyleType::USER2),
    USER3  = int(mu::engraving::TextStyleType::USER3),
    USER4  = int(mu::engraving::TextStyleType::USER4),
    USER5  = int(mu::engraving::TextStyleType::USER5),
    USER6  = int(mu::engraving::TextStyleType::USER6),
    USER7  = int(mu::engraving::TextStyleType::USER7),
    USER8  = int(mu::engraving::TextStyleType::USER8),
    USER9  = int(mu::engraving::TextStyleType::USER9),
    USER10 = int(mu::engraving::TextStyleType::USER10),
    USER11 = int(mu::engraving::TextStyleType::USER11),
    USER12 = int(mu::engraving::TextStyleType::USER12),

    /// special, no-contents, styles used while importing older scores
    TEXT_TYPES    = int(mu::engraving::TextStyleType::TEXT_TYPES),        // used for user-defined types
    IGNORED_TYPES = int(mu::engraving::TextStyleType::IGNORED_TYPES),     // used for types no longer relevant (mainly Figured bass text type)
};
Q_ENUM_NS(Tid);

enum class Syllabic {
    SINGLE = int(mu::engraving::LyricsSyllabic::SINGLE),
    BEGIN  = int(mu::engraving::LyricsSyllabic::BEGIN),
    END    = int(mu::engraving::LyricsSyllabic::END),
    MIDDLE = int(mu::engraving::LyricsSyllabic::MIDDLE),
};
Q_ENUM_NS(Syllabic);

enum class Anchor {
    SEGMENT = int(mu::engraving::Spanner::Anchor::SEGMENT),
    MEASURE = int(mu::engraving::Spanner::Anchor::MEASURE),
    CHORD   = int(mu::engraving::Spanner::Anchor::CHORD),
    NOTE    = int(mu::engraving::Spanner::Anchor::NOTE),
};
Q_ENUM_NS(Anchor);

enum class MMRestRangeBracketType {
    BRACKETS    = int(mu::engraving::MMRestRangeBracketType::BRACKETS),
    PARENTHESES = int(mu::engraving::MMRestRangeBracketType::PARENTHESES),
    NONE        = int(mu::engraving::MMRestRangeBracketType::NONE),
};
Q_ENUM_NS(MMRestRangeBracketType);

enum class TupletNumberType {
    SHOW_NUMBER   = int(mu::engraving::TupletNumberType::SHOW_NUMBER),
    SHOW_RELATION = int(mu::engraving::TupletNumberType::SHOW_RELATION),
    NO_TEXT       = int(mu::engraving::TupletNumberType::NO_TEXT),
};
Q_ENUM_NS(TupletNumberType);

enum class TupletBracketType {
    AUTO_BRACKET    = int(mu::engraving::TupletBracketType::AUTO_BRACKET),
    SHOW_BRACKET    = int(mu::engraving::TupletBracketType::SHOW_BRACKET),
    SHOW_NO_BRACKET = int(mu::engraving::TupletBracketType::SHOW_NO_BRACKET),
};
Q_ENUM_NS(TupletBracketType);

enum class TripletFeelType {
    NONE          = int(mu::engraving::TripletFeelType::NONE),
    TRIPLET_8TH   = int(mu::engraving::TripletFeelType::TRIPLET_8TH),
    TRIPLET_16TH  = int(mu::engraving::TripletFeelType::TRIPLET_16TH),
    DOTTED_8TH    = int(mu::engraving::TripletFeelType::DOTTED_8TH),
    DOTTED_16TH   = int(mu::engraving::TripletFeelType::DOTTED_16TH),
    SCOTTISH_8TH  = int(mu::engraving::TripletFeelType::SCOTTISH_8TH),
    SCOTTISH_16TH = int(mu::engraving::TripletFeelType::SCOTTISH_16TH),
};
Q_ENUM_NS(TripletFeelType);

enum class GuitarBendType {
    BEND            = int(mu::engraving::GuitarBendType::BEND),
    PRE_BEND        = int(mu::engraving::GuitarBendType::PRE_BEND),
    GRACE_NOTE_BEND = int(mu::engraving::GuitarBendType::GRACE_NOTE_BEND),
    SLIGHT_BEND     = int(mu::engraving::GuitarBendType::SLIGHT_BEND),
};
Q_ENUM_NS(GuitarBendType);

enum class GuitarBendShowHoldLine {
    AUTO = int(mu::engraving::GuitarBendShowHoldLine::AUTO),
    SHOW = int(mu::engraving::GuitarBendShowHoldLine::SHOW),
    HIDE = int(mu::engraving::GuitarBendShowHoldLine::HIDE),
};
Q_ENUM_NS(GuitarBendShowHoldLine);

enum class ClefType {
    INVALID    = int(mu::engraving::ClefType::INVALID),
    G          = int(mu::engraving::ClefType::G),
    G15_MB     = int(mu::engraving::ClefType::G15_MB),
    G8_VB      = int(mu::engraving::ClefType::G8_VB),
    G8_VA      = int(mu::engraving::ClefType::G8_VA),
    G15_MA     = int(mu::engraving::ClefType::G15_MA),
    G8_VB_O    = int(mu::engraving::ClefType::G8_VB_O),
    G8_VB_P    = int(mu::engraving::ClefType::G8_VB_P),
    G_1        = int(mu::engraving::ClefType::G_1),
    C1         = int(mu::engraving::ClefType::C1),
    C2         = int(mu::engraving::ClefType::C2),
    C3         = int(mu::engraving::ClefType::C3),
    C4         = int(mu::engraving::ClefType::C4),
    C5         = int(mu::engraving::ClefType::C5),
    C_19C      = int(mu::engraving::ClefType::C_19C),
    C1_F18C    = int(mu::engraving::ClefType::C1_F18C),
    C3_F18C    = int(mu::engraving::ClefType::C3_F18C),
    C4_F18C    = int(mu::engraving::ClefType::C4_F18C),
    C1_F20C    = int(mu::engraving::ClefType::C1_F20C),
    C3_F20C    = int(mu::engraving::ClefType::C3_F20C),
    C4_F20C    = int(mu::engraving::ClefType::C4_F20C),
    F          = int(mu::engraving::ClefType::F),
    F15_MB     = int(mu::engraving::ClefType::F15_MB),
    F8_VB      = int(mu::engraving::ClefType::F8_VB),
    F_8VA      = int(mu::engraving::ClefType::F_8VA),
    F_15MA     = int(mu::engraving::ClefType::F_15MA),
    F_B        = int(mu::engraving::ClefType::F_B),
    F_C        = int(mu::engraving::ClefType::F_C),
    F_F18C     = int(mu::engraving::ClefType::F_F18C),
    F_19C      = int(mu::engraving::ClefType::F_19C),
    PERC       = int(mu::engraving::ClefType::PERC),
    PERC2      = int(mu::engraving::ClefType::PERC2),
    TAB        = int(mu::engraving::ClefType::TAB),
    TAB4       = int(mu::engraving::ClefType::TAB4),
    TAB_SERIF  = int(mu::engraving::ClefType::TAB_SERIF),
    TAB4_SERIF = int(mu::engraving::ClefType::TAB4_SERIF),
    C4_8VB     = int(mu::engraving::ClefType::C4_8VB),
    G8_VB_C    = int(mu::engraving::ClefType::G8_VB_C),
    MAX        = int(mu::engraving::ClefType::MAX),
};
Q_ENUM_NS(ClefType);

enum class ClefToBarlinePosition {
    AUTO   = int(mu::engraving::ClefToBarlinePosition::AUTO),
    BEFORE = int(mu::engraving::ClefToBarlinePosition::BEFORE),
    AFTER  = int(mu::engraving::ClefToBarlinePosition::AFTER),
};
Q_ENUM_NS(ClefToBarlinePosition);

enum class DynamicType {
    OTHER  = int(mu::engraving::DynamicType::OTHER),
    PPPPPP = int(mu::engraving::DynamicType::PPPPPP),
    PPPPP  = int(mu::engraving::DynamicType::PPPPP),
    PPPP   = int(mu::engraving::DynamicType::PPPP),
    PPP    = int(mu::engraving::DynamicType::PPP),
    PP     = int(mu::engraving::DynamicType::PP),
    P      = int(mu::engraving::DynamicType::P),
    MP     = int(mu::engraving::DynamicType::MP),
    MF     = int(mu::engraving::DynamicType::MF),
    F      = int(mu::engraving::DynamicType::F),
    FF     = int(mu::engraving::DynamicType::FF),
    FFF    = int(mu::engraving::DynamicType::FFF),
    FFFF   = int(mu::engraving::DynamicType::FFFF),
    FFFFF  = int(mu::engraving::DynamicType::FFFFF),
    FFFFFF = int(mu::engraving::DynamicType::FFFFFF),
    FP     = int(mu::engraving::DynamicType::FP),
    PF     = int(mu::engraving::DynamicType::PF),
    SF     = int(mu::engraving::DynamicType::SF),
    SFZ    = int(mu::engraving::DynamicType::SFZ),
    SFF    = int(mu::engraving::DynamicType::SFF),
    SFFZ   = int(mu::engraving::DynamicType::SFFZ),
    SFFF   = int(mu::engraving::DynamicType::SFFF),
    SFFFZ  = int(mu::engraving::DynamicType::SFFFZ),
    SFP    = int(mu::engraving::DynamicType::SFP),
    SFPP   = int(mu::engraving::DynamicType::SFPP),
    RFZ    = int(mu::engraving::DynamicType::RFZ),
    RF     = int(mu::engraving::DynamicType::RF),
    FZ     = int(mu::engraving::DynamicType::FZ),
    M      = int(mu::engraving::DynamicType::M),
    R      = int(mu::engraving::DynamicType::R),
    S      = int(mu::engraving::DynamicType::S),
    Z      = int(mu::engraving::DynamicType::Z),
    N      = int(mu::engraving::DynamicType::N),
    LAST   = int(mu::engraving::DynamicType::LAST),
};
Q_ENUM_NS(DynamicType);

enum class DynamicSpeed {
    SLOW   = int(mu::engraving::DynamicSpeed::SLOW),
    NORMAL = int(mu::engraving::DynamicSpeed::NORMAL),
    FAST   = int(mu::engraving::DynamicSpeed::FAST),
};
Q_ENUM_NS(DynamicSpeed);

enum class LineType {
    SOLID  = int(mu::engraving::LineType::SOLID),
    DASHED = int(mu::engraving::LineType::DASHED),
    DOTTED = int(mu::engraving::LineType::DOTTED),
};
Q_ENUM_NS(LineType);

enum class HookType {
    NONE     = int(mu::engraving::HookType::NONE),
    HOOK_90  = int(mu::engraving::HookType::HOOK_90),
    HOOK_45  = int(mu::engraving::HookType::HOOK_45),
    HOOK_90T = int(mu::engraving::HookType::HOOK_90T),
};
Q_ENUM_NS(HookType);

enum class KeyMode {
    UNKNOWN    = int(mu::engraving::KeyMode::UNKNOWN),
    NONE       = int(mu::engraving::KeyMode::NONE),
    MAJOR      = int(mu::engraving::KeyMode::MAJOR),
    MINOR      = int(mu::engraving::KeyMode::MINOR),
    DORIAN     = int(mu::engraving::KeyMode::DORIAN),
    PHRYGIAN   = int(mu::engraving::KeyMode::PHRYGIAN),
    LYDIAN     = int(mu::engraving::KeyMode::LYDIAN),
    MIXOLYDIAN = int(mu::engraving::KeyMode::MIXOLYDIAN),
    AEOLIAN    = int(mu::engraving::KeyMode::AEOLIAN),
    IONIAN     = int(mu::engraving::KeyMode::IONIAN),
    LOCRIAN    = int(mu::engraving::KeyMode::LOCRIAN),
};
Q_ENUM_NS(KeyMode);

enum class ArpeggioType {
    NORMAL        = int(mu::engraving::ArpeggioType::NORMAL),
    UP            = int(mu::engraving::ArpeggioType::UP),
    DOWN          = int(mu::engraving::ArpeggioType::DOWN),
    BRACKET       = int(mu::engraving::ArpeggioType::BRACKET),
    UP_STRAIGHT   = int(mu::engraving::ArpeggioType::UP_STRAIGHT),
    DOWN_STRAIGHT = int(mu::engraving::ArpeggioType::DOWN_STRAIGHT),
};
Q_ENUM_NS(ArpeggioType);

enum class IntervalStep {
    UNISON  = int(mu::engraving::IntervalStep::UNISON),
    SECOND  = int(mu::engraving::IntervalStep::SECOND),
    THIRD   = int(mu::engraving::IntervalStep::THIRD),
    FOURTH  = int(mu::engraving::IntervalStep::FOURTH),
    FIFTH   = int(mu::engraving::IntervalStep::FIFTH),
    SIXTH   = int(mu::engraving::IntervalStep::SIXTH),
    SEVENTH = int(mu::engraving::IntervalStep::SEVENTH),
    OCTAVE  = int(mu::engraving::IntervalStep::OCTAVE),
};
Q_ENUM_NS(IntervalStep);

enum class IntervalType {
    AUTO       = int(mu::engraving::IntervalType::AUTO),
    AUGMENTED  = int(mu::engraving::IntervalType::AUGMENTED),
    MAJOR      = int(mu::engraving::IntervalType::MAJOR),
    PERFECT    = int(mu::engraving::IntervalType::PERFECT),
    MINOR      = int(mu::engraving::IntervalType::MINOR),
    DIMINISHED = int(mu::engraving::IntervalType::DIMINISHED),
};
Q_ENUM_NS(IntervalType);

enum class InstrumentLabelVisibility {
    LONG  = int(mu::engraving::InstrumentLabelVisibility::LONG),
    SHORT = int(mu::engraving::InstrumentLabelVisibility::SHORT),
    HIDE  = int(mu::engraving::InstrumentLabelVisibility::HIDE),
};
Q_ENUM_NS(InstrumentLabelVisibility);

enum class OrnamentShowAccidental {
    DEFAULT        = int(mu::engraving::OrnamentShowAccidental::DEFAULT),
    ANY_ALTERATION = int(mu::engraving::OrnamentShowAccidental::ANY_ALTERATION),
    ALWAYS         = int(mu::engraving::OrnamentShowAccidental::ALWAYS),
};
Q_ENUM_NS(OrnamentShowAccidental);

enum class PartialSpannerDirection {
    NONE     = int(mu::engraving::PartialSpannerDirection::NONE),
    INCOMING = int(mu::engraving::PartialSpannerDirection::INCOMING),
    OUTGOING = int(mu::engraving::PartialSpannerDirection::OUTGOING),
    BOTH     = int(mu::engraving::PartialSpannerDirection::BOTH),
};
Q_ENUM_NS(PartialSpannerDirection);

enum class ChordStylePreset {
    STANDARD = int(mu::engraving::ChordStylePreset::STANDARD),
    JAZZ     = int(mu::engraving::ChordStylePreset::JAZZ),
    LEGACY   = int(mu::engraving::ChordStylePreset::LEGACY),
    CUSTOM   = int(mu::engraving::ChordStylePreset::CUSTOM),
};
Q_ENUM_NS(ChordStylePreset);

enum class AnnotationCategory {
    Undefined         = int(mu::engraving::AnnotationCategory::Undefined),
    TempoAnnotation   = int(mu::engraving::AnnotationCategory::TempoAnnotation),
    PlayingAnnotation = int(mu::engraving::AnnotationCategory::PlayingAnnotation),
    Other             = int(mu::engraving::AnnotationCategory::Other),
};
Q_ENUM_NS(AnnotationCategory);

enum class PlayingTechniqueType {
    Undefined          = int(mu::engraving::PlayingTechniqueType::Undefined),
    Natural            = int(mu::engraving::PlayingTechniqueType::Natural),
    Pizzicato          = int(mu::engraving::PlayingTechniqueType::Pizzicato),
    Open               = int(mu::engraving::PlayingTechniqueType::Open),
    Mute               = int(mu::engraving::PlayingTechniqueType::Mute),
    Tremolo            = int(mu::engraving::PlayingTechniqueType::Tremolo),
    Detache            = int(mu::engraving::PlayingTechniqueType::Detache),
    Martele            = int(mu::engraving::PlayingTechniqueType::Martele),
    ColLegno           = int(mu::engraving::PlayingTechniqueType::ColLegno),
    SulPonticello      = int(mu::engraving::PlayingTechniqueType::SulPonticello),
    SulTasto           = int(mu::engraving::PlayingTechniqueType::SulTasto),
    Vibrato            = int(mu::engraving::PlayingTechniqueType::Vibrato),
    Legato             = int(mu::engraving::PlayingTechniqueType::Legato),
    Distortion         = int(mu::engraving::PlayingTechniqueType::Distortion),
    Overdrive          = int(mu::engraving::PlayingTechniqueType::Overdrive),
    Harmonics          = int(mu::engraving::PlayingTechniqueType::Harmonics),
    JazzTone           = int(mu::engraving::PlayingTechniqueType::JazzTone),
    HandbellsSwing     = int(mu::engraving::PlayingTechniqueType::HandbellsSwing),
    HandbellsSwingUp   = int(mu::engraving::PlayingTechniqueType::HandbellsSwingUp),
    HandbellsSwingDown = int(mu::engraving::PlayingTechniqueType::HandbellsSwingDown),
    HandbellsEcho1     = int(mu::engraving::PlayingTechniqueType::HandbellsEcho1),
    HandbellsEcho2     = int(mu::engraving::PlayingTechniqueType::HandbellsEcho2),
    HandbellsDamp      = int(mu::engraving::PlayingTechniqueType::HandbellsDamp),
    HandbellsLV        = int(mu::engraving::PlayingTechniqueType::HandbellsLV),
    HandbellsR         = int(mu::engraving::PlayingTechniqueType::HandbellsR),
};
Q_ENUM_NS(PlayingTechniqueType);

enum class GradualTempoChangeType {
    Undefined    = int(mu::engraving::GradualTempoChangeType::Undefined),
    Accelerando  = int(mu::engraving::GradualTempoChangeType::Accelerando),
    Allargando   = int(mu::engraving::GradualTempoChangeType::Allargando),
    Calando      = int(mu::engraving::GradualTempoChangeType::Calando),
    Lentando     = int(mu::engraving::GradualTempoChangeType::Lentando),
    Morendo      = int(mu::engraving::GradualTempoChangeType::Morendo),
    Precipitando = int(mu::engraving::GradualTempoChangeType::Precipitando),
    Rallentando  = int(mu::engraving::GradualTempoChangeType::Rallentando),
    Ritardando   = int(mu::engraving::GradualTempoChangeType::Ritardando),
    Smorzando    = int(mu::engraving::GradualTempoChangeType::Smorzando),
    Sostenuto    = int(mu::engraving::GradualTempoChangeType::Sostenuto),
    Stringendo   = int(mu::engraving::GradualTempoChangeType::Stringendo),
};
Q_ENUM_NS(GradualTempoChangeType);

enum class ChangeMethod {
    NORMAL      = int(mu::engraving::ChangeMethod::NORMAL),
    EXPONENTIAL = int(mu::engraving::ChangeMethod::EXPONENTIAL),
    EASE_IN     = int(mu::engraving::ChangeMethod::EASE_IN),
    EASE_OUT    = int(mu::engraving::ChangeMethod::EASE_OUT),
    EASE_IN_OUT = int(mu::engraving::ChangeMethod::EASE_IN_OUT),
};
Q_ENUM_NS(ChangeMethod);

enum class ChangeDirection {
    INCREASING = int(mu::engraving::ChangeDirection::INCREASING),
    DECREASING = int(mu::engraving::ChangeDirection::DECREASING),
};
Q_ENUM_NS(ChangeDirection);

enum class AccidentalRole {
    AUTO = int(mu::engraving::AccidentalRole::AUTO),
    USER = int(mu::engraving::AccidentalRole::USER),
};
Q_ENUM_NS(AccidentalRole);

enum class AccidentalVal {
    SHARP3  = int(mu::engraving::AccidentalVal::SHARP3),
    SHARP2  = int(mu::engraving::AccidentalVal::SHARP2),
    SHARP   = int(mu::engraving::AccidentalVal::SHARP),
    NATURAL = int(mu::engraving::AccidentalVal::NATURAL),
    FLAT    = int(mu::engraving::AccidentalVal::FLAT),
    FLAT2   = int(mu::engraving::AccidentalVal::FLAT2),
    FLAT3   = int(mu::engraving::AccidentalVal::FLAT3),
    MIN     = int(mu::engraving::AccidentalVal::MIN),
    MAX     = int(mu::engraving::AccidentalVal::MAX),
};
Q_ENUM_NS(AccidentalVal);

enum class FermataType {
    Undefined  = int(mu::engraving::FermataType::Undefined),
    VeryShort  = int(mu::engraving::FermataType::VeryShort),
    Short      = int(mu::engraving::FermataType::Short),
    ShortHenze = int(mu::engraving::FermataType::ShortHenze),
    Normal     = int(mu::engraving::FermataType::Normal),
    Long       = int(mu::engraving::FermataType::Long),
    LongHenze  = int(mu::engraving::FermataType::LongHenze),
    VeryLong   = int(mu::engraving::FermataType::VeryLong),
};
Q_ENUM_NS(FermataType);

enum class ChordLineType {
    NOTYPE = int(mu::engraving::ChordLineType::NOTYPE),
    FALL   = int(mu::engraving::ChordLineType::FALL),
    DOIT   = int(mu::engraving::ChordLineType::DOIT),
    PLOP   = int(mu::engraving::ChordLineType::PLOP),
    SCOOP  = int(mu::engraving::ChordLineType::SCOOP),
};
Q_ENUM_NS(ChordLineType);

enum class SlurStyleType {
    Undefined  = int(mu::engraving::SlurStyleType::Undefined),
    Solid      = int(mu::engraving::SlurStyleType::Solid),
    Dotted     = int(mu::engraving::SlurStyleType::Dotted),
    Dashed     = int(mu::engraving::SlurStyleType::Dashed),
    WideDashed = int(mu::engraving::SlurStyleType::WideDashed),
};
Q_ENUM_NS(SlurStyleType);

enum class TremoloType {
    INVALID_TREMOLO = int(mu::engraving::TremoloType::INVALID_TREMOLO),

    /// single-chord tremolos
    R8        = int(mu::engraving::TremoloType::R8),
    R16       = int(mu::engraving::TremoloType::R16),
    R32       = int(mu::engraving::TremoloType::R32),
    R64       = int(mu::engraving::TremoloType::R64),
    BUZZ_ROLL = int(mu::engraving::TremoloType::BUZZ_ROLL),

    /// two-chord tremolos
    C8  = int(mu::engraving::TremoloType::C8),
    C16 = int(mu::engraving::TremoloType::C16),
    C32 = int(mu::engraving::TremoloType::C32),
    C64 = int(mu::engraving::TremoloType::C64),
};
Q_ENUM_NS(TremoloType);

// only applicable to minim two-note tremolo in non-TAB staves
enum class TremoloStyle {
    DEFAULT               = int(mu::engraving::TremoloStyle::DEFAULT),
    TRADITIONAL           = int(mu::engraving::TremoloStyle::TRADITIONAL),
    TRADITIONAL_ALTERNATE = int(mu::engraving::TremoloStyle::TRADITIONAL_ALTERNATE),
};
Q_ENUM_NS(TremoloStyle);

enum class TremoloChordType {
    TremoloNone        = int(mu::engraving::TremoloChordType::TremoloNone),
    TremoloSingle      = int(mu::engraving::TremoloChordType::TremoloSingle),
    TremoloFirstChord  = int(mu::engraving::TremoloChordType::TremoloFirstChord),
    TremoloSecondChord = int(mu::engraving::TremoloChordType::TremoloSecondChord),
};
Q_ENUM_NS(TremoloChordType);

enum class BracketType {
    NORMAL     = int(mu::engraving::BracketType::NORMAL),
    BRACE      = int(mu::engraving::BracketType::BRACE),
    SQUARE     = int(mu::engraving::BracketType::SQUARE),
    LINE       = int(mu::engraving::BracketType::LINE),
    NO_BRACKET = int(mu::engraving::BracketType::NO_BRACKET),
};
Q_ENUM_NS(BracketType);

enum class JumpType {
    DC             = int(mu::engraving::JumpType::DC),
    DC_AL_FINE     = int(mu::engraving::JumpType::DC_AL_FINE),
    DC_AL_CODA     = int(mu::engraving::JumpType::DC_AL_CODA),
    DS_AL_CODA     = int(mu::engraving::JumpType::DS_AL_CODA),
    DS_AL_FINE     = int(mu::engraving::JumpType::DS_AL_FINE),
    DS             = int(mu::engraving::JumpType::DS),
    DC_AL_DBLCODA  = int(mu::engraving::JumpType::DC_AL_DBLCODA),
    DS_AL_DBLCODA  = int(mu::engraving::JumpType::DS_AL_DBLCODA),
    DSS            = int(mu::engraving::JumpType::DSS),
    DSS_AL_CODA    = int(mu::engraving::JumpType::DSS_AL_CODA),
    DSS_AL_DBLCODA = int(mu::engraving::JumpType::DSS_AL_DBLCODA),
    DSS_AL_FINE    = int(mu::engraving::JumpType::DSS_AL_FINE),
    USER           = int(mu::engraving::JumpType::USER),
};
Q_ENUM_NS(JumpType);

enum class MarkerType {
    SEGNO      = int(mu::engraving::MarkerType::SEGNO),
    VARSEGNO   = int(mu::engraving::MarkerType::VARSEGNO),
    CODA       = int(mu::engraving::MarkerType::CODA),
    VARCODA    = int(mu::engraving::MarkerType::VARCODA),
    CODETTA    = int(mu::engraving::MarkerType::CODETTA),
    FINE       = int(mu::engraving::MarkerType::FINE),
    TOCODA     = int(mu::engraving::MarkerType::TOCODA),
    TOCODASYM  = int(mu::engraving::MarkerType::TOCODASYM),
    DA_CODA    = int(mu::engraving::MarkerType::DA_CODA),
    DA_DBLCODA = int(mu::engraving::MarkerType::DA_DBLCODA),
    USER       = int(mu::engraving::MarkerType::USER),
};
Q_ENUM_NS(MarkerType);

enum class MeasureNumberMode {
    AUTO = int(mu::engraving::MeasureNumberMode::AUTO),
    SHOW = int(mu::engraving::MeasureNumberMode::SHOW),
    HIDE = int(mu::engraving::MeasureNumberMode::HIDE),
};
Q_ENUM_NS(MeasureNumberMode);

enum class StaffGroup {
    STANDARD   = int(mu::engraving::StaffGroup::STANDARD),
    PERCUSSION = int(mu::engraving::StaffGroup::PERCUSSION),
    TAB        = int(mu::engraving::StaffGroup::TAB),
};
Q_ENUM_NS(StaffGroup);

enum class OttavaType {
    OTTAVA_8VA  = int(mu::engraving::OttavaType::OTTAVA_8VA),
    OTTAVA_8VB  = int(mu::engraving::OttavaType::OTTAVA_8VB),
    OTTAVA_15MA = int(mu::engraving::OttavaType::OTTAVA_15MA),
    OTTAVA_15MB = int(mu::engraving::OttavaType::OTTAVA_15MB),
    OTTAVA_22MA = int(mu::engraving::OttavaType::OTTAVA_22MA),
    OTTAVA_22MB = int(mu::engraving::OttavaType::OTTAVA_22MB),
};
Q_ENUM_NS(OttavaType);

enum class HairpinType {
    INVALID         = int(mu::engraving::HairpinType::INVALID),
    CRESC_HAIRPIN   = int(mu::engraving::HairpinType::CRESC_HAIRPIN),
    DIM_HAIRPIN     = int(mu::engraving::HairpinType::DIM_HAIRPIN),
    CRESC_LINE      = int(mu::engraving::HairpinType::CRESC_LINE),
    DIM_LINE        = int(mu::engraving::HairpinType::DIM_LINE),

    // obsolete
    DECRESC_HAIRPIN = int(mu::engraving::HairpinType::DIM_HAIRPIN),
    DECRESC_LINE    = int(mu::engraving::HairpinType::DIM_LINE)
};
Q_ENUM_NS(HairpinType);

enum class TrillType {
    TRILL_LINE      = int(mu::engraving::TrillType::TRILL_LINE),
    UPPRALL_LINE    = int(mu::engraving::TrillType::UPPRALL_LINE),
    DOWNPRALL_LINE  = int(mu::engraving::TrillType::DOWNPRALL_LINE),
    PRALLPRALL_LINE = int(mu::engraving::TrillType::PRALLPRALL_LINE),
};
Q_ENUM_NS(TrillType);

enum class VibratoType {
    GUITAR_VIBRATO        = int(mu::engraving::VibratoType::GUITAR_VIBRATO),
    GUITAR_VIBRATO_WIDE   = int(mu::engraving::VibratoType::GUITAR_VIBRATO_WIDE),
    VIBRATO_SAWTOOTH      = int(mu::engraving::VibratoType::VIBRATO_SAWTOOTH),
    VIBRATO_SAWTOOTH_WIDE = int(mu::engraving::VibratoType::VIBRATO_SAWTOOTH_WIDE),
};
Q_ENUM_NS(VibratoType);

enum class ArticulationTextType {
    NO_TEXT = int(mu::engraving::ArticulationTextType::NO_TEXT),
    SLAP    = int(mu::engraving::ArticulationTextType::SLAP),
    POP     = int(mu::engraving::ArticulationTextType::POP),
    TD      = int(mu::engraving::ArticulationTextType::TD),
    BD      = int(mu::engraving::ArticulationTextType::BD),
    RT      = int(mu::engraving::ArticulationTextType::RT),
    PL      = int(mu::engraving::ArticulationTextType::PL),
    SB      = int(mu::engraving::ArticulationTextType::SB),
    VIB     = int(mu::engraving::ArticulationTextType::VIB),
};
Q_ENUM_NS(ArticulationTextType);

enum class LyricsDashSystemStart {
    STANDARD         = int(mu::engraving::LyricsDashSystemStart::STANDARD),
    UNDER_HEADER     = int(mu::engraving::LyricsDashSystemStart::UNDER_HEADER),
    UNDER_FIRST_NOTE = int(mu::engraving::LyricsDashSystemStart::UNDER_FIRST_NOTE),
};
Q_ENUM_NS(LyricsDashSystemStart);

enum class NoteLineEndPlacement {
    LEFT_EDGE   = int(mu::engraving::NoteLineEndPlacement::LEFT_EDGE),
    OFFSET_ENDS = int(mu::engraving::NoteLineEndPlacement::OFFSET_ENDS),
};
Q_ENUM_NS(NoteLineEndPlacement);

enum class SpannerSegmentType {
    SINGLE = int(mu::engraving::SpannerSegmentType::SINGLE),
    BEGIN  = int(mu::engraving::SpannerSegmentType::BEGIN),
    MIDDLE = int(mu::engraving::SpannerSegmentType::MIDDLE),
    END    = int(mu::engraving::SpannerSegmentType::END),
};
Q_ENUM_NS(SpannerSegmentType);

enum class TiePlacement {
    AUTO    = int(mu::engraving::TiePlacement::AUTO),
    INSIDE  = int(mu::engraving::TiePlacement::INSIDE),
    OUTSIDE = int(mu::engraving::TiePlacement::OUTSIDE),
};
Q_ENUM_NS(TiePlacement);

enum class TieDotsPlacement {
    AUTO        = int(mu::engraving::TieDotsPlacement::AUTO),
    BEFORE_DOTS = int(mu::engraving::TieDotsPlacement::BEFORE_DOTS),
    AFTER_DOTS  = int(mu::engraving::TieDotsPlacement::AFTER_DOTS),
};
Q_ENUM_NS(TieDotsPlacement);

enum class TimeSigType {
    NORMAL     = int(mu::engraving::TimeSigType::NORMAL),
    FOUR_FOUR  = int(mu::engraving::TimeSigType::FOUR_FOUR),
    ALLA_BREVE = int(mu::engraving::TimeSigType::ALLA_BREVE),
    CUT_BACH   = int(mu::engraving::TimeSigType::CUT_BACH),
    CUT_TRIPLE = int(mu::engraving::TimeSigType::CUT_TRIPLE),
};
Q_ENUM_NS(TimeSigType);

enum class TimeSigPlacement {
    NORMAL        = int(mu::engraving::TimeSigPlacement::NORMAL),
    ABOVE_STAVES  = int(mu::engraving::TimeSigPlacement::ABOVE_STAVES),
    ACROSS_STAVES = int(mu::engraving::TimeSigPlacement::ACROSS_STAVES),
};
Q_ENUM_NS(TimeSigPlacement);

enum class TimeSigStyle {
    NORMAL = int(mu::engraving::TimeSigStyle::NORMAL),
    NARROW = int(mu::engraving::TimeSigStyle::NARROW),
    LARGE  = int(mu::engraving::TimeSigStyle::LARGE),
};
Q_ENUM_NS(TimeSigStyle);

enum class TimeSigVSMargin {
    HANG_INTO_MARGIN       = int(mu::engraving::TimeSigVSMargin::HANG_INTO_MARGIN),
    RIGHT_ALIGN_TO_BARLINE = int(mu::engraving::TimeSigVSMargin::RIGHT_ALIGN_TO_BARLINE),
    CREATE_SPACE           = int(mu::engraving::TimeSigVSMargin::CREATE_SPACE),
};
Q_ENUM_NS(TimeSigVSMargin);

enum class NoteSpellingType {
    STANDARD    = int(mu::engraving::NoteSpellingType::STANDARD),
    GERMAN      = int(mu::engraving::NoteSpellingType::GERMAN),
    GERMAN_PURE = int(mu::engraving::NoteSpellingType::GERMAN_PURE),
    SOLFEGGIO   = int(mu::engraving::NoteSpellingType::SOLFEGGIO),
    FRENCH      = int(mu::engraving::NoteSpellingType::FRENCH),
};
Q_ENUM_NS(NoteSpellingType);

enum class Key {
    C_B              = int(mu::engraving::Key::C_B),
    G_B              = int(mu::engraving::Key::G_B),
    D_B              = int(mu::engraving::Key::D_B),
    A_B              = int(mu::engraving::Key::A_B),
    E_B              = int(mu::engraving::Key::E_B),
    B_B              = int(mu::engraving::Key::B_B),
    F                = int(mu::engraving::Key::F),
    C                = int(mu::engraving::Key::C),
    G                = int(mu::engraving::Key::G),
    D                = int(mu::engraving::Key::D),
    A                = int(mu::engraving::Key::A),
    E                = int(mu::engraving::Key::E),
    B                = int(mu::engraving::Key::B),
    F_S              = int(mu::engraving::Key::F_S),
    C_S              = int(mu::engraving::Key::C_S),
    MIN              = int(mu::engraving::Key::MIN),
    MAX              = int(mu::engraving::Key::MAX),
    INVALID          = int(mu::engraving::Key::INVALID),
    NUM_OF           = int(mu::engraving::Key::NUM_OF),
    DELTA_ENHARMONIC = int(mu::engraving::Key::DELTA_ENHARMONIC),
};
Q_ENUM_NS(Key);

enum class UpdateMode {
    DoNothing = int(mu::engraving::UpdateMode::DoNothing),
    Update    = int(mu::engraving::UpdateMode::Update),
    UpdateAll = int(mu::engraving::UpdateMode::UpdateAll),
    Layout    = int(mu::engraving::UpdateMode::Layout),
};
Q_ENUM_NS(UpdateMode);

enum class LayoutFlag {
    NO_FLAGS             = int(mu::engraving::LayoutFlag::NO_FLAGS),
    PLAY_EVENTS          = int(mu::engraving::LayoutFlag::PLAY_EVENTS),
    REBUILD_MIDI_MAPPING = int(mu::engraving::LayoutFlag::REBUILD_MIDI_MAPPING),
};
Q_ENUM_NS(LayoutFlag);

enum class LayoutMode {
    PAGE             = int(mu::engraving::LayoutMode::PAGE),
    FLOAT            = int(mu::engraving::LayoutMode::FLOAT),
    LINE             = int(mu::engraving::LayoutMode::LINE),
    SYSTEM           = int(mu::engraving::LayoutMode::SYSTEM),
    HORIZONTAL_FIXED = int(mu::engraving::LayoutMode::HORIZONTAL_FIXED),
};
Q_ENUM_NS(LayoutMode);

enum class TappingHand {
    INVALID = int(mu::engraving::TappingHand::INVALID),
    LEFT    = int(mu::engraving::TappingHand::LEFT),
    RIGHT   = int(mu::engraving::TappingHand::RIGHT),
};
Q_ENUM_NS(TappingHand);

enum class LHTappingSymbol {
    DOT       = int(mu::engraving::LHTappingSymbol::DOT),
    CIRCLED_T = int(mu::engraving::LHTappingSymbol::CIRCLED_T),
};
Q_ENUM_NS(LHTappingSymbol);

enum class RHTappingSymbol {
    T    = int(mu::engraving::RHTappingSymbol::T),
    PLUS = int(mu::engraving::RHTappingSymbol::PLUS),
};
Q_ENUM_NS(RHTappingSymbol);

enum class LHTappingShowItems {
    HALF_SLUR = int(mu::engraving::LHTappingShowItems::HALF_SLUR),
    SYMBOL    = int(mu::engraving::LHTappingShowItems::SYMBOL),
    BOTH      = int(mu::engraving::LHTappingShowItems::BOTH),
};
Q_ENUM_NS(LHTappingShowItems);

enum class ParenthesesMode {
    NONE  = int(mu::engraving::ParenthesesMode::NONE),
    LEFT  = int(mu::engraving::ParenthesesMode::LEFT),
    RIGHT = int(mu::engraving::ParenthesesMode::RIGHT),
    BOTH  = int(mu::engraving::ParenthesesMode::BOTH),
};
Q_ENUM_NS(ParenthesesMode);

enum class RepeatPlayCountPreset {
    X_N          = int(mu::engraving::RepeatPlayCountPreset::X_N),
    N_X          = int(mu::engraving::RepeatPlayCountPreset::N_X),
    PLAY_N_TIMES = int(mu::engraving::RepeatPlayCountPreset::PLAY_N_TIMES),
    N_REPEATS    = int(mu::engraving::RepeatPlayCountPreset::N_REPEATS),
};
Q_ENUM_NS(RepeatPlayCountPreset);

enum class MeasureNumberPlacement {
    ABOVE_SYSTEM            = int(mu::engraving::MeasureNumberPlacement::ABOVE_SYSTEM),
    BELOW_SYSTEM            = int(mu::engraving::MeasureNumberPlacement::BELOW_SYSTEM),
    ON_SYSTEM_OBJECT_STAVES = int(mu::engraving::MeasureNumberPlacement::ON_SYSTEM_OBJECT_STAVES),
    ON_ALL_STAVES           = int(mu::engraving::MeasureNumberPlacement::ON_ALL_STAVES),
};
Q_ENUM_NS(MeasureNumberPlacement);

enum class SymId {
    noSym = int(mu::engraving::SymId::noSym),

    // SMuFL standard symbol IDs {{{
    fourStringTabClef = int(mu::engraving::SymId::fourStringTabClef),
    sixStringTabClef = int(mu::engraving::SymId::sixStringTabClef),
    accSagittal11LargeDiesisDown = int(mu::engraving::SymId::accSagittal11LargeDiesisDown),
    accSagittal11LargeDiesisUp = int(mu::engraving::SymId::accSagittal11LargeDiesisUp),
    accSagittal11MediumDiesisDown = int(mu::engraving::SymId::accSagittal11MediumDiesisDown),
    accSagittal11MediumDiesisUp = int(mu::engraving::SymId::accSagittal11MediumDiesisUp),
    accSagittal11v19LargeDiesisDown = int(mu::engraving::SymId::accSagittal11v19LargeDiesisDown),
    accSagittal11v19LargeDiesisUp = int(mu::engraving::SymId::accSagittal11v19LargeDiesisUp),
    accSagittal11v19MediumDiesisDown = int(mu::engraving::SymId::accSagittal11v19MediumDiesisDown),
    accSagittal11v19MediumDiesisUp = int(mu::engraving::SymId::accSagittal11v19MediumDiesisUp),
    accSagittal11v49CommaDown = int(mu::engraving::SymId::accSagittal11v49CommaDown),
    accSagittal11v49CommaUp = int(mu::engraving::SymId::accSagittal11v49CommaUp),
    accSagittal143CommaDown = int(mu::engraving::SymId::accSagittal143CommaDown),
    accSagittal143CommaUp = int(mu::engraving::SymId::accSagittal143CommaUp),
    accSagittal17CommaDown = int(mu::engraving::SymId::accSagittal17CommaDown),
    accSagittal17CommaUp = int(mu::engraving::SymId::accSagittal17CommaUp),
    accSagittal17KleismaDown = int(mu::engraving::SymId::accSagittal17KleismaDown),
    accSagittal17KleismaUp = int(mu::engraving::SymId::accSagittal17KleismaUp),
    accSagittal19CommaDown = int(mu::engraving::SymId::accSagittal19CommaDown),
    accSagittal19CommaUp = int(mu::engraving::SymId::accSagittal19CommaUp),
    accSagittal19SchismaDown = int(mu::engraving::SymId::accSagittal19SchismaDown),
    accSagittal19SchismaUp = int(mu::engraving::SymId::accSagittal19SchismaUp),
    accSagittal1MinaDown = int(mu::engraving::SymId::accSagittal1MinaDown),
    accSagittal1MinaUp = int(mu::engraving::SymId::accSagittal1MinaUp),
    accSagittal1TinaDown = int(mu::engraving::SymId::accSagittal1TinaDown),
    accSagittal1TinaUp = int(mu::engraving::SymId::accSagittal1TinaUp),
    accSagittal23CommaDown = int(mu::engraving::SymId::accSagittal23CommaDown),
    accSagittal23CommaUp = int(mu::engraving::SymId::accSagittal23CommaUp),
    accSagittal23SmallDiesisDown = int(mu::engraving::SymId::accSagittal23SmallDiesisDown),
    accSagittal23SmallDiesisUp = int(mu::engraving::SymId::accSagittal23SmallDiesisUp),
    accSagittal25SmallDiesisDown = int(mu::engraving::SymId::accSagittal25SmallDiesisDown),
    accSagittal25SmallDiesisUp = int(mu::engraving::SymId::accSagittal25SmallDiesisUp),
    accSagittal2MinasDown = int(mu::engraving::SymId::accSagittal2MinasDown),
    accSagittal2MinasUp = int(mu::engraving::SymId::accSagittal2MinasUp),
    accSagittal2TinasDown = int(mu::engraving::SymId::accSagittal2TinasDown),
    accSagittal2TinasUp = int(mu::engraving::SymId::accSagittal2TinasUp),
    accSagittal35LargeDiesisDown = int(mu::engraving::SymId::accSagittal35LargeDiesisDown),
    accSagittal35LargeDiesisUp = int(mu::engraving::SymId::accSagittal35LargeDiesisUp),
    accSagittal35MediumDiesisDown = int(mu::engraving::SymId::accSagittal35MediumDiesisDown),
    accSagittal35MediumDiesisUp = int(mu::engraving::SymId::accSagittal35MediumDiesisUp),
    accSagittal3TinasDown = int(mu::engraving::SymId::accSagittal3TinasDown),
    accSagittal3TinasUp = int(mu::engraving::SymId::accSagittal3TinasUp),
    accSagittal49LargeDiesisDown = int(mu::engraving::SymId::accSagittal49LargeDiesisDown),
    accSagittal49LargeDiesisUp = int(mu::engraving::SymId::accSagittal49LargeDiesisUp),
    accSagittal49MediumDiesisDown = int(mu::engraving::SymId::accSagittal49MediumDiesisDown),
    accSagittal49MediumDiesisUp = int(mu::engraving::SymId::accSagittal49MediumDiesisUp),
    accSagittal49SmallDiesisDown = int(mu::engraving::SymId::accSagittal49SmallDiesisDown),
    accSagittal49SmallDiesisUp = int(mu::engraving::SymId::accSagittal49SmallDiesisUp),
    accSagittal4TinasDown = int(mu::engraving::SymId::accSagittal4TinasDown),
    accSagittal4TinasUp = int(mu::engraving::SymId::accSagittal4TinasUp),
    accSagittal55CommaDown = int(mu::engraving::SymId::accSagittal55CommaDown),
    accSagittal55CommaUp = int(mu::engraving::SymId::accSagittal55CommaUp),
    accSagittal5CommaDown = int(mu::engraving::SymId::accSagittal5CommaDown),
    accSagittal5CommaUp = int(mu::engraving::SymId::accSagittal5CommaUp),
    accSagittal5TinasDown = int(mu::engraving::SymId::accSagittal5TinasDown),
    accSagittal5TinasUp = int(mu::engraving::SymId::accSagittal5TinasUp),
    accSagittal5v11SmallDiesisDown = int(mu::engraving::SymId::accSagittal5v11SmallDiesisDown),
    accSagittal5v11SmallDiesisUp = int(mu::engraving::SymId::accSagittal5v11SmallDiesisUp),
    accSagittal5v13LargeDiesisDown = int(mu::engraving::SymId::accSagittal5v13LargeDiesisDown),
    accSagittal5v13LargeDiesisUp = int(mu::engraving::SymId::accSagittal5v13LargeDiesisUp),
    accSagittal5v13MediumDiesisDown = int(mu::engraving::SymId::accSagittal5v13MediumDiesisDown),
    accSagittal5v13MediumDiesisUp = int(mu::engraving::SymId::accSagittal5v13MediumDiesisUp),
    accSagittal5v19CommaDown = int(mu::engraving::SymId::accSagittal5v19CommaDown),
    accSagittal5v19CommaUp = int(mu::engraving::SymId::accSagittal5v19CommaUp),
    accSagittal5v23SmallDiesisDown = int(mu::engraving::SymId::accSagittal5v23SmallDiesisDown),
    accSagittal5v23SmallDiesisUp = int(mu::engraving::SymId::accSagittal5v23SmallDiesisUp),
    accSagittal5v49MediumDiesisDown = int(mu::engraving::SymId::accSagittal5v49MediumDiesisDown),
    accSagittal5v49MediumDiesisUp = int(mu::engraving::SymId::accSagittal5v49MediumDiesisUp),
    accSagittal5v7KleismaDown = int(mu::engraving::SymId::accSagittal5v7KleismaDown),
    accSagittal5v7KleismaUp = int(mu::engraving::SymId::accSagittal5v7KleismaUp),
    accSagittal6TinasDown = int(mu::engraving::SymId::accSagittal6TinasDown),
    accSagittal6TinasUp = int(mu::engraving::SymId::accSagittal6TinasUp),
    accSagittal7CommaDown = int(mu::engraving::SymId::accSagittal7CommaDown),
    accSagittal7CommaUp = int(mu::engraving::SymId::accSagittal7CommaUp),
    accSagittal7TinasDown = int(mu::engraving::SymId::accSagittal7TinasDown),
    accSagittal7TinasUp = int(mu::engraving::SymId::accSagittal7TinasUp),
    accSagittal7v11CommaDown = int(mu::engraving::SymId::accSagittal7v11CommaDown),
    accSagittal7v11CommaUp = int(mu::engraving::SymId::accSagittal7v11CommaUp),
    accSagittal7v11KleismaDown = int(mu::engraving::SymId::accSagittal7v11KleismaDown),
    accSagittal7v11KleismaUp = int(mu::engraving::SymId::accSagittal7v11KleismaUp),
    accSagittal7v19CommaDown = int(mu::engraving::SymId::accSagittal7v19CommaDown),
    accSagittal7v19CommaUp = int(mu::engraving::SymId::accSagittal7v19CommaUp),
    accSagittal8TinasDown = int(mu::engraving::SymId::accSagittal8TinasDown),
    accSagittal8TinasUp = int(mu::engraving::SymId::accSagittal8TinasUp),
    accSagittal9TinasDown = int(mu::engraving::SymId::accSagittal9TinasDown),
    accSagittal9TinasUp = int(mu::engraving::SymId::accSagittal9TinasUp),
    accSagittalAcute = int(mu::engraving::SymId::accSagittalAcute),
    accSagittalDoubleFlat = int(mu::engraving::SymId::accSagittalDoubleFlat),
    accSagittalDoubleFlat11v49CUp = int(mu::engraving::SymId::accSagittalDoubleFlat11v49CUp),
    accSagittalDoubleFlat143CUp = int(mu::engraving::SymId::accSagittalDoubleFlat143CUp),
    accSagittalDoubleFlat17CUp = int(mu::engraving::SymId::accSagittalDoubleFlat17CUp),
    accSagittalDoubleFlat17kUp = int(mu::engraving::SymId::accSagittalDoubleFlat17kUp),
    accSagittalDoubleFlat19CUp = int(mu::engraving::SymId::accSagittalDoubleFlat19CUp),
    accSagittalDoubleFlat19sUp = int(mu::engraving::SymId::accSagittalDoubleFlat19sUp),
    accSagittalDoubleFlat23CUp = int(mu::engraving::SymId::accSagittalDoubleFlat23CUp),
    accSagittalDoubleFlat23SUp = int(mu::engraving::SymId::accSagittalDoubleFlat23SUp),
    accSagittalDoubleFlat25SUp = int(mu::engraving::SymId::accSagittalDoubleFlat25SUp),
    accSagittalDoubleFlat49SUp = int(mu::engraving::SymId::accSagittalDoubleFlat49SUp),
    accSagittalDoubleFlat55CUp = int(mu::engraving::SymId::accSagittalDoubleFlat55CUp),
    accSagittalDoubleFlat5CUp = int(mu::engraving::SymId::accSagittalDoubleFlat5CUp),
    accSagittalDoubleFlat5v11SUp = int(mu::engraving::SymId::accSagittalDoubleFlat5v11SUp),
    accSagittalDoubleFlat5v19CUp = int(mu::engraving::SymId::accSagittalDoubleFlat5v19CUp),
    accSagittalDoubleFlat5v23SUp = int(mu::engraving::SymId::accSagittalDoubleFlat5v23SUp),
    accSagittalDoubleFlat5v7kUp = int(mu::engraving::SymId::accSagittalDoubleFlat5v7kUp),
    accSagittalDoubleFlat7CUp = int(mu::engraving::SymId::accSagittalDoubleFlat7CUp),
    accSagittalDoubleFlat7v11CUp = int(mu::engraving::SymId::accSagittalDoubleFlat7v11CUp),
    accSagittalDoubleFlat7v11kUp = int(mu::engraving::SymId::accSagittalDoubleFlat7v11kUp),
    accSagittalDoubleFlat7v19CUp = int(mu::engraving::SymId::accSagittalDoubleFlat7v19CUp),
    accSagittalDoubleSharp = int(mu::engraving::SymId::accSagittalDoubleSharp),
    accSagittalDoubleSharp11v49CDown = int(mu::engraving::SymId::accSagittalDoubleSharp11v49CDown),
    accSagittalDoubleSharp143CDown = int(mu::engraving::SymId::accSagittalDoubleSharp143CDown),
    accSagittalDoubleSharp17CDown = int(mu::engraving::SymId::accSagittalDoubleSharp17CDown),
    accSagittalDoubleSharp17kDown = int(mu::engraving::SymId::accSagittalDoubleSharp17kDown),
    accSagittalDoubleSharp19CDown = int(mu::engraving::SymId::accSagittalDoubleSharp19CDown),
    accSagittalDoubleSharp19sDown = int(mu::engraving::SymId::accSagittalDoubleSharp19sDown),
    accSagittalDoubleSharp23CDown = int(mu::engraving::SymId::accSagittalDoubleSharp23CDown),
    accSagittalDoubleSharp23SDown = int(mu::engraving::SymId::accSagittalDoubleSharp23SDown),
    accSagittalDoubleSharp25SDown = int(mu::engraving::SymId::accSagittalDoubleSharp25SDown),
    accSagittalDoubleSharp49SDown = int(mu::engraving::SymId::accSagittalDoubleSharp49SDown),
    accSagittalDoubleSharp55CDown = int(mu::engraving::SymId::accSagittalDoubleSharp55CDown),
    accSagittalDoubleSharp5CDown = int(mu::engraving::SymId::accSagittalDoubleSharp5CDown),
    accSagittalDoubleSharp5v11SDown = int(mu::engraving::SymId::accSagittalDoubleSharp5v11SDown),
    accSagittalDoubleSharp5v19CDown = int(mu::engraving::SymId::accSagittalDoubleSharp5v19CDown),
    accSagittalDoubleSharp5v23SDown = int(mu::engraving::SymId::accSagittalDoubleSharp5v23SDown),
    accSagittalDoubleSharp5v7kDown = int(mu::engraving::SymId::accSagittalDoubleSharp5v7kDown),
    accSagittalDoubleSharp7CDown = int(mu::engraving::SymId::accSagittalDoubleSharp7CDown),
    accSagittalDoubleSharp7v11CDown = int(mu::engraving::SymId::accSagittalDoubleSharp7v11CDown),
    accSagittalDoubleSharp7v11kDown = int(mu::engraving::SymId::accSagittalDoubleSharp7v11kDown),
    accSagittalDoubleSharp7v19CDown = int(mu::engraving::SymId::accSagittalDoubleSharp7v19CDown),
    accSagittalFlat = int(mu::engraving::SymId::accSagittalFlat),
    accSagittalFlat11LDown = int(mu::engraving::SymId::accSagittalFlat11LDown),
    accSagittalFlat11MDown = int(mu::engraving::SymId::accSagittalFlat11MDown),
    accSagittalFlat11v19LDown = int(mu::engraving::SymId::accSagittalFlat11v19LDown),
    accSagittalFlat11v19MDown = int(mu::engraving::SymId::accSagittalFlat11v19MDown),
    accSagittalFlat11v49CDown = int(mu::engraving::SymId::accSagittalFlat11v49CDown),
    accSagittalFlat11v49CUp = int(mu::engraving::SymId::accSagittalFlat11v49CUp),
    accSagittalFlat143CDown = int(mu::engraving::SymId::accSagittalFlat143CDown),
    accSagittalFlat143CUp = int(mu::engraving::SymId::accSagittalFlat143CUp),
    accSagittalFlat17CDown = int(mu::engraving::SymId::accSagittalFlat17CDown),
    accSagittalFlat17CUp = int(mu::engraving::SymId::accSagittalFlat17CUp),
    accSagittalFlat17kDown = int(mu::engraving::SymId::accSagittalFlat17kDown),
    accSagittalFlat17kUp = int(mu::engraving::SymId::accSagittalFlat17kUp),
    accSagittalFlat19CDown = int(mu::engraving::SymId::accSagittalFlat19CDown),
    accSagittalFlat19CUp = int(mu::engraving::SymId::accSagittalFlat19CUp),
    accSagittalFlat19sDown = int(mu::engraving::SymId::accSagittalFlat19sDown),
    accSagittalFlat19sUp = int(mu::engraving::SymId::accSagittalFlat19sUp),
    accSagittalFlat23CDown = int(mu::engraving::SymId::accSagittalFlat23CDown),
    accSagittalFlat23CUp = int(mu::engraving::SymId::accSagittalFlat23CUp),
    accSagittalFlat23SDown = int(mu::engraving::SymId::accSagittalFlat23SDown),
    accSagittalFlat23SUp = int(mu::engraving::SymId::accSagittalFlat23SUp),
    accSagittalFlat25SDown = int(mu::engraving::SymId::accSagittalFlat25SDown),
    accSagittalFlat25SUp = int(mu::engraving::SymId::accSagittalFlat25SUp),
    accSagittalFlat35LDown = int(mu::engraving::SymId::accSagittalFlat35LDown),
    accSagittalFlat35MDown = int(mu::engraving::SymId::accSagittalFlat35MDown),
    accSagittalFlat49LDown = int(mu::engraving::SymId::accSagittalFlat49LDown),
    accSagittalFlat49MDown = int(mu::engraving::SymId::accSagittalFlat49MDown),
    accSagittalFlat49SDown = int(mu::engraving::SymId::accSagittalFlat49SDown),
    accSagittalFlat49SUp = int(mu::engraving::SymId::accSagittalFlat49SUp),
    accSagittalFlat55CDown = int(mu::engraving::SymId::accSagittalFlat55CDown),
    accSagittalFlat55CUp = int(mu::engraving::SymId::accSagittalFlat55CUp),
    accSagittalFlat5CDown = int(mu::engraving::SymId::accSagittalFlat5CDown),
    accSagittalFlat5CUp = int(mu::engraving::SymId::accSagittalFlat5CUp),
    accSagittalFlat5v11SDown = int(mu::engraving::SymId::accSagittalFlat5v11SDown),
    accSagittalFlat5v11SUp = int(mu::engraving::SymId::accSagittalFlat5v11SUp),
    accSagittalFlat5v13LDown = int(mu::engraving::SymId::accSagittalFlat5v13LDown),
    accSagittalFlat5v13MDown = int(mu::engraving::SymId::accSagittalFlat5v13MDown),
    accSagittalFlat5v19CDown = int(mu::engraving::SymId::accSagittalFlat5v19CDown),
    accSagittalFlat5v19CUp = int(mu::engraving::SymId::accSagittalFlat5v19CUp),
    accSagittalFlat5v23SDown = int(mu::engraving::SymId::accSagittalFlat5v23SDown),
    accSagittalFlat5v23SUp = int(mu::engraving::SymId::accSagittalFlat5v23SUp),
    accSagittalFlat5v49MDown = int(mu::engraving::SymId::accSagittalFlat5v49MDown),
    accSagittalFlat5v7kDown = int(mu::engraving::SymId::accSagittalFlat5v7kDown),
    accSagittalFlat5v7kUp = int(mu::engraving::SymId::accSagittalFlat5v7kUp),
    accSagittalFlat7CDown = int(mu::engraving::SymId::accSagittalFlat7CDown),
    accSagittalFlat7CUp = int(mu::engraving::SymId::accSagittalFlat7CUp),
    accSagittalFlat7v11CDown = int(mu::engraving::SymId::accSagittalFlat7v11CDown),
    accSagittalFlat7v11CUp = int(mu::engraving::SymId::accSagittalFlat7v11CUp),
    accSagittalFlat7v11kDown = int(mu::engraving::SymId::accSagittalFlat7v11kDown),
    accSagittalFlat7v11kUp = int(mu::engraving::SymId::accSagittalFlat7v11kUp),
    accSagittalFlat7v19CDown = int(mu::engraving::SymId::accSagittalFlat7v19CDown),
    accSagittalFlat7v19CUp = int(mu::engraving::SymId::accSagittalFlat7v19CUp),
    accSagittalFractionalTinaDown = int(mu::engraving::SymId::accSagittalFractionalTinaDown),
    accSagittalFractionalTinaUp = int(mu::engraving::SymId::accSagittalFractionalTinaUp),
    accSagittalGrave = int(mu::engraving::SymId::accSagittalGrave),
    accSagittalShaftDown = int(mu::engraving::SymId::accSagittalShaftDown),
    accSagittalShaftUp = int(mu::engraving::SymId::accSagittalShaftUp),
    accSagittalSharp = int(mu::engraving::SymId::accSagittalSharp),
    accSagittalSharp11LUp = int(mu::engraving::SymId::accSagittalSharp11LUp),
    accSagittalSharp11MUp = int(mu::engraving::SymId::accSagittalSharp11MUp),
    accSagittalSharp11v19LUp = int(mu::engraving::SymId::accSagittalSharp11v19LUp),
    accSagittalSharp11v19MUp = int(mu::engraving::SymId::accSagittalSharp11v19MUp),
    accSagittalSharp11v49CDown = int(mu::engraving::SymId::accSagittalSharp11v49CDown),
    accSagittalSharp11v49CUp = int(mu::engraving::SymId::accSagittalSharp11v49CUp),
    accSagittalSharp143CDown = int(mu::engraving::SymId::accSagittalSharp143CDown),
    accSagittalSharp143CUp = int(mu::engraving::SymId::accSagittalSharp143CUp),
    accSagittalSharp17CDown = int(mu::engraving::SymId::accSagittalSharp17CDown),
    accSagittalSharp17CUp = int(mu::engraving::SymId::accSagittalSharp17CUp),
    accSagittalSharp17kDown = int(mu::engraving::SymId::accSagittalSharp17kDown),
    accSagittalSharp17kUp = int(mu::engraving::SymId::accSagittalSharp17kUp),
    accSagittalSharp19CDown = int(mu::engraving::SymId::accSagittalSharp19CDown),
    accSagittalSharp19CUp = int(mu::engraving::SymId::accSagittalSharp19CUp),
    accSagittalSharp19sDown = int(mu::engraving::SymId::accSagittalSharp19sDown),
    accSagittalSharp19sUp = int(mu::engraving::SymId::accSagittalSharp19sUp),
    accSagittalSharp23CDown = int(mu::engraving::SymId::accSagittalSharp23CDown),
    accSagittalSharp23CUp = int(mu::engraving::SymId::accSagittalSharp23CUp),
    accSagittalSharp23SDown = int(mu::engraving::SymId::accSagittalSharp23SDown),
    accSagittalSharp23SUp = int(mu::engraving::SymId::accSagittalSharp23SUp),
    accSagittalSharp25SDown = int(mu::engraving::SymId::accSagittalSharp25SDown),
    accSagittalSharp25SUp = int(mu::engraving::SymId::accSagittalSharp25SUp),
    accSagittalSharp35LUp = int(mu::engraving::SymId::accSagittalSharp35LUp),
    accSagittalSharp35MUp = int(mu::engraving::SymId::accSagittalSharp35MUp),
    accSagittalSharp49LUp = int(mu::engraving::SymId::accSagittalSharp49LUp),
    accSagittalSharp49MUp = int(mu::engraving::SymId::accSagittalSharp49MUp),
    accSagittalSharp49SDown = int(mu::engraving::SymId::accSagittalSharp49SDown),
    accSagittalSharp49SUp = int(mu::engraving::SymId::accSagittalSharp49SUp),
    accSagittalSharp55CDown = int(mu::engraving::SymId::accSagittalSharp55CDown),
    accSagittalSharp55CUp = int(mu::engraving::SymId::accSagittalSharp55CUp),
    accSagittalSharp5CDown = int(mu::engraving::SymId::accSagittalSharp5CDown),
    accSagittalSharp5CUp = int(mu::engraving::SymId::accSagittalSharp5CUp),
    accSagittalSharp5v11SDown = int(mu::engraving::SymId::accSagittalSharp5v11SDown),
    accSagittalSharp5v11SUp = int(mu::engraving::SymId::accSagittalSharp5v11SUp),
    accSagittalSharp5v13LUp = int(mu::engraving::SymId::accSagittalSharp5v13LUp),
    accSagittalSharp5v13MUp = int(mu::engraving::SymId::accSagittalSharp5v13MUp),
    accSagittalSharp5v19CDown = int(mu::engraving::SymId::accSagittalSharp5v19CDown),
    accSagittalSharp5v19CUp = int(mu::engraving::SymId::accSagittalSharp5v19CUp),
    accSagittalSharp5v23SDown = int(mu::engraving::SymId::accSagittalSharp5v23SDown),
    accSagittalSharp5v23SUp = int(mu::engraving::SymId::accSagittalSharp5v23SUp),
    accSagittalSharp5v49MUp = int(mu::engraving::SymId::accSagittalSharp5v49MUp),
    accSagittalSharp5v7kDown = int(mu::engraving::SymId::accSagittalSharp5v7kDown),
    accSagittalSharp5v7kUp = int(mu::engraving::SymId::accSagittalSharp5v7kUp),
    accSagittalSharp7CDown = int(mu::engraving::SymId::accSagittalSharp7CDown),
    accSagittalSharp7CUp = int(mu::engraving::SymId::accSagittalSharp7CUp),
    accSagittalSharp7v11CDown = int(mu::engraving::SymId::accSagittalSharp7v11CDown),
    accSagittalSharp7v11CUp = int(mu::engraving::SymId::accSagittalSharp7v11CUp),
    accSagittalSharp7v11kDown = int(mu::engraving::SymId::accSagittalSharp7v11kDown),
    accSagittalSharp7v11kUp = int(mu::engraving::SymId::accSagittalSharp7v11kUp),
    accSagittalSharp7v19CDown = int(mu::engraving::SymId::accSagittalSharp7v19CDown),
    accSagittalSharp7v19CUp = int(mu::engraving::SymId::accSagittalSharp7v19CUp),
    accSagittalUnused1 = int(mu::engraving::SymId::accSagittalUnused1),
    accSagittalUnused2 = int(mu::engraving::SymId::accSagittalUnused2),
    accSagittalUnused3 = int(mu::engraving::SymId::accSagittalUnused3),
    accSagittalUnused4 = int(mu::engraving::SymId::accSagittalUnused4),
    accdnCombDot = int(mu::engraving::SymId::accdnCombDot),
    accdnCombLH2RanksEmpty = int(mu::engraving::SymId::accdnCombLH2RanksEmpty),
    accdnCombLH3RanksEmptySquare = int(mu::engraving::SymId::accdnCombLH3RanksEmptySquare),
    accdnCombRH3RanksEmpty = int(mu::engraving::SymId::accdnCombRH3RanksEmpty),
    accdnCombRH4RanksEmpty = int(mu::engraving::SymId::accdnCombRH4RanksEmpty),
    accdnDiatonicClef = int(mu::engraving::SymId::accdnDiatonicClef),
    accdnLH2Ranks16Round = int(mu::engraving::SymId::accdnLH2Ranks16Round),
    accdnLH2Ranks8Plus16Round = int(mu::engraving::SymId::accdnLH2Ranks8Plus16Round),
    accdnLH2Ranks8Round = int(mu::engraving::SymId::accdnLH2Ranks8Round),
    accdnLH2RanksFullMasterRound = int(mu::engraving::SymId::accdnLH2RanksFullMasterRound),
    accdnLH2RanksMasterPlus16Round = int(mu::engraving::SymId::accdnLH2RanksMasterPlus16Round),
    accdnLH2RanksMasterRound = int(mu::engraving::SymId::accdnLH2RanksMasterRound),
    accdnLH3Ranks2Plus8Square = int(mu::engraving::SymId::accdnLH3Ranks2Plus8Square),
    accdnLH3Ranks2Square = int(mu::engraving::SymId::accdnLH3Ranks2Square),
    accdnLH3Ranks8Square = int(mu::engraving::SymId::accdnLH3Ranks8Square),
    accdnLH3RanksDouble8Square = int(mu::engraving::SymId::accdnLH3RanksDouble8Square),
    accdnLH3RanksTuttiSquare = int(mu::engraving::SymId::accdnLH3RanksTuttiSquare),
    accdnPull = int(mu::engraving::SymId::accdnPull),
    accdnPush = int(mu::engraving::SymId::accdnPush),
    accdnRH3RanksAccordion = int(mu::engraving::SymId::accdnRH3RanksAccordion),
    accdnRH3RanksAuthenticMusette = int(mu::engraving::SymId::accdnRH3RanksAuthenticMusette),
    accdnRH3RanksBandoneon = int(mu::engraving::SymId::accdnRH3RanksBandoneon),
    accdnRH3RanksBassoon = int(mu::engraving::SymId::accdnRH3RanksBassoon),
    accdnRH3RanksClarinet = int(mu::engraving::SymId::accdnRH3RanksClarinet),
    accdnRH3RanksDoubleTremoloLower8ve = int(mu::engraving::SymId::accdnRH3RanksDoubleTremoloLower8ve),
    accdnRH3RanksDoubleTremoloUpper8ve = int(mu::engraving::SymId::accdnRH3RanksDoubleTremoloUpper8ve),
    accdnRH3RanksFullFactory = int(mu::engraving::SymId::accdnRH3RanksFullFactory),
    accdnRH3RanksHarmonium = int(mu::engraving::SymId::accdnRH3RanksHarmonium),
    accdnRH3RanksImitationMusette = int(mu::engraving::SymId::accdnRH3RanksImitationMusette),
    accdnRH3RanksLowerTremolo8 = int(mu::engraving::SymId::accdnRH3RanksLowerTremolo8),
    accdnRH3RanksMaster = int(mu::engraving::SymId::accdnRH3RanksMaster),
    accdnRH3RanksOboe = int(mu::engraving::SymId::accdnRH3RanksOboe),
    accdnRH3RanksOrgan = int(mu::engraving::SymId::accdnRH3RanksOrgan),
    accdnRH3RanksPiccolo = int(mu::engraving::SymId::accdnRH3RanksPiccolo),
    accdnRH3RanksTremoloLower8ve = int(mu::engraving::SymId::accdnRH3RanksTremoloLower8ve),
    accdnRH3RanksTremoloUpper8ve = int(mu::engraving::SymId::accdnRH3RanksTremoloUpper8ve),
    accdnRH3RanksTwoChoirs = int(mu::engraving::SymId::accdnRH3RanksTwoChoirs),
    accdnRH3RanksUpperTremolo8 = int(mu::engraving::SymId::accdnRH3RanksUpperTremolo8),
    accdnRH3RanksViolin = int(mu::engraving::SymId::accdnRH3RanksViolin),
    accdnRH4RanksAlto = int(mu::engraving::SymId::accdnRH4RanksAlto),
    accdnRH4RanksBassAlto = int(mu::engraving::SymId::accdnRH4RanksBassAlto),
    accdnRH4RanksMaster = int(mu::engraving::SymId::accdnRH4RanksMaster),
    accdnRH4RanksSoftBass = int(mu::engraving::SymId::accdnRH4RanksSoftBass),
    accdnRH4RanksSoftTenor = int(mu::engraving::SymId::accdnRH4RanksSoftTenor),
    accdnRH4RanksSoprano = int(mu::engraving::SymId::accdnRH4RanksSoprano),
    accdnRH4RanksTenor = int(mu::engraving::SymId::accdnRH4RanksTenor),
    accdnRicochet2 = int(mu::engraving::SymId::accdnRicochet2),
    accdnRicochet3 = int(mu::engraving::SymId::accdnRicochet3),
    accdnRicochet4 = int(mu::engraving::SymId::accdnRicochet4),
    accdnRicochet5 = int(mu::engraving::SymId::accdnRicochet5),
    accdnRicochet6 = int(mu::engraving::SymId::accdnRicochet6),
    accdnRicochetStem2 = int(mu::engraving::SymId::accdnRicochetStem2),
    accdnRicochetStem3 = int(mu::engraving::SymId::accdnRicochetStem3),
    accdnRicochetStem4 = int(mu::engraving::SymId::accdnRicochetStem4),
    accdnRicochetStem5 = int(mu::engraving::SymId::accdnRicochetStem5),
    accdnRicochetStem6 = int(mu::engraving::SymId::accdnRicochetStem6),
    accidental1CommaFlat = int(mu::engraving::SymId::accidental1CommaFlat),
    accidental1CommaSharp = int(mu::engraving::SymId::accidental1CommaSharp),
    accidental2CommaFlat = int(mu::engraving::SymId::accidental2CommaFlat),
    accidental2CommaSharp = int(mu::engraving::SymId::accidental2CommaSharp),
    accidental3CommaFlat = int(mu::engraving::SymId::accidental3CommaFlat),
    accidental3CommaSharp = int(mu::engraving::SymId::accidental3CommaSharp),
    accidental4CommaFlat = int(mu::engraving::SymId::accidental4CommaFlat),
    accidental5CommaSharp = int(mu::engraving::SymId::accidental5CommaSharp),
    accidentalArrowDown = int(mu::engraving::SymId::accidentalArrowDown),
    accidentalArrowUp = int(mu::engraving::SymId::accidentalArrowUp),
    accidentalBakiyeFlat = int(mu::engraving::SymId::accidentalBakiyeFlat),
    accidentalBakiyeSharp = int(mu::engraving::SymId::accidentalBakiyeSharp),
    accidentalBracketLeft = int(mu::engraving::SymId::accidentalBracketLeft),
    accidentalBracketRight = int(mu::engraving::SymId::accidentalBracketRight),
    accidentalBuyukMucennebFlat = int(mu::engraving::SymId::accidentalBuyukMucennebFlat),
    accidentalBuyukMucennebSharp = int(mu::engraving::SymId::accidentalBuyukMucennebSharp),
    accidentalCombiningCloseCurlyBrace = int(mu::engraving::SymId::accidentalCombiningCloseCurlyBrace),
    accidentalCombiningLower17Schisma = int(mu::engraving::SymId::accidentalCombiningLower17Schisma),
    accidentalCombiningLower19Schisma = int(mu::engraving::SymId::accidentalCombiningLower19Schisma),
    accidentalCombiningLower23Limit29LimitComma = int(mu::engraving::SymId::accidentalCombiningLower23Limit29LimitComma),
    accidentalCombiningLower29LimitComma = int(mu::engraving::SymId::accidentalCombiningLower29LimitComma),
    accidentalCombiningLower31Schisma = int(mu::engraving::SymId::accidentalCombiningLower31Schisma),
    accidentalCombiningLower37Quartertone = int(mu::engraving::SymId::accidentalCombiningLower37Quartertone),
    accidentalCombiningLower41Comma = int(mu::engraving::SymId::accidentalCombiningLower41Comma),
    accidentalCombiningLower43Comma = int(mu::engraving::SymId::accidentalCombiningLower43Comma),
    accidentalCombiningLower47Quartertone = int(mu::engraving::SymId::accidentalCombiningLower47Quartertone),
    accidentalCombiningLower53LimitComma = int(mu::engraving::SymId::accidentalCombiningLower53LimitComma),
    accidentalCombiningOpenCurlyBrace = int(mu::engraving::SymId::accidentalCombiningOpenCurlyBrace),
    accidentalCombiningRaise17Schisma = int(mu::engraving::SymId::accidentalCombiningRaise17Schisma),
    accidentalCombiningRaise19Schisma = int(mu::engraving::SymId::accidentalCombiningRaise19Schisma),
    accidentalCombiningRaise23Limit29LimitComma = int(mu::engraving::SymId::accidentalCombiningRaise23Limit29LimitComma),
    accidentalCombiningRaise29LimitComma = int(mu::engraving::SymId::accidentalCombiningRaise29LimitComma),
    accidentalCombiningRaise31Schisma = int(mu::engraving::SymId::accidentalCombiningRaise31Schisma),
    accidentalCombiningRaise37Quartertone = int(mu::engraving::SymId::accidentalCombiningRaise37Quartertone),
    accidentalCombiningRaise41Comma = int(mu::engraving::SymId::accidentalCombiningRaise41Comma),
    accidentalCombiningRaise43Comma = int(mu::engraving::SymId::accidentalCombiningRaise43Comma),
    accidentalCombiningRaise47Quartertone = int(mu::engraving::SymId::accidentalCombiningRaise47Quartertone),
    accidentalCombiningRaise53LimitComma = int(mu::engraving::SymId::accidentalCombiningRaise53LimitComma),
    accidentalCommaSlashDown = int(mu::engraving::SymId::accidentalCommaSlashDown),
    accidentalCommaSlashUp = int(mu::engraving::SymId::accidentalCommaSlashUp),
    accidentalDoubleFlat = int(mu::engraving::SymId::accidentalDoubleFlat),
    accidentalDoubleFlatArabic = int(mu::engraving::SymId::accidentalDoubleFlatArabic),
    accidentalDoubleFlatEqualTempered = int(mu::engraving::SymId::accidentalDoubleFlatEqualTempered),
    accidentalDoubleFlatOneArrowDown = int(mu::engraving::SymId::accidentalDoubleFlatOneArrowDown),
    accidentalDoubleFlatOneArrowUp = int(mu::engraving::SymId::accidentalDoubleFlatOneArrowUp),
    accidentalDoubleFlatReversed = int(mu::engraving::SymId::accidentalDoubleFlatReversed),
    accidentalDoubleFlatThreeArrowsDown = int(mu::engraving::SymId::accidentalDoubleFlatThreeArrowsDown),
    accidentalDoubleFlatThreeArrowsUp = int(mu::engraving::SymId::accidentalDoubleFlatThreeArrowsUp),
    accidentalDoubleFlatTurned = int(mu::engraving::SymId::accidentalDoubleFlatTurned),
    accidentalDoubleFlatTwoArrowsDown = int(mu::engraving::SymId::accidentalDoubleFlatTwoArrowsDown),
    accidentalDoubleFlatTwoArrowsUp = int(mu::engraving::SymId::accidentalDoubleFlatTwoArrowsUp),
    accidentalDoubleSharp = int(mu::engraving::SymId::accidentalDoubleSharp),
    accidentalDoubleSharpArabic = int(mu::engraving::SymId::accidentalDoubleSharpArabic),
    accidentalDoubleSharpEqualTempered = int(mu::engraving::SymId::accidentalDoubleSharpEqualTempered),
    accidentalDoubleSharpOneArrowDown = int(mu::engraving::SymId::accidentalDoubleSharpOneArrowDown),
    accidentalDoubleSharpOneArrowUp = int(mu::engraving::SymId::accidentalDoubleSharpOneArrowUp),
    accidentalDoubleSharpThreeArrowsDown = int(mu::engraving::SymId::accidentalDoubleSharpThreeArrowsDown),
    accidentalDoubleSharpThreeArrowsUp = int(mu::engraving::SymId::accidentalDoubleSharpThreeArrowsUp),
    accidentalDoubleSharpTwoArrowsDown = int(mu::engraving::SymId::accidentalDoubleSharpTwoArrowsDown),
    accidentalDoubleSharpTwoArrowsUp = int(mu::engraving::SymId::accidentalDoubleSharpTwoArrowsUp),
    accidentalEnharmonicAlmostEqualTo = int(mu::engraving::SymId::accidentalEnharmonicAlmostEqualTo),
    accidentalEnharmonicEquals = int(mu::engraving::SymId::accidentalEnharmonicEquals),
    accidentalEnharmonicTilde = int(mu::engraving::SymId::accidentalEnharmonicTilde),
    accidentalFilledReversedFlatAndFlat = int(mu::engraving::SymId::accidentalFilledReversedFlatAndFlat),
    accidentalFilledReversedFlatAndFlatArrowDown = int(mu::engraving::SymId::accidentalFilledReversedFlatAndFlatArrowDown),
    accidentalFilledReversedFlatAndFlatArrowUp = int(mu::engraving::SymId::accidentalFilledReversedFlatAndFlatArrowUp),
    accidentalFilledReversedFlatArrowDown = int(mu::engraving::SymId::accidentalFilledReversedFlatArrowDown),
    accidentalFilledReversedFlatArrowUp = int(mu::engraving::SymId::accidentalFilledReversedFlatArrowUp),
    accidentalFiveQuarterTonesFlatArrowDown = int(mu::engraving::SymId::accidentalFiveQuarterTonesFlatArrowDown),
    accidentalFiveQuarterTonesSharpArrowUp = int(mu::engraving::SymId::accidentalFiveQuarterTonesSharpArrowUp),
    accidentalFlat = int(mu::engraving::SymId::accidentalFlat),
    accidentalFlatArabic = int(mu::engraving::SymId::accidentalFlatArabic),
    accidentalFlatEqualTempered = int(mu::engraving::SymId::accidentalFlatEqualTempered),
    accidentalFlatLoweredStockhausen = int(mu::engraving::SymId::accidentalFlatLoweredStockhausen),
    accidentalFlatOneArrowDown = int(mu::engraving::SymId::accidentalFlatOneArrowDown),
    accidentalFlatOneArrowUp = int(mu::engraving::SymId::accidentalFlatOneArrowUp),
    accidentalFlatRaisedStockhausen = int(mu::engraving::SymId::accidentalFlatRaisedStockhausen),
    accidentalFlatRepeatedLineStockhausen = int(mu::engraving::SymId::accidentalFlatRepeatedLineStockhausen),
    accidentalFlatRepeatedSpaceStockhausen = int(mu::engraving::SymId::accidentalFlatRepeatedSpaceStockhausen),
    accidentalFlatThreeArrowsDown = int(mu::engraving::SymId::accidentalFlatThreeArrowsDown),
    accidentalFlatThreeArrowsUp = int(mu::engraving::SymId::accidentalFlatThreeArrowsUp),
    accidentalFlatTurned = int(mu::engraving::SymId::accidentalFlatTurned),
    accidentalFlatTwoArrowsDown = int(mu::engraving::SymId::accidentalFlatTwoArrowsDown),
    accidentalFlatTwoArrowsUp = int(mu::engraving::SymId::accidentalFlatTwoArrowsUp),
    accidentalHabaFlatQuarterToneHigher = int(mu::engraving::SymId::accidentalHabaFlatQuarterToneHigher),
    accidentalHabaFlatThreeQuarterTonesLower = int(mu::engraving::SymId::accidentalHabaFlatThreeQuarterTonesLower),
    accidentalHabaQuarterToneHigher = int(mu::engraving::SymId::accidentalHabaQuarterToneHigher),
    accidentalHabaQuarterToneLower = int(mu::engraving::SymId::accidentalHabaQuarterToneLower),
    accidentalHabaSharpQuarterToneLower = int(mu::engraving::SymId::accidentalHabaSharpQuarterToneLower),
    accidentalHabaSharpThreeQuarterTonesHigher = int(mu::engraving::SymId::accidentalHabaSharpThreeQuarterTonesHigher),
    accidentalHalfSharpArrowDown = int(mu::engraving::SymId::accidentalHalfSharpArrowDown),
    accidentalHalfSharpArrowUp = int(mu::engraving::SymId::accidentalHalfSharpArrowUp),
    accidentalJohnston13 = int(mu::engraving::SymId::accidentalJohnston13),
    accidentalJohnston31 = int(mu::engraving::SymId::accidentalJohnston31),
    accidentalJohnstonDown = int(mu::engraving::SymId::accidentalJohnstonDown),
    accidentalJohnstonEl = int(mu::engraving::SymId::accidentalJohnstonEl),
    accidentalJohnstonMinus = int(mu::engraving::SymId::accidentalJohnstonMinus),
    accidentalJohnstonPlus = int(mu::engraving::SymId::accidentalJohnstonPlus),
    accidentalJohnstonSeven = int(mu::engraving::SymId::accidentalJohnstonSeven),
    accidentalJohnstonUp = int(mu::engraving::SymId::accidentalJohnstonUp),
    accidentalKomaFlat = int(mu::engraving::SymId::accidentalKomaFlat),
    accidentalKomaSharp = int(mu::engraving::SymId::accidentalKomaSharp),
    accidentalKoron = int(mu::engraving::SymId::accidentalKoron),
    accidentalKucukMucennebFlat = int(mu::engraving::SymId::accidentalKucukMucennebFlat),
    accidentalKucukMucennebSharp = int(mu::engraving::SymId::accidentalKucukMucennebSharp),
    accidentalLargeDoubleSharp = int(mu::engraving::SymId::accidentalLargeDoubleSharp),
    accidentalLowerOneSeptimalComma = int(mu::engraving::SymId::accidentalLowerOneSeptimalComma),
    accidentalLowerOneTridecimalQuartertone = int(mu::engraving::SymId::accidentalLowerOneTridecimalQuartertone),
    accidentalLowerOneUndecimalQuartertone = int(mu::engraving::SymId::accidentalLowerOneUndecimalQuartertone),
    accidentalLowerTwoSeptimalCommas = int(mu::engraving::SymId::accidentalLowerTwoSeptimalCommas),
    accidentalLoweredStockhausen = int(mu::engraving::SymId::accidentalLoweredStockhausen),
    accidentalNarrowReversedFlat = int(mu::engraving::SymId::accidentalNarrowReversedFlat),
    accidentalNarrowReversedFlatAndFlat = int(mu::engraving::SymId::accidentalNarrowReversedFlatAndFlat),
    accidentalNatural = int(mu::engraving::SymId::accidentalNatural),
    accidentalNaturalArabic = int(mu::engraving::SymId::accidentalNaturalArabic),
    accidentalNaturalEqualTempered = int(mu::engraving::SymId::accidentalNaturalEqualTempered),
    accidentalNaturalFlat = int(mu::engraving::SymId::accidentalNaturalFlat),
    accidentalNaturalLoweredStockhausen = int(mu::engraving::SymId::accidentalNaturalLoweredStockhausen),
    accidentalNaturalOneArrowDown = int(mu::engraving::SymId::accidentalNaturalOneArrowDown),
    accidentalNaturalOneArrowUp = int(mu::engraving::SymId::accidentalNaturalOneArrowUp),
    accidentalNaturalRaisedStockhausen = int(mu::engraving::SymId::accidentalNaturalRaisedStockhausen),
    accidentalNaturalReversed = int(mu::engraving::SymId::accidentalNaturalReversed),
    accidentalNaturalSharp = int(mu::engraving::SymId::accidentalNaturalSharp),
    accidentalNaturalThreeArrowsDown = int(mu::engraving::SymId::accidentalNaturalThreeArrowsDown),
    accidentalNaturalThreeArrowsUp = int(mu::engraving::SymId::accidentalNaturalThreeArrowsUp),
    accidentalNaturalTwoArrowsDown = int(mu::engraving::SymId::accidentalNaturalTwoArrowsDown),
    accidentalNaturalTwoArrowsUp = int(mu::engraving::SymId::accidentalNaturalTwoArrowsUp),
    accidentalOneAndAHalfSharpsArrowDown = int(mu::engraving::SymId::accidentalOneAndAHalfSharpsArrowDown),
    accidentalOneAndAHalfSharpsArrowUp = int(mu::engraving::SymId::accidentalOneAndAHalfSharpsArrowUp),
    accidentalOneQuarterToneFlatFerneyhough = int(mu::engraving::SymId::accidentalOneQuarterToneFlatFerneyhough),
    accidentalOneQuarterToneFlatStockhausen = int(mu::engraving::SymId::accidentalOneQuarterToneFlatStockhausen),
    accidentalOneQuarterToneSharpFerneyhough = int(mu::engraving::SymId::accidentalOneQuarterToneSharpFerneyhough),
    accidentalOneQuarterToneSharpStockhausen = int(mu::engraving::SymId::accidentalOneQuarterToneSharpStockhausen),
    accidentalOneThirdToneFlatFerneyhough = int(mu::engraving::SymId::accidentalOneThirdToneFlatFerneyhough),
    accidentalOneThirdToneSharpFerneyhough = int(mu::engraving::SymId::accidentalOneThirdToneSharpFerneyhough),
    accidentalParensLeft = int(mu::engraving::SymId::accidentalParensLeft),
    accidentalParensRight = int(mu::engraving::SymId::accidentalParensRight),
    accidentalQuarterFlatEqualTempered = int(mu::engraving::SymId::accidentalQuarterFlatEqualTempered),
    accidentalQuarterSharpEqualTempered = int(mu::engraving::SymId::accidentalQuarterSharpEqualTempered),
    accidentalQuarterToneFlat4 = int(mu::engraving::SymId::accidentalQuarterToneFlat4),
    accidentalQuarterToneFlatArabic = int(mu::engraving::SymId::accidentalQuarterToneFlatArabic),
    accidentalQuarterToneFlatArrowUp = int(mu::engraving::SymId::accidentalQuarterToneFlatArrowUp),
    accidentalQuarterToneFlatFilledReversed = int(mu::engraving::SymId::accidentalQuarterToneFlatFilledReversed),
    accidentalQuarterToneFlatNaturalArrowDown = int(mu::engraving::SymId::accidentalQuarterToneFlatNaturalArrowDown),
    accidentalQuarterToneFlatPenderecki = int(mu::engraving::SymId::accidentalQuarterToneFlatPenderecki),
    accidentalQuarterToneFlatStein = int(mu::engraving::SymId::accidentalQuarterToneFlatStein),
    accidentalQuarterToneFlatVanBlankenburg = int(mu::engraving::SymId::accidentalQuarterToneFlatVanBlankenburg),
    accidentalQuarterToneSharp4 = int(mu::engraving::SymId::accidentalQuarterToneSharp4),
    accidentalQuarterToneSharpArabic = int(mu::engraving::SymId::accidentalQuarterToneSharpArabic),
    accidentalQuarterToneSharpArrowDown = int(mu::engraving::SymId::accidentalQuarterToneSharpArrowDown),
    accidentalQuarterToneSharpBusotti = int(mu::engraving::SymId::accidentalQuarterToneSharpBusotti),
    accidentalQuarterToneSharpNaturalArrowUp = int(mu::engraving::SymId::accidentalQuarterToneSharpNaturalArrowUp),
    accidentalQuarterToneSharpStein = int(mu::engraving::SymId::accidentalQuarterToneSharpStein),
    accidentalQuarterToneSharpWiggle = int(mu::engraving::SymId::accidentalQuarterToneSharpWiggle),
    accidentalRaiseOneSeptimalComma = int(mu::engraving::SymId::accidentalRaiseOneSeptimalComma),
    accidentalRaiseOneTridecimalQuartertone = int(mu::engraving::SymId::accidentalRaiseOneTridecimalQuartertone),
    accidentalRaiseOneUndecimalQuartertone = int(mu::engraving::SymId::accidentalRaiseOneUndecimalQuartertone),
    accidentalRaiseTwoSeptimalCommas = int(mu::engraving::SymId::accidentalRaiseTwoSeptimalCommas),
    accidentalRaisedStockhausen = int(mu::engraving::SymId::accidentalRaisedStockhausen),
    accidentalReversedFlatAndFlatArrowDown = int(mu::engraving::SymId::accidentalReversedFlatAndFlatArrowDown),
    accidentalReversedFlatAndFlatArrowUp = int(mu::engraving::SymId::accidentalReversedFlatAndFlatArrowUp),
    accidentalReversedFlatArrowDown = int(mu::engraving::SymId::accidentalReversedFlatArrowDown),
    accidentalReversedFlatArrowUp = int(mu::engraving::SymId::accidentalReversedFlatArrowUp),
    accidentalSharp = int(mu::engraving::SymId::accidentalSharp),
    accidentalSharpArabic = int(mu::engraving::SymId::accidentalSharpArabic),
    accidentalSharpEqualTempered = int(mu::engraving::SymId::accidentalSharpEqualTempered),
    accidentalSharpLoweredStockhausen = int(mu::engraving::SymId::accidentalSharpLoweredStockhausen),
    accidentalSharpOneArrowDown = int(mu::engraving::SymId::accidentalSharpOneArrowDown),
    accidentalSharpOneArrowUp = int(mu::engraving::SymId::accidentalSharpOneArrowUp),
    accidentalSharpOneHorizontalStroke = int(mu::engraving::SymId::accidentalSharpOneHorizontalStroke),
    accidentalSharpRaisedStockhausen = int(mu::engraving::SymId::accidentalSharpRaisedStockhausen),
    accidentalSharpRepeatedLineStockhausen = int(mu::engraving::SymId::accidentalSharpRepeatedLineStockhausen),
    accidentalSharpRepeatedSpaceStockhausen = int(mu::engraving::SymId::accidentalSharpRepeatedSpaceStockhausen),
    accidentalSharpReversed = int(mu::engraving::SymId::accidentalSharpReversed),
    accidentalSharpSharp = int(mu::engraving::SymId::accidentalSharpSharp),
    accidentalSharpThreeArrowsDown = int(mu::engraving::SymId::accidentalSharpThreeArrowsDown),
    accidentalSharpThreeArrowsUp = int(mu::engraving::SymId::accidentalSharpThreeArrowsUp),
    accidentalSharpTwoArrowsDown = int(mu::engraving::SymId::accidentalSharpTwoArrowsDown),
    accidentalSharpTwoArrowsUp = int(mu::engraving::SymId::accidentalSharpTwoArrowsUp),
    accidentalSims12Down = int(mu::engraving::SymId::accidentalSims12Down),
    accidentalSims12Up = int(mu::engraving::SymId::accidentalSims12Up),
    accidentalSims4Down = int(mu::engraving::SymId::accidentalSims4Down),
    accidentalSims4Up = int(mu::engraving::SymId::accidentalSims4Up),
    accidentalSims6Down = int(mu::engraving::SymId::accidentalSims6Down),
    accidentalSims6Up = int(mu::engraving::SymId::accidentalSims6Up),
    accidentalSori = int(mu::engraving::SymId::accidentalSori),
    accidentalTavenerFlat = int(mu::engraving::SymId::accidentalTavenerFlat),
    accidentalTavenerSharp = int(mu::engraving::SymId::accidentalTavenerSharp),
    accidentalThreeQuarterTonesFlatArabic = int(mu::engraving::SymId::accidentalThreeQuarterTonesFlatArabic),
    accidentalThreeQuarterTonesFlatArrowDown = int(mu::engraving::SymId::accidentalThreeQuarterTonesFlatArrowDown),
    accidentalThreeQuarterTonesFlatArrowUp = int(mu::engraving::SymId::accidentalThreeQuarterTonesFlatArrowUp),
    accidentalThreeQuarterTonesFlatCouper = int(mu::engraving::SymId::accidentalThreeQuarterTonesFlatCouper),
    accidentalThreeQuarterTonesFlatGrisey = int(mu::engraving::SymId::accidentalThreeQuarterTonesFlatGrisey),
    accidentalThreeQuarterTonesFlatTartini = int(mu::engraving::SymId::accidentalThreeQuarterTonesFlatTartini),
    accidentalThreeQuarterTonesFlatZimmermann = int(mu::engraving::SymId::accidentalThreeQuarterTonesFlatZimmermann),
    accidentalThreeQuarterTonesSharpArabic = int(mu::engraving::SymId::accidentalThreeQuarterTonesSharpArabic),
    accidentalThreeQuarterTonesSharpArrowDown = int(mu::engraving::SymId::accidentalThreeQuarterTonesSharpArrowDown),
    accidentalThreeQuarterTonesSharpArrowUp = int(mu::engraving::SymId::accidentalThreeQuarterTonesSharpArrowUp),
    accidentalThreeQuarterTonesSharpBusotti = int(mu::engraving::SymId::accidentalThreeQuarterTonesSharpBusotti),
    accidentalThreeQuarterTonesSharpStein = int(mu::engraving::SymId::accidentalThreeQuarterTonesSharpStein),
    accidentalThreeQuarterTonesSharpStockhausen = int(mu::engraving::SymId::accidentalThreeQuarterTonesSharpStockhausen),
    accidentalTripleFlat = int(mu::engraving::SymId::accidentalTripleFlat),
    accidentalTripleSharp = int(mu::engraving::SymId::accidentalTripleSharp),
    accidentalTwoThirdTonesFlatFerneyhough = int(mu::engraving::SymId::accidentalTwoThirdTonesFlatFerneyhough),
    accidentalTwoThirdTonesSharpFerneyhough = int(mu::engraving::SymId::accidentalTwoThirdTonesSharpFerneyhough),
    accidentalUpsAndDownsDown = int(mu::engraving::SymId::accidentalUpsAndDownsDown),
    accidentalUpsAndDownsLess = int(mu::engraving::SymId::accidentalUpsAndDownsLess),
    accidentalUpsAndDownsMore = int(mu::engraving::SymId::accidentalUpsAndDownsMore),
    accidentalUpsAndDownsUp = int(mu::engraving::SymId::accidentalUpsAndDownsUp),
    accidentalWilsonMinus = int(mu::engraving::SymId::accidentalWilsonMinus),
    accidentalWilsonPlus = int(mu::engraving::SymId::accidentalWilsonPlus),
    accidentalWyschnegradsky10TwelfthsFlat = int(mu::engraving::SymId::accidentalWyschnegradsky10TwelfthsFlat),
    accidentalWyschnegradsky10TwelfthsSharp = int(mu::engraving::SymId::accidentalWyschnegradsky10TwelfthsSharp),
    accidentalWyschnegradsky11TwelfthsFlat = int(mu::engraving::SymId::accidentalWyschnegradsky11TwelfthsFlat),
    accidentalWyschnegradsky11TwelfthsSharp = int(mu::engraving::SymId::accidentalWyschnegradsky11TwelfthsSharp),
    accidentalWyschnegradsky1TwelfthsFlat = int(mu::engraving::SymId::accidentalWyschnegradsky1TwelfthsFlat),
    accidentalWyschnegradsky1TwelfthsSharp = int(mu::engraving::SymId::accidentalWyschnegradsky1TwelfthsSharp),
    accidentalWyschnegradsky2TwelfthsFlat = int(mu::engraving::SymId::accidentalWyschnegradsky2TwelfthsFlat),
    accidentalWyschnegradsky2TwelfthsSharp = int(mu::engraving::SymId::accidentalWyschnegradsky2TwelfthsSharp),
    accidentalWyschnegradsky3TwelfthsFlat = int(mu::engraving::SymId::accidentalWyschnegradsky3TwelfthsFlat),
    accidentalWyschnegradsky3TwelfthsSharp = int(mu::engraving::SymId::accidentalWyschnegradsky3TwelfthsSharp),
    accidentalWyschnegradsky4TwelfthsFlat = int(mu::engraving::SymId::accidentalWyschnegradsky4TwelfthsFlat),
    accidentalWyschnegradsky4TwelfthsSharp = int(mu::engraving::SymId::accidentalWyschnegradsky4TwelfthsSharp),
    accidentalWyschnegradsky5TwelfthsFlat = int(mu::engraving::SymId::accidentalWyschnegradsky5TwelfthsFlat),
    accidentalWyschnegradsky5TwelfthsSharp = int(mu::engraving::SymId::accidentalWyschnegradsky5TwelfthsSharp),
    accidentalWyschnegradsky6TwelfthsFlat = int(mu::engraving::SymId::accidentalWyschnegradsky6TwelfthsFlat),
    accidentalWyschnegradsky6TwelfthsSharp = int(mu::engraving::SymId::accidentalWyschnegradsky6TwelfthsSharp),
    accidentalWyschnegradsky7TwelfthsFlat = int(mu::engraving::SymId::accidentalWyschnegradsky7TwelfthsFlat),
    accidentalWyschnegradsky7TwelfthsSharp = int(mu::engraving::SymId::accidentalWyschnegradsky7TwelfthsSharp),
    accidentalWyschnegradsky8TwelfthsFlat = int(mu::engraving::SymId::accidentalWyschnegradsky8TwelfthsFlat),
    accidentalWyschnegradsky8TwelfthsSharp = int(mu::engraving::SymId::accidentalWyschnegradsky8TwelfthsSharp),
    accidentalWyschnegradsky9TwelfthsFlat = int(mu::engraving::SymId::accidentalWyschnegradsky9TwelfthsFlat),
    accidentalWyschnegradsky9TwelfthsSharp = int(mu::engraving::SymId::accidentalWyschnegradsky9TwelfthsSharp),
    accidentalXenakisOneThirdToneSharp = int(mu::engraving::SymId::accidentalXenakisOneThirdToneSharp),
    accidentalXenakisTwoThirdTonesSharp = int(mu::engraving::SymId::accidentalXenakisTwoThirdTonesSharp),
    analyticsChoralmelodie = int(mu::engraving::SymId::analyticsChoralmelodie),
    analyticsEndStimme = int(mu::engraving::SymId::analyticsEndStimme),
    analyticsHauptrhythmus = int(mu::engraving::SymId::analyticsHauptrhythmus),
    analyticsHauptstimme = int(mu::engraving::SymId::analyticsHauptstimme),
    analyticsInversion1 = int(mu::engraving::SymId::analyticsInversion1),
    analyticsNebenstimme = int(mu::engraving::SymId::analyticsNebenstimme),
    analyticsStartStimme = int(mu::engraving::SymId::analyticsStartStimme),
    analyticsTheme = int(mu::engraving::SymId::analyticsTheme),
    analyticsTheme1 = int(mu::engraving::SymId::analyticsTheme1),
    analyticsThemeInversion = int(mu::engraving::SymId::analyticsThemeInversion),
    analyticsThemerograde = int(mu::engraving::SymId::analyticsThemerograde),
    analyticsThemerogradeInversion = int(mu::engraving::SymId::analyticsThemerogradeInversion),
    arpeggiato = int(mu::engraving::SymId::arpeggiato),
    arpeggiatoDown = int(mu::engraving::SymId::arpeggiatoDown),
    arpeggiatoUp = int(mu::engraving::SymId::arpeggiatoUp),
    arrowBlackDown = int(mu::engraving::SymId::arrowBlackDown),
    arrowBlackDownLeft = int(mu::engraving::SymId::arrowBlackDownLeft),
    arrowBlackDownRight = int(mu::engraving::SymId::arrowBlackDownRight),
    arrowBlackLeft = int(mu::engraving::SymId::arrowBlackLeft),
    arrowBlackRight = int(mu::engraving::SymId::arrowBlackRight),
    arrowBlackUp = int(mu::engraving::SymId::arrowBlackUp),
    arrowBlackUpLeft = int(mu::engraving::SymId::arrowBlackUpLeft),
    arrowBlackUpRight = int(mu::engraving::SymId::arrowBlackUpRight),
    arrowOpenDown = int(mu::engraving::SymId::arrowOpenDown),
    arrowOpenDownLeft = int(mu::engraving::SymId::arrowOpenDownLeft),
    arrowOpenDownRight = int(mu::engraving::SymId::arrowOpenDownRight),
    arrowOpenLeft = int(mu::engraving::SymId::arrowOpenLeft),
    arrowOpenRight = int(mu::engraving::SymId::arrowOpenRight),
    arrowOpenUp = int(mu::engraving::SymId::arrowOpenUp),
    arrowOpenUpLeft = int(mu::engraving::SymId::arrowOpenUpLeft),
    arrowOpenUpRight = int(mu::engraving::SymId::arrowOpenUpRight),
    arrowWhiteDown = int(mu::engraving::SymId::arrowWhiteDown),
    arrowWhiteDownLeft = int(mu::engraving::SymId::arrowWhiteDownLeft),
    arrowWhiteDownRight = int(mu::engraving::SymId::arrowWhiteDownRight),
    arrowWhiteLeft = int(mu::engraving::SymId::arrowWhiteLeft),
    arrowWhiteRight = int(mu::engraving::SymId::arrowWhiteRight),
    arrowWhiteUp = int(mu::engraving::SymId::arrowWhiteUp),
    arrowWhiteUpLeft = int(mu::engraving::SymId::arrowWhiteUpLeft),
    arrowWhiteUpRight = int(mu::engraving::SymId::arrowWhiteUpRight),
    arrowheadBlackDown = int(mu::engraving::SymId::arrowheadBlackDown),
    arrowheadBlackDownLeft = int(mu::engraving::SymId::arrowheadBlackDownLeft),
    arrowheadBlackDownRight = int(mu::engraving::SymId::arrowheadBlackDownRight),
    arrowheadBlackLeft = int(mu::engraving::SymId::arrowheadBlackLeft),
    arrowheadBlackRight = int(mu::engraving::SymId::arrowheadBlackRight),
    arrowheadBlackUp = int(mu::engraving::SymId::arrowheadBlackUp),
    arrowheadBlackUpLeft = int(mu::engraving::SymId::arrowheadBlackUpLeft),
    arrowheadBlackUpRight = int(mu::engraving::SymId::arrowheadBlackUpRight),
    arrowheadOpenDown = int(mu::engraving::SymId::arrowheadOpenDown),
    arrowheadOpenDownLeft = int(mu::engraving::SymId::arrowheadOpenDownLeft),
    arrowheadOpenDownRight = int(mu::engraving::SymId::arrowheadOpenDownRight),
    arrowheadOpenLeft = int(mu::engraving::SymId::arrowheadOpenLeft),
    arrowheadOpenRight = int(mu::engraving::SymId::arrowheadOpenRight),
    arrowheadOpenUp = int(mu::engraving::SymId::arrowheadOpenUp),
    arrowheadOpenUpLeft = int(mu::engraving::SymId::arrowheadOpenUpLeft),
    arrowheadOpenUpRight = int(mu::engraving::SymId::arrowheadOpenUpRight),
    arrowheadWhiteDown = int(mu::engraving::SymId::arrowheadWhiteDown),
    arrowheadWhiteDownLeft = int(mu::engraving::SymId::arrowheadWhiteDownLeft),
    arrowheadWhiteDownRight = int(mu::engraving::SymId::arrowheadWhiteDownRight),
    arrowheadWhiteLeft = int(mu::engraving::SymId::arrowheadWhiteLeft),
    arrowheadWhiteRight = int(mu::engraving::SymId::arrowheadWhiteRight),
    arrowheadWhiteUp = int(mu::engraving::SymId::arrowheadWhiteUp),
    arrowheadWhiteUpLeft = int(mu::engraving::SymId::arrowheadWhiteUpLeft),
    arrowheadWhiteUpRight = int(mu::engraving::SymId::arrowheadWhiteUpRight),
    articAccentAbove = int(mu::engraving::SymId::articAccentAbove),
    articAccentBelow = int(mu::engraving::SymId::articAccentBelow),
    articAccentStaccatoAbove = int(mu::engraving::SymId::articAccentStaccatoAbove),
    articAccentStaccatoBelow = int(mu::engraving::SymId::articAccentStaccatoBelow),
    articLaissezVibrerAbove = int(mu::engraving::SymId::articLaissezVibrerAbove),
    articLaissezVibrerBelow = int(mu::engraving::SymId::articLaissezVibrerBelow),
    articMarcatoAbove = int(mu::engraving::SymId::articMarcatoAbove),
    articMarcatoBelow = int(mu::engraving::SymId::articMarcatoBelow),
    articMarcatoStaccatoAbove = int(mu::engraving::SymId::articMarcatoStaccatoAbove),
    articMarcatoStaccatoBelow = int(mu::engraving::SymId::articMarcatoStaccatoBelow),
    articMarcatoTenutoAbove = int(mu::engraving::SymId::articMarcatoTenutoAbove),
    articMarcatoTenutoBelow = int(mu::engraving::SymId::articMarcatoTenutoBelow),
    articSoftAccentAbove = int(mu::engraving::SymId::articSoftAccentAbove),
    articSoftAccentBelow = int(mu::engraving::SymId::articSoftAccentBelow),
    articSoftAccentStaccatoAbove = int(mu::engraving::SymId::articSoftAccentStaccatoAbove),
    articSoftAccentStaccatoBelow = int(mu::engraving::SymId::articSoftAccentStaccatoBelow),
    articSoftAccentTenutoAbove = int(mu::engraving::SymId::articSoftAccentTenutoAbove),
    articSoftAccentTenutoBelow = int(mu::engraving::SymId::articSoftAccentTenutoBelow),
    articSoftAccentTenutoStaccatoAbove = int(mu::engraving::SymId::articSoftAccentTenutoStaccatoAbove),
    articSoftAccentTenutoStaccatoBelow = int(mu::engraving::SymId::articSoftAccentTenutoStaccatoBelow),
    articStaccatissimoAbove = int(mu::engraving::SymId::articStaccatissimoAbove),
    articStaccatissimoBelow = int(mu::engraving::SymId::articStaccatissimoBelow),
    articStaccatissimoStrokeAbove = int(mu::engraving::SymId::articStaccatissimoStrokeAbove),
    articStaccatissimoStrokeBelow = int(mu::engraving::SymId::articStaccatissimoStrokeBelow),
    articStaccatissimoWedgeAbove = int(mu::engraving::SymId::articStaccatissimoWedgeAbove),
    articStaccatissimoWedgeBelow = int(mu::engraving::SymId::articStaccatissimoWedgeBelow),
    articStaccatoAbove = int(mu::engraving::SymId::articStaccatoAbove),
    articStaccatoBelow = int(mu::engraving::SymId::articStaccatoBelow),
    articStressAbove = int(mu::engraving::SymId::articStressAbove),
    articStressBelow = int(mu::engraving::SymId::articStressBelow),
    articTenutoAbove = int(mu::engraving::SymId::articTenutoAbove),
    articTenutoAccentAbove = int(mu::engraving::SymId::articTenutoAccentAbove),
    articTenutoAccentBelow = int(mu::engraving::SymId::articTenutoAccentBelow),
    articTenutoBelow = int(mu::engraving::SymId::articTenutoBelow),
    articTenutoStaccatoAbove = int(mu::engraving::SymId::articTenutoStaccatoAbove),
    articTenutoStaccatoBelow = int(mu::engraving::SymId::articTenutoStaccatoBelow),
    articUnstressAbove = int(mu::engraving::SymId::articUnstressAbove),
    articUnstressBelow = int(mu::engraving::SymId::articUnstressBelow),
    augmentationDot = int(mu::engraving::SymId::augmentationDot),
    barlineDashed = int(mu::engraving::SymId::barlineDashed),
    barlineDotted = int(mu::engraving::SymId::barlineDotted),
    barlineDouble = int(mu::engraving::SymId::barlineDouble),
    barlineFinal = int(mu::engraving::SymId::barlineFinal),
    barlineHeavy = int(mu::engraving::SymId::barlineHeavy),
    barlineHeavyHeavy = int(mu::engraving::SymId::barlineHeavyHeavy),
    barlineReverseFinal = int(mu::engraving::SymId::barlineReverseFinal),
    barlineShort = int(mu::engraving::SymId::barlineShort),
    barlineSingle = int(mu::engraving::SymId::barlineSingle),
    barlineTick = int(mu::engraving::SymId::barlineTick),
    beamAccelRit1 = int(mu::engraving::SymId::beamAccelRit1),
    beamAccelRit10 = int(mu::engraving::SymId::beamAccelRit10),
    beamAccelRit11 = int(mu::engraving::SymId::beamAccelRit11),
    beamAccelRit12 = int(mu::engraving::SymId::beamAccelRit12),
    beamAccelRit13 = int(mu::engraving::SymId::beamAccelRit13),
    beamAccelRit14 = int(mu::engraving::SymId::beamAccelRit14),
    beamAccelRit15 = int(mu::engraving::SymId::beamAccelRit15),
    beamAccelRit2 = int(mu::engraving::SymId::beamAccelRit2),
    beamAccelRit3 = int(mu::engraving::SymId::beamAccelRit3),
    beamAccelRit4 = int(mu::engraving::SymId::beamAccelRit4),
    beamAccelRit5 = int(mu::engraving::SymId::beamAccelRit5),
    beamAccelRit6 = int(mu::engraving::SymId::beamAccelRit6),
    beamAccelRit7 = int(mu::engraving::SymId::beamAccelRit7),
    beamAccelRit8 = int(mu::engraving::SymId::beamAccelRit8),
    beamAccelRit9 = int(mu::engraving::SymId::beamAccelRit9),
    beamAccelRitFinal = int(mu::engraving::SymId::beamAccelRitFinal),
    brace = int(mu::engraving::SymId::brace),
    bracket = int(mu::engraving::SymId::bracket),
    bracketBottom = int(mu::engraving::SymId::bracketBottom),
    bracketTop = int(mu::engraving::SymId::bracketTop),
    brassBend = int(mu::engraving::SymId::brassBend),
    brassDoitLong = int(mu::engraving::SymId::brassDoitLong),
    brassDoitMedium = int(mu::engraving::SymId::brassDoitMedium),
    brassDoitShort = int(mu::engraving::SymId::brassDoitShort),
    brassFallLipLong = int(mu::engraving::SymId::brassFallLipLong),
    brassFallLipMedium = int(mu::engraving::SymId::brassFallLipMedium),
    brassFallLipShort = int(mu::engraving::SymId::brassFallLipShort),
    brassFallRoughLong = int(mu::engraving::SymId::brassFallRoughLong),
    brassFallRoughMedium = int(mu::engraving::SymId::brassFallRoughMedium),
    brassFallRoughShort = int(mu::engraving::SymId::brassFallRoughShort),
    brassFallSmoothLong = int(mu::engraving::SymId::brassFallSmoothLong),
    brassFallSmoothMedium = int(mu::engraving::SymId::brassFallSmoothMedium),
    brassFallSmoothShort = int(mu::engraving::SymId::brassFallSmoothShort),
    brassFlip = int(mu::engraving::SymId::brassFlip),
    brassHarmonMuteClosed = int(mu::engraving::SymId::brassHarmonMuteClosed),
    brassHarmonMuteStemHalfLeft = int(mu::engraving::SymId::brassHarmonMuteStemHalfLeft),
    brassHarmonMuteStemHalfRight = int(mu::engraving::SymId::brassHarmonMuteStemHalfRight),
    brassHarmonMuteStemOpen = int(mu::engraving::SymId::brassHarmonMuteStemOpen),
    brassJazzTurn = int(mu::engraving::SymId::brassJazzTurn),
    brassLiftLong = int(mu::engraving::SymId::brassLiftLong),
    brassLiftMedium = int(mu::engraving::SymId::brassLiftMedium),
    brassLiftShort = int(mu::engraving::SymId::brassLiftShort),
    brassLiftSmoothLong = int(mu::engraving::SymId::brassLiftSmoothLong),
    brassLiftSmoothMedium = int(mu::engraving::SymId::brassLiftSmoothMedium),
    brassLiftSmoothShort = int(mu::engraving::SymId::brassLiftSmoothShort),
    brassMuteClosed = int(mu::engraving::SymId::brassMuteClosed),
    brassMuteHalfClosed = int(mu::engraving::SymId::brassMuteHalfClosed),
    brassMuteOpen = int(mu::engraving::SymId::brassMuteOpen),
    brassPlop = int(mu::engraving::SymId::brassPlop),
    brassScoop = int(mu::engraving::SymId::brassScoop),
    brassSmear = int(mu::engraving::SymId::brassSmear),
    brassValveTrill = int(mu::engraving::SymId::brassValveTrill),
    breathMarkComma = int(mu::engraving::SymId::breathMarkComma),
    breathMarkSalzedo = int(mu::engraving::SymId::breathMarkSalzedo),
    breathMarkTick = int(mu::engraving::SymId::breathMarkTick),
    breathMarkUpbow = int(mu::engraving::SymId::breathMarkUpbow),
    bridgeClef = int(mu::engraving::SymId::bridgeClef),
    buzzRoll = int(mu::engraving::SymId::buzzRoll),
    cClef = int(mu::engraving::SymId::cClef),
    cClef8vb = int(mu::engraving::SymId::cClef8vb),
    cClefArrowDown = int(mu::engraving::SymId::cClefArrowDown),
    cClefArrowUp = int(mu::engraving::SymId::cClefArrowUp),
    cClefChange = int(mu::engraving::SymId::cClefChange),
    cClefCombining = int(mu::engraving::SymId::cClefCombining),
    cClefReversed = int(mu::engraving::SymId::cClefReversed),
    cClefSquare = int(mu::engraving::SymId::cClefSquare),
    caesura = int(mu::engraving::SymId::caesura),
    caesuraCurved = int(mu::engraving::SymId::caesuraCurved),
    caesuraShort = int(mu::engraving::SymId::caesuraShort),
    caesuraSingleStroke = int(mu::engraving::SymId::caesuraSingleStroke),
    caesuraThick = int(mu::engraving::SymId::caesuraThick),
    chantAccentusAbove = int(mu::engraving::SymId::chantAccentusAbove),
    chantAccentusBelow = int(mu::engraving::SymId::chantAccentusBelow),
    chantAuctumAsc = int(mu::engraving::SymId::chantAuctumAsc),
    chantAuctumDesc = int(mu::engraving::SymId::chantAuctumDesc),
    chantAugmentum = int(mu::engraving::SymId::chantAugmentum),
    chantCaesura = int(mu::engraving::SymId::chantCaesura),
    chantCclef = int(mu::engraving::SymId::chantCclef),
    chantCirculusAbove = int(mu::engraving::SymId::chantCirculusAbove),
    chantCirculusBelow = int(mu::engraving::SymId::chantCirculusBelow),
    chantConnectingLineAsc2nd = int(mu::engraving::SymId::chantConnectingLineAsc2nd),
    chantConnectingLineAsc3rd = int(mu::engraving::SymId::chantConnectingLineAsc3rd),
    chantConnectingLineAsc4th = int(mu::engraving::SymId::chantConnectingLineAsc4th),
    chantConnectingLineAsc5th = int(mu::engraving::SymId::chantConnectingLineAsc5th),
    chantConnectingLineAsc6th = int(mu::engraving::SymId::chantConnectingLineAsc6th),
    chantCustosStemDownPosHigh = int(mu::engraving::SymId::chantCustosStemDownPosHigh),
    chantCustosStemDownPosHighest = int(mu::engraving::SymId::chantCustosStemDownPosHighest),
    chantCustosStemDownPosMiddle = int(mu::engraving::SymId::chantCustosStemDownPosMiddle),
    chantCustosStemUpPosLow = int(mu::engraving::SymId::chantCustosStemUpPosLow),
    chantCustosStemUpPosLowest = int(mu::engraving::SymId::chantCustosStemUpPosLowest),
    chantCustosStemUpPosMiddle = int(mu::engraving::SymId::chantCustosStemUpPosMiddle),
    chantDeminutumLower = int(mu::engraving::SymId::chantDeminutumLower),
    chantDeminutumUpper = int(mu::engraving::SymId::chantDeminutumUpper),
    chantDivisioFinalis = int(mu::engraving::SymId::chantDivisioFinalis),
    chantDivisioMaior = int(mu::engraving::SymId::chantDivisioMaior),
    chantDivisioMaxima = int(mu::engraving::SymId::chantDivisioMaxima),
    chantDivisioMinima = int(mu::engraving::SymId::chantDivisioMinima),
    chantEntryLineAsc2nd = int(mu::engraving::SymId::chantEntryLineAsc2nd),
    chantEntryLineAsc3rd = int(mu::engraving::SymId::chantEntryLineAsc3rd),
    chantEntryLineAsc4th = int(mu::engraving::SymId::chantEntryLineAsc4th),
    chantEntryLineAsc5th = int(mu::engraving::SymId::chantEntryLineAsc5th),
    chantEntryLineAsc6th = int(mu::engraving::SymId::chantEntryLineAsc6th),
    chantEpisema = int(mu::engraving::SymId::chantEpisema),
    chantFclef = int(mu::engraving::SymId::chantFclef),
    chantIctusAbove = int(mu::engraving::SymId::chantIctusAbove),
    chantIctusBelow = int(mu::engraving::SymId::chantIctusBelow),
    chantLigaturaDesc2nd = int(mu::engraving::SymId::chantLigaturaDesc2nd),
    chantLigaturaDesc3rd = int(mu::engraving::SymId::chantLigaturaDesc3rd),
    chantLigaturaDesc4th = int(mu::engraving::SymId::chantLigaturaDesc4th),
    chantLigaturaDesc5th = int(mu::engraving::SymId::chantLigaturaDesc5th),
    chantOriscusAscending = int(mu::engraving::SymId::chantOriscusAscending),
    chantOriscusDescending = int(mu::engraving::SymId::chantOriscusDescending),
    chantOriscusLiquescens = int(mu::engraving::SymId::chantOriscusLiquescens),
    chantPodatusLower = int(mu::engraving::SymId::chantPodatusLower),
    chantPodatusUpper = int(mu::engraving::SymId::chantPodatusUpper),
    chantPunctum = int(mu::engraving::SymId::chantPunctum),
    chantPunctumCavum = int(mu::engraving::SymId::chantPunctumCavum),
    chantPunctumDeminutum = int(mu::engraving::SymId::chantPunctumDeminutum),
    chantPunctumInclinatum = int(mu::engraving::SymId::chantPunctumInclinatum),
    chantPunctumInclinatumAuctum = int(mu::engraving::SymId::chantPunctumInclinatumAuctum),
    chantPunctumInclinatumDeminutum = int(mu::engraving::SymId::chantPunctumInclinatumDeminutum),
    chantPunctumLinea = int(mu::engraving::SymId::chantPunctumLinea),
    chantPunctumLineaCavum = int(mu::engraving::SymId::chantPunctumLineaCavum),
    chantPunctumVirga = int(mu::engraving::SymId::chantPunctumVirga),
    chantPunctumVirgaReversed = int(mu::engraving::SymId::chantPunctumVirgaReversed),
    chantQuilisma = int(mu::engraving::SymId::chantQuilisma),
    chantSemicirculusAbove = int(mu::engraving::SymId::chantSemicirculusAbove),
    chantSemicirculusBelow = int(mu::engraving::SymId::chantSemicirculusBelow),
    chantStaff = int(mu::engraving::SymId::chantStaff),
    chantStaffNarrow = int(mu::engraving::SymId::chantStaffNarrow),
    chantStaffWide = int(mu::engraving::SymId::chantStaffWide),
    chantStrophicus = int(mu::engraving::SymId::chantStrophicus),
    chantStrophicusAuctus = int(mu::engraving::SymId::chantStrophicusAuctus),
    chantStrophicusLiquescens2nd = int(mu::engraving::SymId::chantStrophicusLiquescens2nd),
    chantStrophicusLiquescens3rd = int(mu::engraving::SymId::chantStrophicusLiquescens3rd),
    chantStrophicusLiquescens4th = int(mu::engraving::SymId::chantStrophicusLiquescens4th),
    chantStrophicusLiquescens5th = int(mu::engraving::SymId::chantStrophicusLiquescens5th),
    chantVirgula = int(mu::engraving::SymId::chantVirgula),
    clef15 = int(mu::engraving::SymId::clef15),
    clef8 = int(mu::engraving::SymId::clef8),
    clefChangeCombining = int(mu::engraving::SymId::clefChangeCombining),
    coda = int(mu::engraving::SymId::coda),
    codaSquare = int(mu::engraving::SymId::codaSquare),
    conductorBeat2Compound = int(mu::engraving::SymId::conductorBeat2Compound),
    conductorBeat2Simple = int(mu::engraving::SymId::conductorBeat2Simple),
    conductorBeat3Compound = int(mu::engraving::SymId::conductorBeat3Compound),
    conductorBeat3Simple = int(mu::engraving::SymId::conductorBeat3Simple),
    conductorBeat4Compound = int(mu::engraving::SymId::conductorBeat4Compound),
    conductorBeat4Simple = int(mu::engraving::SymId::conductorBeat4Simple),
    conductorLeftBeat = int(mu::engraving::SymId::conductorLeftBeat),
    conductorRightBeat = int(mu::engraving::SymId::conductorRightBeat),
    conductorStrongBeat = int(mu::engraving::SymId::conductorStrongBeat),
    conductorUnconducted = int(mu::engraving::SymId::conductorUnconducted),
    conductorWeakBeat = int(mu::engraving::SymId::conductorWeakBeat),
    controlBeginBeam = int(mu::engraving::SymId::controlBeginBeam),
    controlBeginPhrase = int(mu::engraving::SymId::controlBeginPhrase),
    controlBeginSlur = int(mu::engraving::SymId::controlBeginSlur),
    controlBeginTie = int(mu::engraving::SymId::controlBeginTie),
    controlEndBeam = int(mu::engraving::SymId::controlEndBeam),
    controlEndPhrase = int(mu::engraving::SymId::controlEndPhrase),
    controlEndSlur = int(mu::engraving::SymId::controlEndSlur),
    controlEndTie = int(mu::engraving::SymId::controlEndTie),
    csymAccidentalDoubleFlat = int(mu::engraving::SymId::csymAccidentalDoubleFlat),
    csymAccidentalDoubleSharp = int(mu::engraving::SymId::csymAccidentalDoubleSharp),
    csymAccidentalFlat = int(mu::engraving::SymId::csymAccidentalFlat),
    csymAccidentalNatural = int(mu::engraving::SymId::csymAccidentalNatural),
    csymAccidentalSharp = int(mu::engraving::SymId::csymAccidentalSharp),
    csymAccidentalTripleFlat = int(mu::engraving::SymId::csymAccidentalTripleFlat),
    csymAccidentalTripleSharp = int(mu::engraving::SymId::csymAccidentalTripleSharp),
    csymAlteredBassSlash = int(mu::engraving::SymId::csymAlteredBassSlash),
    csymAugmented = int(mu::engraving::SymId::csymAugmented),
    csymBracketLeftTall = int(mu::engraving::SymId::csymBracketLeftTall),
    csymBracketRightTall = int(mu::engraving::SymId::csymBracketRightTall),
    csymDiagonalArrangementSlash = int(mu::engraving::SymId::csymDiagonalArrangementSlash),
    csymDiminished = int(mu::engraving::SymId::csymDiminished),
    csymHalfDiminished = int(mu::engraving::SymId::csymHalfDiminished),
    csymMajorSeventh = int(mu::engraving::SymId::csymMajorSeventh),
    csymMinor = int(mu::engraving::SymId::csymMinor),
    csymParensLeftTall = int(mu::engraving::SymId::csymParensLeftTall),
    csymParensLeftVeryTall = int(mu::engraving::SymId::csymParensLeftVeryTall),
    csymParensRightTall = int(mu::engraving::SymId::csymParensRightTall),
    csymParensRightVeryTall = int(mu::engraving::SymId::csymParensRightVeryTall),
    curlewSign = int(mu::engraving::SymId::curlewSign),
    daCapo = int(mu::engraving::SymId::daCapo),
    dalSegno = int(mu::engraving::SymId::dalSegno),
    daseianExcellentes1 = int(mu::engraving::SymId::daseianExcellentes1),
    daseianExcellentes2 = int(mu::engraving::SymId::daseianExcellentes2),
    daseianExcellentes3 = int(mu::engraving::SymId::daseianExcellentes3),
    daseianExcellentes4 = int(mu::engraving::SymId::daseianExcellentes4),
    daseianFinales1 = int(mu::engraving::SymId::daseianFinales1),
    daseianFinales2 = int(mu::engraving::SymId::daseianFinales2),
    daseianFinales3 = int(mu::engraving::SymId::daseianFinales3),
    daseianFinales4 = int(mu::engraving::SymId::daseianFinales4),
    daseianGraves1 = int(mu::engraving::SymId::daseianGraves1),
    daseianGraves2 = int(mu::engraving::SymId::daseianGraves2),
    daseianGraves3 = int(mu::engraving::SymId::daseianGraves3),
    daseianGraves4 = int(mu::engraving::SymId::daseianGraves4),
    daseianResidua1 = int(mu::engraving::SymId::daseianResidua1),
    daseianResidua2 = int(mu::engraving::SymId::daseianResidua2),
    daseianSuperiores1 = int(mu::engraving::SymId::daseianSuperiores1),
    daseianSuperiores2 = int(mu::engraving::SymId::daseianSuperiores2),
    daseianSuperiores3 = int(mu::engraving::SymId::daseianSuperiores3),
    daseianSuperiores4 = int(mu::engraving::SymId::daseianSuperiores4),
    doubleLateralRollStevens = int(mu::engraving::SymId::doubleLateralRollStevens),
    doubleTongueAbove = int(mu::engraving::SymId::doubleTongueAbove),
    doubleTongueBelow = int(mu::engraving::SymId::doubleTongueBelow),
    dynamicCombinedSeparatorColon = int(mu::engraving::SymId::dynamicCombinedSeparatorColon),
    dynamicCombinedSeparatorHyphen = int(mu::engraving::SymId::dynamicCombinedSeparatorHyphen),
    dynamicCombinedSeparatorSlash = int(mu::engraving::SymId::dynamicCombinedSeparatorSlash),
    dynamicCombinedSeparatorSpace = int(mu::engraving::SymId::dynamicCombinedSeparatorSpace),
    dynamicCrescendoHairpin = int(mu::engraving::SymId::dynamicCrescendoHairpin),
    dynamicDiminuendoHairpin = int(mu::engraving::SymId::dynamicDiminuendoHairpin),
    dynamicFF = int(mu::engraving::SymId::dynamicFF),
    dynamicFFF = int(mu::engraving::SymId::dynamicFFF),
    dynamicFFFF = int(mu::engraving::SymId::dynamicFFFF),
    dynamicFFFFF = int(mu::engraving::SymId::dynamicFFFFF),
    dynamicFFFFFF = int(mu::engraving::SymId::dynamicFFFFFF),
    dynamicForte = int(mu::engraving::SymId::dynamicForte),
    dynamicFortePiano = int(mu::engraving::SymId::dynamicFortePiano),
    dynamicForzando = int(mu::engraving::SymId::dynamicForzando),
    dynamicHairpinBracketLeft = int(mu::engraving::SymId::dynamicHairpinBracketLeft),
    dynamicHairpinBracketRight = int(mu::engraving::SymId::dynamicHairpinBracketRight),
    dynamicHairpinParenthesisLeft = int(mu::engraving::SymId::dynamicHairpinParenthesisLeft),
    dynamicHairpinParenthesisRight = int(mu::engraving::SymId::dynamicHairpinParenthesisRight),
    dynamicMF = int(mu::engraving::SymId::dynamicMF),
    dynamicMP = int(mu::engraving::SymId::dynamicMP),
    dynamicMessaDiVoce = int(mu::engraving::SymId::dynamicMessaDiVoce),
    dynamicMezzo = int(mu::engraving::SymId::dynamicMezzo),
    dynamicNiente = int(mu::engraving::SymId::dynamicNiente),
    dynamicNienteForHairpin = int(mu::engraving::SymId::dynamicNienteForHairpin),
    dynamicPF = int(mu::engraving::SymId::dynamicPF),
    dynamicPP = int(mu::engraving::SymId::dynamicPP),
    dynamicPPP = int(mu::engraving::SymId::dynamicPPP),
    dynamicPPPP = int(mu::engraving::SymId::dynamicPPPP),
    dynamicPPPPP = int(mu::engraving::SymId::dynamicPPPPP),
    dynamicPPPPPP = int(mu::engraving::SymId::dynamicPPPPPP),
    dynamicPiano = int(mu::engraving::SymId::dynamicPiano),
    dynamicRinforzando = int(mu::engraving::SymId::dynamicRinforzando),
    dynamicRinforzando1 = int(mu::engraving::SymId::dynamicRinforzando1),
    dynamicRinforzando2 = int(mu::engraving::SymId::dynamicRinforzando2),
    dynamicSforzando = int(mu::engraving::SymId::dynamicSforzando),
    dynamicSforzando1 = int(mu::engraving::SymId::dynamicSforzando1),
    dynamicSforzandoPianissimo = int(mu::engraving::SymId::dynamicSforzandoPianissimo),
    dynamicSforzandoPiano = int(mu::engraving::SymId::dynamicSforzandoPiano),
    dynamicSforzato = int(mu::engraving::SymId::dynamicSforzato),
    dynamicSforzatoFF = int(mu::engraving::SymId::dynamicSforzatoFF),
    dynamicSforzatoPiano = int(mu::engraving::SymId::dynamicSforzatoPiano),
    dynamicZ = int(mu::engraving::SymId::dynamicZ),
    elecAudioChannelsEight = int(mu::engraving::SymId::elecAudioChannelsEight),
    elecAudioChannelsFive = int(mu::engraving::SymId::elecAudioChannelsFive),
    elecAudioChannelsFour = int(mu::engraving::SymId::elecAudioChannelsFour),
    elecAudioChannelsOne = int(mu::engraving::SymId::elecAudioChannelsOne),
    elecAudioChannelsSeven = int(mu::engraving::SymId::elecAudioChannelsSeven),
    elecAudioChannelsSix = int(mu::engraving::SymId::elecAudioChannelsSix),
    elecAudioChannelsThreeFrontal = int(mu::engraving::SymId::elecAudioChannelsThreeFrontal),
    elecAudioChannelsThreeSurround = int(mu::engraving::SymId::elecAudioChannelsThreeSurround),
    elecAudioChannelsTwo = int(mu::engraving::SymId::elecAudioChannelsTwo),
    elecAudioIn = int(mu::engraving::SymId::elecAudioIn),
    elecAudioMono = int(mu::engraving::SymId::elecAudioMono),
    elecAudioOut = int(mu::engraving::SymId::elecAudioOut),
    elecAudioStereo = int(mu::engraving::SymId::elecAudioStereo),
    elecCamera = int(mu::engraving::SymId::elecCamera),
    elecDataIn = int(mu::engraving::SymId::elecDataIn),
    elecDataOut = int(mu::engraving::SymId::elecDataOut),
    elecDisc = int(mu::engraving::SymId::elecDisc),
    elecDownload = int(mu::engraving::SymId::elecDownload),
    elecEject = int(mu::engraving::SymId::elecEject),
    elecFastForward = int(mu::engraving::SymId::elecFastForward),
    elecHeadphones = int(mu::engraving::SymId::elecHeadphones),
    elecHeadset = int(mu::engraving::SymId::elecHeadset),
    elecLineIn = int(mu::engraving::SymId::elecLineIn),
    elecLineOut = int(mu::engraving::SymId::elecLineOut),
    elecLoop = int(mu::engraving::SymId::elecLoop),
    elecLoudspeaker = int(mu::engraving::SymId::elecLoudspeaker),
    elecMIDIController0 = int(mu::engraving::SymId::elecMIDIController0),
    elecMIDIController100 = int(mu::engraving::SymId::elecMIDIController100),
    elecMIDIController20 = int(mu::engraving::SymId::elecMIDIController20),
    elecMIDIController40 = int(mu::engraving::SymId::elecMIDIController40),
    elecMIDIController60 = int(mu::engraving::SymId::elecMIDIController60),
    elecMIDIController80 = int(mu::engraving::SymId::elecMIDIController80),
    elecMIDIIn = int(mu::engraving::SymId::elecMIDIIn),
    elecMIDIOut = int(mu::engraving::SymId::elecMIDIOut),
    elecMicrophone = int(mu::engraving::SymId::elecMicrophone),
    elecMicrophoneMute = int(mu::engraving::SymId::elecMicrophoneMute),
    elecMicrophoneUnmute = int(mu::engraving::SymId::elecMicrophoneUnmute),
    elecMixingConsole = int(mu::engraving::SymId::elecMixingConsole),
    elecMonitor = int(mu::engraving::SymId::elecMonitor),
    elecMute = int(mu::engraving::SymId::elecMute),
    elecPause = int(mu::engraving::SymId::elecPause),
    elecPlay = int(mu::engraving::SymId::elecPlay),
    elecPowerOnOff = int(mu::engraving::SymId::elecPowerOnOff),
    elecProjector = int(mu::engraving::SymId::elecProjector),
    elecReplay = int(mu::engraving::SymId::elecReplay),
    elecRewind = int(mu::engraving::SymId::elecRewind),
    elecShuffle = int(mu::engraving::SymId::elecShuffle),
    elecSkipBackwards = int(mu::engraving::SymId::elecSkipBackwards),
    elecSkipForwards = int(mu::engraving::SymId::elecSkipForwards),
    elecStop = int(mu::engraving::SymId::elecStop),
    elecTape = int(mu::engraving::SymId::elecTape),
    elecUSB = int(mu::engraving::SymId::elecUSB),
    elecUnmute = int(mu::engraving::SymId::elecUnmute),
    elecUpload = int(mu::engraving::SymId::elecUpload),
    elecVideoCamera = int(mu::engraving::SymId::elecVideoCamera),
    elecVideoIn = int(mu::engraving::SymId::elecVideoIn),
    elecVideoOut = int(mu::engraving::SymId::elecVideoOut),
    elecVolumeFader = int(mu::engraving::SymId::elecVolumeFader),
    elecVolumeFaderThumb = int(mu::engraving::SymId::elecVolumeFaderThumb),
    elecVolumeLevel0 = int(mu::engraving::SymId::elecVolumeLevel0),
    elecVolumeLevel100 = int(mu::engraving::SymId::elecVolumeLevel100),
    elecVolumeLevel20 = int(mu::engraving::SymId::elecVolumeLevel20),
    elecVolumeLevel40 = int(mu::engraving::SymId::elecVolumeLevel40),
    elecVolumeLevel60 = int(mu::engraving::SymId::elecVolumeLevel60),
    elecVolumeLevel80 = int(mu::engraving::SymId::elecVolumeLevel80),
    fClef = int(mu::engraving::SymId::fClef),
    fClef15ma = int(mu::engraving::SymId::fClef15ma),
    fClef15mb = int(mu::engraving::SymId::fClef15mb),
    fClef8va = int(mu::engraving::SymId::fClef8va),
    fClef8vb = int(mu::engraving::SymId::fClef8vb),
    fClefArrowDown = int(mu::engraving::SymId::fClefArrowDown),
    fClefArrowUp = int(mu::engraving::SymId::fClefArrowUp),
    fClefChange = int(mu::engraving::SymId::fClefChange),
    fClefReversed = int(mu::engraving::SymId::fClefReversed),
    fClefTurned = int(mu::engraving::SymId::fClefTurned),
    fermataAbove = int(mu::engraving::SymId::fermataAbove),
    fermataBelow = int(mu::engraving::SymId::fermataBelow),
    fermataLongAbove = int(mu::engraving::SymId::fermataLongAbove),
    fermataLongBelow = int(mu::engraving::SymId::fermataLongBelow),
    fermataLongHenzeAbove = int(mu::engraving::SymId::fermataLongHenzeAbove),
    fermataLongHenzeBelow = int(mu::engraving::SymId::fermataLongHenzeBelow),
    fermataShortAbove = int(mu::engraving::SymId::fermataShortAbove),
    fermataShortBelow = int(mu::engraving::SymId::fermataShortBelow),
    fermataShortHenzeAbove = int(mu::engraving::SymId::fermataShortHenzeAbove),
    fermataShortHenzeBelow = int(mu::engraving::SymId::fermataShortHenzeBelow),
    fermataVeryLongAbove = int(mu::engraving::SymId::fermataVeryLongAbove),
    fermataVeryLongBelow = int(mu::engraving::SymId::fermataVeryLongBelow),
    fermataVeryShortAbove = int(mu::engraving::SymId::fermataVeryShortAbove),
    fermataVeryShortBelow = int(mu::engraving::SymId::fermataVeryShortBelow),
    figbass0 = int(mu::engraving::SymId::figbass0),
    figbass1 = int(mu::engraving::SymId::figbass1),
    figbass2 = int(mu::engraving::SymId::figbass2),
    figbass2Raised = int(mu::engraving::SymId::figbass2Raised),
    figbass3 = int(mu::engraving::SymId::figbass3),
    figbass4 = int(mu::engraving::SymId::figbass4),
    figbass4Raised = int(mu::engraving::SymId::figbass4Raised),
    figbass5 = int(mu::engraving::SymId::figbass5),
    figbass5Raised1 = int(mu::engraving::SymId::figbass5Raised1),
    figbass5Raised2 = int(mu::engraving::SymId::figbass5Raised2),
    figbass5Raised3 = int(mu::engraving::SymId::figbass5Raised3),
    figbass6 = int(mu::engraving::SymId::figbass6),
    figbass6Raised = int(mu::engraving::SymId::figbass6Raised),
    figbass6Raised2 = int(mu::engraving::SymId::figbass6Raised2),
    figbass7 = int(mu::engraving::SymId::figbass7),
    figbass7Diminished = int(mu::engraving::SymId::figbass7Diminished),
    figbass7Raised1 = int(mu::engraving::SymId::figbass7Raised1),
    figbass7Raised2 = int(mu::engraving::SymId::figbass7Raised2),
    figbass8 = int(mu::engraving::SymId::figbass8),
    figbass9 = int(mu::engraving::SymId::figbass9),
    figbass9Raised = int(mu::engraving::SymId::figbass9Raised),
    figbassBracketLeft = int(mu::engraving::SymId::figbassBracketLeft),
    figbassBracketRight = int(mu::engraving::SymId::figbassBracketRight),
    figbassCombiningLowering = int(mu::engraving::SymId::figbassCombiningLowering),
    figbassCombiningRaising = int(mu::engraving::SymId::figbassCombiningRaising),
    figbassDoubleFlat = int(mu::engraving::SymId::figbassDoubleFlat),
    figbassDoubleSharp = int(mu::engraving::SymId::figbassDoubleSharp),
    figbassFlat = int(mu::engraving::SymId::figbassFlat),
    figbassNatural = int(mu::engraving::SymId::figbassNatural),
    figbassParensLeft = int(mu::engraving::SymId::figbassParensLeft),
    figbassParensRight = int(mu::engraving::SymId::figbassParensRight),
    figbassPlus = int(mu::engraving::SymId::figbassPlus),
    figbassSharp = int(mu::engraving::SymId::figbassSharp),
    figbassTripleFlat = int(mu::engraving::SymId::figbassTripleFlat),
    figbassTripleSharp = int(mu::engraving::SymId::figbassTripleSharp),
    fingering0 = int(mu::engraving::SymId::fingering0),
    fingering0Italic = int(mu::engraving::SymId::fingering0Italic),
    fingering1 = int(mu::engraving::SymId::fingering1),
    fingering1Italic = int(mu::engraving::SymId::fingering1Italic),
    fingering2 = int(mu::engraving::SymId::fingering2),
    fingering2Italic = int(mu::engraving::SymId::fingering2Italic),
    fingering3 = int(mu::engraving::SymId::fingering3),
    fingering3Italic = int(mu::engraving::SymId::fingering3Italic),
    fingering4 = int(mu::engraving::SymId::fingering4),
    fingering4Italic = int(mu::engraving::SymId::fingering4Italic),
    fingering5 = int(mu::engraving::SymId::fingering5),
    fingering5Italic = int(mu::engraving::SymId::fingering5Italic),
    fingering6 = int(mu::engraving::SymId::fingering6),
    fingering6Italic = int(mu::engraving::SymId::fingering6Italic),
    fingering7 = int(mu::engraving::SymId::fingering7),
    fingering7Italic = int(mu::engraving::SymId::fingering7Italic),
    fingering8 = int(mu::engraving::SymId::fingering8),
    fingering8Italic = int(mu::engraving::SymId::fingering8Italic),
    fingering9 = int(mu::engraving::SymId::fingering9),
    fingering9Italic = int(mu::engraving::SymId::fingering9Italic),
    fingeringALower = int(mu::engraving::SymId::fingeringALower),
    fingeringCLower = int(mu::engraving::SymId::fingeringCLower),
    fingeringELower = int(mu::engraving::SymId::fingeringELower),
    fingeringILower = int(mu::engraving::SymId::fingeringILower),
    fingeringLeftBracket = int(mu::engraving::SymId::fingeringLeftBracket),
    fingeringLeftBracketItalic = int(mu::engraving::SymId::fingeringLeftBracketItalic),
    fingeringLeftParenthesis = int(mu::engraving::SymId::fingeringLeftParenthesis),
    fingeringLeftParenthesisItalic = int(mu::engraving::SymId::fingeringLeftParenthesisItalic),
    fingeringMLower = int(mu::engraving::SymId::fingeringMLower),
    fingeringMultipleNotes = int(mu::engraving::SymId::fingeringMultipleNotes),
    fingeringOLower = int(mu::engraving::SymId::fingeringOLower),
    fingeringPLower = int(mu::engraving::SymId::fingeringPLower),
    fingeringQLower = int(mu::engraving::SymId::fingeringQLower),
    fingeringRightBracket = int(mu::engraving::SymId::fingeringRightBracket),
    fingeringRightBracketItalic = int(mu::engraving::SymId::fingeringRightBracketItalic),
    fingeringRightParenthesis = int(mu::engraving::SymId::fingeringRightParenthesis),
    fingeringRightParenthesisItalic = int(mu::engraving::SymId::fingeringRightParenthesisItalic),
    fingeringSLower = int(mu::engraving::SymId::fingeringSLower),
    fingeringSeparatorMiddleDot = int(mu::engraving::SymId::fingeringSeparatorMiddleDot),
    fingeringSeparatorMiddleDotWhite = int(mu::engraving::SymId::fingeringSeparatorMiddleDotWhite),
    fingeringSeparatorSlash = int(mu::engraving::SymId::fingeringSeparatorSlash),
    fingeringSubstitutionAbove = int(mu::engraving::SymId::fingeringSubstitutionAbove),
    fingeringSubstitutionBelow = int(mu::engraving::SymId::fingeringSubstitutionBelow),
    fingeringSubstitutionDash = int(mu::engraving::SymId::fingeringSubstitutionDash),
    fingeringTLower = int(mu::engraving::SymId::fingeringTLower),
    fingeringTUpper = int(mu::engraving::SymId::fingeringTUpper),
    fingeringXLower = int(mu::engraving::SymId::fingeringXLower),
    flag1024thDown = int(mu::engraving::SymId::flag1024thDown),
    flag1024thUp = int(mu::engraving::SymId::flag1024thUp),
    flag128thDown = int(mu::engraving::SymId::flag128thDown),
    flag128thUp = int(mu::engraving::SymId::flag128thUp),
    flag16thDown = int(mu::engraving::SymId::flag16thDown),
    flag16thUp = int(mu::engraving::SymId::flag16thUp),
    flag256thDown = int(mu::engraving::SymId::flag256thDown),
    flag256thUp = int(mu::engraving::SymId::flag256thUp),
    flag32ndDown = int(mu::engraving::SymId::flag32ndDown),
    flag32ndUp = int(mu::engraving::SymId::flag32ndUp),
    flag512thDown = int(mu::engraving::SymId::flag512thDown),
    flag512thUp = int(mu::engraving::SymId::flag512thUp),
    flag64thDown = int(mu::engraving::SymId::flag64thDown),
    flag64thUp = int(mu::engraving::SymId::flag64thUp),
    flag8thDown = int(mu::engraving::SymId::flag8thDown),
    flag8thUp = int(mu::engraving::SymId::flag8thUp),
    flagInternalDown = int(mu::engraving::SymId::flagInternalDown),
    flagInternalUp = int(mu::engraving::SymId::flagInternalUp),
    fretboard3String = int(mu::engraving::SymId::fretboard3String),
    fretboard3StringNut = int(mu::engraving::SymId::fretboard3StringNut),
    fretboard4String = int(mu::engraving::SymId::fretboard4String),
    fretboard4StringNut = int(mu::engraving::SymId::fretboard4StringNut),
    fretboard5String = int(mu::engraving::SymId::fretboard5String),
    fretboard5StringNut = int(mu::engraving::SymId::fretboard5StringNut),
    fretboard6String = int(mu::engraving::SymId::fretboard6String),
    fretboard6StringNut = int(mu::engraving::SymId::fretboard6StringNut),
    fretboardFilledCircle = int(mu::engraving::SymId::fretboardFilledCircle),
    fretboardO = int(mu::engraving::SymId::fretboardO),
    fretboardX = int(mu::engraving::SymId::fretboardX),
    functionAngleLeft = int(mu::engraving::SymId::functionAngleLeft),
    functionAngleRight = int(mu::engraving::SymId::functionAngleRight),
    functionBracketLeft = int(mu::engraving::SymId::functionBracketLeft),
    functionBracketRight = int(mu::engraving::SymId::functionBracketRight),
    functionDD = int(mu::engraving::SymId::functionDD),
    functionDLower = int(mu::engraving::SymId::functionDLower),
    functionDUpper = int(mu::engraving::SymId::functionDUpper),
    functionEight = int(mu::engraving::SymId::functionEight),
    functionFUpper = int(mu::engraving::SymId::functionFUpper),
    functionFive = int(mu::engraving::SymId::functionFive),
    functionFour = int(mu::engraving::SymId::functionFour),
    functionGLower = int(mu::engraving::SymId::functionGLower),
    functionGUpper = int(mu::engraving::SymId::functionGUpper),
    functionGreaterThan = int(mu::engraving::SymId::functionGreaterThan),
    functionILower = int(mu::engraving::SymId::functionILower),
    functionIUpper = int(mu::engraving::SymId::functionIUpper),
    functionKLower = int(mu::engraving::SymId::functionKLower),
    functionKUpper = int(mu::engraving::SymId::functionKUpper),
    functionLLower = int(mu::engraving::SymId::functionLLower),
    functionLUpper = int(mu::engraving::SymId::functionLUpper),
    functionLessThan = int(mu::engraving::SymId::functionLessThan),
    functionMLower = int(mu::engraving::SymId::functionMLower),
    functionMUpper = int(mu::engraving::SymId::functionMUpper),
    functionMinus = int(mu::engraving::SymId::functionMinus),
    functionNLower = int(mu::engraving::SymId::functionNLower),
    functionNUpper = int(mu::engraving::SymId::functionNUpper),
    functionNUpperSuperscript = int(mu::engraving::SymId::functionNUpperSuperscript),
    functionNine = int(mu::engraving::SymId::functionNine),
    functionOne = int(mu::engraving::SymId::functionOne),
    functionPLower = int(mu::engraving::SymId::functionPLower),
    functionPUpper = int(mu::engraving::SymId::functionPUpper),
    functionParensLeft = int(mu::engraving::SymId::functionParensLeft),
    functionParensRight = int(mu::engraving::SymId::functionParensRight),
    functionPlus = int(mu::engraving::SymId::functionPlus),
    functionRLower = int(mu::engraving::SymId::functionRLower),
    functionRepetition1 = int(mu::engraving::SymId::functionRepetition1),
    functionRepetition2 = int(mu::engraving::SymId::functionRepetition2),
    functionRing = int(mu::engraving::SymId::functionRing),
    functionSLower = int(mu::engraving::SymId::functionSLower),
    functionSSLower = int(mu::engraving::SymId::functionSSLower),
    functionSSUpper = int(mu::engraving::SymId::functionSSUpper),
    functionSUpper = int(mu::engraving::SymId::functionSUpper),
    functionSeven = int(mu::engraving::SymId::functionSeven),
    functionSix = int(mu::engraving::SymId::functionSix),
    functionSlashedDD = int(mu::engraving::SymId::functionSlashedDD),
    functionTLower = int(mu::engraving::SymId::functionTLower),
    functionTUpper = int(mu::engraving::SymId::functionTUpper),
    functionThree = int(mu::engraving::SymId::functionThree),
    functionTwo = int(mu::engraving::SymId::functionTwo),
    functionVLower = int(mu::engraving::SymId::functionVLower),
    functionVUpper = int(mu::engraving::SymId::functionVUpper),
    functionZero = int(mu::engraving::SymId::functionZero),
    gClef = int(mu::engraving::SymId::gClef),
    gClef15ma = int(mu::engraving::SymId::gClef15ma),
    gClef15mb = int(mu::engraving::SymId::gClef15mb),
    gClef8va = int(mu::engraving::SymId::gClef8va),
    gClef8vb = int(mu::engraving::SymId::gClef8vb),
    gClef8vbCClef = int(mu::engraving::SymId::gClef8vbCClef),
    gClef8vbOld = int(mu::engraving::SymId::gClef8vbOld),
    gClef8vbParens = int(mu::engraving::SymId::gClef8vbParens),
    gClefArrowDown = int(mu::engraving::SymId::gClefArrowDown),
    gClefArrowUp = int(mu::engraving::SymId::gClefArrowUp),
    gClefChange = int(mu::engraving::SymId::gClefChange),
    gClefLigatedNumberAbove = int(mu::engraving::SymId::gClefLigatedNumberAbove),
    gClefLigatedNumberBelow = int(mu::engraving::SymId::gClefLigatedNumberBelow),
    gClefReversed = int(mu::engraving::SymId::gClefReversed),
    gClefTurned = int(mu::engraving::SymId::gClefTurned),
    glissandoDown = int(mu::engraving::SymId::glissandoDown),
    glissandoUp = int(mu::engraving::SymId::glissandoUp),
    graceNoteAcciaccaturaStemDown = int(mu::engraving::SymId::graceNoteAcciaccaturaStemDown),
    graceNoteAcciaccaturaStemUp = int(mu::engraving::SymId::graceNoteAcciaccaturaStemUp),
    graceNoteAppoggiaturaStemDown = int(mu::engraving::SymId::graceNoteAppoggiaturaStemDown),
    graceNoteAppoggiaturaStemUp = int(mu::engraving::SymId::graceNoteAppoggiaturaStemUp),
    graceNoteSlashStemDown = int(mu::engraving::SymId::graceNoteSlashStemDown),
    graceNoteSlashStemUp = int(mu::engraving::SymId::graceNoteSlashStemUp),
    guitarBarreFull = int(mu::engraving::SymId::guitarBarreFull),
    guitarBarreHalf = int(mu::engraving::SymId::guitarBarreHalf),
    guitarClosePedal = int(mu::engraving::SymId::guitarClosePedal),
    guitarFadeIn = int(mu::engraving::SymId::guitarFadeIn),
    guitarFadeOut = int(mu::engraving::SymId::guitarFadeOut),
    guitarGolpe = int(mu::engraving::SymId::guitarGolpe),
    guitarHalfOpenPedal = int(mu::engraving::SymId::guitarHalfOpenPedal),
    guitarLeftHandTapping = int(mu::engraving::SymId::guitarLeftHandTapping),
    guitarOpenPedal = int(mu::engraving::SymId::guitarOpenPedal),
    guitarRightHandTapping = int(mu::engraving::SymId::guitarRightHandTapping),
    guitarShake = int(mu::engraving::SymId::guitarShake),
    guitarString0 = int(mu::engraving::SymId::guitarString0),
    guitarString1 = int(mu::engraving::SymId::guitarString1),
    guitarString10 = int(mu::engraving::SymId::guitarString10),
    guitarString11 = int(mu::engraving::SymId::guitarString11),
    guitarString12 = int(mu::engraving::SymId::guitarString12),
    guitarString13 = int(mu::engraving::SymId::guitarString13),
    guitarString2 = int(mu::engraving::SymId::guitarString2),
    guitarString3 = int(mu::engraving::SymId::guitarString3),
    guitarString4 = int(mu::engraving::SymId::guitarString4),
    guitarString5 = int(mu::engraving::SymId::guitarString5),
    guitarString6 = int(mu::engraving::SymId::guitarString6),
    guitarString7 = int(mu::engraving::SymId::guitarString7),
    guitarString8 = int(mu::engraving::SymId::guitarString8),
    guitarString9 = int(mu::engraving::SymId::guitarString9),
    guitarStrumDown = int(mu::engraving::SymId::guitarStrumDown),
    guitarStrumUp = int(mu::engraving::SymId::guitarStrumUp),
    guitarVibratoBarDip = int(mu::engraving::SymId::guitarVibratoBarDip),
    guitarVibratoBarScoop = int(mu::engraving::SymId::guitarVibratoBarScoop),
    guitarVibratoStroke = int(mu::engraving::SymId::guitarVibratoStroke),
    guitarVolumeSwell = int(mu::engraving::SymId::guitarVolumeSwell),
    guitarWideVibratoStroke = int(mu::engraving::SymId::guitarWideVibratoStroke),
    handbellsBelltree = int(mu::engraving::SymId::handbellsBelltree),
    handbellsDamp3 = int(mu::engraving::SymId::handbellsDamp3),
    handbellsEcho1 = int(mu::engraving::SymId::handbellsEcho1),
    handbellsEcho2 = int(mu::engraving::SymId::handbellsEcho2),
    handbellsGyro = int(mu::engraving::SymId::handbellsGyro),
    handbellsHandMartellato = int(mu::engraving::SymId::handbellsHandMartellato),
    handbellsMalletBellOnTable = int(mu::engraving::SymId::handbellsMalletBellOnTable),
    handbellsMalletBellSuspended = int(mu::engraving::SymId::handbellsMalletBellSuspended),
    handbellsMalletLft = int(mu::engraving::SymId::handbellsMalletLft),
    handbellsMartellato = int(mu::engraving::SymId::handbellsMartellato),
    handbellsMartellatoLift = int(mu::engraving::SymId::handbellsMartellatoLift),
    handbellsMutedMartellato = int(mu::engraving::SymId::handbellsMutedMartellato),
    handbellsPluckLift = int(mu::engraving::SymId::handbellsPluckLift),
    handbellsSwing = int(mu::engraving::SymId::handbellsSwing),
    handbellsSwingDown = int(mu::engraving::SymId::handbellsSwingDown),
    handbellsSwingUp = int(mu::engraving::SymId::handbellsSwingUp),
    handbellsTablePairBells = int(mu::engraving::SymId::handbellsTablePairBells),
    handbellsTableSingleBell = int(mu::engraving::SymId::handbellsTableSingleBell),
    harpMetalRod = int(mu::engraving::SymId::harpMetalRod),
    harpPedalCentered = int(mu::engraving::SymId::harpPedalCentered),
    harpPedalDivider = int(mu::engraving::SymId::harpPedalDivider),
    harpPedalLowered = int(mu::engraving::SymId::harpPedalLowered),
    harpPedalRaised = int(mu::engraving::SymId::harpPedalRaised),
    harpSalzedoAeolianAscending = int(mu::engraving::SymId::harpSalzedoAeolianAscending),
    harpSalzedoAeolianDescending = int(mu::engraving::SymId::harpSalzedoAeolianDescending),
    harpSalzedoDampAbove = int(mu::engraving::SymId::harpSalzedoDampAbove),
    harpSalzedoDampBelow = int(mu::engraving::SymId::harpSalzedoDampBelow),
    harpSalzedoDampBothHands = int(mu::engraving::SymId::harpSalzedoDampBothHands),
    harpSalzedoDampLowStrings = int(mu::engraving::SymId::harpSalzedoDampLowStrings),
    harpSalzedoFluidicSoundsLeft = int(mu::engraving::SymId::harpSalzedoFluidicSoundsLeft),
    harpSalzedoFluidicSoundsRight = int(mu::engraving::SymId::harpSalzedoFluidicSoundsRight),
    harpSalzedoIsolatedSounds = int(mu::engraving::SymId::harpSalzedoIsolatedSounds),
    harpSalzedoMetallicSounds = int(mu::engraving::SymId::harpSalzedoMetallicSounds),
    harpSalzedoMetallicSoundsOneString = int(mu::engraving::SymId::harpSalzedoMetallicSoundsOneString),
    harpSalzedoMuffleTotally = int(mu::engraving::SymId::harpSalzedoMuffleTotally),
    harpSalzedoOboicFlux = int(mu::engraving::SymId::harpSalzedoOboicFlux),
    harpSalzedoPlayUpperEnd = int(mu::engraving::SymId::harpSalzedoPlayUpperEnd),
    harpSalzedoSlideWithSuppleness = int(mu::engraving::SymId::harpSalzedoSlideWithSuppleness),
    harpSalzedoSnareDrum = int(mu::engraving::SymId::harpSalzedoSnareDrum),
    harpSalzedoTamTamSounds = int(mu::engraving::SymId::harpSalzedoTamTamSounds),
    harpSalzedoThunderEffect = int(mu::engraving::SymId::harpSalzedoThunderEffect),
    harpSalzedoTimpanicSounds = int(mu::engraving::SymId::harpSalzedoTimpanicSounds),
    harpSalzedoWhistlingSounds = int(mu::engraving::SymId::harpSalzedoWhistlingSounds),
    harpStringNoiseStem = int(mu::engraving::SymId::harpStringNoiseStem),
    harpTuningKey = int(mu::engraving::SymId::harpTuningKey),
    harpTuningKeyGlissando = int(mu::engraving::SymId::harpTuningKeyGlissando),
    harpTuningKeyHandle = int(mu::engraving::SymId::harpTuningKeyHandle),
    harpTuningKeyShank = int(mu::engraving::SymId::harpTuningKeyShank),
    indianDrumClef = int(mu::engraving::SymId::indianDrumClef),
    kahnBackChug = int(mu::engraving::SymId::kahnBackChug),
    kahnBackFlap = int(mu::engraving::SymId::kahnBackFlap),
    kahnBackRiff = int(mu::engraving::SymId::kahnBackRiff),
    kahnBackRip = int(mu::engraving::SymId::kahnBackRip),
    kahnBallChange = int(mu::engraving::SymId::kahnBallChange),
    kahnBallDig = int(mu::engraving::SymId::kahnBallDig),
    kahnBrushBackward = int(mu::engraving::SymId::kahnBrushBackward),
    kahnBrushForward = int(mu::engraving::SymId::kahnBrushForward),
    kahnChug = int(mu::engraving::SymId::kahnChug),
    kahnClap = int(mu::engraving::SymId::kahnClap),
    kahnDoubleSnap = int(mu::engraving::SymId::kahnDoubleSnap),
    kahnDoubleWing = int(mu::engraving::SymId::kahnDoubleWing),
    kahnDrawStep = int(mu::engraving::SymId::kahnDrawStep),
    kahnDrawTap = int(mu::engraving::SymId::kahnDrawTap),
    kahnFlam = int(mu::engraving::SymId::kahnFlam),
    kahnFlap = int(mu::engraving::SymId::kahnFlap),
    kahnFlapStep = int(mu::engraving::SymId::kahnFlapStep),
    kahnFlat = int(mu::engraving::SymId::kahnFlat),
    kahnFleaHop = int(mu::engraving::SymId::kahnFleaHop),
    kahnFleaTap = int(mu::engraving::SymId::kahnFleaTap),
    kahnGraceTap = int(mu::engraving::SymId::kahnGraceTap),
    kahnGraceTapChange = int(mu::engraving::SymId::kahnGraceTapChange),
    kahnGraceTapHop = int(mu::engraving::SymId::kahnGraceTapHop),
    kahnGraceTapStamp = int(mu::engraving::SymId::kahnGraceTapStamp),
    kahnHeel = int(mu::engraving::SymId::kahnHeel),
    kahnHeelChange = int(mu::engraving::SymId::kahnHeelChange),
    kahnHeelClick = int(mu::engraving::SymId::kahnHeelClick),
    kahnHeelDrop = int(mu::engraving::SymId::kahnHeelDrop),
    kahnHeelStep = int(mu::engraving::SymId::kahnHeelStep),
    kahnHeelTap = int(mu::engraving::SymId::kahnHeelTap),
    kahnHop = int(mu::engraving::SymId::kahnHop),
    kahnJumpApart = int(mu::engraving::SymId::kahnJumpApart),
    kahnJumpTogether = int(mu::engraving::SymId::kahnJumpTogether),
    kahnKneeInward = int(mu::engraving::SymId::kahnKneeInward),
    kahnKneeOutward = int(mu::engraving::SymId::kahnKneeOutward),
    kahnLeap = int(mu::engraving::SymId::kahnLeap),
    kahnLeapFlatFoot = int(mu::engraving::SymId::kahnLeapFlatFoot),
    kahnLeapHeelClick = int(mu::engraving::SymId::kahnLeapHeelClick),
    kahnLeftCatch = int(mu::engraving::SymId::kahnLeftCatch),
    kahnLeftCross = int(mu::engraving::SymId::kahnLeftCross),
    kahnLeftFoot = int(mu::engraving::SymId::kahnLeftFoot),
    kahnLeftToeStrike = int(mu::engraving::SymId::kahnLeftToeStrike),
    kahnLeftTurn = int(mu::engraving::SymId::kahnLeftTurn),
    kahnOverTheTop = int(mu::engraving::SymId::kahnOverTheTop),
    kahnOverTheTopTap = int(mu::engraving::SymId::kahnOverTheTopTap),
    kahnPull = int(mu::engraving::SymId::kahnPull),
    kahnPush = int(mu::engraving::SymId::kahnPush),
    kahnRiff = int(mu::engraving::SymId::kahnRiff),
    kahnRiffle = int(mu::engraving::SymId::kahnRiffle),
    kahnRightCatch = int(mu::engraving::SymId::kahnRightCatch),
    kahnRightCross = int(mu::engraving::SymId::kahnRightCross),
    kahnRightFoot = int(mu::engraving::SymId::kahnRightFoot),
    kahnRightToeStrike = int(mu::engraving::SymId::kahnRightToeStrike),
    kahnRightTurn = int(mu::engraving::SymId::kahnRightTurn),
    kahnRip = int(mu::engraving::SymId::kahnRip),
    kahnRipple = int(mu::engraving::SymId::kahnRipple),
    kahnScrape = int(mu::engraving::SymId::kahnScrape),
    kahnScuff = int(mu::engraving::SymId::kahnScuff),
    kahnScuffle = int(mu::engraving::SymId::kahnScuffle),
    kahnShuffle = int(mu::engraving::SymId::kahnShuffle),
    kahnSlam = int(mu::engraving::SymId::kahnSlam),
    kahnSlap = int(mu::engraving::SymId::kahnSlap),
    kahnSlideStep = int(mu::engraving::SymId::kahnSlideStep),
    kahnSlideTap = int(mu::engraving::SymId::kahnSlideTap),
    kahnSnap = int(mu::engraving::SymId::kahnSnap),
    kahnStamp = int(mu::engraving::SymId::kahnStamp),
    kahnStampStamp = int(mu::engraving::SymId::kahnStampStamp),
    kahnStep = int(mu::engraving::SymId::kahnStep),
    kahnStepStamp = int(mu::engraving::SymId::kahnStepStamp),
    kahnStomp = int(mu::engraving::SymId::kahnStomp),
    kahnStompBrush = int(mu::engraving::SymId::kahnStompBrush),
    kahnTap = int(mu::engraving::SymId::kahnTap),
    kahnToe = int(mu::engraving::SymId::kahnToe),
    kahnToeClick = int(mu::engraving::SymId::kahnToeClick),
    kahnToeDrop = int(mu::engraving::SymId::kahnToeDrop),
    kahnToeStep = int(mu::engraving::SymId::kahnToeStep),
    kahnToeTap = int(mu::engraving::SymId::kahnToeTap),
    kahnTrench = int(mu::engraving::SymId::kahnTrench),
    kahnWing = int(mu::engraving::SymId::kahnWing),
    kahnWingChange = int(mu::engraving::SymId::kahnWingChange),
    kahnZank = int(mu::engraving::SymId::kahnZank),
    kahnZink = int(mu::engraving::SymId::kahnZink),
    keyboardBebung2DotsAbove = int(mu::engraving::SymId::keyboardBebung2DotsAbove),
    keyboardBebung2DotsBelow = int(mu::engraving::SymId::keyboardBebung2DotsBelow),
    keyboardBebung3DotsAbove = int(mu::engraving::SymId::keyboardBebung3DotsAbove),
    keyboardBebung3DotsBelow = int(mu::engraving::SymId::keyboardBebung3DotsBelow),
    keyboardBebung4DotsAbove = int(mu::engraving::SymId::keyboardBebung4DotsAbove),
    keyboardBebung4DotsBelow = int(mu::engraving::SymId::keyboardBebung4DotsBelow),
    keyboardLeftPedalPictogram = int(mu::engraving::SymId::keyboardLeftPedalPictogram),
    keyboardMiddlePedalPictogram = int(mu::engraving::SymId::keyboardMiddlePedalPictogram),
    keyboardPedalD = int(mu::engraving::SymId::keyboardPedalD),
    keyboardPedalDot = int(mu::engraving::SymId::keyboardPedalDot),
    keyboardPedalE = int(mu::engraving::SymId::keyboardPedalE),
    keyboardPedalHalf = int(mu::engraving::SymId::keyboardPedalHalf),
    keyboardPedalHalf2 = int(mu::engraving::SymId::keyboardPedalHalf2),
    keyboardPedalHalf3 = int(mu::engraving::SymId::keyboardPedalHalf3),
    keyboardPedalHeel1 = int(mu::engraving::SymId::keyboardPedalHeel1),
    keyboardPedalHeel2 = int(mu::engraving::SymId::keyboardPedalHeel2),
    keyboardPedalHeel3 = int(mu::engraving::SymId::keyboardPedalHeel3),
    keyboardPedalHeelToToe = int(mu::engraving::SymId::keyboardPedalHeelToToe),
    keyboardPedalHeelToe = int(mu::engraving::SymId::keyboardPedalHeelToe),
    keyboardPedalHookEnd = int(mu::engraving::SymId::keyboardPedalHookEnd),
    keyboardPedalHookStart = int(mu::engraving::SymId::keyboardPedalHookStart),
    keyboardPedalHyphen = int(mu::engraving::SymId::keyboardPedalHyphen),
    keyboardPedalP = int(mu::engraving::SymId::keyboardPedalP),
    keyboardPedalParensLeft = int(mu::engraving::SymId::keyboardPedalParensLeft),
    keyboardPedalParensRight = int(mu::engraving::SymId::keyboardPedalParensRight),
    keyboardPedalPed = int(mu::engraving::SymId::keyboardPedalPed),
    keyboardPedalS = int(mu::engraving::SymId::keyboardPedalS),
    keyboardPedalSost = int(mu::engraving::SymId::keyboardPedalSost),
    keyboardPedalToe1 = int(mu::engraving::SymId::keyboardPedalToe1),
    keyboardPedalToe2 = int(mu::engraving::SymId::keyboardPedalToe2),
    keyboardPedalToeToHeel = int(mu::engraving::SymId::keyboardPedalToeToHeel),
    keyboardPedalUp = int(mu::engraving::SymId::keyboardPedalUp),
    keyboardPedalUpNotch = int(mu::engraving::SymId::keyboardPedalUpNotch),
    keyboardPedalUpSpecial = int(mu::engraving::SymId::keyboardPedalUpSpecial),
    keyboardPlayWithLH = int(mu::engraving::SymId::keyboardPlayWithLH),
    keyboardPlayWithLHEnd = int(mu::engraving::SymId::keyboardPlayWithLHEnd),
    keyboardPlayWithRH = int(mu::engraving::SymId::keyboardPlayWithRH),
    keyboardPlayWithRHEnd = int(mu::engraving::SymId::keyboardPlayWithRHEnd),
    keyboardPluckInside = int(mu::engraving::SymId::keyboardPluckInside),
    keyboardRightPedalPictogram = int(mu::engraving::SymId::keyboardRightPedalPictogram),
    kievanAccidentalFlat = int(mu::engraving::SymId::kievanAccidentalFlat),
    kievanAccidentalSharp = int(mu::engraving::SymId::kievanAccidentalSharp),
    kievanAugmentationDot = int(mu::engraving::SymId::kievanAugmentationDot),
    kievanCClef = int(mu::engraving::SymId::kievanCClef),
    kievanEndingSymbol = int(mu::engraving::SymId::kievanEndingSymbol),
    kievanNote8thStemDown = int(mu::engraving::SymId::kievanNote8thStemDown),
    kievanNote8thStemUp = int(mu::engraving::SymId::kievanNote8thStemUp),
    kievanNoteBeam = int(mu::engraving::SymId::kievanNoteBeam),
    kievanNoteHalfStaffLine = int(mu::engraving::SymId::kievanNoteHalfStaffLine),
    kievanNoteHalfStaffSpace = int(mu::engraving::SymId::kievanNoteHalfStaffSpace),
    kievanNoteQuarterStemDown = int(mu::engraving::SymId::kievanNoteQuarterStemDown),
    kievanNoteQuarterStemUp = int(mu::engraving::SymId::kievanNoteQuarterStemUp),
    kievanNoteReciting = int(mu::engraving::SymId::kievanNoteReciting),
    kievanNoteWhole = int(mu::engraving::SymId::kievanNoteWhole),
    kievanNoteWholeFinal = int(mu::engraving::SymId::kievanNoteWholeFinal),
    kodalyHandDo = int(mu::engraving::SymId::kodalyHandDo),
    kodalyHandFa = int(mu::engraving::SymId::kodalyHandFa),
    kodalyHandLa = int(mu::engraving::SymId::kodalyHandLa),
    kodalyHandMi = int(mu::engraving::SymId::kodalyHandMi),
    kodalyHandRe = int(mu::engraving::SymId::kodalyHandRe),
    kodalyHandSo = int(mu::engraving::SymId::kodalyHandSo),
    kodalyHandTi = int(mu::engraving::SymId::kodalyHandTi),
    leftRepeatSmall = int(mu::engraving::SymId::leftRepeatSmall),
    legerLine = int(mu::engraving::SymId::legerLine),
    legerLineNarrow = int(mu::engraving::SymId::legerLineNarrow),
    legerLineWide = int(mu::engraving::SymId::legerLineWide),
    luteBarlineEndRepeat = int(mu::engraving::SymId::luteBarlineEndRepeat),
    luteBarlineFinal = int(mu::engraving::SymId::luteBarlineFinal),
    luteBarlineStartRepeat = int(mu::engraving::SymId::luteBarlineStartRepeat),
    luteDuration16th = int(mu::engraving::SymId::luteDuration16th),
    luteDuration32nd = int(mu::engraving::SymId::luteDuration32nd),
    luteDuration8th = int(mu::engraving::SymId::luteDuration8th),
    luteDurationDoubleWhole = int(mu::engraving::SymId::luteDurationDoubleWhole),
    luteDurationHalf = int(mu::engraving::SymId::luteDurationHalf),
    luteDurationQuarter = int(mu::engraving::SymId::luteDurationQuarter),
    luteDurationWhole = int(mu::engraving::SymId::luteDurationWhole),
    luteFingeringRHFirst = int(mu::engraving::SymId::luteFingeringRHFirst),
    luteFingeringRHSecond = int(mu::engraving::SymId::luteFingeringRHSecond),
    luteFingeringRHThird = int(mu::engraving::SymId::luteFingeringRHThird),
    luteFingeringRHThumb = int(mu::engraving::SymId::luteFingeringRHThumb),
    luteFrench10thCourse = int(mu::engraving::SymId::luteFrench10thCourse),
    luteFrench7thCourse = int(mu::engraving::SymId::luteFrench7thCourse),
    luteFrench8thCourse = int(mu::engraving::SymId::luteFrench8thCourse),
    luteFrench9thCourse = int(mu::engraving::SymId::luteFrench9thCourse),
    luteFrenchAppoggiaturaAbove = int(mu::engraving::SymId::luteFrenchAppoggiaturaAbove),
    luteFrenchAppoggiaturaBelow = int(mu::engraving::SymId::luteFrenchAppoggiaturaBelow),
    luteFrenchFretA = int(mu::engraving::SymId::luteFrenchFretA),
    luteFrenchFretB = int(mu::engraving::SymId::luteFrenchFretB),
    luteFrenchFretC = int(mu::engraving::SymId::luteFrenchFretC),
    luteFrenchFretD = int(mu::engraving::SymId::luteFrenchFretD),
    luteFrenchFretE = int(mu::engraving::SymId::luteFrenchFretE),
    luteFrenchFretF = int(mu::engraving::SymId::luteFrenchFretF),
    luteFrenchFretG = int(mu::engraving::SymId::luteFrenchFretG),
    luteFrenchFretH = int(mu::engraving::SymId::luteFrenchFretH),
    luteFrenchFretI = int(mu::engraving::SymId::luteFrenchFretI),
    luteFrenchFretK = int(mu::engraving::SymId::luteFrenchFretK),
    luteFrenchFretL = int(mu::engraving::SymId::luteFrenchFretL),
    luteFrenchFretM = int(mu::engraving::SymId::luteFrenchFretM),
    luteFrenchFretN = int(mu::engraving::SymId::luteFrenchFretN),
    luteFrenchMordentInverted = int(mu::engraving::SymId::luteFrenchMordentInverted),
    luteFrenchMordentLower = int(mu::engraving::SymId::luteFrenchMordentLower),
    luteFrenchMordentUpper = int(mu::engraving::SymId::luteFrenchMordentUpper),
    luteGermanALower = int(mu::engraving::SymId::luteGermanALower),
    luteGermanAUpper = int(mu::engraving::SymId::luteGermanAUpper),
    luteGermanBLower = int(mu::engraving::SymId::luteGermanBLower),
    luteGermanBUpper = int(mu::engraving::SymId::luteGermanBUpper),
    luteGermanCLower = int(mu::engraving::SymId::luteGermanCLower),
    luteGermanCUpper = int(mu::engraving::SymId::luteGermanCUpper),
    luteGermanDLower = int(mu::engraving::SymId::luteGermanDLower),
    luteGermanDUpper = int(mu::engraving::SymId::luteGermanDUpper),
    luteGermanELower = int(mu::engraving::SymId::luteGermanELower),
    luteGermanEUpper = int(mu::engraving::SymId::luteGermanEUpper),
    luteGermanFLower = int(mu::engraving::SymId::luteGermanFLower),
    luteGermanFUpper = int(mu::engraving::SymId::luteGermanFUpper),
    luteGermanGLower = int(mu::engraving::SymId::luteGermanGLower),
    luteGermanGUpper = int(mu::engraving::SymId::luteGermanGUpper),
    luteGermanHLower = int(mu::engraving::SymId::luteGermanHLower),
    luteGermanHUpper = int(mu::engraving::SymId::luteGermanHUpper),
    luteGermanILower = int(mu::engraving::SymId::luteGermanILower),
    luteGermanIUpper = int(mu::engraving::SymId::luteGermanIUpper),
    luteGermanKLower = int(mu::engraving::SymId::luteGermanKLower),
    luteGermanKUpper = int(mu::engraving::SymId::luteGermanKUpper),
    luteGermanLLower = int(mu::engraving::SymId::luteGermanLLower),
    luteGermanLUpper = int(mu::engraving::SymId::luteGermanLUpper),
    luteGermanMLower = int(mu::engraving::SymId::luteGermanMLower),
    luteGermanMUpper = int(mu::engraving::SymId::luteGermanMUpper),
    luteGermanNLower = int(mu::engraving::SymId::luteGermanNLower),
    luteGermanNUpper = int(mu::engraving::SymId::luteGermanNUpper),
    luteGermanOLower = int(mu::engraving::SymId::luteGermanOLower),
    luteGermanPLower = int(mu::engraving::SymId::luteGermanPLower),
    luteGermanQLower = int(mu::engraving::SymId::luteGermanQLower),
    luteGermanRLower = int(mu::engraving::SymId::luteGermanRLower),
    luteGermanSLower = int(mu::engraving::SymId::luteGermanSLower),
    luteGermanTLower = int(mu::engraving::SymId::luteGermanTLower),
    luteGermanVLower = int(mu::engraving::SymId::luteGermanVLower),
    luteGermanXLower = int(mu::engraving::SymId::luteGermanXLower),
    luteGermanYLower = int(mu::engraving::SymId::luteGermanYLower),
    luteGermanZLower = int(mu::engraving::SymId::luteGermanZLower),
    luteItalianClefCSolFaUt = int(mu::engraving::SymId::luteItalianClefCSolFaUt),
    luteItalianClefFFaUt = int(mu::engraving::SymId::luteItalianClefFFaUt),
    luteItalianFret0 = int(mu::engraving::SymId::luteItalianFret0),
    luteItalianFret1 = int(mu::engraving::SymId::luteItalianFret1),
    luteItalianFret2 = int(mu::engraving::SymId::luteItalianFret2),
    luteItalianFret3 = int(mu::engraving::SymId::luteItalianFret3),
    luteItalianFret4 = int(mu::engraving::SymId::luteItalianFret4),
    luteItalianFret5 = int(mu::engraving::SymId::luteItalianFret5),
    luteItalianFret6 = int(mu::engraving::SymId::luteItalianFret6),
    luteItalianFret7 = int(mu::engraving::SymId::luteItalianFret7),
    luteItalianFret8 = int(mu::engraving::SymId::luteItalianFret8),
    luteItalianFret9 = int(mu::engraving::SymId::luteItalianFret9),
    luteItalianHoldFinger = int(mu::engraving::SymId::luteItalianHoldFinger),
    luteItalianHoldNote = int(mu::engraving::SymId::luteItalianHoldNote),
    luteItalianReleaseFinger = int(mu::engraving::SymId::luteItalianReleaseFinger),
    luteItalianTempoFast = int(mu::engraving::SymId::luteItalianTempoFast),
    luteItalianTempoNeitherFastNorSlow = int(mu::engraving::SymId::luteItalianTempoNeitherFastNorSlow),
    luteItalianTempoSlow = int(mu::engraving::SymId::luteItalianTempoSlow),
    luteItalianTempoSomewhatFast = int(mu::engraving::SymId::luteItalianTempoSomewhatFast),
    luteItalianTempoVerySlow = int(mu::engraving::SymId::luteItalianTempoVerySlow),
    luteItalianTimeTriple = int(mu::engraving::SymId::luteItalianTimeTriple),
    luteItalianTremolo = int(mu::engraving::SymId::luteItalianTremolo),
    luteItalianVibrato = int(mu::engraving::SymId::luteItalianVibrato),
    luteStaff6Lines = int(mu::engraving::SymId::luteStaff6Lines),
    luteStaff6LinesNarrow = int(mu::engraving::SymId::luteStaff6LinesNarrow),
    luteStaff6LinesWide = int(mu::engraving::SymId::luteStaff6LinesWide),
    lyricsElision = int(mu::engraving::SymId::lyricsElision),
    lyricsElisionNarrow = int(mu::engraving::SymId::lyricsElisionNarrow),
    lyricsElisionWide = int(mu::engraving::SymId::lyricsElisionWide),
    lyricsHyphenBaseline = int(mu::engraving::SymId::lyricsHyphenBaseline),
    lyricsHyphenBaselineNonBreaking = int(mu::engraving::SymId::lyricsHyphenBaselineNonBreaking),
    lyricsTextRepeat = int(mu::engraving::SymId::lyricsTextRepeat),
    medRenFlatHardB = int(mu::engraving::SymId::medRenFlatHardB),
    medRenFlatSoftB = int(mu::engraving::SymId::medRenFlatSoftB),
    medRenFlatWithDot = int(mu::engraving::SymId::medRenFlatWithDot),
    medRenGClefCMN = int(mu::engraving::SymId::medRenGClefCMN),
    medRenLiquescenceCMN = int(mu::engraving::SymId::medRenLiquescenceCMN),
    medRenLiquescentAscCMN = int(mu::engraving::SymId::medRenLiquescentAscCMN),
    medRenLiquescentDescCMN = int(mu::engraving::SymId::medRenLiquescentDescCMN),
    medRenNatural = int(mu::engraving::SymId::medRenNatural),
    medRenNaturalWithCross = int(mu::engraving::SymId::medRenNaturalWithCross),
    medRenOriscusCMN = int(mu::engraving::SymId::medRenOriscusCMN),
    medRenPlicaCMN = int(mu::engraving::SymId::medRenPlicaCMN),
    medRenPunctumCMN = int(mu::engraving::SymId::medRenPunctumCMN),
    medRenQuilismaCMN = int(mu::engraving::SymId::medRenQuilismaCMN),
    medRenSharpCroix = int(mu::engraving::SymId::medRenSharpCroix),
    medRenStrophicusCMN = int(mu::engraving::SymId::medRenStrophicusCMN),
    mensuralAlterationSign = int(mu::engraving::SymId::mensuralAlterationSign),
    mensuralBlackBrevis = int(mu::engraving::SymId::mensuralBlackBrevis),
    mensuralBlackBrevisVoid = int(mu::engraving::SymId::mensuralBlackBrevisVoid),
    mensuralBlackDragma = int(mu::engraving::SymId::mensuralBlackDragma),
    mensuralBlackLonga = int(mu::engraving::SymId::mensuralBlackLonga),
    mensuralBlackMaxima = int(mu::engraving::SymId::mensuralBlackMaxima),
    mensuralBlackMinima = int(mu::engraving::SymId::mensuralBlackMinima),
    mensuralBlackMinimaVoid = int(mu::engraving::SymId::mensuralBlackMinimaVoid),
    mensuralBlackSemibrevis = int(mu::engraving::SymId::mensuralBlackSemibrevis),
    mensuralBlackSemibrevisCaudata = int(mu::engraving::SymId::mensuralBlackSemibrevisCaudata),
    mensuralBlackSemibrevisOblique = int(mu::engraving::SymId::mensuralBlackSemibrevisOblique),
    mensuralBlackSemibrevisVoid = int(mu::engraving::SymId::mensuralBlackSemibrevisVoid),
    mensuralBlackSemiminima = int(mu::engraving::SymId::mensuralBlackSemiminima),
    mensuralCclef = int(mu::engraving::SymId::mensuralCclef),
    mensuralCclefPetrucciPosHigh = int(mu::engraving::SymId::mensuralCclefPetrucciPosHigh),
    mensuralCclefPetrucciPosHighest = int(mu::engraving::SymId::mensuralCclefPetrucciPosHighest),
    mensuralCclefPetrucciPosLow = int(mu::engraving::SymId::mensuralCclefPetrucciPosLow),
    mensuralCclefPetrucciPosLowest = int(mu::engraving::SymId::mensuralCclefPetrucciPosLowest),
    mensuralCclefPetrucciPosMiddle = int(mu::engraving::SymId::mensuralCclefPetrucciPosMiddle),
    mensuralColorationEndRound = int(mu::engraving::SymId::mensuralColorationEndRound),
    mensuralColorationEndSquare = int(mu::engraving::SymId::mensuralColorationEndSquare),
    mensuralColorationStartRound = int(mu::engraving::SymId::mensuralColorationStartRound),
    mensuralColorationStartSquare = int(mu::engraving::SymId::mensuralColorationStartSquare),
    mensuralCombStemDiagonal = int(mu::engraving::SymId::mensuralCombStemDiagonal),
    mensuralCombStemDown = int(mu::engraving::SymId::mensuralCombStemDown),
    mensuralCombStemDownFlagExtended = int(mu::engraving::SymId::mensuralCombStemDownFlagExtended),
    mensuralCombStemDownFlagFlared = int(mu::engraving::SymId::mensuralCombStemDownFlagFlared),
    mensuralCombStemDownFlagFusa = int(mu::engraving::SymId::mensuralCombStemDownFlagFusa),
    mensuralCombStemDownFlagLeft = int(mu::engraving::SymId::mensuralCombStemDownFlagLeft),
    mensuralCombStemDownFlagRight = int(mu::engraving::SymId::mensuralCombStemDownFlagRight),
    mensuralCombStemDownFlagSemiminima = int(mu::engraving::SymId::mensuralCombStemDownFlagSemiminima),
    mensuralCombStemUp = int(mu::engraving::SymId::mensuralCombStemUp),
    mensuralCombStemUpFlagExtended = int(mu::engraving::SymId::mensuralCombStemUpFlagExtended),
    mensuralCombStemUpFlagFlared = int(mu::engraving::SymId::mensuralCombStemUpFlagFlared),
    mensuralCombStemUpFlagFusa = int(mu::engraving::SymId::mensuralCombStemUpFlagFusa),
    mensuralCombStemUpFlagLeft = int(mu::engraving::SymId::mensuralCombStemUpFlagLeft),
    mensuralCombStemUpFlagRight = int(mu::engraving::SymId::mensuralCombStemUpFlagRight),
    mensuralCombStemUpFlagSemiminima = int(mu::engraving::SymId::mensuralCombStemUpFlagSemiminima),
    mensuralCustosCheckmark = int(mu::engraving::SymId::mensuralCustosCheckmark),
    mensuralCustosDown = int(mu::engraving::SymId::mensuralCustosDown),
    mensuralCustosTurn = int(mu::engraving::SymId::mensuralCustosTurn),
    mensuralCustosUp = int(mu::engraving::SymId::mensuralCustosUp),
    mensuralFclef = int(mu::engraving::SymId::mensuralFclef),
    mensuralFclefPetrucci = int(mu::engraving::SymId::mensuralFclefPetrucci),
    mensuralGclef = int(mu::engraving::SymId::mensuralGclef),
    mensuralGclefPetrucci = int(mu::engraving::SymId::mensuralGclefPetrucci),
    mensuralModusImperfectumVert = int(mu::engraving::SymId::mensuralModusImperfectumVert),
    mensuralModusPerfectumVert = int(mu::engraving::SymId::mensuralModusPerfectumVert),
    mensuralNoteheadLongaBlack = int(mu::engraving::SymId::mensuralNoteheadLongaBlack),
    mensuralNoteheadLongaBlackVoid = int(mu::engraving::SymId::mensuralNoteheadLongaBlackVoid),
    mensuralNoteheadLongaVoid = int(mu::engraving::SymId::mensuralNoteheadLongaVoid),
    mensuralNoteheadLongaWhite = int(mu::engraving::SymId::mensuralNoteheadLongaWhite),
    mensuralNoteheadMaximaBlack = int(mu::engraving::SymId::mensuralNoteheadMaximaBlack),
    mensuralNoteheadMaximaBlackVoid = int(mu::engraving::SymId::mensuralNoteheadMaximaBlackVoid),
    mensuralNoteheadMaximaVoid = int(mu::engraving::SymId::mensuralNoteheadMaximaVoid),
    mensuralNoteheadMaximaWhite = int(mu::engraving::SymId::mensuralNoteheadMaximaWhite),
    mensuralNoteheadMinimaWhite = int(mu::engraving::SymId::mensuralNoteheadMinimaWhite),
    mensuralNoteheadSemibrevisBlack = int(mu::engraving::SymId::mensuralNoteheadSemibrevisBlack),
    mensuralNoteheadSemibrevisBlackVoid = int(mu::engraving::SymId::mensuralNoteheadSemibrevisBlackVoid),
    mensuralNoteheadSemibrevisBlackVoidTurned = int(mu::engraving::SymId::mensuralNoteheadSemibrevisBlackVoidTurned),
    mensuralNoteheadSemibrevisVoid = int(mu::engraving::SymId::mensuralNoteheadSemibrevisVoid),
    mensuralNoteheadSemiminimaWhite = int(mu::engraving::SymId::mensuralNoteheadSemiminimaWhite),
    mensuralObliqueAsc2ndBlack = int(mu::engraving::SymId::mensuralObliqueAsc2ndBlack),
    mensuralObliqueAsc2ndBlackVoid = int(mu::engraving::SymId::mensuralObliqueAsc2ndBlackVoid),
    mensuralObliqueAsc2ndVoid = int(mu::engraving::SymId::mensuralObliqueAsc2ndVoid),
    mensuralObliqueAsc2ndWhite = int(mu::engraving::SymId::mensuralObliqueAsc2ndWhite),
    mensuralObliqueAsc3rdBlack = int(mu::engraving::SymId::mensuralObliqueAsc3rdBlack),
    mensuralObliqueAsc3rdBlackVoid = int(mu::engraving::SymId::mensuralObliqueAsc3rdBlackVoid),
    mensuralObliqueAsc3rdVoid = int(mu::engraving::SymId::mensuralObliqueAsc3rdVoid),
    mensuralObliqueAsc3rdWhite = int(mu::engraving::SymId::mensuralObliqueAsc3rdWhite),
    mensuralObliqueAsc4thBlack = int(mu::engraving::SymId::mensuralObliqueAsc4thBlack),
    mensuralObliqueAsc4thBlackVoid = int(mu::engraving::SymId::mensuralObliqueAsc4thBlackVoid),
    mensuralObliqueAsc4thVoid = int(mu::engraving::SymId::mensuralObliqueAsc4thVoid),
    mensuralObliqueAsc4thWhite = int(mu::engraving::SymId::mensuralObliqueAsc4thWhite),
    mensuralObliqueAsc5thBlack = int(mu::engraving::SymId::mensuralObliqueAsc5thBlack),
    mensuralObliqueAsc5thBlackVoid = int(mu::engraving::SymId::mensuralObliqueAsc5thBlackVoid),
    mensuralObliqueAsc5thVoid = int(mu::engraving::SymId::mensuralObliqueAsc5thVoid),
    mensuralObliqueAsc5thWhite = int(mu::engraving::SymId::mensuralObliqueAsc5thWhite),
    mensuralObliqueDesc2ndBlack = int(mu::engraving::SymId::mensuralObliqueDesc2ndBlack),
    mensuralObliqueDesc2ndBlackVoid = int(mu::engraving::SymId::mensuralObliqueDesc2ndBlackVoid),
    mensuralObliqueDesc2ndVoid = int(mu::engraving::SymId::mensuralObliqueDesc2ndVoid),
    mensuralObliqueDesc2ndWhite = int(mu::engraving::SymId::mensuralObliqueDesc2ndWhite),
    mensuralObliqueDesc3rdBlack = int(mu::engraving::SymId::mensuralObliqueDesc3rdBlack),
    mensuralObliqueDesc3rdBlackVoid = int(mu::engraving::SymId::mensuralObliqueDesc3rdBlackVoid),
    mensuralObliqueDesc3rdVoid = int(mu::engraving::SymId::mensuralObliqueDesc3rdVoid),
    mensuralObliqueDesc3rdWhite = int(mu::engraving::SymId::mensuralObliqueDesc3rdWhite),
    mensuralObliqueDesc4thBlack = int(mu::engraving::SymId::mensuralObliqueDesc4thBlack),
    mensuralObliqueDesc4thBlackVoid = int(mu::engraving::SymId::mensuralObliqueDesc4thBlackVoid),
    mensuralObliqueDesc4thVoid = int(mu::engraving::SymId::mensuralObliqueDesc4thVoid),
    mensuralObliqueDesc4thWhite = int(mu::engraving::SymId::mensuralObliqueDesc4thWhite),
    mensuralObliqueDesc5thBlack = int(mu::engraving::SymId::mensuralObliqueDesc5thBlack),
    mensuralObliqueDesc5thBlackVoid = int(mu::engraving::SymId::mensuralObliqueDesc5thBlackVoid),
    mensuralObliqueDesc5thVoid = int(mu::engraving::SymId::mensuralObliqueDesc5thVoid),
    mensuralObliqueDesc5thWhite = int(mu::engraving::SymId::mensuralObliqueDesc5thWhite),
    mensuralProlation1 = int(mu::engraving::SymId::mensuralProlation1),
    mensuralProlation10 = int(mu::engraving::SymId::mensuralProlation10),
    mensuralProlation11 = int(mu::engraving::SymId::mensuralProlation11),
    mensuralProlation2 = int(mu::engraving::SymId::mensuralProlation2),
    mensuralProlation3 = int(mu::engraving::SymId::mensuralProlation3),
    mensuralProlation4 = int(mu::engraving::SymId::mensuralProlation4),
    mensuralProlation5 = int(mu::engraving::SymId::mensuralProlation5),
    mensuralProlation6 = int(mu::engraving::SymId::mensuralProlation6),
    mensuralProlation7 = int(mu::engraving::SymId::mensuralProlation7),
    mensuralProlation8 = int(mu::engraving::SymId::mensuralProlation8),
    mensuralProlation9 = int(mu::engraving::SymId::mensuralProlation9),
    mensuralProlationCombiningDot = int(mu::engraving::SymId::mensuralProlationCombiningDot),
    mensuralProlationCombiningDotVoid = int(mu::engraving::SymId::mensuralProlationCombiningDotVoid),
    mensuralProlationCombiningStroke = int(mu::engraving::SymId::mensuralProlationCombiningStroke),
    mensuralProlationCombiningThreeDots = int(mu::engraving::SymId::mensuralProlationCombiningThreeDots),
    mensuralProlationCombiningThreeDotsTri = int(mu::engraving::SymId::mensuralProlationCombiningThreeDotsTri),
    mensuralProlationCombiningTwoDots = int(mu::engraving::SymId::mensuralProlationCombiningTwoDots),
    mensuralProportion1 = int(mu::engraving::SymId::mensuralProportion1),
    mensuralProportion2 = int(mu::engraving::SymId::mensuralProportion2),
    mensuralProportion3 = int(mu::engraving::SymId::mensuralProportion3),
    mensuralProportion4 = int(mu::engraving::SymId::mensuralProportion4),
    mensuralProportion5 = int(mu::engraving::SymId::mensuralProportion5),
    mensuralProportion6 = int(mu::engraving::SymId::mensuralProportion6),
    mensuralProportion7 = int(mu::engraving::SymId::mensuralProportion7),
    mensuralProportion8 = int(mu::engraving::SymId::mensuralProportion8),
    mensuralProportion9 = int(mu::engraving::SymId::mensuralProportion9),
    mensuralProportionMajor = int(mu::engraving::SymId::mensuralProportionMajor),
    mensuralProportionMinor = int(mu::engraving::SymId::mensuralProportionMinor),
    mensuralProportionProportioDupla1 = int(mu::engraving::SymId::mensuralProportionProportioDupla1),
    mensuralProportionProportioDupla2 = int(mu::engraving::SymId::mensuralProportionProportioDupla2),
    mensuralProportionProportioQuadrupla = int(mu::engraving::SymId::mensuralProportionProportioQuadrupla),
    mensuralProportionProportioTripla = int(mu::engraving::SymId::mensuralProportionProportioTripla),
    mensuralProportionTempusPerfectum = int(mu::engraving::SymId::mensuralProportionTempusPerfectum),
    mensuralRestBrevis = int(mu::engraving::SymId::mensuralRestBrevis),
    mensuralRestFusa = int(mu::engraving::SymId::mensuralRestFusa),
    mensuralRestLongaImperfecta = int(mu::engraving::SymId::mensuralRestLongaImperfecta),
    mensuralRestLongaPerfecta = int(mu::engraving::SymId::mensuralRestLongaPerfecta),
    mensuralRestMaxima = int(mu::engraving::SymId::mensuralRestMaxima),
    mensuralRestMinima = int(mu::engraving::SymId::mensuralRestMinima),
    mensuralRestSemibrevis = int(mu::engraving::SymId::mensuralRestSemibrevis),
    mensuralRestSemifusa = int(mu::engraving::SymId::mensuralRestSemifusa),
    mensuralRestSemiminima = int(mu::engraving::SymId::mensuralRestSemiminima),
    mensuralSignumDown = int(mu::engraving::SymId::mensuralSignumDown),
    mensuralSignumUp = int(mu::engraving::SymId::mensuralSignumUp),
    mensuralTempusImperfectumHoriz = int(mu::engraving::SymId::mensuralTempusImperfectumHoriz),
    mensuralTempusPerfectumHoriz = int(mu::engraving::SymId::mensuralTempusPerfectumHoriz),
    mensuralWhiteBrevis = int(mu::engraving::SymId::mensuralWhiteBrevis),
    mensuralWhiteFusa = int(mu::engraving::SymId::mensuralWhiteFusa),
    mensuralWhiteLonga = int(mu::engraving::SymId::mensuralWhiteLonga),
    mensuralWhiteMaxima = int(mu::engraving::SymId::mensuralWhiteMaxima),
    mensuralWhiteMinima = int(mu::engraving::SymId::mensuralWhiteMinima),
    mensuralWhiteSemibrevis = int(mu::engraving::SymId::mensuralWhiteSemibrevis),
    mensuralWhiteSemiminima = int(mu::engraving::SymId::mensuralWhiteSemiminima),
    metAugmentationDot = int(mu::engraving::SymId::metAugmentationDot),
    metNote1024thDown = int(mu::engraving::SymId::metNote1024thDown),
    metNote1024thUp = int(mu::engraving::SymId::metNote1024thUp),
    metNote128thDown = int(mu::engraving::SymId::metNote128thDown),
    metNote128thUp = int(mu::engraving::SymId::metNote128thUp),
    metNote16thDown = int(mu::engraving::SymId::metNote16thDown),
    metNote16thUp = int(mu::engraving::SymId::metNote16thUp),
    metNote256thDown = int(mu::engraving::SymId::metNote256thDown),
    metNote256thUp = int(mu::engraving::SymId::metNote256thUp),
    metNote32ndDown = int(mu::engraving::SymId::metNote32ndDown),
    metNote32ndUp = int(mu::engraving::SymId::metNote32ndUp),
    metNote512thDown = int(mu::engraving::SymId::metNote512thDown),
    metNote512thUp = int(mu::engraving::SymId::metNote512thUp),
    metNote64thDown = int(mu::engraving::SymId::metNote64thDown),
    metNote64thUp = int(mu::engraving::SymId::metNote64thUp),
    metNote8thDown = int(mu::engraving::SymId::metNote8thDown),
    metNote8thUp = int(mu::engraving::SymId::metNote8thUp),
    metNoteDoubleWhole = int(mu::engraving::SymId::metNoteDoubleWhole),
    metNoteDoubleWholeSquare = int(mu::engraving::SymId::metNoteDoubleWholeSquare),
    metNoteHalfDown = int(mu::engraving::SymId::metNoteHalfDown),
    metNoteHalfUp = int(mu::engraving::SymId::metNoteHalfUp),
    metNoteQuarterDown = int(mu::engraving::SymId::metNoteQuarterDown),
    metNoteQuarterUp = int(mu::engraving::SymId::metNoteQuarterUp),
    metNoteWhole = int(mu::engraving::SymId::metNoteWhole),
    metricModulationArrowLeft = int(mu::engraving::SymId::metricModulationArrowLeft),
    metricModulationArrowRight = int(mu::engraving::SymId::metricModulationArrowRight),
    miscDoNotCopy = int(mu::engraving::SymId::miscDoNotCopy),
    miscDoNotPhotocopy = int(mu::engraving::SymId::miscDoNotPhotocopy),
    miscEyeglasses = int(mu::engraving::SymId::miscEyeglasses),
    note1024thDown = int(mu::engraving::SymId::note1024thDown),
    note1024thUp = int(mu::engraving::SymId::note1024thUp),
    note128thDown = int(mu::engraving::SymId::note128thDown),
    note128thUp = int(mu::engraving::SymId::note128thUp),
    note16thDown = int(mu::engraving::SymId::note16thDown),
    note16thUp = int(mu::engraving::SymId::note16thUp),
    note256thDown = int(mu::engraving::SymId::note256thDown),
    note256thUp = int(mu::engraving::SymId::note256thUp),
    note32ndDown = int(mu::engraving::SymId::note32ndDown),
    note32ndUp = int(mu::engraving::SymId::note32ndUp),
    note512thDown = int(mu::engraving::SymId::note512thDown),
    note512thUp = int(mu::engraving::SymId::note512thUp),
    note64thDown = int(mu::engraving::SymId::note64thDown),
    note64thUp = int(mu::engraving::SymId::note64thUp),
    note8thDown = int(mu::engraving::SymId::note8thDown),
    note8thUp = int(mu::engraving::SymId::note8thUp),
    noteABlack = int(mu::engraving::SymId::noteABlack),
    noteAFlatBlack = int(mu::engraving::SymId::noteAFlatBlack),
    noteAFlatHalf = int(mu::engraving::SymId::noteAFlatHalf),
    noteAFlatWhole = int(mu::engraving::SymId::noteAFlatWhole),
    noteAHalf = int(mu::engraving::SymId::noteAHalf),
    noteASharpBlack = int(mu::engraving::SymId::noteASharpBlack),
    noteASharpHalf = int(mu::engraving::SymId::noteASharpHalf),
    noteASharpWhole = int(mu::engraving::SymId::noteASharpWhole),
    noteAWhole = int(mu::engraving::SymId::noteAWhole),
    noteBBlack = int(mu::engraving::SymId::noteBBlack),
    noteBFlatBlack = int(mu::engraving::SymId::noteBFlatBlack),
    noteBFlatHalf = int(mu::engraving::SymId::noteBFlatHalf),
    noteBFlatWhole = int(mu::engraving::SymId::noteBFlatWhole),
    noteBHalf = int(mu::engraving::SymId::noteBHalf),
    noteBSharpBlack = int(mu::engraving::SymId::noteBSharpBlack),
    noteBSharpHalf = int(mu::engraving::SymId::noteBSharpHalf),
    noteBSharpWhole = int(mu::engraving::SymId::noteBSharpWhole),
    noteBWhole = int(mu::engraving::SymId::noteBWhole),
    noteCBlack = int(mu::engraving::SymId::noteCBlack),
    noteCFlatBlack = int(mu::engraving::SymId::noteCFlatBlack),
    noteCFlatHalf = int(mu::engraving::SymId::noteCFlatHalf),
    noteCFlatWhole = int(mu::engraving::SymId::noteCFlatWhole),
    noteCHalf = int(mu::engraving::SymId::noteCHalf),
    noteCSharpBlack = int(mu::engraving::SymId::noteCSharpBlack),
    noteCSharpHalf = int(mu::engraving::SymId::noteCSharpHalf),
    noteCSharpWhole = int(mu::engraving::SymId::noteCSharpWhole),
    noteCWhole = int(mu::engraving::SymId::noteCWhole),
    noteDBlack = int(mu::engraving::SymId::noteDBlack),
    noteDFlatBlack = int(mu::engraving::SymId::noteDFlatBlack),
    noteDFlatHalf = int(mu::engraving::SymId::noteDFlatHalf),
    noteDFlatWhole = int(mu::engraving::SymId::noteDFlatWhole),
    noteDHalf = int(mu::engraving::SymId::noteDHalf),
    noteDSharpBlack = int(mu::engraving::SymId::noteDSharpBlack),
    noteDSharpHalf = int(mu::engraving::SymId::noteDSharpHalf),
    noteDSharpWhole = int(mu::engraving::SymId::noteDSharpWhole),
    noteDWhole = int(mu::engraving::SymId::noteDWhole),
    noteDiBlack = int(mu::engraving::SymId::noteDiBlack),
    noteDiHalf = int(mu::engraving::SymId::noteDiHalf),
    noteDiWhole = int(mu::engraving::SymId::noteDiWhole),
    noteDoBlack = int(mu::engraving::SymId::noteDoBlack),
    noteDoHalf = int(mu::engraving::SymId::noteDoHalf),
    noteDoWhole = int(mu::engraving::SymId::noteDoWhole),
    noteDoubleWhole = int(mu::engraving::SymId::noteDoubleWhole),
    noteDoubleWholeSquare = int(mu::engraving::SymId::noteDoubleWholeSquare),
    noteEBlack = int(mu::engraving::SymId::noteEBlack),
    noteEFlatBlack = int(mu::engraving::SymId::noteEFlatBlack),
    noteEFlatHalf = int(mu::engraving::SymId::noteEFlatHalf),
    noteEFlatWhole = int(mu::engraving::SymId::noteEFlatWhole),
    noteEHalf = int(mu::engraving::SymId::noteEHalf),
    noteESharpBlack = int(mu::engraving::SymId::noteESharpBlack),
    noteESharpHalf = int(mu::engraving::SymId::noteESharpHalf),
    noteESharpWhole = int(mu::engraving::SymId::noteESharpWhole),
    noteEWhole = int(mu::engraving::SymId::noteEWhole),
    noteEmptyBlack = int(mu::engraving::SymId::noteEmptyBlack),
    noteEmptyHalf = int(mu::engraving::SymId::noteEmptyHalf),
    noteEmptyWhole = int(mu::engraving::SymId::noteEmptyWhole),
    noteFBlack = int(mu::engraving::SymId::noteFBlack),
    noteFFlatBlack = int(mu::engraving::SymId::noteFFlatBlack),
    noteFFlatHalf = int(mu::engraving::SymId::noteFFlatHalf),
    noteFFlatWhole = int(mu::engraving::SymId::noteFFlatWhole),
    noteFHalf = int(mu::engraving::SymId::noteFHalf),
    noteFSharpBlack = int(mu::engraving::SymId::noteFSharpBlack),
    noteFSharpHalf = int(mu::engraving::SymId::noteFSharpHalf),
    noteFSharpWhole = int(mu::engraving::SymId::noteFSharpWhole),
    noteFWhole = int(mu::engraving::SymId::noteFWhole),
    noteFaBlack = int(mu::engraving::SymId::noteFaBlack),
    noteFaHalf = int(mu::engraving::SymId::noteFaHalf),
    noteFaWhole = int(mu::engraving::SymId::noteFaWhole),
    noteFiBlack = int(mu::engraving::SymId::noteFiBlack),
    noteFiHalf = int(mu::engraving::SymId::noteFiHalf),
    noteFiWhole = int(mu::engraving::SymId::noteFiWhole),
    noteGBlack = int(mu::engraving::SymId::noteGBlack),
    noteGFlatBlack = int(mu::engraving::SymId::noteGFlatBlack),
    noteGFlatHalf = int(mu::engraving::SymId::noteGFlatHalf),
    noteGFlatWhole = int(mu::engraving::SymId::noteGFlatWhole),
    noteGHalf = int(mu::engraving::SymId::noteGHalf),
    noteGSharpBlack = int(mu::engraving::SymId::noteGSharpBlack),
    noteGSharpHalf = int(mu::engraving::SymId::noteGSharpHalf),
    noteGSharpWhole = int(mu::engraving::SymId::noteGSharpWhole),
    noteGWhole = int(mu::engraving::SymId::noteGWhole),
    noteHBlack = int(mu::engraving::SymId::noteHBlack),
    noteHHalf = int(mu::engraving::SymId::noteHHalf),
    noteHSharpBlack = int(mu::engraving::SymId::noteHSharpBlack),
    noteHSharpHalf = int(mu::engraving::SymId::noteHSharpHalf),
    noteHSharpWhole = int(mu::engraving::SymId::noteHSharpWhole),
    noteHWhole = int(mu::engraving::SymId::noteHWhole),
    noteHalfDown = int(mu::engraving::SymId::noteHalfDown),
    noteHalfUp = int(mu::engraving::SymId::noteHalfUp),
    noteLaBlack = int(mu::engraving::SymId::noteLaBlack),
    noteLaHalf = int(mu::engraving::SymId::noteLaHalf),
    noteLaWhole = int(mu::engraving::SymId::noteLaWhole),
    noteLeBlack = int(mu::engraving::SymId::noteLeBlack),
    noteLeHalf = int(mu::engraving::SymId::noteLeHalf),
    noteLeWhole = int(mu::engraving::SymId::noteLeWhole),
    noteLiBlack = int(mu::engraving::SymId::noteLiBlack),
    noteLiHalf = int(mu::engraving::SymId::noteLiHalf),
    noteLiWhole = int(mu::engraving::SymId::noteLiWhole),
    noteMeBlack = int(mu::engraving::SymId::noteMeBlack),
    noteMeHalf = int(mu::engraving::SymId::noteMeHalf),
    noteMeWhole = int(mu::engraving::SymId::noteMeWhole),
    noteMiBlack = int(mu::engraving::SymId::noteMiBlack),
    noteMiHalf = int(mu::engraving::SymId::noteMiHalf),
    noteMiWhole = int(mu::engraving::SymId::noteMiWhole),
    noteQuarterDown = int(mu::engraving::SymId::noteQuarterDown),
    noteQuarterUp = int(mu::engraving::SymId::noteQuarterUp),
    noteRaBlack = int(mu::engraving::SymId::noteRaBlack),
    noteRaHalf = int(mu::engraving::SymId::noteRaHalf),
    noteRaWhole = int(mu::engraving::SymId::noteRaWhole),
    noteReBlack = int(mu::engraving::SymId::noteReBlack),
    noteReHalf = int(mu::engraving::SymId::noteReHalf),
    noteReWhole = int(mu::engraving::SymId::noteReWhole),
    noteRiBlack = int(mu::engraving::SymId::noteRiBlack),
    noteRiHalf = int(mu::engraving::SymId::noteRiHalf),
    noteRiWhole = int(mu::engraving::SymId::noteRiWhole),
    noteSeBlack = int(mu::engraving::SymId::noteSeBlack),
    noteSeHalf = int(mu::engraving::SymId::noteSeHalf),
    noteSeWhole = int(mu::engraving::SymId::noteSeWhole),
    noteShapeArrowheadLeftBlack = int(mu::engraving::SymId::noteShapeArrowheadLeftBlack),
    noteShapeArrowheadLeftDoubleWhole = int(mu::engraving::SymId::noteShapeArrowheadLeftDoubleWhole),
    noteShapeArrowheadLeftWhite = int(mu::engraving::SymId::noteShapeArrowheadLeftWhite),
    noteShapeDiamondBlack = int(mu::engraving::SymId::noteShapeDiamondBlack),
    noteShapeDiamondDoubleWhole = int(mu::engraving::SymId::noteShapeDiamondDoubleWhole),
    noteShapeDiamondWhite = int(mu::engraving::SymId::noteShapeDiamondWhite),
    noteShapeIsoscelesTriangleBlack = int(mu::engraving::SymId::noteShapeIsoscelesTriangleBlack),
    noteShapeIsoscelesTriangleDoubleWhole = int(mu::engraving::SymId::noteShapeIsoscelesTriangleDoubleWhole),
    noteShapeIsoscelesTriangleWhite = int(mu::engraving::SymId::noteShapeIsoscelesTriangleWhite),
    noteShapeKeystoneBlack = int(mu::engraving::SymId::noteShapeKeystoneBlack),
    noteShapeKeystoneDoubleWhole = int(mu::engraving::SymId::noteShapeKeystoneDoubleWhole),
    noteShapeKeystoneWhite = int(mu::engraving::SymId::noteShapeKeystoneWhite),
    noteShapeMoonBlack = int(mu::engraving::SymId::noteShapeMoonBlack),
    noteShapeMoonDoubleWhole = int(mu::engraving::SymId::noteShapeMoonDoubleWhole),
    noteShapeMoonLeftBlack = int(mu::engraving::SymId::noteShapeMoonLeftBlack),
    noteShapeMoonLeftDoubleWhole = int(mu::engraving::SymId::noteShapeMoonLeftDoubleWhole),
    noteShapeMoonLeftWhite = int(mu::engraving::SymId::noteShapeMoonLeftWhite),
    noteShapeMoonWhite = int(mu::engraving::SymId::noteShapeMoonWhite),
    noteShapeQuarterMoonBlack = int(mu::engraving::SymId::noteShapeQuarterMoonBlack),
    noteShapeQuarterMoonDoubleWhole = int(mu::engraving::SymId::noteShapeQuarterMoonDoubleWhole),
    noteShapeQuarterMoonWhite = int(mu::engraving::SymId::noteShapeQuarterMoonWhite),
    noteShapeRoundBlack = int(mu::engraving::SymId::noteShapeRoundBlack),
    noteShapeRoundDoubleWhole = int(mu::engraving::SymId::noteShapeRoundDoubleWhole),
    noteShapeRoundWhite = int(mu::engraving::SymId::noteShapeRoundWhite),
    noteShapeSquareBlack = int(mu::engraving::SymId::noteShapeSquareBlack),
    noteShapeSquareDoubleWhole = int(mu::engraving::SymId::noteShapeSquareDoubleWhole),
    noteShapeSquareWhite = int(mu::engraving::SymId::noteShapeSquareWhite),
    noteShapeTriangleLeftBlack = int(mu::engraving::SymId::noteShapeTriangleLeftBlack),
    noteShapeTriangleLeftDoubleWhole = int(mu::engraving::SymId::noteShapeTriangleLeftDoubleWhole),
    noteShapeTriangleLeftWhite = int(mu::engraving::SymId::noteShapeTriangleLeftWhite),
    noteShapeTriangleRightBlack = int(mu::engraving::SymId::noteShapeTriangleRightBlack),
    noteShapeTriangleRightDoubleWhole = int(mu::engraving::SymId::noteShapeTriangleRightDoubleWhole),
    noteShapeTriangleRightWhite = int(mu::engraving::SymId::noteShapeTriangleRightWhite),
    noteShapeTriangleRoundBlack = int(mu::engraving::SymId::noteShapeTriangleRoundBlack),
    noteShapeTriangleRoundDoubleWhole = int(mu::engraving::SymId::noteShapeTriangleRoundDoubleWhole),
    noteShapeTriangleRoundLeftBlack = int(mu::engraving::SymId::noteShapeTriangleRoundLeftBlack),
    noteShapeTriangleRoundLeftDoubleWhole = int(mu::engraving::SymId::noteShapeTriangleRoundLeftDoubleWhole),
    noteShapeTriangleRoundLeftWhite = int(mu::engraving::SymId::noteShapeTriangleRoundLeftWhite),
    noteShapeTriangleRoundWhite = int(mu::engraving::SymId::noteShapeTriangleRoundWhite),
    noteShapeTriangleUpBlack = int(mu::engraving::SymId::noteShapeTriangleUpBlack),
    noteShapeTriangleUpDoubleWhole = int(mu::engraving::SymId::noteShapeTriangleUpDoubleWhole),
    noteShapeTriangleUpWhite = int(mu::engraving::SymId::noteShapeTriangleUpWhite),
    noteSiBlack = int(mu::engraving::SymId::noteSiBlack),
    noteSiHalf = int(mu::engraving::SymId::noteSiHalf),
    noteSiWhole = int(mu::engraving::SymId::noteSiWhole),
    noteSoBlack = int(mu::engraving::SymId::noteSoBlack),
    noteSoHalf = int(mu::engraving::SymId::noteSoHalf),
    noteSoWhole = int(mu::engraving::SymId::noteSoWhole),
    noteTeBlack = int(mu::engraving::SymId::noteTeBlack),
    noteTeHalf = int(mu::engraving::SymId::noteTeHalf),
    noteTeWhole = int(mu::engraving::SymId::noteTeWhole),
    noteTiBlack = int(mu::engraving::SymId::noteTiBlack),
    noteTiHalf = int(mu::engraving::SymId::noteTiHalf),
    noteTiWhole = int(mu::engraving::SymId::noteTiWhole),
    noteWhole = int(mu::engraving::SymId::noteWhole),
    noteheadBlack = int(mu::engraving::SymId::noteheadBlack),
    noteheadCircleSlash = int(mu::engraving::SymId::noteheadCircleSlash),
    noteheadCircleX = int(mu::engraving::SymId::noteheadCircleX),
    noteheadCircleXDoubleWhole = int(mu::engraving::SymId::noteheadCircleXDoubleWhole),
    noteheadCircleXHalf = int(mu::engraving::SymId::noteheadCircleXHalf),
    noteheadCircleXWhole = int(mu::engraving::SymId::noteheadCircleXWhole),
    noteheadCircledBlack = int(mu::engraving::SymId::noteheadCircledBlack),
    noteheadCircledBlackLarge = int(mu::engraving::SymId::noteheadCircledBlackLarge),
    noteheadCircledDoubleWhole = int(mu::engraving::SymId::noteheadCircledDoubleWhole),
    noteheadCircledDoubleWholeLarge = int(mu::engraving::SymId::noteheadCircledDoubleWholeLarge),
    noteheadCircledHalf = int(mu::engraving::SymId::noteheadCircledHalf),
    noteheadCircledHalfLarge = int(mu::engraving::SymId::noteheadCircledHalfLarge),
    noteheadCircledWhole = int(mu::engraving::SymId::noteheadCircledWhole),
    noteheadCircledWholeLarge = int(mu::engraving::SymId::noteheadCircledWholeLarge),
    noteheadCircledXLarge = int(mu::engraving::SymId::noteheadCircledXLarge),
    noteheadClusterDoubleWhole2nd = int(mu::engraving::SymId::noteheadClusterDoubleWhole2nd),
    noteheadClusterDoubleWhole3rd = int(mu::engraving::SymId::noteheadClusterDoubleWhole3rd),
    noteheadClusterDoubleWholeBottom = int(mu::engraving::SymId::noteheadClusterDoubleWholeBottom),
    noteheadClusterDoubleWholeMiddle = int(mu::engraving::SymId::noteheadClusterDoubleWholeMiddle),
    noteheadClusterDoubleWholeTop = int(mu::engraving::SymId::noteheadClusterDoubleWholeTop),
    noteheadClusterHalf2nd = int(mu::engraving::SymId::noteheadClusterHalf2nd),
    noteheadClusterHalf3rd = int(mu::engraving::SymId::noteheadClusterHalf3rd),
    noteheadClusterHalfBottom = int(mu::engraving::SymId::noteheadClusterHalfBottom),
    noteheadClusterHalfMiddle = int(mu::engraving::SymId::noteheadClusterHalfMiddle),
    noteheadClusterHalfTop = int(mu::engraving::SymId::noteheadClusterHalfTop),
    noteheadClusterQuarter2nd = int(mu::engraving::SymId::noteheadClusterQuarter2nd),
    noteheadClusterQuarter3rd = int(mu::engraving::SymId::noteheadClusterQuarter3rd),
    noteheadClusterQuarterBottom = int(mu::engraving::SymId::noteheadClusterQuarterBottom),
    noteheadClusterQuarterMiddle = int(mu::engraving::SymId::noteheadClusterQuarterMiddle),
    noteheadClusterQuarterTop = int(mu::engraving::SymId::noteheadClusterQuarterTop),
    noteheadClusterRoundBlack = int(mu::engraving::SymId::noteheadClusterRoundBlack),
    noteheadClusterRoundWhite = int(mu::engraving::SymId::noteheadClusterRoundWhite),
    noteheadClusterSquareBlack = int(mu::engraving::SymId::noteheadClusterSquareBlack),
    noteheadClusterSquareWhite = int(mu::engraving::SymId::noteheadClusterSquareWhite),
    noteheadClusterWhole2nd = int(mu::engraving::SymId::noteheadClusterWhole2nd),
    noteheadClusterWhole3rd = int(mu::engraving::SymId::noteheadClusterWhole3rd),
    noteheadClusterWholeBottom = int(mu::engraving::SymId::noteheadClusterWholeBottom),
    noteheadClusterWholeMiddle = int(mu::engraving::SymId::noteheadClusterWholeMiddle),
    noteheadClusterWholeTop = int(mu::engraving::SymId::noteheadClusterWholeTop),
    noteheadCowellEleventhNoteSeriesHalf = int(mu::engraving::SymId::noteheadCowellEleventhNoteSeriesHalf),
    noteheadCowellEleventhNoteSeriesWhole = int(mu::engraving::SymId::noteheadCowellEleventhNoteSeriesWhole),
    noteheadCowellEleventhSeriesBlack = int(mu::engraving::SymId::noteheadCowellEleventhSeriesBlack),
    noteheadCowellFifteenthNoteSeriesBlack = int(mu::engraving::SymId::noteheadCowellFifteenthNoteSeriesBlack),
    noteheadCowellFifteenthNoteSeriesHalf = int(mu::engraving::SymId::noteheadCowellFifteenthNoteSeriesHalf),
    noteheadCowellFifteenthNoteSeriesWhole = int(mu::engraving::SymId::noteheadCowellFifteenthNoteSeriesWhole),
    noteheadCowellFifthNoteSeriesBlack = int(mu::engraving::SymId::noteheadCowellFifthNoteSeriesBlack),
    noteheadCowellFifthNoteSeriesHalf = int(mu::engraving::SymId::noteheadCowellFifthNoteSeriesHalf),
    noteheadCowellFifthNoteSeriesWhole = int(mu::engraving::SymId::noteheadCowellFifthNoteSeriesWhole),
    noteheadCowellNinthNoteSeriesBlack = int(mu::engraving::SymId::noteheadCowellNinthNoteSeriesBlack),
    noteheadCowellNinthNoteSeriesHalf = int(mu::engraving::SymId::noteheadCowellNinthNoteSeriesHalf),
    noteheadCowellNinthNoteSeriesWhole = int(mu::engraving::SymId::noteheadCowellNinthNoteSeriesWhole),
    noteheadCowellSeventhNoteSeriesBlack = int(mu::engraving::SymId::noteheadCowellSeventhNoteSeriesBlack),
    noteheadCowellSeventhNoteSeriesHalf = int(mu::engraving::SymId::noteheadCowellSeventhNoteSeriesHalf),
    noteheadCowellSeventhNoteSeriesWhole = int(mu::engraving::SymId::noteheadCowellSeventhNoteSeriesWhole),
    noteheadCowellThirdNoteSeriesBlack = int(mu::engraving::SymId::noteheadCowellThirdNoteSeriesBlack),
    noteheadCowellThirdNoteSeriesHalf = int(mu::engraving::SymId::noteheadCowellThirdNoteSeriesHalf),
    noteheadCowellThirdNoteSeriesWhole = int(mu::engraving::SymId::noteheadCowellThirdNoteSeriesWhole),
    noteheadCowellThirteenthNoteSeriesBlack = int(mu::engraving::SymId::noteheadCowellThirteenthNoteSeriesBlack),
    noteheadCowellThirteenthNoteSeriesHalf = int(mu::engraving::SymId::noteheadCowellThirteenthNoteSeriesHalf),
    noteheadCowellThirteenthNoteSeriesWhole = int(mu::engraving::SymId::noteheadCowellThirteenthNoteSeriesWhole),
    noteheadDiamondBlack = int(mu::engraving::SymId::noteheadDiamondBlack),
    noteheadDiamondBlackOld = int(mu::engraving::SymId::noteheadDiamondBlackOld),
    noteheadDiamondBlackWide = int(mu::engraving::SymId::noteheadDiamondBlackWide),
    noteheadDiamondClusterBlack2nd = int(mu::engraving::SymId::noteheadDiamondClusterBlack2nd),
    noteheadDiamondClusterBlack3rd = int(mu::engraving::SymId::noteheadDiamondClusterBlack3rd),
    noteheadDiamondClusterBlackBottom = int(mu::engraving::SymId::noteheadDiamondClusterBlackBottom),
    noteheadDiamondClusterBlackMiddle = int(mu::engraving::SymId::noteheadDiamondClusterBlackMiddle),
    noteheadDiamondClusterBlackTop = int(mu::engraving::SymId::noteheadDiamondClusterBlackTop),
    noteheadDiamondClusterWhite2nd = int(mu::engraving::SymId::noteheadDiamondClusterWhite2nd),
    noteheadDiamondClusterWhite3rd = int(mu::engraving::SymId::noteheadDiamondClusterWhite3rd),
    noteheadDiamondClusterWhiteBottom = int(mu::engraving::SymId::noteheadDiamondClusterWhiteBottom),
    noteheadDiamondClusterWhiteMiddle = int(mu::engraving::SymId::noteheadDiamondClusterWhiteMiddle),
    noteheadDiamondClusterWhiteTop = int(mu::engraving::SymId::noteheadDiamondClusterWhiteTop),
    noteheadDiamondDoubleWhole = int(mu::engraving::SymId::noteheadDiamondDoubleWhole),
    noteheadDiamondDoubleWholeOld = int(mu::engraving::SymId::noteheadDiamondDoubleWholeOld),
    noteheadDiamondHalf = int(mu::engraving::SymId::noteheadDiamondHalf),
    noteheadDiamondHalfFilled = int(mu::engraving::SymId::noteheadDiamondHalfFilled),
    noteheadDiamondHalfOld = int(mu::engraving::SymId::noteheadDiamondHalfOld),
    noteheadDiamondHalfWide = int(mu::engraving::SymId::noteheadDiamondHalfWide),
    noteheadDiamondOpen = int(mu::engraving::SymId::noteheadDiamondOpen),
    noteheadDiamondWhite = int(mu::engraving::SymId::noteheadDiamondWhite),
    noteheadDiamondWhiteWide = int(mu::engraving::SymId::noteheadDiamondWhiteWide),
    noteheadDiamondWhole = int(mu::engraving::SymId::noteheadDiamondWhole),
    noteheadDiamondWholeOld = int(mu::engraving::SymId::noteheadDiamondWholeOld),
    noteheadDoubleWhole = int(mu::engraving::SymId::noteheadDoubleWhole),
    noteheadDoubleWholeSquare = int(mu::engraving::SymId::noteheadDoubleWholeSquare),
    noteheadDoubleWholeWithX = int(mu::engraving::SymId::noteheadDoubleWholeWithX),
    noteheadHalf = int(mu::engraving::SymId::noteheadHalf),
    noteheadHalfFilled = int(mu::engraving::SymId::noteheadHalfFilled),
    noteheadHalfWithX = int(mu::engraving::SymId::noteheadHalfWithX),
    noteheadHeavyX = int(mu::engraving::SymId::noteheadHeavyX),
    noteheadHeavyXHat = int(mu::engraving::SymId::noteheadHeavyXHat),
    noteheadLargeArrowDownBlack = int(mu::engraving::SymId::noteheadLargeArrowDownBlack),
    noteheadLargeArrowDownDoubleWhole = int(mu::engraving::SymId::noteheadLargeArrowDownDoubleWhole),
    noteheadLargeArrowDownHalf = int(mu::engraving::SymId::noteheadLargeArrowDownHalf),
    noteheadLargeArrowDownWhole = int(mu::engraving::SymId::noteheadLargeArrowDownWhole),
    noteheadLargeArrowUpBlack = int(mu::engraving::SymId::noteheadLargeArrowUpBlack),
    noteheadLargeArrowUpDoubleWhole = int(mu::engraving::SymId::noteheadLargeArrowUpDoubleWhole),
    noteheadLargeArrowUpHalf = int(mu::engraving::SymId::noteheadLargeArrowUpHalf),
    noteheadLargeArrowUpWhole = int(mu::engraving::SymId::noteheadLargeArrowUpWhole),
    noteheadMoonBlack = int(mu::engraving::SymId::noteheadMoonBlack),
    noteheadMoonWhite = int(mu::engraving::SymId::noteheadMoonWhite),
    noteheadNancarrowSine = int(mu::engraving::SymId::noteheadNancarrowSine),
    noteheadNull = int(mu::engraving::SymId::noteheadNull),
    noteheadParenthesis = int(mu::engraving::SymId::noteheadParenthesis),
    noteheadParenthesisLeft = int(mu::engraving::SymId::noteheadParenthesisLeft),
    noteheadParenthesisRight = int(mu::engraving::SymId::noteheadParenthesisRight),
    noteheadPlusBlack = int(mu::engraving::SymId::noteheadPlusBlack),
    noteheadPlusDoubleWhole = int(mu::engraving::SymId::noteheadPlusDoubleWhole),
    noteheadPlusHalf = int(mu::engraving::SymId::noteheadPlusHalf),
    noteheadPlusWhole = int(mu::engraving::SymId::noteheadPlusWhole),
    noteheadRectangularClusterBlackBottom = int(mu::engraving::SymId::noteheadRectangularClusterBlackBottom),
    noteheadRectangularClusterBlackMiddle = int(mu::engraving::SymId::noteheadRectangularClusterBlackMiddle),
    noteheadRectangularClusterBlackTop = int(mu::engraving::SymId::noteheadRectangularClusterBlackTop),
    noteheadRectangularClusterWhiteBottom = int(mu::engraving::SymId::noteheadRectangularClusterWhiteBottom),
    noteheadRectangularClusterWhiteMiddle = int(mu::engraving::SymId::noteheadRectangularClusterWhiteMiddle),
    noteheadRectangularClusterWhiteTop = int(mu::engraving::SymId::noteheadRectangularClusterWhiteTop),
    noteheadRoundBlack = int(mu::engraving::SymId::noteheadRoundBlack),
    noteheadRoundBlackDoubleSlashed = int(mu::engraving::SymId::noteheadRoundBlackDoubleSlashed),
    noteheadRoundBlackLarge = int(mu::engraving::SymId::noteheadRoundBlackLarge),
    noteheadRoundBlackSlashed = int(mu::engraving::SymId::noteheadRoundBlackSlashed),
    noteheadRoundBlackSlashedLarge = int(mu::engraving::SymId::noteheadRoundBlackSlashedLarge),
    noteheadRoundWhite = int(mu::engraving::SymId::noteheadRoundWhite),
    noteheadRoundWhiteDoubleSlashed = int(mu::engraving::SymId::noteheadRoundWhiteDoubleSlashed),
    noteheadRoundWhiteLarge = int(mu::engraving::SymId::noteheadRoundWhiteLarge),
    noteheadRoundWhiteSlashed = int(mu::engraving::SymId::noteheadRoundWhiteSlashed),
    noteheadRoundWhiteSlashedLarge = int(mu::engraving::SymId::noteheadRoundWhiteSlashedLarge),
    noteheadRoundWhiteWithDot = int(mu::engraving::SymId::noteheadRoundWhiteWithDot),
    noteheadRoundWhiteWithDotLarge = int(mu::engraving::SymId::noteheadRoundWhiteWithDotLarge),
    noteheadSlashDiamondWhite = int(mu::engraving::SymId::noteheadSlashDiamondWhite),
    noteheadSlashHorizontalEnds = int(mu::engraving::SymId::noteheadSlashHorizontalEnds),
    noteheadSlashHorizontalEndsMuted = int(mu::engraving::SymId::noteheadSlashHorizontalEndsMuted),
    noteheadSlashVerticalEnds = int(mu::engraving::SymId::noteheadSlashVerticalEnds),
    noteheadSlashVerticalEndsMuted = int(mu::engraving::SymId::noteheadSlashVerticalEndsMuted),
    noteheadSlashVerticalEndsSmall = int(mu::engraving::SymId::noteheadSlashVerticalEndsSmall),
    noteheadSlashWhiteDoubleWhole = int(mu::engraving::SymId::noteheadSlashWhiteDoubleWhole),
    noteheadSlashWhiteHalf = int(mu::engraving::SymId::noteheadSlashWhiteHalf),
    noteheadSlashWhiteMuted = int(mu::engraving::SymId::noteheadSlashWhiteMuted),
    noteheadSlashWhiteWhole = int(mu::engraving::SymId::noteheadSlashWhiteWhole),
    noteheadSlashX = int(mu::engraving::SymId::noteheadSlashX),
    noteheadSlashedBlack1 = int(mu::engraving::SymId::noteheadSlashedBlack1),
    noteheadSlashedBlack2 = int(mu::engraving::SymId::noteheadSlashedBlack2),
    noteheadSlashedDoubleWhole1 = int(mu::engraving::SymId::noteheadSlashedDoubleWhole1),
    noteheadSlashedDoubleWhole2 = int(mu::engraving::SymId::noteheadSlashedDoubleWhole2),
    noteheadSlashedHalf1 = int(mu::engraving::SymId::noteheadSlashedHalf1),
    noteheadSlashedHalf2 = int(mu::engraving::SymId::noteheadSlashedHalf2),
    noteheadSlashedWhole1 = int(mu::engraving::SymId::noteheadSlashedWhole1),
    noteheadSlashedWhole2 = int(mu::engraving::SymId::noteheadSlashedWhole2),
    noteheadSquareBlack = int(mu::engraving::SymId::noteheadSquareBlack),
    noteheadSquareBlackLarge = int(mu::engraving::SymId::noteheadSquareBlackLarge),
    noteheadSquareBlackWhite = int(mu::engraving::SymId::noteheadSquareBlackWhite),
    noteheadSquareWhite = int(mu::engraving::SymId::noteheadSquareWhite),
    noteheadTriangleDownBlack = int(mu::engraving::SymId::noteheadTriangleDownBlack),
    noteheadTriangleDownDoubleWhole = int(mu::engraving::SymId::noteheadTriangleDownDoubleWhole),
    noteheadTriangleDownHalf = int(mu::engraving::SymId::noteheadTriangleDownHalf),
    noteheadTriangleDownWhite = int(mu::engraving::SymId::noteheadTriangleDownWhite),
    noteheadTriangleDownWhole = int(mu::engraving::SymId::noteheadTriangleDownWhole),
    noteheadTriangleLeftBlack = int(mu::engraving::SymId::noteheadTriangleLeftBlack),
    noteheadTriangleLeftWhite = int(mu::engraving::SymId::noteheadTriangleLeftWhite),
    noteheadTriangleRightBlack = int(mu::engraving::SymId::noteheadTriangleRightBlack),
    noteheadTriangleRightWhite = int(mu::engraving::SymId::noteheadTriangleRightWhite),
    noteheadTriangleRoundDownBlack = int(mu::engraving::SymId::noteheadTriangleRoundDownBlack),
    noteheadTriangleRoundDownWhite = int(mu::engraving::SymId::noteheadTriangleRoundDownWhite),
    noteheadTriangleUpBlack = int(mu::engraving::SymId::noteheadTriangleUpBlack),
    noteheadTriangleUpDoubleWhole = int(mu::engraving::SymId::noteheadTriangleUpDoubleWhole),
    noteheadTriangleUpHalf = int(mu::engraving::SymId::noteheadTriangleUpHalf),
    noteheadTriangleUpRightBlack = int(mu::engraving::SymId::noteheadTriangleUpRightBlack),
    noteheadTriangleUpRightWhite = int(mu::engraving::SymId::noteheadTriangleUpRightWhite),
    noteheadTriangleUpWhite = int(mu::engraving::SymId::noteheadTriangleUpWhite),
    noteheadTriangleUpWhole = int(mu::engraving::SymId::noteheadTriangleUpWhole),
    noteheadVoidWithX = int(mu::engraving::SymId::noteheadVoidWithX),
    noteheadWhole = int(mu::engraving::SymId::noteheadWhole),
    noteheadWholeFilled = int(mu::engraving::SymId::noteheadWholeFilled),
    noteheadWholeWithX = int(mu::engraving::SymId::noteheadWholeWithX),
    noteheadXBlack = int(mu::engraving::SymId::noteheadXBlack),
    noteheadXDoubleWhole = int(mu::engraving::SymId::noteheadXDoubleWhole),
    noteheadXHalf = int(mu::engraving::SymId::noteheadXHalf),
    noteheadXOrnate = int(mu::engraving::SymId::noteheadXOrnate),
    noteheadXOrnateEllipse = int(mu::engraving::SymId::noteheadXOrnateEllipse),
    noteheadXWhole = int(mu::engraving::SymId::noteheadXWhole),
    octaveBaselineA = int(mu::engraving::SymId::octaveBaselineA),
    octaveBaselineB = int(mu::engraving::SymId::octaveBaselineB),
    octaveBaselineM = int(mu::engraving::SymId::octaveBaselineM),
    octaveBaselineV = int(mu::engraving::SymId::octaveBaselineV),
    octaveBassa = int(mu::engraving::SymId::octaveBassa),
    octaveLoco = int(mu::engraving::SymId::octaveLoco),
    octaveParensLeft = int(mu::engraving::SymId::octaveParensLeft),
    octaveParensRight = int(mu::engraving::SymId::octaveParensRight),
    octaveSuperscriptA = int(mu::engraving::SymId::octaveSuperscriptA),
    octaveSuperscriptB = int(mu::engraving::SymId::octaveSuperscriptB),
    octaveSuperscriptM = int(mu::engraving::SymId::octaveSuperscriptM),
    octaveSuperscriptV = int(mu::engraving::SymId::octaveSuperscriptV),
    oneHandedRollStevens = int(mu::engraving::SymId::oneHandedRollStevens),
    organGerman2Fusae = int(mu::engraving::SymId::organGerman2Fusae),
    organGerman2Minimae = int(mu::engraving::SymId::organGerman2Minimae),
    organGerman2OctaveUp = int(mu::engraving::SymId::organGerman2OctaveUp),
    organGerman2Semifusae = int(mu::engraving::SymId::organGerman2Semifusae),
    organGerman2Semiminimae = int(mu::engraving::SymId::organGerman2Semiminimae),
    organGerman3Fusae = int(mu::engraving::SymId::organGerman3Fusae),
    organGerman3Minimae = int(mu::engraving::SymId::organGerman3Minimae),
    organGerman3Semifusae = int(mu::engraving::SymId::organGerman3Semifusae),
    organGerman3Semiminimae = int(mu::engraving::SymId::organGerman3Semiminimae),
    organGerman4Fusae = int(mu::engraving::SymId::organGerman4Fusae),
    organGerman4Minimae = int(mu::engraving::SymId::organGerman4Minimae),
    organGerman4Semifusae = int(mu::engraving::SymId::organGerman4Semifusae),
    organGerman4Semiminimae = int(mu::engraving::SymId::organGerman4Semiminimae),
    organGerman5Fusae = int(mu::engraving::SymId::organGerman5Fusae),
    organGerman5Minimae = int(mu::engraving::SymId::organGerman5Minimae),
    organGerman5Semifusae = int(mu::engraving::SymId::organGerman5Semifusae),
    organGerman5Semiminimae = int(mu::engraving::SymId::organGerman5Semiminimae),
    organGerman6Fusae = int(mu::engraving::SymId::organGerman6Fusae),
    organGerman6Minimae = int(mu::engraving::SymId::organGerman6Minimae),
    organGerman6Semifusae = int(mu::engraving::SymId::organGerman6Semifusae),
    organGerman6Semiminimae = int(mu::engraving::SymId::organGerman6Semiminimae),
    organGermanALower = int(mu::engraving::SymId::organGermanALower),
    organGermanAUpper = int(mu::engraving::SymId::organGermanAUpper),
    organGermanAugmentationDot = int(mu::engraving::SymId::organGermanAugmentationDot),
    organGermanBLower = int(mu::engraving::SymId::organGermanBLower),
    organGermanBUpper = int(mu::engraving::SymId::organGermanBUpper),
    organGermanBuxheimerBrevis2 = int(mu::engraving::SymId::organGermanBuxheimerBrevis2),
    organGermanBuxheimerBrevis3 = int(mu::engraving::SymId::organGermanBuxheimerBrevis3),
    organGermanBuxheimerMinimaRest = int(mu::engraving::SymId::organGermanBuxheimerMinimaRest),
    organGermanBuxheimerSemibrevis = int(mu::engraving::SymId::organGermanBuxheimerSemibrevis),
    organGermanBuxheimerSemibrevisRest = int(mu::engraving::SymId::organGermanBuxheimerSemibrevisRest),
    organGermanCLower = int(mu::engraving::SymId::organGermanCLower),
    organGermanCUpper = int(mu::engraving::SymId::organGermanCUpper),
    organGermanCisLower = int(mu::engraving::SymId::organGermanCisLower),
    organGermanCisUpper = int(mu::engraving::SymId::organGermanCisUpper),
    organGermanDLower = int(mu::engraving::SymId::organGermanDLower),
    organGermanDUpper = int(mu::engraving::SymId::organGermanDUpper),
    organGermanDisLower = int(mu::engraving::SymId::organGermanDisLower),
    organGermanDisUpper = int(mu::engraving::SymId::organGermanDisUpper),
    organGermanELower = int(mu::engraving::SymId::organGermanELower),
    organGermanEUpper = int(mu::engraving::SymId::organGermanEUpper),
    organGermanFLower = int(mu::engraving::SymId::organGermanFLower),
    organGermanFUpper = int(mu::engraving::SymId::organGermanFUpper),
    organGermanFisLower = int(mu::engraving::SymId::organGermanFisLower),
    organGermanFisUpper = int(mu::engraving::SymId::organGermanFisUpper),
    organGermanFusa = int(mu::engraving::SymId::organGermanFusa),
    organGermanFusaRest = int(mu::engraving::SymId::organGermanFusaRest),
    organGermanGLower = int(mu::engraving::SymId::organGermanGLower),
    organGermanGUpper = int(mu::engraving::SymId::organGermanGUpper),
    organGermanGisLower = int(mu::engraving::SymId::organGermanGisLower),
    organGermanGisUpper = int(mu::engraving::SymId::organGermanGisUpper),
    organGermanHLower = int(mu::engraving::SymId::organGermanHLower),
    organGermanHUpper = int(mu::engraving::SymId::organGermanHUpper),
    organGermanMinima = int(mu::engraving::SymId::organGermanMinima),
    organGermanMinimaRest = int(mu::engraving::SymId::organGermanMinimaRest),
    organGermanOctaveDown = int(mu::engraving::SymId::organGermanOctaveDown),
    organGermanOctaveUp = int(mu::engraving::SymId::organGermanOctaveUp),
    organGermanSemibrevis = int(mu::engraving::SymId::organGermanSemibrevis),
    organGermanSemibrevisRest = int(mu::engraving::SymId::organGermanSemibrevisRest),
    organGermanSemifusa = int(mu::engraving::SymId::organGermanSemifusa),
    organGermanSemifusaRest = int(mu::engraving::SymId::organGermanSemifusaRest),
    organGermanSemiminima = int(mu::engraving::SymId::organGermanSemiminima),
    organGermanSemiminimaRest = int(mu::engraving::SymId::organGermanSemiminimaRest),
    organGermanTie = int(mu::engraving::SymId::organGermanTie),
    ornamentBottomLeftConcaveStroke = int(mu::engraving::SymId::ornamentBottomLeftConcaveStroke),
    ornamentBottomLeftConcaveStrokeLarge = int(mu::engraving::SymId::ornamentBottomLeftConcaveStrokeLarge),
    ornamentBottomLeftConvexStroke = int(mu::engraving::SymId::ornamentBottomLeftConvexStroke),
    ornamentBottomRightConcaveStroke = int(mu::engraving::SymId::ornamentBottomRightConcaveStroke),
    ornamentBottomRightConvexStroke = int(mu::engraving::SymId::ornamentBottomRightConvexStroke),
    ornamentComma = int(mu::engraving::SymId::ornamentComma),
    ornamentDoubleObliqueLinesAfterNote = int(mu::engraving::SymId::ornamentDoubleObliqueLinesAfterNote),
    ornamentDoubleObliqueLinesBeforeNote = int(mu::engraving::SymId::ornamentDoubleObliqueLinesBeforeNote),
    ornamentDownCurve = int(mu::engraving::SymId::ornamentDownCurve),
    ornamentHaydn = int(mu::engraving::SymId::ornamentHaydn),
    ornamentHighLeftConcaveStroke = int(mu::engraving::SymId::ornamentHighLeftConcaveStroke),
    ornamentHighLeftConvexStroke = int(mu::engraving::SymId::ornamentHighLeftConvexStroke),
    ornamentHighRightConcaveStroke = int(mu::engraving::SymId::ornamentHighRightConcaveStroke),
    ornamentHighRightConvexStroke = int(mu::engraving::SymId::ornamentHighRightConvexStroke),
    ornamentHookAfterNote = int(mu::engraving::SymId::ornamentHookAfterNote),
    ornamentHookBeforeNote = int(mu::engraving::SymId::ornamentHookBeforeNote),
    ornamentLeftFacingHalfCircle = int(mu::engraving::SymId::ornamentLeftFacingHalfCircle),
    ornamentLeftFacingHook = int(mu::engraving::SymId::ornamentLeftFacingHook),
    ornamentLeftPlus = int(mu::engraving::SymId::ornamentLeftPlus),
    ornamentLeftShakeT = int(mu::engraving::SymId::ornamentLeftShakeT),
    ornamentLeftVerticalStroke = int(mu::engraving::SymId::ornamentLeftVerticalStroke),
    ornamentLeftVerticalStrokeWithCross = int(mu::engraving::SymId::ornamentLeftVerticalStrokeWithCross),
    ornamentLowLeftConcaveStroke = int(mu::engraving::SymId::ornamentLowLeftConcaveStroke),
    ornamentLowLeftConvexStroke = int(mu::engraving::SymId::ornamentLowLeftConvexStroke),
    ornamentLowRightConcaveStroke = int(mu::engraving::SymId::ornamentLowRightConcaveStroke),
    ornamentLowRightConvexStroke = int(mu::engraving::SymId::ornamentLowRightConvexStroke),
    ornamentMiddleVerticalStroke = int(mu::engraving::SymId::ornamentMiddleVerticalStroke),
    ornamentMordent = int(mu::engraving::SymId::ornamentMordent),
    ornamentObliqueLineAfterNote = int(mu::engraving::SymId::ornamentObliqueLineAfterNote),
    ornamentObliqueLineBeforeNote = int(mu::engraving::SymId::ornamentObliqueLineBeforeNote),
    ornamentObliqueLineHorizAfterNote = int(mu::engraving::SymId::ornamentObliqueLineHorizAfterNote),
    ornamentObliqueLineHorizBeforeNote = int(mu::engraving::SymId::ornamentObliqueLineHorizBeforeNote),
    ornamentOriscus = int(mu::engraving::SymId::ornamentOriscus),
    ornamentPinceCouperin = int(mu::engraving::SymId::ornamentPinceCouperin),
    ornamentPortDeVoixV = int(mu::engraving::SymId::ornamentPortDeVoixV),
    ornamentPrecompAppoggTrill = int(mu::engraving::SymId::ornamentPrecompAppoggTrill),
    ornamentPrecompAppoggTrillSuffix = int(mu::engraving::SymId::ornamentPrecompAppoggTrillSuffix),
    ornamentPrecompCadence = int(mu::engraving::SymId::ornamentPrecompCadence),
    ornamentPrecompCadenceUpperPrefix = int(mu::engraving::SymId::ornamentPrecompCadenceUpperPrefix),
    ornamentPrecompCadenceUpperPrefixTurn = int(mu::engraving::SymId::ornamentPrecompCadenceUpperPrefixTurn),
    ornamentPrecompCadenceWithTurn = int(mu::engraving::SymId::ornamentPrecompCadenceWithTurn),
    ornamentPrecompDescendingSlide = int(mu::engraving::SymId::ornamentPrecompDescendingSlide),
    ornamentPrecompDoubleCadenceLowerPrefix = int(mu::engraving::SymId::ornamentPrecompDoubleCadenceLowerPrefix),
    ornamentPrecompDoubleCadenceUpperPrefix = int(mu::engraving::SymId::ornamentPrecompDoubleCadenceUpperPrefix),
    ornamentPrecompDoubleCadenceUpperPrefixTurn = int(mu::engraving::SymId::ornamentPrecompDoubleCadenceUpperPrefixTurn),
    ornamentPrecompInvertedMordentUpperPrefix = int(mu::engraving::SymId::ornamentPrecompInvertedMordentUpperPrefix),
    ornamentPrecompMordentRelease = int(mu::engraving::SymId::ornamentPrecompMordentRelease),
    ornamentPrecompMordentUpperPrefix = int(mu::engraving::SymId::ornamentPrecompMordentUpperPrefix),
    ornamentPrecompPortDeVoixMordent = int(mu::engraving::SymId::ornamentPrecompPortDeVoixMordent),
    ornamentPrecompSlide = int(mu::engraving::SymId::ornamentPrecompSlide),
    ornamentPrecompSlideTrillBach = int(mu::engraving::SymId::ornamentPrecompSlideTrillBach),
    ornamentPrecompSlideTrillDAnglebert = int(mu::engraving::SymId::ornamentPrecompSlideTrillDAnglebert),
    ornamentPrecompSlideTrillMarpurg = int(mu::engraving::SymId::ornamentPrecompSlideTrillMarpurg),
    ornamentPrecompSlideTrillMuffat = int(mu::engraving::SymId::ornamentPrecompSlideTrillMuffat),
    ornamentPrecompSlideTrillSuffixMuffat = int(mu::engraving::SymId::ornamentPrecompSlideTrillSuffixMuffat),
    ornamentPrecompTrillLowerSuffix = int(mu::engraving::SymId::ornamentPrecompTrillLowerSuffix),
    ornamentPrecompTrillSuffixDandrieu = int(mu::engraving::SymId::ornamentPrecompTrillSuffixDandrieu),
    ornamentPrecompTrillWithMordent = int(mu::engraving::SymId::ornamentPrecompTrillWithMordent),
    ornamentPrecompTurnTrillBach = int(mu::engraving::SymId::ornamentPrecompTurnTrillBach),
    ornamentPrecompTurnTrillDAnglebert = int(mu::engraving::SymId::ornamentPrecompTurnTrillDAnglebert),
    ornamentQuilisma = int(mu::engraving::SymId::ornamentQuilisma),
    ornamentRightFacingHalfCircle = int(mu::engraving::SymId::ornamentRightFacingHalfCircle),
    ornamentRightFacingHook = int(mu::engraving::SymId::ornamentRightFacingHook),
    ornamentRightVerticalStroke = int(mu::engraving::SymId::ornamentRightVerticalStroke),
    ornamentSchleifer = int(mu::engraving::SymId::ornamentSchleifer),
    ornamentShake3 = int(mu::engraving::SymId::ornamentShake3),
    ornamentShakeMuffat1 = int(mu::engraving::SymId::ornamentShakeMuffat1),
    ornamentShortObliqueLineAfterNote = int(mu::engraving::SymId::ornamentShortObliqueLineAfterNote),
    ornamentShortObliqueLineBeforeNote = int(mu::engraving::SymId::ornamentShortObliqueLineBeforeNote),
    ornamentShortTrill = int(mu::engraving::SymId::ornamentShortTrill),
    ornamentTopLeftConcaveStroke = int(mu::engraving::SymId::ornamentTopLeftConcaveStroke),
    ornamentTopLeftConvexStroke = int(mu::engraving::SymId::ornamentTopLeftConvexStroke),
    ornamentTopRightConcaveStroke = int(mu::engraving::SymId::ornamentTopRightConcaveStroke),
    ornamentTopRightConvexStroke = int(mu::engraving::SymId::ornamentTopRightConvexStroke),
    ornamentTremblement = int(mu::engraving::SymId::ornamentTremblement),
    ornamentTremblementCouperin = int(mu::engraving::SymId::ornamentTremblementCouperin),
    ornamentTrill = int(mu::engraving::SymId::ornamentTrill),
    ornamentTurn = int(mu::engraving::SymId::ornamentTurn),
    ornamentTurnInverted = int(mu::engraving::SymId::ornamentTurnInverted),
    ornamentTurnSlash = int(mu::engraving::SymId::ornamentTurnSlash),
    ornamentTurnUp = int(mu::engraving::SymId::ornamentTurnUp),
    ornamentTurnUpS = int(mu::engraving::SymId::ornamentTurnUpS),
    ornamentUpCurve = int(mu::engraving::SymId::ornamentUpCurve),
    ornamentVerticalLine = int(mu::engraving::SymId::ornamentVerticalLine),
    ornamentZigZagLineNoRightEnd = int(mu::engraving::SymId::ornamentZigZagLineNoRightEnd),
    ornamentZigZagLineWithRightEnd = int(mu::engraving::SymId::ornamentZigZagLineWithRightEnd),
    ottava = int(mu::engraving::SymId::ottava),
    ottavaAlta = int(mu::engraving::SymId::ottavaAlta),
    ottavaBassa = int(mu::engraving::SymId::ottavaBassa),
    ottavaBassaBa = int(mu::engraving::SymId::ottavaBassaBa),
    ottavaBassaVb = int(mu::engraving::SymId::ottavaBassaVb),
    pendereckiTremolo = int(mu::engraving::SymId::pendereckiTremolo),
    pictAgogo = int(mu::engraving::SymId::pictAgogo),
    pictAlmglocken = int(mu::engraving::SymId::pictAlmglocken),
    pictAnvil = int(mu::engraving::SymId::pictAnvil),
    pictBambooChimes = int(mu::engraving::SymId::pictBambooChimes),
    pictBambooScraper = int(mu::engraving::SymId::pictBambooScraper),
    pictBassDrum = int(mu::engraving::SymId::pictBassDrum),
    pictBassDrumOnSide = int(mu::engraving::SymId::pictBassDrumOnSide),
    pictBeaterBow = int(mu::engraving::SymId::pictBeaterBow),
    pictBeaterBox = int(mu::engraving::SymId::pictBeaterBox),
    pictBeaterBrassMalletsDown = int(mu::engraving::SymId::pictBeaterBrassMalletsDown),
    pictBeaterBrassMalletsLeft = int(mu::engraving::SymId::pictBeaterBrassMalletsLeft),
    pictBeaterBrassMalletsRight = int(mu::engraving::SymId::pictBeaterBrassMalletsRight),
    pictBeaterBrassMalletsUp = int(mu::engraving::SymId::pictBeaterBrassMalletsUp),
    pictBeaterCombiningDashedCircle = int(mu::engraving::SymId::pictBeaterCombiningDashedCircle),
    pictBeaterCombiningParentheses = int(mu::engraving::SymId::pictBeaterCombiningParentheses),
    pictBeaterDoubleBassDrumDown = int(mu::engraving::SymId::pictBeaterDoubleBassDrumDown),
    pictBeaterDoubleBassDrumUp = int(mu::engraving::SymId::pictBeaterDoubleBassDrumUp),
    pictBeaterFinger = int(mu::engraving::SymId::pictBeaterFinger),
    pictBeaterFingernails = int(mu::engraving::SymId::pictBeaterFingernails),
    pictBeaterFist = int(mu::engraving::SymId::pictBeaterFist),
    pictBeaterGuiroScraper = int(mu::engraving::SymId::pictBeaterGuiroScraper),
    pictBeaterHammer = int(mu::engraving::SymId::pictBeaterHammer),
    pictBeaterHammerMetalDown = int(mu::engraving::SymId::pictBeaterHammerMetalDown),
    pictBeaterHammerMetalUp = int(mu::engraving::SymId::pictBeaterHammerMetalUp),
    pictBeaterHammerPlasticDown = int(mu::engraving::SymId::pictBeaterHammerPlasticDown),
    pictBeaterHammerPlasticUp = int(mu::engraving::SymId::pictBeaterHammerPlasticUp),
    pictBeaterHammerWoodDown = int(mu::engraving::SymId::pictBeaterHammerWoodDown),
    pictBeaterHammerWoodUp = int(mu::engraving::SymId::pictBeaterHammerWoodUp),
    pictBeaterHand = int(mu::engraving::SymId::pictBeaterHand),
    pictBeaterHardBassDrumDown = int(mu::engraving::SymId::pictBeaterHardBassDrumDown),
    pictBeaterHardBassDrumUp = int(mu::engraving::SymId::pictBeaterHardBassDrumUp),
    pictBeaterHardGlockenspielDown = int(mu::engraving::SymId::pictBeaterHardGlockenspielDown),
    pictBeaterHardGlockenspielLeft = int(mu::engraving::SymId::pictBeaterHardGlockenspielLeft),
    pictBeaterHardGlockenspielRight = int(mu::engraving::SymId::pictBeaterHardGlockenspielRight),
    pictBeaterHardGlockenspielUp = int(mu::engraving::SymId::pictBeaterHardGlockenspielUp),
    pictBeaterHardTimpaniDown = int(mu::engraving::SymId::pictBeaterHardTimpaniDown),
    pictBeaterHardTimpaniLeft = int(mu::engraving::SymId::pictBeaterHardTimpaniLeft),
    pictBeaterHardTimpaniRight = int(mu::engraving::SymId::pictBeaterHardTimpaniRight),
    pictBeaterHardTimpaniUp = int(mu::engraving::SymId::pictBeaterHardTimpaniUp),
    pictBeaterHardXylophoneDown = int(mu::engraving::SymId::pictBeaterHardXylophoneDown),
    pictBeaterHardXylophoneLeft = int(mu::engraving::SymId::pictBeaterHardXylophoneLeft),
    pictBeaterHardXylophoneRight = int(mu::engraving::SymId::pictBeaterHardXylophoneRight),
    pictBeaterHardXylophoneUp = int(mu::engraving::SymId::pictBeaterHardXylophoneUp),
    pictBeaterHardYarnDown = int(mu::engraving::SymId::pictBeaterHardYarnDown),
    pictBeaterHardYarnLeft = int(mu::engraving::SymId::pictBeaterHardYarnLeft),
    pictBeaterHardYarnRight = int(mu::engraving::SymId::pictBeaterHardYarnRight),
    pictBeaterHardYarnUp = int(mu::engraving::SymId::pictBeaterHardYarnUp),
    pictBeaterJazzSticksDown = int(mu::engraving::SymId::pictBeaterJazzSticksDown),
    pictBeaterJazzSticksUp = int(mu::engraving::SymId::pictBeaterJazzSticksUp),
    pictBeaterKnittingNeedle = int(mu::engraving::SymId::pictBeaterKnittingNeedle),
    pictBeaterMallet = int(mu::engraving::SymId::pictBeaterMallet),
    pictBeaterMalletDown = int(mu::engraving::SymId::pictBeaterMalletDown),
    pictBeaterMediumBassDrumDown = int(mu::engraving::SymId::pictBeaterMediumBassDrumDown),
    pictBeaterMediumBassDrumUp = int(mu::engraving::SymId::pictBeaterMediumBassDrumUp),
    pictBeaterMediumTimpaniDown = int(mu::engraving::SymId::pictBeaterMediumTimpaniDown),
    pictBeaterMediumTimpaniLeft = int(mu::engraving::SymId::pictBeaterMediumTimpaniLeft),
    pictBeaterMediumTimpaniRight = int(mu::engraving::SymId::pictBeaterMediumTimpaniRight),
    pictBeaterMediumTimpaniUp = int(mu::engraving::SymId::pictBeaterMediumTimpaniUp),
    pictBeaterMediumXylophoneDown = int(mu::engraving::SymId::pictBeaterMediumXylophoneDown),
    pictBeaterMediumXylophoneLeft = int(mu::engraving::SymId::pictBeaterMediumXylophoneLeft),
    pictBeaterMediumXylophoneRight = int(mu::engraving::SymId::pictBeaterMediumXylophoneRight),
    pictBeaterMediumXylophoneUp = int(mu::engraving::SymId::pictBeaterMediumXylophoneUp),
    pictBeaterMediumYarnDown = int(mu::engraving::SymId::pictBeaterMediumYarnDown),
    pictBeaterMediumYarnLeft = int(mu::engraving::SymId::pictBeaterMediumYarnLeft),
    pictBeaterMediumYarnRight = int(mu::engraving::SymId::pictBeaterMediumYarnRight),
    pictBeaterMediumYarnUp = int(mu::engraving::SymId::pictBeaterMediumYarnUp),
    pictBeaterMetalBassDrumDown = int(mu::engraving::SymId::pictBeaterMetalBassDrumDown),
    pictBeaterMetalBassDrumUp = int(mu::engraving::SymId::pictBeaterMetalBassDrumUp),
    pictBeaterMetalDown = int(mu::engraving::SymId::pictBeaterMetalDown),
    pictBeaterMetalHammer = int(mu::engraving::SymId::pictBeaterMetalHammer),
    pictBeaterMetalLeft = int(mu::engraving::SymId::pictBeaterMetalLeft),
    pictBeaterMetalRight = int(mu::engraving::SymId::pictBeaterMetalRight),
    pictBeaterMetalUp = int(mu::engraving::SymId::pictBeaterMetalUp),
    pictBeaterSnareSticksDown = int(mu::engraving::SymId::pictBeaterSnareSticksDown),
    pictBeaterSnareSticksUp = int(mu::engraving::SymId::pictBeaterSnareSticksUp),
    pictBeaterSoftBassDrumDown = int(mu::engraving::SymId::pictBeaterSoftBassDrumDown),
    pictBeaterSoftBassDrumUp = int(mu::engraving::SymId::pictBeaterSoftBassDrumUp),
    pictBeaterSoftGlockenspielDown = int(mu::engraving::SymId::pictBeaterSoftGlockenspielDown),
    pictBeaterSoftGlockenspielLeft = int(mu::engraving::SymId::pictBeaterSoftGlockenspielLeft),
    pictBeaterSoftGlockenspielRight = int(mu::engraving::SymId::pictBeaterSoftGlockenspielRight),
    pictBeaterSoftGlockenspielUp = int(mu::engraving::SymId::pictBeaterSoftGlockenspielUp),
    pictBeaterSoftTimpaniDown = int(mu::engraving::SymId::pictBeaterSoftTimpaniDown),
    pictBeaterSoftTimpaniLeft = int(mu::engraving::SymId::pictBeaterSoftTimpaniLeft),
    pictBeaterSoftTimpaniRight = int(mu::engraving::SymId::pictBeaterSoftTimpaniRight),
    pictBeaterSoftTimpaniUp = int(mu::engraving::SymId::pictBeaterSoftTimpaniUp),
    pictBeaterSoftXylophone = int(mu::engraving::SymId::pictBeaterSoftXylophone),
    pictBeaterSoftXylophoneDown = int(mu::engraving::SymId::pictBeaterSoftXylophoneDown),
    pictBeaterSoftXylophoneLeft = int(mu::engraving::SymId::pictBeaterSoftXylophoneLeft),
    pictBeaterSoftXylophoneRight = int(mu::engraving::SymId::pictBeaterSoftXylophoneRight),
    pictBeaterSoftXylophoneUp = int(mu::engraving::SymId::pictBeaterSoftXylophoneUp),
    pictBeaterSoftYarnDown = int(mu::engraving::SymId::pictBeaterSoftYarnDown),
    pictBeaterSoftYarnLeft = int(mu::engraving::SymId::pictBeaterSoftYarnLeft),
    pictBeaterSoftYarnRight = int(mu::engraving::SymId::pictBeaterSoftYarnRight),
    pictBeaterSoftYarnUp = int(mu::engraving::SymId::pictBeaterSoftYarnUp),
    pictBeaterSpoonWoodenMallet = int(mu::engraving::SymId::pictBeaterSpoonWoodenMallet),
    pictBeaterSuperballDown = int(mu::engraving::SymId::pictBeaterSuperballDown),
    pictBeaterSuperballLeft = int(mu::engraving::SymId::pictBeaterSuperballLeft),
    pictBeaterSuperballRight = int(mu::engraving::SymId::pictBeaterSuperballRight),
    pictBeaterSuperballUp = int(mu::engraving::SymId::pictBeaterSuperballUp),
    pictBeaterTriangleDown = int(mu::engraving::SymId::pictBeaterTriangleDown),
    pictBeaterTrianglePlain = int(mu::engraving::SymId::pictBeaterTrianglePlain),
    pictBeaterTriangleUp = int(mu::engraving::SymId::pictBeaterTriangleUp),
    pictBeaterWireBrushesDown = int(mu::engraving::SymId::pictBeaterWireBrushesDown),
    pictBeaterWireBrushesUp = int(mu::engraving::SymId::pictBeaterWireBrushesUp),
    pictBeaterWoodTimpaniDown = int(mu::engraving::SymId::pictBeaterWoodTimpaniDown),
    pictBeaterWoodTimpaniLeft = int(mu::engraving::SymId::pictBeaterWoodTimpaniLeft),
    pictBeaterWoodTimpaniRight = int(mu::engraving::SymId::pictBeaterWoodTimpaniRight),
    pictBeaterWoodTimpaniUp = int(mu::engraving::SymId::pictBeaterWoodTimpaniUp),
    pictBeaterWoodXylophoneDown = int(mu::engraving::SymId::pictBeaterWoodXylophoneDown),
    pictBeaterWoodXylophoneLeft = int(mu::engraving::SymId::pictBeaterWoodXylophoneLeft),
    pictBeaterWoodXylophoneRight = int(mu::engraving::SymId::pictBeaterWoodXylophoneRight),
    pictBeaterWoodXylophoneUp = int(mu::engraving::SymId::pictBeaterWoodXylophoneUp),
    pictBell = int(mu::engraving::SymId::pictBell),
    pictBellOfCymbal = int(mu::engraving::SymId::pictBellOfCymbal),
    pictBellPlate = int(mu::engraving::SymId::pictBellPlate),
    pictBellTree = int(mu::engraving::SymId::pictBellTree),
    pictBirdWhistle = int(mu::engraving::SymId::pictBirdWhistle),
    pictBoardClapper = int(mu::engraving::SymId::pictBoardClapper),
    pictBongos = int(mu::engraving::SymId::pictBongos),
    pictBrakeDrum = int(mu::engraving::SymId::pictBrakeDrum),
    pictCabasa = int(mu::engraving::SymId::pictCabasa),
    pictCannon = int(mu::engraving::SymId::pictCannon),
    pictCarHorn = int(mu::engraving::SymId::pictCarHorn),
    pictCastanets = int(mu::engraving::SymId::pictCastanets),
    pictCastanetsWithHandle = int(mu::engraving::SymId::pictCastanetsWithHandle),
    pictCelesta = int(mu::engraving::SymId::pictCelesta),
    pictCencerro = int(mu::engraving::SymId::pictCencerro),
    pictCenter1 = int(mu::engraving::SymId::pictCenter1),
    pictCenter2 = int(mu::engraving::SymId::pictCenter2),
    pictCenter3 = int(mu::engraving::SymId::pictCenter3),
    pictChainRattle = int(mu::engraving::SymId::pictChainRattle),
    pictChimes = int(mu::engraving::SymId::pictChimes),
    pictChineseCymbal = int(mu::engraving::SymId::pictChineseCymbal),
    pictChokeCymbal = int(mu::engraving::SymId::pictChokeCymbal),
    pictClaves = int(mu::engraving::SymId::pictClaves),
    pictCoins = int(mu::engraving::SymId::pictCoins),
    pictConga = int(mu::engraving::SymId::pictConga),
    pictCowBell = int(mu::engraving::SymId::pictCowBell),
    pictCrashCymbals = int(mu::engraving::SymId::pictCrashCymbals),
    pictCrotales = int(mu::engraving::SymId::pictCrotales),
    pictCrushStem = int(mu::engraving::SymId::pictCrushStem),
    pictCuica = int(mu::engraving::SymId::pictCuica),
    pictCymbalTongs = int(mu::engraving::SymId::pictCymbalTongs),
    pictDamp1 = int(mu::engraving::SymId::pictDamp1),
    pictDamp2 = int(mu::engraving::SymId::pictDamp2),
    pictDamp3 = int(mu::engraving::SymId::pictDamp3),
    pictDamp4 = int(mu::engraving::SymId::pictDamp4),
    pictDeadNoteStem = int(mu::engraving::SymId::pictDeadNoteStem),
    pictDrumStick = int(mu::engraving::SymId::pictDrumStick),
    pictDuckCall = int(mu::engraving::SymId::pictDuckCall),
    pictEdgeOfCymbal = int(mu::engraving::SymId::pictEdgeOfCymbal),
    pictEmptyTrap = int(mu::engraving::SymId::pictEmptyTrap),
    pictFingerCymbals = int(mu::engraving::SymId::pictFingerCymbals),
    pictFlexatone = int(mu::engraving::SymId::pictFlexatone),
    pictFootballRatchet = int(mu::engraving::SymId::pictFootballRatchet),
    pictGlassHarmonica = int(mu::engraving::SymId::pictGlassHarmonica),
    pictGlassHarp = int(mu::engraving::SymId::pictGlassHarp),
    pictGlassPlateChimes = int(mu::engraving::SymId::pictGlassPlateChimes),
    pictGlassTubeChimes = int(mu::engraving::SymId::pictGlassTubeChimes),
    pictGlsp = int(mu::engraving::SymId::pictGlsp),
    pictGlspSmithBrindle = int(mu::engraving::SymId::pictGlspSmithBrindle),
    pictGobletDrum = int(mu::engraving::SymId::pictGobletDrum),
    pictGong = int(mu::engraving::SymId::pictGong),
    pictGongWithButton = int(mu::engraving::SymId::pictGongWithButton),
    pictGuiro = int(mu::engraving::SymId::pictGuiro),
    pictGumHardDown = int(mu::engraving::SymId::pictGumHardDown),
    pictGumHardLeft = int(mu::engraving::SymId::pictGumHardLeft),
    pictGumHardRight = int(mu::engraving::SymId::pictGumHardRight),
    pictGumHardUp = int(mu::engraving::SymId::pictGumHardUp),
    pictGumMediumDown = int(mu::engraving::SymId::pictGumMediumDown),
    pictGumMediumLeft = int(mu::engraving::SymId::pictGumMediumLeft),
    pictGumMediumRight = int(mu::engraving::SymId::pictGumMediumRight),
    pictGumMediumUp = int(mu::engraving::SymId::pictGumMediumUp),
    pictGumSoftDown = int(mu::engraving::SymId::pictGumSoftDown),
    pictGumSoftLeft = int(mu::engraving::SymId::pictGumSoftLeft),
    pictGumSoftRight = int(mu::engraving::SymId::pictGumSoftRight),
    pictGumSoftUp = int(mu::engraving::SymId::pictGumSoftUp),
    pictHalfOpen1 = int(mu::engraving::SymId::pictHalfOpen1),
    pictHalfOpen2 = int(mu::engraving::SymId::pictHalfOpen2),
    pictHandbell = int(mu::engraving::SymId::pictHandbell),
    pictHiHat = int(mu::engraving::SymId::pictHiHat),
    pictHiHatOnStand = int(mu::engraving::SymId::pictHiHatOnStand),
    pictJawHarp = int(mu::engraving::SymId::pictJawHarp),
    pictJingleBells = int(mu::engraving::SymId::pictJingleBells),
    pictKlaxonHorn = int(mu::engraving::SymId::pictKlaxonHorn),
    pictLeftHandCircle = int(mu::engraving::SymId::pictLeftHandCircle),
    pictLionsRoar = int(mu::engraving::SymId::pictLionsRoar),
    pictLithophone = int(mu::engraving::SymId::pictLithophone),
    pictLogDrum = int(mu::engraving::SymId::pictLogDrum),
    pictLotusFlute = int(mu::engraving::SymId::pictLotusFlute),
    pictMar = int(mu::engraving::SymId::pictMar),
    pictMarSmithBrindle = int(mu::engraving::SymId::pictMarSmithBrindle),
    pictMaraca = int(mu::engraving::SymId::pictMaraca),
    pictMaracas = int(mu::engraving::SymId::pictMaracas),
    pictMegaphone = int(mu::engraving::SymId::pictMegaphone),
    pictMetalPlateChimes = int(mu::engraving::SymId::pictMetalPlateChimes),
    pictMetalTubeChimes = int(mu::engraving::SymId::pictMetalTubeChimes),
    pictMusicalSaw = int(mu::engraving::SymId::pictMusicalSaw),
    pictNormalPosition = int(mu::engraving::SymId::pictNormalPosition),
    pictOnRim = int(mu::engraving::SymId::pictOnRim),
    pictOpen = int(mu::engraving::SymId::pictOpen),
    pictOpenRimShot = int(mu::engraving::SymId::pictOpenRimShot),
    pictPistolShot = int(mu::engraving::SymId::pictPistolShot),
    pictPoliceWhistle = int(mu::engraving::SymId::pictPoliceWhistle),
    pictQuijada = int(mu::engraving::SymId::pictQuijada),
    pictRainstick = int(mu::engraving::SymId::pictRainstick),
    pictRatchet = int(mu::engraving::SymId::pictRatchet),
    pictRecoReco = int(mu::engraving::SymId::pictRecoReco),
    pictRightHandSquare = int(mu::engraving::SymId::pictRightHandSquare),
    pictRim1 = int(mu::engraving::SymId::pictRim1),
    pictRim2 = int(mu::engraving::SymId::pictRim2),
    pictRim3 = int(mu::engraving::SymId::pictRim3),
    pictRimShotOnStem = int(mu::engraving::SymId::pictRimShotOnStem),
    pictSandpaperBlocks = int(mu::engraving::SymId::pictSandpaperBlocks),
    pictScrapeAroundRim = int(mu::engraving::SymId::pictScrapeAroundRim),
    pictScrapeAroundRimClockwise = int(mu::engraving::SymId::pictScrapeAroundRimClockwise),
    pictScrapeCenterToEdge = int(mu::engraving::SymId::pictScrapeCenterToEdge),
    pictScrapeEdgeToCenter = int(mu::engraving::SymId::pictScrapeEdgeToCenter),
    pictShellBells = int(mu::engraving::SymId::pictShellBells),
    pictShellChimes = int(mu::engraving::SymId::pictShellChimes),
    pictSiren = int(mu::engraving::SymId::pictSiren),
    pictSistrum = int(mu::engraving::SymId::pictSistrum),
    pictSizzleCymbal = int(mu::engraving::SymId::pictSizzleCymbal),
    pictSleighBell = int(mu::engraving::SymId::pictSleighBell),
    pictSlideBrushOnGong = int(mu::engraving::SymId::pictSlideBrushOnGong),
    pictSlideWhistle = int(mu::engraving::SymId::pictSlideWhistle),
    pictSlitDrum = int(mu::engraving::SymId::pictSlitDrum),
    pictSnareDrum = int(mu::engraving::SymId::pictSnareDrum),
    pictSnareDrumMilitary = int(mu::engraving::SymId::pictSnareDrumMilitary),
    pictSnareDrumSnaresOff = int(mu::engraving::SymId::pictSnareDrumSnaresOff),
    pictSteelDrums = int(mu::engraving::SymId::pictSteelDrums),
    pictStickShot = int(mu::engraving::SymId::pictStickShot),
    pictSuperball = int(mu::engraving::SymId::pictSuperball),
    pictSuspendedCymbal = int(mu::engraving::SymId::pictSuspendedCymbal),
    pictSwishStem = int(mu::engraving::SymId::pictSwishStem),
    pictTabla = int(mu::engraving::SymId::pictTabla),
    pictTamTam = int(mu::engraving::SymId::pictTamTam),
    pictTamTamWithBeater = int(mu::engraving::SymId::pictTamTamWithBeater),
    pictTambourine = int(mu::engraving::SymId::pictTambourine),
    pictTempleBlocks = int(mu::engraving::SymId::pictTempleBlocks),
    pictTenorDrum = int(mu::engraving::SymId::pictTenorDrum),
    pictThundersheet = int(mu::engraving::SymId::pictThundersheet),
    pictTimbales = int(mu::engraving::SymId::pictTimbales),
    pictTimpani = int(mu::engraving::SymId::pictTimpani),
    pictTomTom = int(mu::engraving::SymId::pictTomTom),
    pictTomTomChinese = int(mu::engraving::SymId::pictTomTomChinese),
    pictTomTomIndoAmerican = int(mu::engraving::SymId::pictTomTomIndoAmerican),
    pictTomTomJapanese = int(mu::engraving::SymId::pictTomTomJapanese),
    pictTriangle = int(mu::engraving::SymId::pictTriangle),
    pictTubaphone = int(mu::engraving::SymId::pictTubaphone),
    pictTubularBells = int(mu::engraving::SymId::pictTubularBells),
    pictTurnLeftStem = int(mu::engraving::SymId::pictTurnLeftStem),
    pictTurnRightLeftStem = int(mu::engraving::SymId::pictTurnRightLeftStem),
    pictTurnRightStem = int(mu::engraving::SymId::pictTurnRightStem),
    pictVib = int(mu::engraving::SymId::pictVib),
    pictVibMotorOff = int(mu::engraving::SymId::pictVibMotorOff),
    pictVibSmithBrindle = int(mu::engraving::SymId::pictVibSmithBrindle),
    pictVibraslap = int(mu::engraving::SymId::pictVibraslap),
    pictVietnameseHat = int(mu::engraving::SymId::pictVietnameseHat),
    pictWhip = int(mu::engraving::SymId::pictWhip),
    pictWindChimesGlass = int(mu::engraving::SymId::pictWindChimesGlass),
    pictWindMachine = int(mu::engraving::SymId::pictWindMachine),
    pictWindWhistle = int(mu::engraving::SymId::pictWindWhistle),
    pictWoodBlock = int(mu::engraving::SymId::pictWoodBlock),
    pictWoundHardDown = int(mu::engraving::SymId::pictWoundHardDown),
    pictWoundHardLeft = int(mu::engraving::SymId::pictWoundHardLeft),
    pictWoundHardRight = int(mu::engraving::SymId::pictWoundHardRight),
    pictWoundHardUp = int(mu::engraving::SymId::pictWoundHardUp),
    pictWoundSoftDown = int(mu::engraving::SymId::pictWoundSoftDown),
    pictWoundSoftLeft = int(mu::engraving::SymId::pictWoundSoftLeft),
    pictWoundSoftRight = int(mu::engraving::SymId::pictWoundSoftRight),
    pictWoundSoftUp = int(mu::engraving::SymId::pictWoundSoftUp),
    pictXyl = int(mu::engraving::SymId::pictXyl),
    pictXylBass = int(mu::engraving::SymId::pictXylBass),
    pictXylSmithBrindle = int(mu::engraving::SymId::pictXylSmithBrindle),
    pictXylTenor = int(mu::engraving::SymId::pictXylTenor),
    pictXylTenorTrough = int(mu::engraving::SymId::pictXylTenorTrough),
    pictXylTrough = int(mu::engraving::SymId::pictXylTrough),
    pluckedBuzzPizzicato = int(mu::engraving::SymId::pluckedBuzzPizzicato),
    pluckedDamp = int(mu::engraving::SymId::pluckedDamp),
    pluckedDampAll = int(mu::engraving::SymId::pluckedDampAll),
    pluckedDampOnStem = int(mu::engraving::SymId::pluckedDampOnStem),
    pluckedFingernailFlick = int(mu::engraving::SymId::pluckedFingernailFlick),
    pluckedLeftHandPizzicato = int(mu::engraving::SymId::pluckedLeftHandPizzicato),
    pluckedPlectrum = int(mu::engraving::SymId::pluckedPlectrum),
    pluckedSnapPizzicatoAbove = int(mu::engraving::SymId::pluckedSnapPizzicatoAbove),
    pluckedSnapPizzicatoBelow = int(mu::engraving::SymId::pluckedSnapPizzicatoBelow),
    pluckedWithFingernails = int(mu::engraving::SymId::pluckedWithFingernails),
    quindicesima = int(mu::engraving::SymId::quindicesima),
    quindicesimaAlta = int(mu::engraving::SymId::quindicesimaAlta),
    quindicesimaBassa = int(mu::engraving::SymId::quindicesimaBassa),
    quindicesimaBassaMb = int(mu::engraving::SymId::quindicesimaBassaMb),
    repeat1Bar = int(mu::engraving::SymId::repeat1Bar),
    repeat2Bars = int(mu::engraving::SymId::repeat2Bars),
    repeat4Bars = int(mu::engraving::SymId::repeat4Bars),
    repeatBarLowerDot = int(mu::engraving::SymId::repeatBarLowerDot),
    repeatBarSlash = int(mu::engraving::SymId::repeatBarSlash),
    repeatBarUpperDot = int(mu::engraving::SymId::repeatBarUpperDot),
    repeatDot = int(mu::engraving::SymId::repeatDot),
    repeatDots = int(mu::engraving::SymId::repeatDots),
    repeatLeft = int(mu::engraving::SymId::repeatLeft),
    repeatRight = int(mu::engraving::SymId::repeatRight),
    repeatRightLeft = int(mu::engraving::SymId::repeatRightLeft),
    rest1024th = int(mu::engraving::SymId::rest1024th),
    rest128th = int(mu::engraving::SymId::rest128th),
    rest16th = int(mu::engraving::SymId::rest16th),
    rest256th = int(mu::engraving::SymId::rest256th),
    rest32nd = int(mu::engraving::SymId::rest32nd),
    rest512th = int(mu::engraving::SymId::rest512th),
    rest64th = int(mu::engraving::SymId::rest64th),
    rest8th = int(mu::engraving::SymId::rest8th),
    restDoubleWhole = int(mu::engraving::SymId::restDoubleWhole),
    restDoubleWholeLegerLine = int(mu::engraving::SymId::restDoubleWholeLegerLine),
    restHBar = int(mu::engraving::SymId::restHBar),
    restHBarLeft = int(mu::engraving::SymId::restHBarLeft),
    restHBarMiddle = int(mu::engraving::SymId::restHBarMiddle),
    restHBarRight = int(mu::engraving::SymId::restHBarRight),
    restHalf = int(mu::engraving::SymId::restHalf),
    restHalfLegerLine = int(mu::engraving::SymId::restHalfLegerLine),
    restLonga = int(mu::engraving::SymId::restLonga),
    restMaxima = int(mu::engraving::SymId::restMaxima),
    restQuarter = int(mu::engraving::SymId::restQuarter),
    restQuarterOld = int(mu::engraving::SymId::restQuarterOld),
    restQuarterZ = int(mu::engraving::SymId::restQuarterZ),
    restWhole = int(mu::engraving::SymId::restWhole),
    restWholeLegerLine = int(mu::engraving::SymId::restWholeLegerLine),
    reversedBrace = int(mu::engraving::SymId::reversedBrace),
    reversedBracketBottom = int(mu::engraving::SymId::reversedBracketBottom),
    reversedBracketTop = int(mu::engraving::SymId::reversedBracketTop),
    rightRepeatSmall = int(mu::engraving::SymId::rightRepeatSmall),
    scaleDegree1 = int(mu::engraving::SymId::scaleDegree1),
    scaleDegree2 = int(mu::engraving::SymId::scaleDegree2),
    scaleDegree3 = int(mu::engraving::SymId::scaleDegree3),
    scaleDegree4 = int(mu::engraving::SymId::scaleDegree4),
    scaleDegree5 = int(mu::engraving::SymId::scaleDegree5),
    scaleDegree6 = int(mu::engraving::SymId::scaleDegree6),
    scaleDegree7 = int(mu::engraving::SymId::scaleDegree7),
    scaleDegree8 = int(mu::engraving::SymId::scaleDegree8),
    scaleDegree9 = int(mu::engraving::SymId::scaleDegree9),
    schaefferClef = int(mu::engraving::SymId::schaefferClef),
    schaefferFClefToGClef = int(mu::engraving::SymId::schaefferFClefToGClef),
    schaefferGClefToFClef = int(mu::engraving::SymId::schaefferGClefToFClef),
    schaefferPreviousClef = int(mu::engraving::SymId::schaefferPreviousClef),
    segno = int(mu::engraving::SymId::segno),
    segnoSerpent1 = int(mu::engraving::SymId::segnoSerpent1),
    segnoSerpent2 = int(mu::engraving::SymId::segnoSerpent2),
    semipitchedPercussionClef1 = int(mu::engraving::SymId::semipitchedPercussionClef1),
    semipitchedPercussionClef2 = int(mu::engraving::SymId::semipitchedPercussionClef2),
    smnFlat = int(mu::engraving::SymId::smnFlat),
    smnFlatWhite = int(mu::engraving::SymId::smnFlatWhite),
    smnHistoryDoubleFlat = int(mu::engraving::SymId::smnHistoryDoubleFlat),
    smnHistoryDoubleSharp = int(mu::engraving::SymId::smnHistoryDoubleSharp),
    smnHistoryFlat = int(mu::engraving::SymId::smnHistoryFlat),
    smnHistorySharp = int(mu::engraving::SymId::smnHistorySharp),
    smnNatural = int(mu::engraving::SymId::smnNatural),
    smnSharp = int(mu::engraving::SymId::smnSharp),
    smnSharpDown = int(mu::engraving::SymId::smnSharpDown),
    smnSharpWhite = int(mu::engraving::SymId::smnSharpWhite),
    smnSharpWhiteDown = int(mu::engraving::SymId::smnSharpWhiteDown),
    splitBarDivider = int(mu::engraving::SymId::splitBarDivider),
    staff1Line = int(mu::engraving::SymId::staff1Line),
    staff1LineNarrow = int(mu::engraving::SymId::staff1LineNarrow),
    staff1LineWide = int(mu::engraving::SymId::staff1LineWide),
    staff2Lines = int(mu::engraving::SymId::staff2Lines),
    staff2LinesNarrow = int(mu::engraving::SymId::staff2LinesNarrow),
    staff2LinesWide = int(mu::engraving::SymId::staff2LinesWide),
    staff3Lines = int(mu::engraving::SymId::staff3Lines),
    staff3LinesNarrow = int(mu::engraving::SymId::staff3LinesNarrow),
    staff3LinesWide = int(mu::engraving::SymId::staff3LinesWide),
    staff4Lines = int(mu::engraving::SymId::staff4Lines),
    staff4LinesNarrow = int(mu::engraving::SymId::staff4LinesNarrow),
    staff4LinesWide = int(mu::engraving::SymId::staff4LinesWide),
    staff5Lines = int(mu::engraving::SymId::staff5Lines),
    staff5LinesNarrow = int(mu::engraving::SymId::staff5LinesNarrow),
    staff5LinesWide = int(mu::engraving::SymId::staff5LinesWide),
    staff6Lines = int(mu::engraving::SymId::staff6Lines),
    staff6LinesNarrow = int(mu::engraving::SymId::staff6LinesNarrow),
    staff6LinesWide = int(mu::engraving::SymId::staff6LinesWide),
    staffDivideArrowDown = int(mu::engraving::SymId::staffDivideArrowDown),
    staffDivideArrowUp = int(mu::engraving::SymId::staffDivideArrowUp),
    staffDivideArrowUpDown = int(mu::engraving::SymId::staffDivideArrowUpDown),
    staffPosLower1 = int(mu::engraving::SymId::staffPosLower1),
    staffPosLower2 = int(mu::engraving::SymId::staffPosLower2),
    staffPosLower3 = int(mu::engraving::SymId::staffPosLower3),
    staffPosLower4 = int(mu::engraving::SymId::staffPosLower4),
    staffPosLower5 = int(mu::engraving::SymId::staffPosLower5),
    staffPosLower6 = int(mu::engraving::SymId::staffPosLower6),
    staffPosLower7 = int(mu::engraving::SymId::staffPosLower7),
    staffPosLower8 = int(mu::engraving::SymId::staffPosLower8),
    staffPosRaise1 = int(mu::engraving::SymId::staffPosRaise1),
    staffPosRaise2 = int(mu::engraving::SymId::staffPosRaise2),
    staffPosRaise3 = int(mu::engraving::SymId::staffPosRaise3),
    staffPosRaise4 = int(mu::engraving::SymId::staffPosRaise4),
    staffPosRaise5 = int(mu::engraving::SymId::staffPosRaise5),
    staffPosRaise6 = int(mu::engraving::SymId::staffPosRaise6),
    staffPosRaise7 = int(mu::engraving::SymId::staffPosRaise7),
    staffPosRaise8 = int(mu::engraving::SymId::staffPosRaise8),
    stem = int(mu::engraving::SymId::stem),
    stemBowOnBridge = int(mu::engraving::SymId::stemBowOnBridge),
    stemBowOnTailpiece = int(mu::engraving::SymId::stemBowOnTailpiece),
    stemBuzzRoll = int(mu::engraving::SymId::stemBuzzRoll),
    stemDamp = int(mu::engraving::SymId::stemDamp),
    stemHarpStringNoise = int(mu::engraving::SymId::stemHarpStringNoise),
    stemMultiphonicsBlack = int(mu::engraving::SymId::stemMultiphonicsBlack),
    stemMultiphonicsBlackWhite = int(mu::engraving::SymId::stemMultiphonicsBlackWhite),
    stemMultiphonicsWhite = int(mu::engraving::SymId::stemMultiphonicsWhite),
    stemPendereckiTremolo = int(mu::engraving::SymId::stemPendereckiTremolo),
    stemRimShot = int(mu::engraving::SymId::stemRimShot),
    stemSprechgesang = int(mu::engraving::SymId::stemSprechgesang),
    stemSulPonticello = int(mu::engraving::SymId::stemSulPonticello),
    stemSussurando = int(mu::engraving::SymId::stemSussurando),
    stemSwished = int(mu::engraving::SymId::stemSwished),
    stemVibratoPulse = int(mu::engraving::SymId::stemVibratoPulse),
    stockhausenTremolo = int(mu::engraving::SymId::stockhausenTremolo),
    stringsBowBehindBridge = int(mu::engraving::SymId::stringsBowBehindBridge),
    stringsBowBehindBridgeFourStrings = int(mu::engraving::SymId::stringsBowBehindBridgeFourStrings),
    stringsBowBehindBridgeOneString = int(mu::engraving::SymId::stringsBowBehindBridgeOneString),
    stringsBowBehindBridgeThreeStrings = int(mu::engraving::SymId::stringsBowBehindBridgeThreeStrings),
    stringsBowBehindBridgeTwoStrings = int(mu::engraving::SymId::stringsBowBehindBridgeTwoStrings),
    stringsBowOnBridge = int(mu::engraving::SymId::stringsBowOnBridge),
    stringsBowOnTailpiece = int(mu::engraving::SymId::stringsBowOnTailpiece),
    stringsChangeBowDirection = int(mu::engraving::SymId::stringsChangeBowDirection),
    stringsDownBow = int(mu::engraving::SymId::stringsDownBow),
    stringsDownBowAwayFromBody = int(mu::engraving::SymId::stringsDownBowAwayFromBody),
    stringsDownBowBeyondBridge = int(mu::engraving::SymId::stringsDownBowBeyondBridge),
    stringsDownBowTowardsBody = int(mu::engraving::SymId::stringsDownBowTowardsBody),
    stringsDownBowTurned = int(mu::engraving::SymId::stringsDownBowTurned),
    stringsFouette = int(mu::engraving::SymId::stringsFouette),
    stringsHalfHarmonic = int(mu::engraving::SymId::stringsHalfHarmonic),
    stringsHarmonic = int(mu::engraving::SymId::stringsHarmonic),
    stringsJeteAbove = int(mu::engraving::SymId::stringsJeteAbove),
    stringsJeteBelow = int(mu::engraving::SymId::stringsJeteBelow),
    stringsMuteOff = int(mu::engraving::SymId::stringsMuteOff),
    stringsMuteOn = int(mu::engraving::SymId::stringsMuteOn),
    stringsOverpressureDownBow = int(mu::engraving::SymId::stringsOverpressureDownBow),
    stringsOverpressureNoDirection = int(mu::engraving::SymId::stringsOverpressureNoDirection),
    stringsOverpressurePossibileDownBow = int(mu::engraving::SymId::stringsOverpressurePossibileDownBow),
    stringsOverpressurePossibileUpBow = int(mu::engraving::SymId::stringsOverpressurePossibileUpBow),
    stringsOverpressureUpBow = int(mu::engraving::SymId::stringsOverpressureUpBow),
    stringsScrapeCircularClockwise = int(mu::engraving::SymId::stringsScrapeCircularClockwise),
    stringsScrapeCircularCounterclockwise = int(mu::engraving::SymId::stringsScrapeCircularCounterclockwise),
    stringsScrapeParallelInward = int(mu::engraving::SymId::stringsScrapeParallelInward),
    stringsScrapeParallelOutward = int(mu::engraving::SymId::stringsScrapeParallelOutward),
    stringsThumbPosition = int(mu::engraving::SymId::stringsThumbPosition),
    stringsThumbPositionTurned = int(mu::engraving::SymId::stringsThumbPositionTurned),
    stringsTripleChopInward = int(mu::engraving::SymId::stringsTripleChopInward),
    stringsTripleChopOutward = int(mu::engraving::SymId::stringsTripleChopOutward),
    stringsUpBow = int(mu::engraving::SymId::stringsUpBow),
    stringsUpBowAwayFromBody = int(mu::engraving::SymId::stringsUpBowAwayFromBody),
    stringsUpBowBeyondBridge = int(mu::engraving::SymId::stringsUpBowBeyondBridge),
    stringsUpBowTowardsBody = int(mu::engraving::SymId::stringsUpBowTowardsBody),
    stringsUpBowTurned = int(mu::engraving::SymId::stringsUpBowTurned),
    stringsVibratoPulse = int(mu::engraving::SymId::stringsVibratoPulse),
    swissRudimentsNoteheadBlackDouble = int(mu::engraving::SymId::swissRudimentsNoteheadBlackDouble),
    swissRudimentsNoteheadBlackFlam = int(mu::engraving::SymId::swissRudimentsNoteheadBlackFlam),
    swissRudimentsNoteheadHalfDouble = int(mu::engraving::SymId::swissRudimentsNoteheadHalfDouble),
    swissRudimentsNoteheadHalfFlam = int(mu::engraving::SymId::swissRudimentsNoteheadHalfFlam),
    systemDivider = int(mu::engraving::SymId::systemDivider),
    systemDividerExtraLong = int(mu::engraving::SymId::systemDividerExtraLong),
    systemDividerLong = int(mu::engraving::SymId::systemDividerLong),
    textAugmentationDot = int(mu::engraving::SymId::textAugmentationDot),
    textBlackNoteFrac16thLongStem = int(mu::engraving::SymId::textBlackNoteFrac16thLongStem),
    textBlackNoteFrac16thShortStem = int(mu::engraving::SymId::textBlackNoteFrac16thShortStem),
    textBlackNoteFrac32ndLongStem = int(mu::engraving::SymId::textBlackNoteFrac32ndLongStem),
    textBlackNoteFrac8thLongStem = int(mu::engraving::SymId::textBlackNoteFrac8thLongStem),
    textBlackNoteFrac8thShortStem = int(mu::engraving::SymId::textBlackNoteFrac8thShortStem),
    textBlackNoteLongStem = int(mu::engraving::SymId::textBlackNoteLongStem),
    textBlackNoteShortStem = int(mu::engraving::SymId::textBlackNoteShortStem),
    textCont16thBeamLongStem = int(mu::engraving::SymId::textCont16thBeamLongStem),
    textCont16thBeamShortStem = int(mu::engraving::SymId::textCont16thBeamShortStem),
    textCont32ndBeamLongStem = int(mu::engraving::SymId::textCont32ndBeamLongStem),
    textCont8thBeamLongStem = int(mu::engraving::SymId::textCont8thBeamLongStem),
    textCont8thBeamShortStem = int(mu::engraving::SymId::textCont8thBeamShortStem),
    textHeadlessBlackNoteFrac16thLongStem = int(mu::engraving::SymId::textHeadlessBlackNoteFrac16thLongStem),
    textHeadlessBlackNoteFrac16thShortStem = int(mu::engraving::SymId::textHeadlessBlackNoteFrac16thShortStem),
    textHeadlessBlackNoteFrac32ndLongStem = int(mu::engraving::SymId::textHeadlessBlackNoteFrac32ndLongStem),
    textHeadlessBlackNoteFrac8thLongStem = int(mu::engraving::SymId::textHeadlessBlackNoteFrac8thLongStem),
    textHeadlessBlackNoteFrac8thShortStem = int(mu::engraving::SymId::textHeadlessBlackNoteFrac8thShortStem),
    textHeadlessBlackNoteLongStem = int(mu::engraving::SymId::textHeadlessBlackNoteLongStem),
    textHeadlessBlackNoteShortStem = int(mu::engraving::SymId::textHeadlessBlackNoteShortStem),
    textTie = int(mu::engraving::SymId::textTie),
    textTuplet3LongStem = int(mu::engraving::SymId::textTuplet3LongStem),
    textTuplet3ShortStem = int(mu::engraving::SymId::textTuplet3ShortStem),
    textTupletBracketEndLongStem = int(mu::engraving::SymId::textTupletBracketEndLongStem),
    textTupletBracketEndShortStem = int(mu::engraving::SymId::textTupletBracketEndShortStem),
    textTupletBracketStartLongStem = int(mu::engraving::SymId::textTupletBracketStartLongStem),
    textTupletBracketStartShortStem = int(mu::engraving::SymId::textTupletBracketStartShortStem),
    timeSig0 = int(mu::engraving::SymId::timeSig0),
    timeSig0Reversed = int(mu::engraving::SymId::timeSig0Reversed),
    timeSig0Turned = int(mu::engraving::SymId::timeSig0Turned),
    timeSig1 = int(mu::engraving::SymId::timeSig1),
    timeSig1Reversed = int(mu::engraving::SymId::timeSig1Reversed),
    timeSig1Turned = int(mu::engraving::SymId::timeSig1Turned),
    timeSig2 = int(mu::engraving::SymId::timeSig2),
    timeSig2Reversed = int(mu::engraving::SymId::timeSig2Reversed),
    timeSig2Turned = int(mu::engraving::SymId::timeSig2Turned),
    timeSig3 = int(mu::engraving::SymId::timeSig3),
    timeSig3Reversed = int(mu::engraving::SymId::timeSig3Reversed),
    timeSig3Turned = int(mu::engraving::SymId::timeSig3Turned),
    timeSig4 = int(mu::engraving::SymId::timeSig4),
    timeSig4Reversed = int(mu::engraving::SymId::timeSig4Reversed),
    timeSig4Turned = int(mu::engraving::SymId::timeSig4Turned),
    timeSig5 = int(mu::engraving::SymId::timeSig5),
    timeSig5Reversed = int(mu::engraving::SymId::timeSig5Reversed),
    timeSig5Turned = int(mu::engraving::SymId::timeSig5Turned),
    timeSig6 = int(mu::engraving::SymId::timeSig6),
    timeSig6Reversed = int(mu::engraving::SymId::timeSig6Reversed),
    timeSig6Turned = int(mu::engraving::SymId::timeSig6Turned),
    timeSig7 = int(mu::engraving::SymId::timeSig7),
    timeSig7Reversed = int(mu::engraving::SymId::timeSig7Reversed),
    timeSig7Turned = int(mu::engraving::SymId::timeSig7Turned),
    timeSig8 = int(mu::engraving::SymId::timeSig8),
    timeSig8Reversed = int(mu::engraving::SymId::timeSig8Reversed),
    timeSig8Turned = int(mu::engraving::SymId::timeSig8Turned),
    timeSig9 = int(mu::engraving::SymId::timeSig9),
    timeSig9Reversed = int(mu::engraving::SymId::timeSig9Reversed),
    timeSig9Turned = int(mu::engraving::SymId::timeSig9Turned),
    timeSigBracketLeft = int(mu::engraving::SymId::timeSigBracketLeft),
    timeSigBracketLeftSmall = int(mu::engraving::SymId::timeSigBracketLeftSmall),
    timeSigBracketRight = int(mu::engraving::SymId::timeSigBracketRight),
    timeSigBracketRightSmall = int(mu::engraving::SymId::timeSigBracketRightSmall),
    timeSigCombDenominator = int(mu::engraving::SymId::timeSigCombDenominator),
    timeSigCombNumerator = int(mu::engraving::SymId::timeSigCombNumerator),
    timeSigComma = int(mu::engraving::SymId::timeSigComma),
    timeSigCommon = int(mu::engraving::SymId::timeSigCommon),
    timeSigCommonReversed = int(mu::engraving::SymId::timeSigCommonReversed),
    timeSigCommonTurned = int(mu::engraving::SymId::timeSigCommonTurned),
    timeSigCut2 = int(mu::engraving::SymId::timeSigCut2),
    timeSigCut3 = int(mu::engraving::SymId::timeSigCut3),
    timeSigCutCommon = int(mu::engraving::SymId::timeSigCutCommon),
    timeSigCutCommonReversed = int(mu::engraving::SymId::timeSigCutCommonReversed),
    timeSigCutCommonTurned = int(mu::engraving::SymId::timeSigCutCommonTurned),
    timeSigEquals = int(mu::engraving::SymId::timeSigEquals),
    timeSigFractionHalf = int(mu::engraving::SymId::timeSigFractionHalf),
    timeSigFractionOneThird = int(mu::engraving::SymId::timeSigFractionOneThird),
    timeSigFractionQuarter = int(mu::engraving::SymId::timeSigFractionQuarter),
    timeSigFractionThreeQuarters = int(mu::engraving::SymId::timeSigFractionThreeQuarters),
    timeSigFractionTwoThirds = int(mu::engraving::SymId::timeSigFractionTwoThirds),
    timeSigFractionalSlash = int(mu::engraving::SymId::timeSigFractionalSlash),
    timeSigMinus = int(mu::engraving::SymId::timeSigMinus),
    timeSigMultiply = int(mu::engraving::SymId::timeSigMultiply),
    timeSigOpenPenderecki = int(mu::engraving::SymId::timeSigOpenPenderecki),
    timeSigParensLeft = int(mu::engraving::SymId::timeSigParensLeft),
    timeSigParensLeftSmall = int(mu::engraving::SymId::timeSigParensLeftSmall),
    timeSigParensRight = int(mu::engraving::SymId::timeSigParensRight),
    timeSigParensRightSmall = int(mu::engraving::SymId::timeSigParensRightSmall),
    timeSigPlus = int(mu::engraving::SymId::timeSigPlus),
    timeSigPlusSmall = int(mu::engraving::SymId::timeSigPlusSmall),
    timeSigSlash = int(mu::engraving::SymId::timeSigSlash),
    timeSigX = int(mu::engraving::SymId::timeSigX),
    tremolo1 = int(mu::engraving::SymId::tremolo1),
    tremolo2 = int(mu::engraving::SymId::tremolo2),
    tremolo3 = int(mu::engraving::SymId::tremolo3),
    tremolo4 = int(mu::engraving::SymId::tremolo4),
    tremolo5 = int(mu::engraving::SymId::tremolo5),
    tremoloDivisiDots2 = int(mu::engraving::SymId::tremoloDivisiDots2),
    tremoloDivisiDots3 = int(mu::engraving::SymId::tremoloDivisiDots3),
    tremoloDivisiDots4 = int(mu::engraving::SymId::tremoloDivisiDots4),
    tremoloDivisiDots6 = int(mu::engraving::SymId::tremoloDivisiDots6),
    tremoloFingered1 = int(mu::engraving::SymId::tremoloFingered1),
    tremoloFingered2 = int(mu::engraving::SymId::tremoloFingered2),
    tremoloFingered3 = int(mu::engraving::SymId::tremoloFingered3),
    tremoloFingered4 = int(mu::engraving::SymId::tremoloFingered4),
    tremoloFingered5 = int(mu::engraving::SymId::tremoloFingered5),
    tripleTongueAbove = int(mu::engraving::SymId::tripleTongueAbove),
    tripleTongueBelow = int(mu::engraving::SymId::tripleTongueBelow),
    tuplet0 = int(mu::engraving::SymId::tuplet0),
    tuplet1 = int(mu::engraving::SymId::tuplet1),
    tuplet2 = int(mu::engraving::SymId::tuplet2),
    tuplet3 = int(mu::engraving::SymId::tuplet3),
    tuplet4 = int(mu::engraving::SymId::tuplet4),
    tuplet5 = int(mu::engraving::SymId::tuplet5),
    tuplet6 = int(mu::engraving::SymId::tuplet6),
    tuplet7 = int(mu::engraving::SymId::tuplet7),
    tuplet8 = int(mu::engraving::SymId::tuplet8),
    tuplet9 = int(mu::engraving::SymId::tuplet9),
    tupletColon = int(mu::engraving::SymId::tupletColon),
    unmeasuredTremolo = int(mu::engraving::SymId::unmeasuredTremolo),
    unmeasuredTremoloSimple = int(mu::engraving::SymId::unmeasuredTremoloSimple),
    unpitchedPercussionClef1 = int(mu::engraving::SymId::unpitchedPercussionClef1),
    unpitchedPercussionClef2 = int(mu::engraving::SymId::unpitchedPercussionClef2),
    ventiduesima = int(mu::engraving::SymId::ventiduesima),
    ventiduesimaAlta = int(mu::engraving::SymId::ventiduesimaAlta),
    ventiduesimaBassa = int(mu::engraving::SymId::ventiduesimaBassa),
    ventiduesimaBassaMb = int(mu::engraving::SymId::ventiduesimaBassaMb),
    vocalFingerClickStockhausen = int(mu::engraving::SymId::vocalFingerClickStockhausen),
    vocalHalbGesungen = int(mu::engraving::SymId::vocalHalbGesungen),
    vocalMouthClosed = int(mu::engraving::SymId::vocalMouthClosed),
    vocalMouthOpen = int(mu::engraving::SymId::vocalMouthOpen),
    vocalMouthPursed = int(mu::engraving::SymId::vocalMouthPursed),
    vocalMouthSlightlyOpen = int(mu::engraving::SymId::vocalMouthSlightlyOpen),
    vocalMouthWideOpen = int(mu::engraving::SymId::vocalMouthWideOpen),
    vocalNasalVoice = int(mu::engraving::SymId::vocalNasalVoice),
    vocalSprechgesang = int(mu::engraving::SymId::vocalSprechgesang),
    vocalTongueClickStockhausen = int(mu::engraving::SymId::vocalTongueClickStockhausen),
    vocalTongueFingerClickStockhausen = int(mu::engraving::SymId::vocalTongueFingerClickStockhausen),
    vocalsSussurando = int(mu::engraving::SymId::vocalsSussurando),
    wiggleArpeggiatoDown = int(mu::engraving::SymId::wiggleArpeggiatoDown),
    wiggleArpeggiatoDownArrow = int(mu::engraving::SymId::wiggleArpeggiatoDownArrow),
    wiggleArpeggiatoDownSwash = int(mu::engraving::SymId::wiggleArpeggiatoDownSwash),
    wiggleArpeggiatoUp = int(mu::engraving::SymId::wiggleArpeggiatoUp),
    wiggleArpeggiatoUpArrow = int(mu::engraving::SymId::wiggleArpeggiatoUpArrow),
    wiggleArpeggiatoUpSwash = int(mu::engraving::SymId::wiggleArpeggiatoUpSwash),
    wiggleCircular = int(mu::engraving::SymId::wiggleCircular),
    wiggleCircularConstant = int(mu::engraving::SymId::wiggleCircularConstant),
    wiggleCircularConstantFlipped = int(mu::engraving::SymId::wiggleCircularConstantFlipped),
    wiggleCircularConstantFlippedLarge = int(mu::engraving::SymId::wiggleCircularConstantFlippedLarge),
    wiggleCircularConstantLarge = int(mu::engraving::SymId::wiggleCircularConstantLarge),
    wiggleCircularEnd = int(mu::engraving::SymId::wiggleCircularEnd),
    wiggleCircularLarge = int(mu::engraving::SymId::wiggleCircularLarge),
    wiggleCircularLarger = int(mu::engraving::SymId::wiggleCircularLarger),
    wiggleCircularLargerStill = int(mu::engraving::SymId::wiggleCircularLargerStill),
    wiggleCircularLargest = int(mu::engraving::SymId::wiggleCircularLargest),
    wiggleCircularSmall = int(mu::engraving::SymId::wiggleCircularSmall),
    wiggleCircularStart = int(mu::engraving::SymId::wiggleCircularStart),
    wiggleGlissando = int(mu::engraving::SymId::wiggleGlissando),
    wiggleGlissandoGroup1 = int(mu::engraving::SymId::wiggleGlissandoGroup1),
    wiggleGlissandoGroup2 = int(mu::engraving::SymId::wiggleGlissandoGroup2),
    wiggleGlissandoGroup3 = int(mu::engraving::SymId::wiggleGlissandoGroup3),
    wiggleRandom1 = int(mu::engraving::SymId::wiggleRandom1),
    wiggleRandom2 = int(mu::engraving::SymId::wiggleRandom2),
    wiggleRandom3 = int(mu::engraving::SymId::wiggleRandom3),
    wiggleRandom4 = int(mu::engraving::SymId::wiggleRandom4),
    wiggleSawtooth = int(mu::engraving::SymId::wiggleSawtooth),
    wiggleSawtoothNarrow = int(mu::engraving::SymId::wiggleSawtoothNarrow),
    wiggleSawtoothWide = int(mu::engraving::SymId::wiggleSawtoothWide),
    wiggleSquareWave = int(mu::engraving::SymId::wiggleSquareWave),
    wiggleSquareWaveNarrow = int(mu::engraving::SymId::wiggleSquareWaveNarrow),
    wiggleSquareWaveWide = int(mu::engraving::SymId::wiggleSquareWaveWide),
    wiggleTrill = int(mu::engraving::SymId::wiggleTrill),
    wiggleTrillFast = int(mu::engraving::SymId::wiggleTrillFast),
    wiggleTrillFaster = int(mu::engraving::SymId::wiggleTrillFaster),
    wiggleTrillFasterStill = int(mu::engraving::SymId::wiggleTrillFasterStill),
    wiggleTrillFastest = int(mu::engraving::SymId::wiggleTrillFastest),
    wiggleTrillSlow = int(mu::engraving::SymId::wiggleTrillSlow),
    wiggleTrillSlower = int(mu::engraving::SymId::wiggleTrillSlower),
    wiggleTrillSlowerStill = int(mu::engraving::SymId::wiggleTrillSlowerStill),
    wiggleTrillSlowest = int(mu::engraving::SymId::wiggleTrillSlowest),
    wiggleVIbratoLargestSlower = int(mu::engraving::SymId::wiggleVIbratoLargestSlower),
    wiggleVIbratoMediumSlower = int(mu::engraving::SymId::wiggleVIbratoMediumSlower),
    wiggleVibrato = int(mu::engraving::SymId::wiggleVibrato),
    wiggleVibratoLargeFast = int(mu::engraving::SymId::wiggleVibratoLargeFast),
    wiggleVibratoLargeFaster = int(mu::engraving::SymId::wiggleVibratoLargeFaster),
    wiggleVibratoLargeFasterStill = int(mu::engraving::SymId::wiggleVibratoLargeFasterStill),
    wiggleVibratoLargeFastest = int(mu::engraving::SymId::wiggleVibratoLargeFastest),
    wiggleVibratoLargeSlow = int(mu::engraving::SymId::wiggleVibratoLargeSlow),
    wiggleVibratoLargeSlower = int(mu::engraving::SymId::wiggleVibratoLargeSlower),
    wiggleVibratoLargeSlowest = int(mu::engraving::SymId::wiggleVibratoLargeSlowest),
    wiggleVibratoLargestFast = int(mu::engraving::SymId::wiggleVibratoLargestFast),
    wiggleVibratoLargestFaster = int(mu::engraving::SymId::wiggleVibratoLargestFaster),
    wiggleVibratoLargestFasterStill = int(mu::engraving::SymId::wiggleVibratoLargestFasterStill),
    wiggleVibratoLargestFastest = int(mu::engraving::SymId::wiggleVibratoLargestFastest),
    wiggleVibratoLargestSlow = int(mu::engraving::SymId::wiggleVibratoLargestSlow),
    wiggleVibratoLargestSlowest = int(mu::engraving::SymId::wiggleVibratoLargestSlowest),
    wiggleVibratoMediumFast = int(mu::engraving::SymId::wiggleVibratoMediumFast),
    wiggleVibratoMediumFaster = int(mu::engraving::SymId::wiggleVibratoMediumFaster),
    wiggleVibratoMediumFasterStill = int(mu::engraving::SymId::wiggleVibratoMediumFasterStill),
    wiggleVibratoMediumFastest = int(mu::engraving::SymId::wiggleVibratoMediumFastest),
    wiggleVibratoMediumSlow = int(mu::engraving::SymId::wiggleVibratoMediumSlow),
    wiggleVibratoMediumSlowest = int(mu::engraving::SymId::wiggleVibratoMediumSlowest),
    wiggleVibratoSmallFast = int(mu::engraving::SymId::wiggleVibratoSmallFast),
    wiggleVibratoSmallFaster = int(mu::engraving::SymId::wiggleVibratoSmallFaster),
    wiggleVibratoSmallFasterStill = int(mu::engraving::SymId::wiggleVibratoSmallFasterStill),
    wiggleVibratoSmallFastest = int(mu::engraving::SymId::wiggleVibratoSmallFastest),
    wiggleVibratoSmallSlow = int(mu::engraving::SymId::wiggleVibratoSmallSlow),
    wiggleVibratoSmallSlower = int(mu::engraving::SymId::wiggleVibratoSmallSlower),
    wiggleVibratoSmallSlowest = int(mu::engraving::SymId::wiggleVibratoSmallSlowest),
    wiggleVibratoSmallestFast = int(mu::engraving::SymId::wiggleVibratoSmallestFast),
    wiggleVibratoSmallestFaster = int(mu::engraving::SymId::wiggleVibratoSmallestFaster),
    wiggleVibratoSmallestFasterStill = int(mu::engraving::SymId::wiggleVibratoSmallestFasterStill),
    wiggleVibratoSmallestFastest = int(mu::engraving::SymId::wiggleVibratoSmallestFastest),
    wiggleVibratoSmallestSlow = int(mu::engraving::SymId::wiggleVibratoSmallestSlow),
    wiggleVibratoSmallestSlower = int(mu::engraving::SymId::wiggleVibratoSmallestSlower),
    wiggleVibratoSmallestSlowest = int(mu::engraving::SymId::wiggleVibratoSmallestSlowest),
    wiggleVibratoStart = int(mu::engraving::SymId::wiggleVibratoStart),
    wiggleVibratoWide = int(mu::engraving::SymId::wiggleVibratoWide),
    wiggleWavy = int(mu::engraving::SymId::wiggleWavy),
    wiggleWavyNarrow = int(mu::engraving::SymId::wiggleWavyNarrow),
    wiggleWavyWide = int(mu::engraving::SymId::wiggleWavyWide),
    windClosedHole = int(mu::engraving::SymId::windClosedHole),
    windFlatEmbouchure = int(mu::engraving::SymId::windFlatEmbouchure),
    windHalfClosedHole1 = int(mu::engraving::SymId::windHalfClosedHole1),
    windHalfClosedHole2 = int(mu::engraving::SymId::windHalfClosedHole2),
    windHalfClosedHole3 = int(mu::engraving::SymId::windHalfClosedHole3),
    windLessRelaxedEmbouchure = int(mu::engraving::SymId::windLessRelaxedEmbouchure),
    windLessTightEmbouchure = int(mu::engraving::SymId::windLessTightEmbouchure),
    windMouthpiecePop = int(mu::engraving::SymId::windMouthpiecePop),
    windMultiphonicsBlackStem = int(mu::engraving::SymId::windMultiphonicsBlackStem),
    windMultiphonicsBlackWhiteStem = int(mu::engraving::SymId::windMultiphonicsBlackWhiteStem),
    windMultiphonicsWhiteStem = int(mu::engraving::SymId::windMultiphonicsWhiteStem),
    windOpenHole = int(mu::engraving::SymId::windOpenHole),
    windReedPositionIn = int(mu::engraving::SymId::windReedPositionIn),
    windReedPositionNormal = int(mu::engraving::SymId::windReedPositionNormal),
    windReedPositionOut = int(mu::engraving::SymId::windReedPositionOut),
    windRelaxedEmbouchure = int(mu::engraving::SymId::windRelaxedEmbouchure),
    windRimOnly = int(mu::engraving::SymId::windRimOnly),
    windSharpEmbouchure = int(mu::engraving::SymId::windSharpEmbouchure),
    windStrongAirPressure = int(mu::engraving::SymId::windStrongAirPressure),
    windThreeQuartersClosedHole = int(mu::engraving::SymId::windThreeQuartersClosedHole),
    windTightEmbouchure = int(mu::engraving::SymId::windTightEmbouchure),
    windTrillKey = int(mu::engraving::SymId::windTrillKey),
    windVeryTightEmbouchure = int(mu::engraving::SymId::windVeryTightEmbouchure),
    windWeakAirPressure = int(mu::engraving::SymId::windWeakAirPressure),
    // SMuFL standard symbol IDs }}}

//    EXTENSIONS
//    SMuFL stylistic alternates which we need to access directly

    noteheadDoubleWholeAlt = int(mu::engraving::SymId::noteheadDoubleWholeAlt), // double whole with double side bars
    fourStringTabClefSerif = int(mu::engraving::SymId::fourStringTabClefSerif), // TAB clef in script style
    sixStringTabClefSerif = int(mu::engraving::SymId::sixStringTabClefSerif),   // TAB clef in script style
    cClefFrench = int(mu::engraving::SymId::cClefFrench),
    cClefFrench20C = int(mu::engraving::SymId::cClefFrench20C),
    fClefFrench = int(mu::engraving::SymId::fClefFrench),
    fClef19thCentury = int(mu::engraving::SymId::fClef19thCentury),
    braceSmall = int(mu::engraving::SymId::braceSmall),
    braceLarge = int(mu::engraving::SymId::braceLarge),
    braceLarger = int(mu::engraving::SymId::braceLarger),

    flag1024thDownStraight = int(mu::engraving::SymId::flag1024thDownStraight),
    flag1024thUpStraight = int(mu::engraving::SymId::flag1024thUpStraight),
    flag128thDownStraight = int(mu::engraving::SymId::flag128thDownStraight),
    flag128thUpStraight = int(mu::engraving::SymId::flag128thUpStraight),
    flag16thDownStraight = int(mu::engraving::SymId::flag16thDownStraight),
    flag16thUpStraight = int(mu::engraving::SymId::flag16thUpStraight),
    flag256thDownStraight = int(mu::engraving::SymId::flag256thDownStraight),
    flag256thUpStraight = int(mu::engraving::SymId::flag256thUpStraight),
    flag32ndDownStraight = int(mu::engraving::SymId::flag32ndDownStraight),
    flag32ndUpStraight = int(mu::engraving::SymId::flag32ndUpStraight),
    flag512thDownStraight = int(mu::engraving::SymId::flag512thDownStraight),
    flag512thUpStraight = int(mu::engraving::SymId::flag512thUpStraight),
    flag64thDownStraight = int(mu::engraving::SymId::flag64thDownStraight),
    flag64thUpStraight = int(mu::engraving::SymId::flag64thUpStraight),
    flag8thDownStraight = int(mu::engraving::SymId::flag8thDownStraight),
    flag8thUpStraight = int(mu::engraving::SymId::flag8thUpStraight),

//    MuseScore-local symbols, precomposed symbols to mimic some Emmentaler glyphs

    ornamentPrallMordent = int(mu::engraving::SymId::ornamentPrallMordent),
    ornamentUpPrall = int(mu::engraving::SymId::ornamentUpPrall),
    ornamentUpMordent = int(mu::engraving::SymId::ornamentUpMordent),
    ornamentPrallDown = int(mu::engraving::SymId::ornamentPrallDown),
//      ornamentDownPrall = int(mu::engraving::SymId::ornamentDownPrall),
    ornamentDownMordent = int(mu::engraving::SymId::ornamentDownMordent),
    ornamentPrallUp = int(mu::engraving::SymId::ornamentPrallUp),
    ornamentLinePrall = int(mu::engraving::SymId::ornamentLinePrall),

//    additional symbols

    accidentalDoubleFlatParens = int(mu::engraving::SymId::accidentalDoubleFlatParens),
    accidentalFlatParens = int(mu::engraving::SymId::accidentalFlatParens),
    accidentalNaturalParens = int(mu::engraving::SymId::accidentalNaturalParens),
    accidentalSharpParens = int(mu::engraving::SymId::accidentalSharpParens),
    accidentalDoubleSharpParens = int(mu::engraving::SymId::accidentalDoubleSharpParens),

    noteLongaUp = int(mu::engraving::SymId::noteLongaUp),
    noteLongaDown = int(mu::engraving::SymId::noteLongaDown),
    noteLongaSquareUp = int(mu::engraving::SymId::noteLongaSquareUp),
    noteLongaSquareDown = int(mu::engraving::SymId::noteLongaSquareDown),
    space = int(mu::engraving::SymId::space),

//    END OF TABLE

    lastSym = int(mu::engraving::SymId::lastSym),
};
Q_ENUM_NS(SymId);
}

Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::AccidentalType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::ElementType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::OrnamentStyle);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::Align);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::Placement);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::PlacementH);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TextPlace);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::Direction);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::DirectionH);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::Orientation);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::AutoOnOff);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::AutoCustomHide);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::VoiceAssignment);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::SpacerType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::LayoutBreakType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::DurationType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::VeloType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::BeamMode);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::GlissandoType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::GlissandoStyle);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::HarmonyType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::HarmonyVoicing);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::HDuration);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::FrameType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::VerticalAlignment);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TremoloBarType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::PreferSharpFlat);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::NoteHeadType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::NoteHeadScheme);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::NoteHeadGroup);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::NoteType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::PlayEventType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::SegmentType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::BarLineType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::Tid);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::Syllabic);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::Anchor);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::MMRestRangeBracketType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TupletNumberType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TupletBracketType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TripletFeelType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::GuitarBendType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::GuitarBendShowHoldLine);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::ClefType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::ClefToBarlinePosition);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::DynamicType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::DynamicSpeed);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::LineType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::HookType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::KeyMode);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::ArpeggioType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::IntervalStep);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::IntervalType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::InstrumentLabelVisibility);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::OrnamentShowAccidental);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::PartialSpannerDirection);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::ChordStylePreset);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::AnnotationCategory);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::PlayingTechniqueType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::GradualTempoChangeType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::ChangeMethod);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::ChangeDirection);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::AccidentalRole);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::AccidentalVal);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::FermataType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::ChordLineType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::SlurStyleType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TremoloType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TremoloChordType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::BracketType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::JumpType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::MarkerType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::MeasureNumberMode);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::StaffGroup);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TrillType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::VibratoType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::ArticulationTextType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::LyricsDashSystemStart);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::NoteLineEndPlacement);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::SpannerSegmentType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TiePlacement);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TieDotsPlacement);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TimeSigPlacement);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TimeSigStyle);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TimeSigVSMargin);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::NoteSpellingType);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::Key);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::UpdateMode);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::LayoutFlag);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::LayoutMode);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::TappingHand);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::LHTappingSymbol);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::RHTappingSymbol);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::LHTappingShowItems);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::ParenthesesMode);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::RepeatPlayCountPreset);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::MeasureNumberPlacement);
Q_DECLARE_METATYPE(mu::engraving::apiv1::enums::SymId);
