/* SPDX-FileCopyrightText: 2024 Greenbone AG
 *
 * SPDX-License-Identifier: AGPL-3.0-or-later
 */

import {describe, test, expect} from '@gsa/testing';
import {
  parseCvssV2BaseFromVector,
  parseCvssV2BaseVector,
  parseCvssV3BaseFromVector,
  parseCvssV3BaseVector,
} from 'gmp/parser/cvss';

const cvssV2Entries = {
  entries: [
    {vector: 'AV:A/AC:H/Au:M/C:N/I:P/A:P', value: 2.7},
    {vector: 'AV:A/AC:H/Au:N/C:C/I:C/A:C', value: 6.8},
    {vector: 'AV:A/AC:H/Au:N/C:C/I:N/A:C', value: 6.2},
    {vector: 'AV:A/AC:H/Au:N/C:C/I:P/A:N', value: 5.3},
    {vector: 'AV:A/AC:H/Au:N/C:N/I:N/A:C', value: 4.6},
    {vector: 'AV:A/AC:H/Au:N/C:N/I:N/A:P', value: 1.8},
    {vector: 'AV:A/AC:H/Au:N/C:N/I:P/A:N', value: 1.8},
    {vector: 'AV:A/AC:H/Au:N/C:P/I:N/A:N', value: 1.8},
    {vector: 'AV:A/AC:H/Au:N/C:P/I:N/A:P', value: 3.2},
    {vector: 'AV:A/AC:H/Au:N/C:P/I:P/A:C', value: 5.8},
    {vector: 'AV:A/AC:H/Au:N/C:P/I:P/A:N', value: 3.2},
    {vector: 'AV:A/AC:H/Au:N/C:P/I:P/A:P', value: 4.3},
    {vector: 'AV:A/AC:H/Au:S/C:C/I:C/A:C', value: 6.5},
    {vector: 'AV:A/AC:H/Au:S/C:C/I:N/A:N', value: 4.3},
    {vector: 'AV:A/AC:H/Au:S/C:N/I:N/A:C', value: 4.3},
    {vector: 'AV:A/AC:H/Au:S/C:N/I:N/A:P', value: 1.4},
    {vector: 'AV:A/AC:H/Au:S/C:P/I:N/A:P', value: 2.9},
    {vector: 'AV:A/AC:H/Au:S/C:P/I:P/A:C', value: 5.5},
    {vector: 'AV:A/AC:H/Au:S/C:P/I:P/A:P', value: 4.0},
    {vector: 'AV:A/AC:L/Au:M/C:C/I:C/A:C', value: 7.2},
    {vector: 'AV:A/AC:L/Au:M/C:P/I:P/A:P', value: 4.7},
    {vector: 'AV:A/AC:L/Au:N/C:C/I:C/A:C', value: 8.3},
    {vector: 'AV:A/AC:L/Au:N/C:C/I:C/A:N', value: 7.8},
    {vector: 'AV:A/AC:L/Au:N/C:C/I:N/A:N', value: 6.1},
    {vector: 'AV:A/AC:L/Au:N/C:C/I:P/A:C', value: 8.0},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:C/A:C', value: 7.8},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:C/A:N', value: 6.1},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:N/A:C', value: 6.1},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:N/A:P', value: 3.3},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:P/A:C', value: 6.8},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:P/A:N', value: 3.3},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:P/A:P', value: 4.8},
    {vector: 'AV:A/AC:L/Au:N/C:P/I:C/A:C', value: 8.0},
    {vector: 'AV:A/AC:L/Au:N/C:P/I:N/A:N', value: 3.3},
    {vector: 'AV:A/AC:L/Au:N/C:P/I:N/A:P', value: 4.8},
    {vector: 'AV:A/AC:L/Au:N/C:P/I:P/A:C', value: 7.3},
    {vector: 'AV:A/AC:L/Au:N/C:P/I:P/A:N', value: 4.8},
    {vector: 'AV:A/AC:L/Au:N/C:P/I:P/A:P', value: 5.8},
    {vector: 'AV:A/AC:L/Au:S/C:C/I:C/A:C', value: 7.7},
    {vector: 'AV:A/AC:L/Au:S/C:C/I:C/A:N', value: 7.1},
    {vector: 'AV:A/AC:L/Au:S/C:C/I:C/A:P', value: 7.4},
    {vector: 'AV:A/AC:L/Au:S/C:C/I:N/A:N', value: 5.5},
    {vector: 'AV:A/AC:L/Au:S/C:C/I:P/A:P', value: 6.7},
    {vector: 'AV:A/AC:L/Au:S/C:N/I:N/A:C', value: 5.5},
    {vector: 'AV:A/AC:L/Au:S/C:N/I:N/A:P', value: 2.7},
    {vector: 'AV:A/AC:L/Au:S/C:N/I:P/A:C', value: 6.2},
    {vector: 'AV:A/AC:L/Au:S/C:N/I:P/A:N', value: 2.7},
    {vector: 'AV:A/AC:L/Au:S/C:N/I:P/A:P', value: 4.1},
    {vector: 'AV:A/AC:L/Au:S/C:P/I:N/A:N', value: 2.7},
    {vector: 'AV:A/AC:L/Au:S/C:P/I:N/A:P', value: 4.1},
    {vector: 'AV:A/AC:L/Au:S/C:P/I:P/A:C', value: 6.7},
    {vector: 'AV:A/AC:L/Au:S/C:P/I:P/A:N', value: 4.1},
    {vector: 'AV:A/AC:L/Au:S/C:P/I:P/A:P', value: 5.2},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:C/A:C', value: 7.9},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:C/A:N', value: 7.3},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:C/A:P', value: 7.6},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:N/A:C', value: 7.3},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:N/A:N', value: 5.7},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:P/A:C', value: 7.6},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:P/A:P', value: 6.9},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:C/A:C', value: 7.3},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:C/A:N', value: 5.7},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:N/A:C', value: 5.7},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:N/A:P', value: 2.9},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:P/A:C', value: 6.4},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:P/A:N', value: 2.9},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:P/A:P', value: 4.3},
    {vector: 'AV:A/AC:M/Au:N/C:P/I:C/A:C', value: 7.6},
    {vector: 'AV:A/AC:M/Au:N/C:P/I:N/A:N', value: 2.9},
    {vector: 'AV:A/AC:M/Au:N/C:P/I:N/A:P', value: 4.3},
    {vector: 'AV:A/AC:M/Au:N/C:P/I:P/A:N', value: 4.3},
    {vector: 'AV:A/AC:M/Au:N/C:P/I:P/A:P', value: 5.4},
    {vector: 'AV:A/AC:M/Au:S/C:C/I:C/A:C', value: 7.4},
    {vector: 'AV:A/AC:M/Au:S/C:C/I:N/A:N', value: 5.2},
    {vector: 'AV:A/AC:M/Au:S/C:N/I:N/A:C', value: 5.2},
    {vector: 'AV:A/AC:M/Au:S/C:N/I:N/A:P', value: 2.3},
    {vector: 'AV:A/AC:M/Au:S/C:N/I:P/A:C', value: 5.8},
    {vector: 'AV:A/AC:M/Au:S/C:N/I:P/A:N', value: 2.3},
    {vector: 'AV:A/AC:M/Au:S/C:N/I:P/A:P', value: 3.8},
    {vector: 'AV:A/AC:M/Au:S/C:P/I:N/A:C', value: 5.8},
    {vector: 'AV:A/AC:M/Au:S/C:P/I:N/A:N', value: 2.3},
    {vector: 'AV:A/AC:M/Au:S/C:P/I:N/A:P', value: 3.8},
    {vector: 'AV:A/AC:M/Au:S/C:P/I:P/A:C', value: 6.3},
    {vector: 'AV:A/AC:M/Au:S/C:P/I:P/A:N', value: 3.8},
    {vector: 'AV:A/AC:M/Au:S/C:P/I:P/A:P', value: 4.9},
    {vector: 'AV:L/AC:H/Au:M/C:C/I:C/A:C', value: 5.9},
    {vector: 'AV:L/AC:H/Au:M/C:N/I:N/A:C', value: 3.7},
    {vector: 'AV:L/AC:H/Au:M/C:P/I:P/A:C', value: 4.9},
    {vector: 'AV:L/AC:H/Au:M/C:P/I:P/A:P', value: 3.4},
    {vector: 'AV:L/AC:H/Au:N/C:C/I:C/A:C', value: 6.2},
    {vector: 'AV:L/AC:H/Au:N/C:C/I:C/A:N', value: 5.6},
    {vector: 'AV:L/AC:H/Au:N/C:C/I:C/A:P', value: 5.9},
    {vector: 'AV:L/AC:H/Au:N/C:C/I:N/A:C', value: 5.6},
    {vector: 'AV:L/AC:H/Au:N/C:C/I:N/A:N', value: 4.0},
    {vector: 'AV:L/AC:H/Au:N/C:N/I:C/A:C', value: 5.6},
    {vector: 'AV:L/AC:H/Au:N/C:N/I:C/A:N', value: 4.0},
    {vector: 'AV:L/AC:H/Au:N/C:N/I:N/A:C', value: 4.0},
    {vector: 'AV:L/AC:H/Au:N/C:N/I:N/A:P', value: 1.2},
    {vector: 'AV:L/AC:H/Au:N/C:N/I:P/A:N', value: 1.2},
    {vector: 'AV:L/AC:H/Au:N/C:N/I:P/A:P', value: 2.6},
    {vector: 'AV:L/AC:H/Au:N/C:P/I:N/A:C', value: 4.7},
    {vector: 'AV:L/AC:H/Au:N/C:P/I:N/A:N', value: 1.2},
    {vector: 'AV:L/AC:H/Au:N/C:P/I:N/A:P', value: 2.6},
    {vector: 'AV:L/AC:H/Au:N/C:P/I:P/A:C', value: 5.2},
    {vector: 'AV:L/AC:H/Au:N/C:P/I:P/A:N', value: 2.6},
    {vector: 'AV:L/AC:H/Au:N/C:P/I:P/A:P', value: 3.7},
    {vector: 'AV:L/AC:H/Au:S/C:C/I:C/A:C', value: 6.0},
    {vector: 'AV:L/AC:H/Au:S/C:C/I:C/A:N', value: 5.5},
    {vector: 'AV:L/AC:H/Au:S/C:C/I:N/A:N', value: 3.8},
    {vector: 'AV:L/AC:H/Au:S/C:N/I:C/A:C', value: 5.5},
    {vector: 'AV:L/AC:H/Au:S/C:N/I:C/A:N', value: 3.8},
    {vector: 'AV:L/AC:H/Au:S/C:N/I:N/A:C', value: 3.8},
    {vector: 'AV:L/AC:H/Au:S/C:N/I:N/A:P', value: 1.0},
    {vector: 'AV:L/AC:H/Au:S/C:N/I:P/A:N', value: 1.0},
    {vector: 'AV:L/AC:H/Au:S/C:N/I:P/A:P', value: 2.4},
    {vector: 'AV:L/AC:H/Au:S/C:P/I:N/A:N', value: 1.0},
    {vector: 'AV:L/AC:H/Au:S/C:P/I:N/A:P', value: 2.4},
    {vector: 'AV:L/AC:H/Au:S/C:P/I:P/A:N', value: 2.4},
    {vector: 'AV:L/AC:H/Au:S/C:P/I:P/A:P', value: 3.5},
    {vector: 'AV:L/AC:L/Au:M/C:C/I:C/A:C', value: 6.5},
    {vector: 'AV:L/AC:L/Au:M/C:N/I:N/A:C', value: 4.3},
    {vector: 'AV:L/AC:L/Au:M/C:P/I:N/A:N', value: 1.4},
    {vector: 'AV:L/AC:L/Au:N/C:C/I:C/A:C', value: 7.2},
    {vector: 'AV:L/AC:L/Au:N/C:C/I:C/A:N', value: 6.6},
    {vector: 'AV:L/AC:L/Au:N/C:C/I:C/A:P', value: 6.8},
    {vector: 'AV:L/AC:L/Au:N/C:C/I:N/A:C', value: 6.6},
    {vector: 'AV:L/AC:L/Au:N/C:C/I:N/A:N', value: 4.9},
    {vector: 'AV:L/AC:L/Au:N/C:C/I:P/A:P', value: 6.1},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:C/A:C', value: 6.6},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:C/A:N', value: 4.9},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:N/A:C', value: 4.9},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:N/A:N', value: 0.0},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:N/A:P', value: 2.1},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:P/A:C', value: 5.6},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:P/A:N', value: 2.1},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:P/A:P', value: 3.6},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:C/A:C', value: 6.8},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:C/A:N', value: 5.6},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:C/A:P', value: 6.1},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:N/A:C', value: 5.6},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:N/A:N', value: 2.1},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:N/A:P', value: 3.6},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:P/A:C', value: 6.1},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:P/A:N', value: 3.6},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:P/A:P', value: 4.6},
    {vector: 'AV:L/AC:L/Au:S/C:C/I:C/A:C', value: 6.8},
    {vector: 'AV:L/AC:L/Au:S/C:C/I:C/A:N', value: 6.2},
    {vector: 'AV:L/AC:L/Au:S/C:C/I:N/A:N', value: 4.6},
    {vector: 'AV:L/AC:L/Au:S/C:C/I:N/A:P', value: 5.2},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:C/A:C', value: 6.2},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:C/A:N', value: 4.6},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:N/A:C', value: 4.6},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:N/A:P', value: 1.7},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:P/A:C', value: 5.2},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:P/A:N', value: 1.7},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:P/A:P', value: 3.2},
    {vector: 'AV:L/AC:L/Au:S/C:P/I:N/A:N', value: 1.7},
    {vector: 'AV:L/AC:L/Au:S/C:P/I:N/A:P', value: 3.2},
    {vector: 'AV:L/AC:L/Au:S/C:P/I:P/A:C', value: 5.7},
    {vector: 'AV:L/AC:L/Au:S/C:P/I:P/A:N', value: 3.2},
    {vector: 'AV:L/AC:L/Au:S/C:P/I:P/A:P', value: 4.3},
    {vector: 'AV:L/AC:M/Au:M/C:C/I:C/A:C', value: 6.3},
    {vector: 'AV:L/AC:M/Au:M/C:N/I:N/A:C', value: 4.1},
    {vector: 'AV:L/AC:M/Au:M/C:N/I:P/A:P', value: 2.7},
    {vector: 'AV:L/AC:M/Au:M/C:P/I:N/A:N', value: 1.3},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:C/A:C', value: 6.9},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:C/A:N', value: 6.3},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:C/A:P', value: 6.6},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:N/A:C', value: 6.3},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:N/A:N', value: 4.7},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:N/A:P', value: 5.4},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:P/A:C', value: 6.6},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:P/A:N', value: 5.4},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:P/A:P', value: 5.9},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:C/A:C', value: 6.3},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:C/A:N', value: 4.7},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:N/A:C', value: 4.7},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:N/A:P', value: 1.9},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:P/A:C', value: 5.4},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:P/A:N', value: 1.9},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:P/A:P', value: 3.3},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:C/A:C', value: 6.6},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:C/A:N', value: 5.4},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:N/A:C', value: 5.4},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:N/A:N', value: 1.9},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:N/A:P', value: 3.3},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:P/A:C', value: 5.9},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:P/A:N', value: 3.3},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:P/A:P', value: 4.4},
    {vector: 'AV:L/AC:M/Au:S/C:C/I:C/A:C', value: 6.6},
    {vector: 'AV:L/AC:M/Au:S/C:C/I:C/A:N', value: 6.0},
    {vector: 'AV:L/AC:M/Au:S/C:C/I:N/A:N', value: 4.4},
    {vector: 'AV:L/AC:M/Au:S/C:N/I:C/A:C', value: 6.0},
    {vector: 'AV:L/AC:M/Au:S/C:N/I:N/A:C', value: 4.4},
    {vector: 'AV:L/AC:M/Au:S/C:N/I:N/A:P', value: 1.5},
    {vector: 'AV:L/AC:M/Au:S/C:N/I:P/A:N', value: 1.5},
    {vector: 'AV:L/AC:M/Au:S/C:N/I:P/A:P', value: 3.0},
    {vector: 'AV:L/AC:M/Au:S/C:P/I:N/A:N', value: 1.5},
    {vector: 'AV:L/AC:M/Au:S/C:P/I:P/A:C', value: 5.5},
    {vector: 'AV:L/AC:M/Au:S/C:P/I:P/A:N', value: 3.0},
    {vector: 'AV:L/AC:M/Au:S/C:P/I:P/A:P', value: 4.1},
    {vector: 'AV:N/AC:H/Au:M/C:C/I:C/A:C', value: 6.8},
    {vector: 'AV:N/AC:H/Au:M/C:N/I:N/A:P', value: 1.7},
    {vector: 'AV:N/AC:H/Au:M/C:N/I:P/A:N', value: 1.7},
    {vector: 'AV:N/AC:H/Au:M/C:P/I:N/A:N', value: 1.7},
    {vector: 'AV:N/AC:H/Au:M/C:P/I:P/A:N', value: 3.2},
    {vector: 'AV:N/AC:H/Au:M/C:P/I:P/A:P', value: 4.3},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:C/A:C', value: 7.6},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:C/A:N', value: 7.1},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:C/A:P', value: 7.3},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:N/A:C', value: 7.1},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:N/A:N', value: 5.4},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:P/A:C', value: 7.3},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:P/A:N', value: 6.1},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:C/A:C', value: 7.1},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:C/A:N', value: 5.4},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:N/A:C', value: 5.4},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:N/A:P', value: 2.6},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:P/A:C', value: 6.1},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:P/A:N', value: 2.6},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:P/A:P', value: 4.0},
    {vector: 'AV:N/AC:H/Au:N/C:P/I:N/A:N', value: 2.6},
    {vector: 'AV:N/AC:H/Au:N/C:P/I:N/A:P', value: 4.0},
    {vector: 'AV:N/AC:H/Au:N/C:P/I:P/A:C', value: 6.6},
    {vector: 'AV:N/AC:H/Au:N/C:P/I:P/A:N', value: 4.0},
    {vector: 'AV:N/AC:H/Au:N/C:P/I:P/A:P', value: 5.1},
    {vector: 'AV:N/AC:H/Au:S/C:C/I:C/A:C', value: 7.1},
    {vector: 'AV:N/AC:H/Au:S/C:C/I:C/A:N', value: 6.6},
    {vector: 'AV:N/AC:H/Au:S/C:C/I:C/A:P', value: 6.8},
    {vector: 'AV:N/AC:H/Au:S/C:C/I:N/A:C', value: 6.6},
    {vector: 'AV:N/AC:H/Au:S/C:C/I:N/A:N', value: 4.9},
    {vector: 'AV:N/AC:H/Au:S/C:C/I:P/A:P', value: 6.1},
    {vector: 'AV:N/AC:H/Au:S/C:N/I:C/A:C', value: 6.6},
    {vector: 'AV:N/AC:H/Au:S/C:N/I:N/A:C', value: 4.9},
    {vector: 'AV:N/AC:H/Au:S/C:N/I:N/A:P', value: 2.1},
    {vector: 'AV:N/AC:H/Au:S/C:N/I:P/A:C', value: 5.6},
    {vector: 'AV:N/AC:H/Au:S/C:N/I:P/A:N', value: 2.1},
    {vector: 'AV:N/AC:H/Au:S/C:N/I:P/A:P', value: 3.6},
    {vector: 'AV:N/AC:H/Au:S/C:P/I:N/A:N', value: 2.1},
    {vector: 'AV:N/AC:H/Au:S/C:P/I:N/A:P', value: 3.6},
    {vector: 'AV:N/AC:H/Au:S/C:P/I:P/A:C', value: 6.1},
    {vector: 'AV:N/AC:H/Au:S/C:P/I:P/A:N', value: 3.6},
    {vector: 'AV:N/AC:H/Au:S/C:P/I:P/A:P', value: 4.6},
    {vector: 'AV:N/AC:L/Au:M/C:C/I:C/A:C', value: 8.3},
    {vector: 'AV:N/AC:L/Au:M/C:C/I:C/A:N', value: 7.7},
    {vector: 'AV:N/AC:L/Au:M/C:N/I:N/A:C', value: 6.1},
    {vector: 'AV:N/AC:L/Au:M/C:N/I:N/A:P', value: 3.3},
    {vector: 'AV:N/AC:L/Au:M/C:N/I:P/A:N', value: 3.3},
    {vector: 'AV:N/AC:L/Au:M/C:P/I:P/A:N', value: 4.7},
    {vector: 'AV:N/AC:L/Au:M/C:P/I:P/A:P', value: 5.8},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:C/A:C', value: 10.0},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:C/A:N', value: 9.4},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:C/A:P', value: 9.7},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:N/A:C', value: 9.4},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:N/A:N', value: 7.8},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:N/A:P', value: 8.5},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:P/A:N', value: 8.5},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:P/A:P', value: 9.0},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:C/A:C', value: 9.4},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:C/A:N', value: 7.8},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:C/A:P', value: 8.5},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:N/A:C', value: 7.8},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:N/A:N', value: 0.0},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:N/A:P', value: 5.0},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:P/A:C', value: 8.5},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:P/A:N', value: 5.0},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:P/A:P', value: 6.4},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:C/A:C', value: 9.7},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:C/A:N', value: 8.5},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:C/A:P', value: 9.0},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:N/A:C', value: 8.5},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:N/A:N', value: 5.0},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:N/A:P', value: 6.4},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:P/A:C', value: 9.0},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:P/A:N', value: 6.4},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:P/A:P', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:C/A:C', value: 9.0},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:C/A:N', value: 8.5},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:C/A:P', value: 8.7},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:N/A:C', value: 8.5},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:N/A:N', value: 6.8},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:N/A:P', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:P/A:C', value: 8.7},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:P/A:N', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:P/A:P', value: 8.0},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:C/A:C', value: 8.5},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:C/A:N', value: 6.8},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:C/A:P', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:N/A:C', value: 6.8},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:N/A:P', value: 4.0},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:P/A:C', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:P/A:N', value: 4.0},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:P/A:P', value: 5.5},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:C/A:C', value: 8.7},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:C/A:N', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:C/A:P', value: 8.0},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:N/A:C', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:N/A:N', value: 4.0},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:N/A:P', value: 5.5},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:P/A:C', value: 8.0},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:P/A:N', value: 5.5},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:P/A:P', value: 6.5},
    {vector: 'AV:N/AC:M/Au:M/C:C/I:C/A:C', value: 7.9},
    {vector: 'AV:N/AC:M/Au:M/C:N/I:N/A:C', value: 5.7},
    {vector: 'AV:N/AC:M/Au:M/C:N/I:N/A:P', value: 2.8},
    {vector: 'AV:N/AC:M/Au:M/C:N/I:P/A:N', value: 2.8},
    {vector: 'AV:N/AC:M/Au:M/C:N/I:P/A:P', value: 4.3},
    {vector: 'AV:N/AC:M/Au:M/C:P/I:N/A:N', value: 2.8},
    {vector: 'AV:N/AC:M/Au:M/C:P/I:P/A:N', value: 4.3},
    {vector: 'AV:N/AC:M/Au:M/C:P/I:P/A:P', value: 5.4},
    {vector: 'AV:N/AC:M/Au:N/C:C/I:C/A:C', value: 9.3},
    {vector: 'AV:N/AC:M/Au:N/C:C/I:C/A:N', value: 8.8},
    {vector: 'AV:N/AC:M/Au:N/C:C/I:N/A:N', value: 7.1},
    {vector: 'AV:N/AC:M/Au:N/C:C/I:P/A:C', value: 9.0},
    {vector: 'AV:N/AC:M/Au:N/C:C/I:P/A:N', value: 7.8},
    {vector: 'AV:N/AC:M/Au:N/C:C/I:P/A:P', value: 8.3},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:C/A:C', value: 8.8},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:C/A:N', value: 7.1},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:N/A:C', value: 7.1},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:N/A:P', value: 4.3},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:P/A:C', value: 7.8},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:P/A:N', value: 4.3},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:P/A:P', value: 5.8},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:C/A:C', value: 9.0},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:C/A:N', value: 7.8},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:C/A:P', value: 8.3},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:N/A:C', value: 7.8},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:N/A:N', value: 4.3},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:N/A:P', value: 5.8},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:P/A:C', value: 8.3},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:P/A:N', value: 5.8},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:P/A:P', value: 6.8},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:C/A:C', value: 8.5},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:C/A:N', value: 7.9},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:C/A:P', value: 8.2},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:N/A:C', value: 7.9},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:N/A:N', value: 6.3},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:P/A:N', value: 7.0},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:P/A:P', value: 7.5},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:C/A:C', value: 7.9},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:C/A:N', value: 6.3},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:N/A:C', value: 6.3},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:N/A:N', value: 0.0},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:N/A:P', value: 3.5},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:P/A:C', value: 7.0},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:P/A:N', value: 3.5},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:P/A:P', value: 4.9},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:C/A:C', value: 8.2},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:C/A:N', value: 7.0},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:N/A:C', value: 7.0},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:N/A:N', value: 3.5},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:N/A:P', value: 4.9},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:P/A:C', value: 7.5},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:P/A:N', value: 4.9},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:P/A:P', value: 6.0},
  ],
};

describe('CVSS v2 parser tests', () => {
  test('should return correct severity value for cvss v2 vector', () => {
    cvssV2Entries.entries.forEach(element => {
      const {
        accessVector,
        accessComplexity,
        authentication,
        confidentialityImpact,
        availabilityImpact,
        integrityImpact,
        cvssScore,
      } = parseCvssV2BaseFromVector(element.vector);
      expect(cvssScore).toEqual(element.value);
      const cvssV2 = parseCvssV2BaseVector({
        accessVector,
        accessComplexity,
        authentication,
        confidentialityImpact,
        availabilityImpact,
        integrityImpact,
      });
      expect(cvssV2[0]).toEqual(element.vector);
      expect(cvssV2[1]).toEqual(element.value);
    });
  });
});

const cvssV3Entries = {
  entries: [
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:C/C:H/I:L/A:L', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:C/C:L/I:N/A:N', value: 2.6},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:U/C:L/I:N/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.2},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:R/S:U/C:N/I:H/A:H', value: 5.6},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:C/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:L/I:H/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:L/I:H/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 3.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 6.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 7.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 5.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:N', value: 6.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:H', value: 5.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 9.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:H', value: 9.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:H/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 8.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 5.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:H/I:L/A:L', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:H/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:L/I:N/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:L/I:L/A:L', value: 3.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 1.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:L/A:H', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:L/A:L', value: 3.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:L/A:N', value: 1.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 7.2},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:C/C:H/I:N/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:C/C:L/I:L/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:C/C:N/I:N/A:H', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:N', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:H/I:N/A:H', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:H/I:N/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:L/I:L/A:H', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:N/I:H/A:H', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:N/I:L/A:N', value: 1.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:N/I:N/A:H', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:N/I:N/A:L', value: 1.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 7.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 7.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:H/I:L/A:N', value: 6.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:N/I:L/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:H/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:H/A:L', value: 5.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 4.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 3.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:L/I:L/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:L/I:N/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:N', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:L/I:H/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:L/I:N/A:N', value: 2.2},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 4.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 2.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 2.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:C/C:H/I:N/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:H', value: 5.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 4.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:N/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:N', value: 7.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:L', value: 5.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:N/I:H/A:H', value: 7.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:N/I:L/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:N/I:L/A:L', value: 4.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:L/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:H/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:L', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:N', value: 3.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:N/A:H', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 2.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:H', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:N', value: 2.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:C/C:L/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:L/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:N/I:N/A:H', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:L', value: 8.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:H/A:H', value: 8.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:H', value: 7.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:L', value: 5.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:N', value: 3.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:L', value: 5.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:L', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:L', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:L', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:L', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:N', value: 7.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:H/I:N/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:L', value: 5.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:N/I:L/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:N/I:N/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:N', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:L/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 4.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:L/I:N/A:N', value: 2.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:L', value: 3.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:N', value: 2.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:H', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:L', value: 2.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 9.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:H', value: 8.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:L', value: 6.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 5.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:L', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 8.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:H/I:N/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:H/I:N/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:L/I:H/A:H', value: 8.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:L', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:L/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:N/I:H/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:N/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:N/I:N/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:L', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:L', value: 3.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:C/C:H/I:H/A:L', value: 7.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:L', value: 6.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:L/I:H/A:L', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:L/I:L/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:L/A:H', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:L/A:N', value: 2.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:L/I:N/A:N', value: 2.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:H/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:H/I:N/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:L/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:L/I:N/A:N', value: 2.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:N/I:H/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:N/I:L/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:N/I:L/A:N', value: 2.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:L/A:L', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:L/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 6.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:L/I:L/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:L/I:N/A:L', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:L/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:L/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:N/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:L', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:N', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:L/I:N/A:N', value: 3.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:N/I:H/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:N/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:N/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:H/A:H', value: 8.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:L/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:N/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:H/A:H', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:L/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 4.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:L', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:L/A:L', value: 7.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:L/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:H/A:L', value: 7.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 7.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 5.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:L', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:H/I:L/A:N', value: 6.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:H/I:N/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:H/A:N', value: 6.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:N/A:N', value: 3.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:N/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:N/I:L/A:N', value: 3.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:N/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:L/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:N/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:L', value: 9.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:N', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:L/A:N', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:H', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:L', value: 6.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:L/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:N/I:H/A:H', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:L/A:L', value: 6.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:L/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:H/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:H/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:N', value: 3.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 2.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:L', value: 3.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:N', value: 2.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:L', value: 2.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:H/I:H/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:H/I:L/A:N', value: 6.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:L/I:H/A:N', value: 6.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:L/I:N/A:N', value: 3.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:H/I:N/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:L/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:N/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:N/I:H/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:N/I:H/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:N/I:N/A:H', value: 4.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:L', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:L/A:L', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:L', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:H/A:H', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:H/A:N', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:N', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:H', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:H', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:N', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:L/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:L/A:N', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:H/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:H/A:H', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:L/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:L/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:N', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:L/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:L/A:L', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:L/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:N/A:L', value: 4.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:L', value: 4.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:L', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:L', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:L/A:L', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:L/A:N', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:H', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:H/A:H', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:H', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:N', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:N/A:L', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:N/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:H/A:N', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:H', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:L', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 9.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:L', value: 9.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:H', value: 9.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:L', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:H/A:H', value: 9.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:H/A:L', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:H/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:N', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:L/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:N/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:H/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:N/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:H/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:N/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:N/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:L/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:L/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 6.9},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:H/UI:R/S:C/C:H/I:H/A:L', value: 6.7},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:L', value: 7.0},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 6.7},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:C/C:H/I:L/A:N', value: 5.6},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:L', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:R/S:U/C:H/I:L/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:C/C:H/I:L/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:L/I:H/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:L', value: 4.8},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 4.2},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:L', value: 3.7},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 2.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 2.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 7.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 2.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 2.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:H/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 7.9},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:L', value: 2.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:H', value: 9.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:L/I:N/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:H/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 8.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 5.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 7.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:H/I:L/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:N/I:L/A:L', value: 3.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:N/I:N/A:L', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:H/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:C/C:L/I:L/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:C/C:N/I:H/A:H', value: 6.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:N/I:N/A:L', value: 1.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 7.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:L/I:H/A:N', value: 6.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:H', value: 5.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:N/A:L', value: 3.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:L', value: 3.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:L', value: 7.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:N', value: 7.2},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:L/I:N/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:N/I:L/A:L', value: 3.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:N/I:L/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:H/I:N/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:N/I:H/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:N/I:H/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:N/I:H/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:L/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:L/A:N', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:L/I:H/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:L/I:N/A:N', value: 3.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:L', value: 3.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:L/A:L', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:L/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:H', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:L', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:N', value: 3.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 2.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:H', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:N', value: 2.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:L', value: 2.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:C/C:N/I:N/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:N/I:H/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:N/I:L/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:N/I:N/A:H', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:H', value: 7.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:N', value: 5.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:L', value: 5.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:L', value: 3.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:L', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:L', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:L', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:H/I:N/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:L/I:H/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:N/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:L', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:N', value: 3.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:L/I:N/A:N', value: 2.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:L', value: 3.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:H', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:L', value: 2.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 9.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 9.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 8.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:H/I:N/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:L', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:N/I:L/A:N', value: 3.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:L', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:L', value: 3.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:C/C:N/I:H/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:C/C:N/I:L/A:N', value: 3.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:N/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:L/I:H/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:L/I:N/A:L', value: 3.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 2.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:L/A:L', value: 3.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:N/A:L', value: 2.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:H/I:H/A:N', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:L/I:L/A:L', value: 5.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:L/I:N/A:N', value: 2.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:H/I:N/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:L/I:N/A:N', value: 2.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:N/I:H/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:N/I:L/A:N', value: 2.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 6.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:L/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:N/I:L/A:N', value: 3.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:L', value: 6.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:N/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:L/I:H/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:L/I:L/A:N', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:N/I:N/A:H', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:H/I:L/A:L', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:L/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:N/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:H/A:H', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:H/A:L', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 4.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:L', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:L/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:L/A:L', value: 7.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:L/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:H/A:L', value: 7.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 7.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 5.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:L', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:L', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:H/I:H/A:N', value: 8.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:H/I:L/A:L', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:L/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:L/A:L', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:N/A:N', value: 3.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:L/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:L/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:N/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:L/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:L/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:N/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:N/I:L/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:L/A:L', value: 6.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:L', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:H/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:N', value: 3.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 2.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:L', value: 3.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:N', value: 2.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:L', value: 2.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:H/I:L/A:N', value: 6.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:H/I:N/A:N', value: 6.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:L/I:L/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:N/I:N/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:H/I:N/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:L/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:L/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:N/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:N/I:H/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:N/I:H/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:N/I:L/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:L', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:L/A:N', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:H/A:N', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:N', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:L', value: 8.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:N', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:L/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:L/A:N', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:L/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:N', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:L/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:H/A:L', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:L', value: 4.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:L', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:L/A:N', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:H', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:L', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:H', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:N', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:N/A:L', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:N/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:H/A:H', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:H/A:N', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 9.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:L', value: 9.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:H', value: 9.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:L', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:H/A:H', value: 9.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:H/A:L', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:H/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:N', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:L/A:L', value: 8.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:L/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:N/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:H/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:N/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:H/A:H', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:H/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:L/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:L', value: 5.6},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:H/UI:R/S:U/C:L/I:L/A:N', value: 2.7},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 4.2},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:R/S:C/C:H/I:H/A:L', value: 6.9},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 7.2},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:L', value: 6.2},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:L', value: 4.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 2.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:C/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:L', value: 5.2},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 2.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 4.3},
  ],
};

describe('CVSS v3 parser tests', () => {
  test('should return correct severity value for cvss v3 vector', () => {
    cvssV3Entries.entries.forEach(element => {
      const {
        attackVector,
        attackComplexity,
        privilegesRequired,
        userInteraction,
        scope,
        confidentialityImpact,
        availabilityImpact,
        integrityImpact,
        cvssScore,
      } = parseCvssV3BaseFromVector(element.vector);
      expect(cvssScore).toEqual(element.value);
      const cvssV3 = parseCvssV3BaseVector({
        attackVector,
        attackComplexity,
        privilegesRequired,
        userInteraction,
        scope,
        confidentialityImpact,
        availabilityImpact,
        integrityImpact,
      });
      expect(cvssV3[0]).toEqual(element.vector);
      expect(cvssV3[1]).toEqual(element.value);
    });
  });
});

describe('parseCvssV2BaseVector tests', () => {
  test('should return undefined', () => {
    expect(parseCvssV2BaseVector()[0]).toBeUndefined();
    expect(parseCvssV2BaseVector()[1]).toBeUndefined();
    expect(parseCvssV2BaseVector({})[0]).toBeUndefined();
    expect(parseCvssV2BaseVector({})[1]).toBeUndefined();
    // @ts-expect-error
    expect(parseCvssV2BaseVector({foo: 'bar'})[0]).toBeUndefined();
    // @ts-expect-error
    expect(parseCvssV2BaseVector({foo: 'bar'})[1]).toBeUndefined();
  });

  test('should parse accessVector', () => {
    expect(parseCvssV2BaseVector({accessVector: 'foo'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({accessVector: 'Local'})).toEqual([
      'AV:L/AC:ERROR/Au:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({accessVector: 'Network'})).toEqual([
      'AV:N/AC:ERROR/Au:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({accessVector: 'Adjacent'})).toEqual([
      'AV:A/AC:ERROR/Au:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
  });

  test('should parse accessComplexity', () => {
    expect(parseCvssV2BaseVector({accessComplexity: 'foo'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({accessComplexity: 'Low'})).toEqual([
      'AV:ERROR/AC:L/Au:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({accessComplexity: 'Medium'})).toEqual([
      'AV:ERROR/AC:M/Au:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({accessComplexity: 'High'})).toEqual([
      'AV:ERROR/AC:H/Au:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
  });

  test('should parse authentication', () => {
    expect(parseCvssV2BaseVector({authentication: 'foo'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({authentication: 'None'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:N/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({authentication: 'Multiple'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:M/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({authentication: 'Single'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:S/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
  });

  test('should parse confidentialityImpact', () => {
    expect(parseCvssV2BaseVector({confidentialityImpact: 'foo'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({confidentialityImpact: 'None'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:N/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({confidentialityImpact: 'Partial'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:P/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({confidentialityImpact: 'Complete'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:C/I:ERROR/A:ERROR',
      undefined,
    ]);
  });

  test('should parse integrityImpact', () => {
    expect(parseCvssV2BaseVector({integrityImpact: 'foo'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({integrityImpact: 'None'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:ERROR/I:N/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({integrityImpact: 'Partial'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:ERROR/I:P/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({integrityImpact: 'Complete'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:ERROR/I:C/A:ERROR',
      undefined,
    ]);
  });

  test('should parse availabilityImpact', () => {
    expect(parseCvssV2BaseVector({availabilityImpact: 'foo'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({availabilityImpact: 'None'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:ERROR/I:ERROR/A:N',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({availabilityImpact: 'Partial'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:ERROR/I:ERROR/A:P',
      undefined,
    ]);
    expect(parseCvssV2BaseVector({availabilityImpact: 'Complete'})).toEqual([
      'AV:ERROR/AC:ERROR/Au:ERROR/C:ERROR/I:ERROR/A:C',
      undefined,
    ]);
  });
});

describe('parseCvssV3BaseVector tests', () => {
  test('should return undefined', () => {
    expect(parseCvssV3BaseVector()[0]).toBeUndefined();
    expect(parseCvssV3BaseVector()[1]).toBeUndefined();
    expect(parseCvssV3BaseVector({})[0]).toBeUndefined();
    expect(parseCvssV3BaseVector({})[1]).toBeUndefined();
    // @ts-expect-error
    expect(parseCvssV3BaseVector({foo: 'bar'})[0]).toBeUndefined();
    // @ts-expect-error
    expect(parseCvssV3BaseVector({foo: 'bar'})[1]).toBeUndefined();
  });

  test('should parse attackVector', () => {
    expect(parseCvssV3BaseVector({attackVector: 'foo'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({attackVector: 'Local'})).toEqual([
      'CVSS:3.1/AV:L/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({attackVector: 'Network'})).toEqual([
      'CVSS:3.1/AV:N/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({attackVector: 'Physical'})).toEqual([
      'CVSS:3.1/AV:P/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
  });

  test('should parse attackComplexity', () => {
    expect(parseCvssV3BaseVector({attackComplexity: 'foo'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({attackComplexity: 'Low'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:L/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({attackComplexity: 'High'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:H/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
  });

  test('should parse privilegesRequired', () => {
    expect(parseCvssV3BaseVector({privilegesRequired: 'foo'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({privilegesRequired: 'None'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:N/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({privilegesRequired: 'High'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:H/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({privilegesRequired: 'Low'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:L/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
  });

  test('should parse userInteraction', () => {
    expect(parseCvssV3BaseVector({userInteraction: 'foo'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({userInteraction: 'None'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:N/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({userInteraction: 'Required'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:R/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
  });

  test('should parse scope', () => {
    expect(parseCvssV3BaseVector({scope: 'foo'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({scope: 'Unchanged'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:U/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({scope: 'Changed'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:C/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
  });

  test('should parse confidentialityImpact', () => {
    expect(parseCvssV3BaseVector({confidentialityImpact: 'foo'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({confidentialityImpact: 'None'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:N/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({confidentialityImpact: 'Low'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:L/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({confidentialityImpact: 'High'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:H/I:ERROR/A:ERROR',
      undefined,
    ]);
  });

  test('should parse integrityImpact', () => {
    expect(parseCvssV3BaseVector({integrityImpact: 'foo'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({integrityImpact: 'None'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:N/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({integrityImpact: 'Low'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:L/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({integrityImpact: 'High'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:H/A:ERROR',
      undefined,
    ]);
  });

  test('should parse availabilityImpact', () => {
    expect(parseCvssV3BaseVector({availabilityImpact: 'foo'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:ERROR',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({availabilityImpact: 'None'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:N',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({availabilityImpact: 'Low'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:L',
      undefined,
    ]);
    expect(parseCvssV3BaseVector({availabilityImpact: 'High'})).toEqual([
      'CVSS:3.1/AV:ERROR/AC:ERROR/PR:ERROR/UI:ERROR/S:ERROR/C:ERROR/I:ERROR/A:H',
      undefined,
    ]);
  });
});

describe('parseCvssV2BaseFromVector tests', () => {
  test('should return empty object', () => {
    expect(parseCvssV2BaseFromVector()).toEqual({});
    expect(parseCvssV2BaseFromVector('')).toEqual({});
    expect(parseCvssV2BaseFromVector(' ')).toEqual({});
  });

  test('should set vectors to undefined', () => {
    expect(parseCvssV2BaseFromVector('foo')).toEqual({
      accessVector: undefined,
      accessComplexity: undefined,
      authentication: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse av', () => {
    expect(parseCvssV2BaseFromVector('AV:L')).toEqual({
      accessVector: 'Local',
      accessComplexity: undefined,
      authentication: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV2BaseFromVector('AV:A')).toEqual({
      accessVector: 'Adjacent',
      accessComplexity: undefined,
      authentication: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV2BaseFromVector('AV:N')).toEqual({
      accessVector: 'Network',
      accessComplexity: undefined,
      authentication: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse ac', () => {
    expect(parseCvssV2BaseFromVector('AC:L')).toEqual({
      accessVector: undefined,
      accessComplexity: 'Low',
      authentication: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV2BaseFromVector('AC:M')).toEqual({
      accessVector: undefined,
      accessComplexity: 'Medium',
      authentication: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV2BaseFromVector('AC:H')).toEqual({
      accessVector: undefined,
      accessComplexity: 'High',
      authentication: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse au', () => {
    expect(parseCvssV2BaseFromVector('AU:M')).toEqual({
      accessVector: undefined,
      accessComplexity: undefined,
      authentication: 'Multiple',
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV2BaseFromVector('AU:S')).toEqual({
      accessVector: undefined,
      accessComplexity: undefined,
      authentication: 'Single',
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV2BaseFromVector('AU:N')).toEqual({
      accessVector: undefined,
      accessComplexity: undefined,
      authentication: 'None',
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse c', () => {
    expect(parseCvssV2BaseFromVector('C:C')).toEqual({
      accessVector: undefined,
      accessComplexity: undefined,
      authentication: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: 'Complete',
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV2BaseFromVector('C:P')).toEqual({
      accessVector: undefined,
      accessComplexity: undefined,
      authentication: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: 'Partial',
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV2BaseFromVector('C:N')).toEqual({
      accessVector: undefined,
      accessComplexity: undefined,
      authentication: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: 'None',
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse i', () => {
    expect(parseCvssV2BaseFromVector('I:C')).toEqual({
      accessVector: undefined,
      accessComplexity: undefined,
      authentication: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: 'Complete',
      cvssScore: undefined,
    });
    expect(parseCvssV2BaseFromVector('I:P')).toEqual({
      accessVector: undefined,
      accessComplexity: undefined,
      authentication: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: 'Partial',
      cvssScore: undefined,
    });
    expect(parseCvssV2BaseFromVector('I:N')).toEqual({
      accessVector: undefined,
      accessComplexity: undefined,
      authentication: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: 'None',
      cvssScore: undefined,
    });
  });

  test('should parse a', () => {
    expect(parseCvssV2BaseFromVector('A:C')).toEqual({
      accessVector: undefined,
      accessComplexity: undefined,
      authentication: undefined,
      availabilityImpact: 'Complete',
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV2BaseFromVector('A:P')).toEqual({
      accessVector: undefined,
      accessComplexity: undefined,
      authentication: undefined,
      availabilityImpact: 'Partial',
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV2BaseFromVector('A:N')).toEqual({
      accessVector: undefined,
      accessComplexity: undefined,
      authentication: undefined,
      availabilityImpact: 'None',
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse full vector', () => {
    expect(parseCvssV2BaseFromVector('AV:N/AC:H/AU:S/C:C/I:C/A:C')).toEqual({
      accessVector: 'Network',
      accessComplexity: 'High',
      authentication: 'Single',
      availabilityImpact: 'Complete',
      confidentialityImpact: 'Complete',
      integrityImpact: 'Complete',
      cvssScore: 7.1,
    });
  });
});

describe('parseCvssV3BaseFromVector tests', () => {
  test('should return empty object', () => {
    expect(parseCvssV3BaseFromVector()).toEqual({});
    expect(parseCvssV3BaseFromVector('')).toEqual({});
    expect(parseCvssV3BaseFromVector(' ')).toEqual({});
  });

  test('should set vectors to undefined', () => {
    expect(parseCvssV3BaseFromVector('foo')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse av', () => {
    expect(parseCvssV3BaseFromVector('AV:L')).toEqual({
      attackVector: 'Local',
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('AV:A')).toEqual({
      attackVector: 'Adjacent',
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('AV:N')).toEqual({
      attackVector: 'Network',
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('AV:L')).toEqual({
      attackVector: 'Local',
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse ac', () => {
    expect(parseCvssV3BaseFromVector('AC:L')).toEqual({
      attackVector: undefined,
      attackComplexity: 'Low',
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('AC:H')).toEqual({
      attackVector: undefined,
      attackComplexity: 'High',
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse pr', () => {
    expect(parseCvssV3BaseFromVector('PR:H')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: 'High',
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('PR:L')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: 'Low',
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('PR:N')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: 'None',
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse ui', () => {
    expect(parseCvssV3BaseFromVector('UI:R')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: 'Required',
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('UI:N')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: 'None',
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse s', () => {
    expect(parseCvssV3BaseFromVector('S:C')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: 'Changed',
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('S:U')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: 'Unchanged',
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse c', () => {
    expect(parseCvssV3BaseFromVector('C:H')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: 'High',
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('C:L')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: 'Low',
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('C:N')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: 'None',
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse i', () => {
    expect(parseCvssV3BaseFromVector('I:H')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: 'High',
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('I:L')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: 'Low',
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('I:N')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: undefined,
      confidentialityImpact: undefined,
      integrityImpact: 'None',
      cvssScore: undefined,
    });
  });

  test('should parse a', () => {
    expect(parseCvssV3BaseFromVector('A:H')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: 'High',
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('A:L')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: 'Low',
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
    expect(parseCvssV3BaseFromVector('A:N')).toEqual({
      attackVector: undefined,
      attackComplexity: undefined,
      privilegesRequired: undefined,
      userInteraction: undefined,
      scope: undefined,
      availabilityImpact: 'None',
      confidentialityImpact: undefined,
      integrityImpact: undefined,
      cvssScore: undefined,
    });
  });

  test('should parse full vector', () => {
    expect(
      parseCvssV3BaseFromVector('CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:C/C:N/I:H/A:N'),
    ).toEqual({
      attackVector: 'Physical',
      attackComplexity: 'Low',
      privilegesRequired: 'None',
      userInteraction: 'None',
      scope: 'Changed',
      availabilityImpact: 'None',
      confidentialityImpact: 'None',
      integrityImpact: 'High',
      cvssScore: 5.3,
    });
  });
});
