#!/usr/bin/env sh

set -ex

: "${TARGET?The TARGET environment variable must be set.}"

# Tests are all super fast anyway, and they fault often enough on travis that
# having only one thread increases debuggability to be worth it.
#export RUST_BACKTRACE=full
#export RUST_TEST_NOCAPTURE=1
#export RUST_TEST_THREADS=1

export RUSTFLAGS="${RUSTFLAGS} -D warnings -Z merge-functions=disabled -Z verify-llvm-ir"
export HOST_RUSTFLAGS="${RUSTFLAGS}"
export PROFILE="${PROFILE:="release"}"

case ${TARGET} in
    # On Windows the linker performs identical COMDAT folding (ICF) by default
    # in release mode which removes identical COMDAT sections. This interferes
    # with our instruction assertions just like LLVM's MergeFunctions pass so
    # we disable it.
    *-pc-windows-msvc)
        export RUSTFLAGS="${RUSTFLAGS} -Clink-args=/OPT:NOICF"
        ;;
    # On 32-bit use a static relocation model which avoids some extra
    # instructions when dealing with static data, notably allowing some
    # instruction assertion checks to pass below the 20 instruction limit. If
    # this is the default, dynamic, then too many instructions are generated
    # when we assert the instruction for a function and it causes tests to fail.
    i686-* | i586-*)
        export RUSTFLAGS="${RUSTFLAGS} -C relocation-model=static"
        ;;
    # Some x86_64 targets enable by default more features beyond SSE2,
    # which cause some instruction assertion checks to fail.
    x86_64-*)
        export RUSTFLAGS="${RUSTFLAGS} -C target-feature=-sse3"
        ;;
    #Unoptimized build uses fast-isel which breaks with msa
    mips-* | mipsel-*)
	export RUSTFLAGS="${RUSTFLAGS} -C llvm-args=-fast-isel=false"
	;;
    armv7-*eabihf | thumbv7-*eabihf)
        export RUSTFLAGS="${RUSTFLAGS} -Ctarget-feature=+neon,+fp16"
        ;;
    amdgcn-*)
        export RUSTFLAGS="${RUSTFLAGS} -Ctarget-cpu=gfx1200"
        ;;
    # Some of our test dependencies use the deprecated `gcc` crates which
    # doesn't detect RISC-V compilers automatically, so do it manually here.
    riscv*)
        export RUSTFLAGS="${RUSTFLAGS} -Ctarget-feature=+zk,+zks,+zbb,+zbc"
        ;;
esac

echo "RUSTFLAGS=${RUSTFLAGS}"
echo "OBJDUMP=${OBJDUMP}"
echo "STDARCH_DISABLE_ASSERT_INSTR=${STDARCH_DISABLE_ASSERT_INSTR}"
echo "STDARCH_TEST_EVERYTHING=${STDARCH_TEST_EVERYTHING}"
echo "STDARCH_TEST_SKIP_FEATURE=${STDARCH_TEST_SKIP_FEATURE}"
echo "STDARCH_TEST_SKIP_FUNCTION=${STDARCH_TEST_SKIP_FUNCTION}"
echo "PROFILE=${PROFILE}"

cargo_test() {
    cmd="cargo"
    subcmd="test"
    if [ "$NORUN" = "1" ]; then
        export subcmd="build"
    fi
    cmd="$cmd ${subcmd} --target=$TARGET --profile=$PROFILE $1"
    cmd="$cmd -- $2"

    case ${TARGET} in
        # wasm targets can't catch panics so if a test failures make sure the test
        # harness isn't trying to capture output, otherwise we won't get any useful
        # output.
        wasm32*)
            if [ "$PROFILE" = "release" ]; then
              dir="release"
            else
              dir="debug"
            fi
            export CARGO_TARGET_WASM32_WASIP1_RUNNER="wasmtime -Wexceptions --dir /checkout/target/wasm32-wasip1/$dir/deps::."
            cmd="$cmd --nocapture"
            ;;
    esac
    $cmd
}

CORE_ARCH="--manifest-path=crates/core_arch/Cargo.toml"
STDARCH_EXAMPLES="--manifest-path=examples/Cargo.toml"

cargo_test "${CORE_ARCH}"

if [ "$NOSTD" != "1" ]; then
    cargo_test "${STDARCH_EXAMPLES}"
fi


# Test targets compiled with extra features.
case ${TARGET} in
    x86_64* | i686*)
        export STDARCH_DISABLE_ASSERT_INSTR=1

        export RUSTFLAGS="${RUSTFLAGS} -C target-feature=+avx"
        cargo_test 
        ;;
    # FIXME: don't build anymore
    #mips-*gnu* | mipsel-*gnu*)
    #    export RUSTFLAGS="${RUSTFLAGS} -C target-feature=+msa,+fp64,+mips32r5"
    #    cargo_test 
	  #    ;;
    mips64*)
        export RUSTFLAGS="${RUSTFLAGS} -C target-feature=+msa"
        cargo_test 
	      ;;
    s390x*)
        export RUSTFLAGS="${RUSTFLAGS} -C target-feature=+vector-enhancements-1"
        cargo_test 
	      ;;
    powerpc64*)
        export RUSTFLAGS="${RUSTFLAGS} -C target-feature=+altivec"
        cargo_test 

        export RUSTFLAGS="${RUSTFLAGS} -C target-feature=+vsx"
        cargo_test 
        ;;
    powerpc*)
        # qemu has a bug in PPC32 which leads to a crash when compiled with `vsx`
        export RUSTFLAGS="${RUSTFLAGS} -C target-feature=+altivec"
        cargo_test 
        ;;
    *)
        ;;

esac

if [ "$NORUN" != "1" ] && [ "$NOSTD" != 1 ]; then
    # Test examples
    (
        cd examples
        cargo test --target "${TARGET}" --profile "${PROFILE}"
        echo test | cargo run --target "${TARGET}" --profile "${PROFILE}" hex
    )
fi
