use std::fmt::Debug;
use std::sync::Arc;
use std::sync::atomic::{AtomicUsize, Ordering};

use pyo3::prelude::*;
use pyo3::types::{PyBytes, PyDict, PyTuple, PyType};
use pyo3::{PyTraverseError, PyVisit};
use type_serializers::any::AnySerializer;

use crate::definitions::{Definitions, DefinitionsBuilder};
use crate::py_gc::PyGcTraverse;

pub(crate) use config::{BytesMode, SerializationConfig};
pub use errors::{PydanticSerializationError, PydanticSerializationUnexpectedValue};
pub(crate) use extra::{Extra, SerMode, SerializationState, WarningsMode};
pub use shared::CombinedSerializer;
use shared::{BuildSerializer, to_json_bytes};

mod computed_fields;
mod config;
mod errors;
mod extra;
mod fields;
mod filter;
mod infer;
mod ob_type;
mod polymorphism_trampoline;
mod prebuilt;
pub mod ser;
mod shared;
mod type_serializers;

#[derive(FromPyObject)]
pub enum WarningsArg {
    Bool(bool),
    Literal(WarningsMode),
}

#[pyclass(module = "pydantic_core._pydantic_core", frozen)]
#[derive(Debug)]
pub struct SchemaSerializer {
    serializer: Arc<CombinedSerializer>,
    definitions: Definitions<Arc<CombinedSerializer>>,
    expected_json_size: AtomicUsize,
    config: SerializationConfig,
    // References to the Python schema and config objects are saved to enable
    // reconstructing the object for pickle support (see `__reduce__`).
    py_schema: Py<PyDict>,
    py_config: Option<Py<PyDict>>,
}

impl_py_gc_traverse!(SchemaSerializer {
    serializer,
    definitions,
    py_schema,
    py_config,
});

#[pymethods]
impl SchemaSerializer {
    #[new]
    #[pyo3(signature = (schema, config=None, _use_prebuilt=true))]
    pub fn py_new(
        schema: Bound<'_, PyDict>,
        config: Option<&Bound<'_, PyDict>>,
        _use_prebuilt: bool,
    ) -> PyResult<Self> {
        // use_prebuilt=true by default, but false during rebuilds to avoid stale references
        // to old serializers (see https://github.com/pydantic/pydantic/issues/12446)
        let mut definitions_builder = DefinitionsBuilder::new(_use_prebuilt);
        let serializer = CombinedSerializer::build(schema.cast()?, config, &mut definitions_builder)?;
        Ok(Self {
            serializer,
            definitions: definitions_builder.finish()?,
            expected_json_size: AtomicUsize::new(1024),
            config: SerializationConfig::from_config(config)?,
            py_schema: schema.into(),
            py_config: match config {
                Some(c) if !c.is_empty() => Some(c.clone().into()),
                _ => None,
            },
        })
    }

    #[allow(clippy::too_many_arguments)]
    #[pyo3(signature = (value, *, mode = None, include = None, exclude = None, by_alias = None,
        exclude_unset = false, exclude_defaults = false, exclude_none = false, exclude_computed_fields = false,
        round_trip = false, warnings = WarningsArg::Bool(true), fallback = None, serialize_as_any = false,
        polymorphic_serialization = None, context = None))]
    pub fn to_python(
        &self,
        py: Python,
        value: &Bound<'_, PyAny>,
        mode: Option<&str>,
        include: Option<Bound<'_, PyAny>>,
        exclude: Option<Bound<'_, PyAny>>,
        by_alias: Option<bool>,
        exclude_unset: bool,
        exclude_defaults: bool,
        exclude_none: bool,
        exclude_computed_fields: bool,
        round_trip: bool,
        warnings: WarningsArg,
        fallback: Option<Bound<'_, PyAny>>,
        serialize_as_any: bool,
        polymorphic_serialization: Option<bool>,
        context: Option<Bound<'_, PyAny>>,
    ) -> PyResult<Py<PyAny>> {
        let mode: SerMode = mode.into();
        let warnings_mode = match warnings {
            WarningsArg::Bool(b) => b.into(),
            WarningsArg::Literal(mode) => mode,
        };
        let extra = Extra::new(
            py,
            mode,
            by_alias,
            exclude_unset,
            exclude_defaults,
            exclude_none,
            exclude_computed_fields,
            round_trip,
            false,
            fallback,
            serialize_as_any,
            polymorphic_serialization,
            context,
        );
        let mut state = SerializationState::new(self.config, warnings_mode, include, exclude, extra)?;
        let v = self.serializer.to_python(value, &mut state)?;
        state.warnings.final_check(py)?;
        Ok(v)
    }

    #[allow(clippy::too_many_arguments)]
    #[pyo3(signature = (value, *, indent = None, ensure_ascii = false, include = None, exclude = None, by_alias = None,
        exclude_unset = false, exclude_defaults = false, exclude_none = false, exclude_computed_fields = false,
        round_trip = false, warnings = WarningsArg::Bool(true), fallback = None, serialize_as_any = false,
        polymorphic_serialization = None, context = None))]
    pub fn to_json(
        &self,
        py: Python,
        value: &Bound<'_, PyAny>,
        indent: Option<usize>,
        ensure_ascii: Option<bool>,
        include: Option<Bound<'_, PyAny>>,
        exclude: Option<Bound<'_, PyAny>>,
        by_alias: Option<bool>,
        exclude_unset: bool,
        exclude_defaults: bool,
        exclude_none: bool,
        exclude_computed_fields: bool,
        round_trip: bool,
        warnings: WarningsArg,
        fallback: Option<Bound<'_, PyAny>>,
        serialize_as_any: bool,
        polymorphic_serialization: Option<bool>,
        context: Option<Bound<'_, PyAny>>,
    ) -> PyResult<Py<PyAny>> {
        let warnings_mode = match warnings {
            WarningsArg::Bool(b) => b.into(),
            WarningsArg::Literal(mode) => mode,
        };
        let extra = Extra::new(
            py,
            SerMode::Json,
            by_alias,
            exclude_unset,
            exclude_defaults,
            exclude_none,
            exclude_computed_fields,
            round_trip,
            false,
            fallback,
            serialize_as_any,
            polymorphic_serialization,
            context,
        );
        let mut state = SerializationState::new(self.config, warnings_mode, include, exclude, extra)?;
        let bytes = to_json_bytes(
            value,
            &self.serializer,
            &mut state,
            indent,
            ensure_ascii.unwrap_or(false),
            self.expected_json_size.load(Ordering::Relaxed),
        )?;

        state.warnings.final_check(py)?;

        self.expected_json_size.store(bytes.len(), Ordering::Relaxed);
        let py_bytes = PyBytes::new(py, &bytes);
        Ok(py_bytes.into())
    }

    pub fn __reduce__<'py>(slf: &Bound<'py, Self>) -> PyResult<(Bound<'py, PyType>, Bound<'py, PyTuple>)> {
        // Passing _use_prebuilt=false avoids reusing prebuilt serializers when unpickling
        let init_args = (&slf.get().py_schema, &slf.get().py_config, false).into_pyobject(slf.py())?;
        Ok((slf.get_type(), init_args))
    }

    pub fn __repr__(&self) -> String {
        format!(
            "SchemaSerializer(serializer={:#?}, definitions={:#?})",
            self.serializer, self.definitions
        )
    }

    fn __traverse__(&self, visit: PyVisit<'_>) -> Result<(), PyTraverseError> {
        self.py_gc_traverse(&visit)
    }
}

#[allow(clippy::too_many_arguments)]
#[pyfunction]
#[pyo3(signature = (value, *, indent = None, ensure_ascii = false, include = None, exclude = None, by_alias = true,
    exclude_none = false, round_trip = false, timedelta_mode = "iso8601", temporal_mode = "iso8601",
    bytes_mode = "utf8",  inf_nan_mode = "constants", serialize_unknown = false, fallback = None,
    serialize_as_any = false, polymorphic_serialization = None, context = None))]
pub fn to_json(
    py: Python,
    value: &Bound<'_, PyAny>,
    indent: Option<usize>,
    ensure_ascii: Option<bool>,
    include: Option<Bound<'_, PyAny>>,
    exclude: Option<Bound<'_, PyAny>>,
    by_alias: bool,
    exclude_none: bool,
    round_trip: bool,
    timedelta_mode: &str,
    temporal_mode: &str,
    bytes_mode: &str,
    inf_nan_mode: &str,
    serialize_unknown: bool,
    fallback: Option<Bound<'_, PyAny>>,
    serialize_as_any: bool,
    polymorphic_serialization: Option<bool>,
    context: Option<Bound<'_, PyAny>>,
) -> PyResult<Py<PyAny>> {
    let config = SerializationConfig::from_args(timedelta_mode, temporal_mode, bytes_mode, inf_nan_mode)?;
    let extra = Extra::new(
        py,
        SerMode::Json,
        Some(by_alias),
        false,
        false,
        exclude_none,
        false,
        round_trip,
        serialize_unknown,
        fallback,
        serialize_as_any,
        polymorphic_serialization,
        context,
    );
    let mut state = SerializationState::new(config, WarningsMode::None, include, exclude, extra)?;
    let bytes = to_json_bytes(
        value,
        AnySerializer::get(),
        &mut state,
        indent,
        ensure_ascii.unwrap_or(false),
        1024,
    )?;
    state.final_check(py)?;
    let py_bytes = PyBytes::new(py, &bytes);
    Ok(py_bytes.into())
}

#[allow(clippy::too_many_arguments)]
#[pyfunction]
#[pyo3(signature = (value, *, include = None, exclude = None, by_alias = true, exclude_none = false, round_trip = false,
    timedelta_mode = "iso8601", temporal_mode = "iso8601", bytes_mode = "utf8", inf_nan_mode = "constants",
    serialize_unknown = false, fallback = None, serialize_as_any = false, polymorphic_serialization = None, context = None))]
pub fn to_jsonable_python(
    py: Python,
    value: &Bound<'_, PyAny>,
    include: Option<Bound<'_, PyAny>>,
    exclude: Option<Bound<'_, PyAny>>,
    by_alias: bool,
    exclude_none: bool,
    round_trip: bool,
    timedelta_mode: &str,
    temporal_mode: &str,
    bytes_mode: &str,
    inf_nan_mode: &str,
    serialize_unknown: bool,
    fallback: Option<Bound<'_, PyAny>>,
    serialize_as_any: bool,
    polymorphic_serialization: Option<bool>,
    context: Option<Bound<'_, PyAny>>,
) -> PyResult<Py<PyAny>> {
    let config = SerializationConfig::from_args(timedelta_mode, temporal_mode, bytes_mode, inf_nan_mode)?;
    let extra = Extra::new(
        py,
        SerMode::Json,
        Some(by_alias),
        false,
        false,
        exclude_none,
        false,
        round_trip,
        serialize_unknown,
        fallback,
        serialize_as_any,
        polymorphic_serialization,
        context,
    );
    let mut state = SerializationState::new(config, WarningsMode::None, include, exclude, extra)?;
    let v = infer::infer_to_python(value, &mut state)?;
    state.final_check(py)?;
    Ok(v)
}
