use crate::context::RenderContext;
use crate::drawing::Drawing;
use crate::html::TextSpan;
use crate::prelude::*;
use crate::string::WStr;
use gc_arena::{Collect, Gc, Mutation};
use ruffle_render::backend::null::NullBitmapSource;
use ruffle_render::backend::{RenderBackend, ShapeHandle};
use ruffle_render::bitmap::{Bitmap, BitmapHandle};
use ruffle_render::error::Error;
use ruffle_render::shape_utils::{DrawCommand, FillRule};
use ruffle_render::transform::Transform;

use std::cell::{Cell, OnceCell, Ref, RefCell};
use std::hash::{Hash, Hasher};
use std::rc::Rc;
use std::sync::Arc;
use swf::FillStyle;

pub use swf::TextGridFit;

#[derive(Clone, Eq, Collect, Debug)]
#[collect(require_static)]
pub struct FontQuery {
    pub font_type: FontType,
    pub name: String,
    pub lowercase_name: String,
    pub is_bold: bool,
    pub is_italic: bool,
}

impl FontQuery {
    pub fn new(font_type: FontType, name: String, is_bold: bool, is_italic: bool) -> Self {
        Self {
            font_type,
            lowercase_name: name.to_lowercase(),
            name,
            is_bold,
            is_italic,
        }
    }

    pub fn from_descriptor(font_type: FontType, descriptor: &FontDescriptor) -> Self {
        Self {
            font_type,
            name: descriptor.name().to_owned(),
            lowercase_name: descriptor.lowercase_name().to_owned(),
            is_bold: descriptor.bold(),
            is_italic: descriptor.italic(),
        }
    }
}

impl PartialEq for FontQuery {
    fn eq(&self, other: &Self) -> bool {
        self.font_type == other.font_type
            && self.lowercase_name == other.lowercase_name
            && self.is_bold == other.is_bold
            && self.is_italic == other.is_italic
    }
}

impl Hash for FontQuery {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.font_type.hash(state);
        self.lowercase_name.hash(state);
        self.is_bold.hash(state);
        self.is_italic.hash(state);
    }
}

#[derive(Debug, Copy, Clone, Eq, PartialEq, Hash, Collect)]
#[collect(require_static)]
pub enum DefaultFont {
    /// `_sans`, a Sans-Serif font (similar to Helvetica or Arial)
    Sans,

    /// `_serif`, a Serif font (similar to Times Roman)
    Serif,

    /// `_typewriter`, a Monospace font (similar to Courier)
    Typewriter,

    /// `_ゴシック`, a Japanese Gothic font
    JapaneseGothic,

    /// `_等幅`, a Japanese Gothic Mono font
    JapaneseGothicMono,

    /// `_明朝`, a Japanese Mincho font
    JapaneseMincho,
}

impl DefaultFont {
    pub fn from_name(name: &str) -> Option<Self> {
        Some(match name {
            "_serif" => DefaultFont::Serif,
            "_sans" => DefaultFont::Sans,
            "_typewriter" => DefaultFont::Typewriter,
            "_ゴシック" => DefaultFont::JapaneseGothic,
            "_等幅" => DefaultFont::JapaneseGothicMono,
            "_明朝" => DefaultFont::JapaneseMincho,
            _ => return None,
        })
    }
}

fn round_to_pixel(t: Twips) -> Twips {
    Twips::from_pixels(t.to_pixels().round())
}

/// Parameters necessary to evaluate a font.
#[derive(Copy, Clone, Debug)]
pub struct EvalParameters {
    /// The height of each glyph, equivalent to a font size.
    pub height: Twips,

    /// Additional letter spacing to be added to or removed from each glyph
    /// after normal or kerned glyph advances are applied.
    pub letter_spacing: Twips,

    /// Whether to allow use of font-provided kerning metrics.
    ///
    /// Fonts can optionally add or remove additional spacing between specific
    /// pairs of letters, separate from the ordinary width between glyphs. This
    /// parameter allows enabling or disabling that feature.
    pub kerning: bool,
}

impl EvalParameters {
    /// Convert the formatting on a text span over to font evaluation
    /// parameters.
    pub fn from_span(span: &TextSpan) -> Self {
        Self {
            height: Twips::from_pixels(span.font.size),
            letter_spacing: Twips::from_pixels(span.font.letter_spacing),
            kerning: span.font.kerning,
        }
    }

    /// Get the height that the font would be evaluated at.
    pub fn height(&self) -> Twips {
        self.height
    }
}

pub trait FontRenderer: std::fmt::Debug {
    fn get_font_metrics(&self) -> FontMetrics;

    fn has_kerning_info(&self) -> bool;

    fn render_glyph(&self, character: char) -> Option<Glyph>;

    fn calculate_kerning(&self, left: char, right: char) -> Twips;
}

struct GlyphToDrawing<'a>(&'a mut Drawing);

/// Convert from a TTF outline, to a flash Drawing.
///
/// Note that the Y axis is flipped. I do not know why, but Flash does this.
impl ttf_parser::OutlineBuilder for GlyphToDrawing<'_> {
    fn move_to(&mut self, x: f32, y: f32) {
        self.0.draw_command(DrawCommand::MoveTo(Point::new(
            Twips::new(x as i32),
            Twips::new(-y as i32),
        )));
    }

    fn line_to(&mut self, x: f32, y: f32) {
        self.0.draw_command(DrawCommand::LineTo(Point::new(
            Twips::new(x as i32),
            Twips::new(-y as i32),
        )));
    }

    fn quad_to(&mut self, x1: f32, y1: f32, x: f32, y: f32) {
        self.0.draw_command(DrawCommand::QuadraticCurveTo {
            control: Point::new(Twips::new(x1 as i32), Twips::new(-y1 as i32)),
            anchor: Point::new(Twips::new(x as i32), Twips::new(-y as i32)),
        });
    }

    fn curve_to(&mut self, x1: f32, y1: f32, x2: f32, y2: f32, x: f32, y: f32) {
        self.0.draw_command(DrawCommand::CubicCurveTo {
            control_a: Point::new(Twips::new(x1 as i32), Twips::new(-y1 as i32)),
            control_b: Point::new(Twips::new(x2 as i32), Twips::new(-y2 as i32)),
            anchor: Point::new(Twips::new(x as i32), Twips::new(-y as i32)),
        });
    }

    fn close(&mut self) {
        self.0.close_path();
    }
}

pub struct FontFileData(Arc<dyn AsRef<[u8]>>);

impl FontFileData {
    pub fn new(data: impl AsRef<[u8]> + 'static) -> Self {
        Self(Arc::new(data))
    }

    pub fn new_shared(data: Arc<dyn AsRef<[u8]>>) -> Self {
        Self(data)
    }
}

impl std::ops::Deref for FontFileData {
    type Target = [u8];

    #[inline]
    fn deref(&self) -> &[u8] {
        self.0.as_ref().as_ref()
    }
}

impl std::fmt::Debug for FontFileData {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.debug_tuple("FontFileData").field(&"<data>").finish()
    }
}

/// Represents a raw font file (ie .ttf).
/// This should be shared and reused where possible, and it's reparsed every time a new glyph is required.
///
/// Parsing of a font is near-free (according to [ttf_parser::Face::parse]), but the storage isn't.
///
/// Font files may contain multiple individual font faces, but those font faces may reuse the same
/// Glyph from the same file. For this reason, glyphs are reused where possible.
#[derive(Debug)]
pub struct FontFace {
    data: FontFileData,
    glyphs: Vec<OnceCell<Option<Glyph>>>,
    font_index: u32,

    ascender: i32,
    descender: i32,
    leading: i16,
    scale: f32,
    might_have_kerning: bool,
}

impl FontFace {
    pub fn new(data: FontFileData, font_index: u32) -> Result<Self, ttf_parser::FaceParsingError> {
        // TODO: Support font collections

        // We validate that the font is good here, so we can just `.expect()` it later
        let face = ttf_parser::Face::parse(&data, font_index)?;

        let ascender = face.ascender() as i32;
        let descender = -face.descender() as i32;
        let leading = face.line_gap();
        let scale = face.units_per_em() as f32;
        let glyphs = vec![OnceCell::new(); face.number_of_glyphs() as usize];

        // [NA] TODO: This is technically correct for just Kerning, but in practice kerning comes in many forms.
        // We need to support GPOS to do better at this, but that's a bigger change to font rendering as a whole.
        let might_have_kerning = face
            .tables()
            .kern
            .map(|k| {
                k.subtables
                    .into_iter()
                    .any(|sub| sub.horizontal && !sub.has_state_machine)
            })
            .unwrap_or_default();

        Ok(Self {
            data,
            font_index,
            glyphs,
            ascender,
            descender,
            leading,
            scale,
            might_have_kerning,
        })
    }

    pub fn get_glyph(&self, character: char) -> Option<&Glyph> {
        let face = ttf_parser::Face::parse(&self.data, self.font_index)
            .expect("Font was already checked to be valid");
        if let Some(glyph_id) = face.glyph_index(character) {
            return self.glyphs[glyph_id.0 as usize]
                .get_or_init(|| {
                    let mut drawing = Drawing::new();
                    // TTF uses NonZero
                    drawing.new_fill(
                        Some(FillStyle::Color(Color::WHITE)),
                        Some(FillRule::NonZero),
                    );
                    if face
                        .outline_glyph(glyph_id, &mut GlyphToDrawing(&mut drawing))
                        .is_some()
                    {
                        let advance = face.glyph_hor_advance(glyph_id).map_or_else(
                            || drawing.self_bounds().width(),
                            |a| Twips::new(a as i32),
                        );
                        Some(Glyph {
                            shape: GlyphShape::Drawing(Box::new(drawing)),
                            advance,
                            character,
                        })
                    } else {
                        let advance = Twips::new(face.glyph_hor_advance(glyph_id)? as i32);
                        // If we have advance, then this is either an image, SVG or simply missing (ie whitespace)
                        Some(Glyph {
                            shape: GlyphShape::None,
                            advance,
                            character,
                        })
                    }
                })
                .as_ref();
        }
        None
    }

    pub fn has_kerning_info(&self) -> bool {
        self.might_have_kerning
    }

    pub fn get_kerning_offset(&self, left: char, right: char) -> Twips {
        let face = ttf_parser::Face::parse(&self.data, self.font_index)
            .expect("Font was already checked to be valid");

        if let Some(kern) = face.tables().kern
            && let (Some(left_glyph), Some(right_glyph)) =
                (face.glyph_index(left), face.glyph_index(right))
        {
            for subtable in kern.subtables {
                if subtable.horizontal
                    && let Some(value) = subtable.glyphs_kerning(left_glyph, right_glyph)
                {
                    return Twips::new(value as i32);
                }
            }
        }

        Twips::ZERO
    }
}

pub enum GlyphRef<'a> {
    Direct(&'a Glyph),
    Ref(Ref<'a, Glyph>),
}

impl<'a> std::ops::Deref for GlyphRef<'a> {
    type Target = Glyph;

    fn deref(&self) -> &Self::Target {
        match self {
            GlyphRef::Direct(r) => r,
            GlyphRef::Ref(r) => r.deref(),
        }
    }
}

#[derive(Debug)]
pub enum GlyphSource {
    Memory {
        /// The list of glyphs defined in the font.
        /// Used directly by `DefineText` tags.
        glyphs: Vec<Glyph>,

        /// A map from a Unicode code point to glyph in the `glyphs` array.
        /// Used by `DefineEditText` tags.
        code_point_to_glyph: fnv::FnvHashMap<u16, usize>,

        /// Kerning information.
        /// Maps from a pair of unicode code points to horizontal offset value.
        kerning_pairs: fnv::FnvHashMap<(u16, u16), Twips>,
    },
    FontFace(FontFace),
    ExternalRenderer {
        /// Maps Unicode code points to glyphs rendered by the renderer.
        glyph_cache: RefCell<fnv::FnvHashMap<u16, Option<Glyph>>>,

        /// Maps Unicode pairs to kerning provided by the renderer.
        kerning_cache: RefCell<fnv::FnvHashMap<(u16, u16), Twips>>,

        font_renderer: Box<dyn FontRenderer>,
    },
    Empty,
}

impl GlyphSource {
    pub fn get_by_index(&self, index: usize) -> Option<GlyphRef<'_>> {
        match self {
            GlyphSource::Memory { glyphs, .. } => glyphs.get(index).map(GlyphRef::Direct),
            GlyphSource::FontFace(_) => None, // Unsupported.
            GlyphSource::ExternalRenderer { .. } => None, // Unsupported.
            GlyphSource::Empty => None,
        }
    }

    pub fn get_by_code_point(&self, code_point: char) -> Option<GlyphRef<'_>> {
        match self {
            GlyphSource::Memory {
                glyphs,
                code_point_to_glyph,
                ..
            } => {
                // TODO: Properly handle UTF-16/out-of-bounds code points.
                let code_point = code_point as u16;
                if let Some(index) = code_point_to_glyph.get(&code_point) {
                    glyphs.get(*index).map(GlyphRef::Direct)
                } else {
                    None
                }
            }
            GlyphSource::FontFace(face) => face.get_glyph(code_point).map(GlyphRef::Direct),
            GlyphSource::ExternalRenderer {
                glyph_cache,
                font_renderer,
                ..
            } => {
                let character = code_point;
                let code_point = code_point as u16;

                glyph_cache
                    .borrow_mut()
                    .entry(code_point)
                    .or_insert_with(|| font_renderer.render_glyph(character));

                let glyph = Ref::filter_map(glyph_cache.borrow(), |v| {
                    v.get(&code_point).unwrap_or(&None).as_ref()
                })
                .ok();

                glyph.map(GlyphRef::Ref)
            }
            GlyphSource::Empty => None,
        }
    }

    pub fn has_kerning_info(&self) -> bool {
        match self {
            GlyphSource::Memory { kerning_pairs, .. } => !kerning_pairs.is_empty(),
            GlyphSource::FontFace(face) => face.has_kerning_info(),
            GlyphSource::ExternalRenderer { font_renderer, .. } => font_renderer.has_kerning_info(),
            GlyphSource::Empty => false,
        }
    }

    pub fn get_kerning_offset(&self, left: char, right: char) -> Twips {
        match self {
            GlyphSource::Memory { kerning_pairs, .. } => {
                // TODO: Properly handle UTF-16/out-of-bounds code points.
                let left_code_point = left as u16;
                let right_code_point = right as u16;
                kerning_pairs
                    .get(&(left_code_point, right_code_point))
                    .cloned()
                    .unwrap_or_default()
            }
            GlyphSource::FontFace(face) => face.get_kerning_offset(left, right),
            GlyphSource::ExternalRenderer {
                kerning_cache,
                font_renderer,
                ..
            } => {
                let (Ok(left_cp), Ok(right_cp)) = (left.try_into(), right.try_into()) else {
                    return Twips::ZERO;
                };
                *kerning_cache
                    .borrow_mut()
                    .entry((left_cp, right_cp))
                    .or_insert_with(|| font_renderer.calculate_kerning(left, right))
            }
            GlyphSource::Empty => Twips::ZERO,
        }
    }
}

#[derive(Debug, Clone, Copy, Eq, PartialEq, Collect, Hash)]
#[collect(require_static)]
pub enum FontType {
    Embedded,
    EmbeddedCFF,
    Device,
}

impl FontType {
    pub fn is_device(self) -> bool {
        self == Self::Device
    }

    pub fn is_embedded(self) -> bool {
        self != Self::Device
    }
}

#[derive(Debug, Clone)]
pub struct FontMetrics {
    /// The scaling applied to the font height to render at the proper size.
    /// This depends on the DefineFont tag version.
    pub scale: f32,

    /// The distance from the top of each glyph to the baseline of the font, in
    /// EM-square coordinates.
    pub ascent: i32,

    /// The distance from the baseline of the font to the bottom of each glyph,
    /// in EM-square coordinates.
    pub descent: i32,

    /// The distance between the bottom of any one glyph and the top of
    /// another, in EM-square coordinates.
    #[allow(dead_code)] // Web build falsely claims it's unused
    pub leading: i16,
}

#[derive(Debug, Clone, Collect, Copy)]
#[collect(no_drop)]
pub struct Font<'gc>(Gc<'gc, FontData>);

#[derive(Debug, Collect)]
#[collect(require_static)]
struct FontData {
    glyphs: GlyphSource,

    metrics: FontMetrics,

    /// The identity of the font.
    #[collect(require_static)]
    descriptor: FontDescriptor,

    font_type: FontType,

    /// Whether this font has a layout defined.
    ///
    /// Fonts without a layout are used only to describe a font,
    /// not to provide glyphs.
    has_layout: bool,
}

impl<'gc> Font<'gc> {
    pub fn from_font_file(
        gc_context: &Mutation<'gc>,
        descriptor: FontDescriptor,
        data: FontFileData,
        font_index: u32,
        font_type: FontType,
    ) -> Result<Font<'gc>, ttf_parser::FaceParsingError> {
        let face = FontFace::new(data, font_index)?;

        Ok(Font(Gc::new(
            gc_context,
            FontData {
                metrics: FontMetrics {
                    scale: face.scale,
                    ascent: face.ascender,
                    descent: face.descender,
                    leading: face.leading,
                },
                glyphs: GlyphSource::FontFace(face),
                descriptor,
                font_type,
                has_layout: true,
            },
        )))
    }

    pub fn from_swf_tag(
        gc_context: &Mutation<'gc>,
        renderer: &mut dyn RenderBackend,
        tag: swf::Font,
        encoding: &'static swf::Encoding,
        font_type: FontType,
    ) -> Font<'gc> {
        let mut code_point_to_glyph = fnv::FnvHashMap::default();

        let descriptor = FontDescriptor::from_swf_tag(&tag, encoding);
        let (ascent, descent, leading) = if let Some(layout) = &tag.layout {
            (layout.ascent as i32, layout.descent as i32, layout.leading)
        } else {
            (0, 0, 0)
        };

        let glyphs: Vec<Glyph> = tag
            .glyphs
            .into_iter()
            .enumerate()
            .map(|(index, swf_glyph)| {
                let code = swf_glyph.code;
                // TODO: Flash doesn't care whether it's a surrogate code point or not.
                //   We should probably rethink using Rust's char for Flash characters.
                let character = char::from_u32(code as u32).unwrap_or(char::REPLACEMENT_CHARACTER);
                code_point_to_glyph.insert(code, index);

                let glyph = Glyph {
                    advance: Twips::new(swf_glyph.advance.into()),
                    shape: GlyphShape::Swf(Box::new(RefCell::new(SwfGlyphOrShape::Glyph(
                        swf_glyph,
                    )))),
                    character,
                };

                // Eager-load ASCII characters.
                if code < 128 {
                    glyph.glyph_render_data(renderer);
                }

                glyph
            })
            .collect();

        let kerning_pairs: fnv::FnvHashMap<(u16, u16), Twips> = if let Some(layout) = &tag.layout {
            layout
                .kerning
                .iter()
                .map(|kerning| ((kerning.left_code, kerning.right_code), kerning.adjustment))
                .collect()
        } else {
            fnv::FnvHashMap::default()
        };

        Font(Gc::new(
            gc_context,
            FontData {
                glyphs: if glyphs.is_empty() {
                    GlyphSource::Empty
                } else {
                    GlyphSource::Memory {
                        glyphs,
                        code_point_to_glyph,
                        kerning_pairs,
                    }
                },

                metrics: FontMetrics {
                    // DefineFont3 stores coordinates at 20x the scale of DefineFont1/2.
                    // (SWF19 p.164)
                    scale: if tag.version >= 3 { 20480.0 } else { 1024.0 },
                    ascent,
                    descent,
                    leading,
                },
                descriptor,
                font_type,
                has_layout: tag.layout.is_some(),
            },
        ))
    }

    pub fn from_font4_tag(
        gc_context: &Mutation<'gc>,
        tag: swf::Font4,
        encoding: &'static swf::Encoding,
    ) -> Result<Font<'gc>, ttf_parser::FaceParsingError> {
        let name = tag.name.to_str_lossy(encoding);
        let descriptor = FontDescriptor::from_parts(&name, tag.is_bold, tag.is_italic);

        if let Some(bytes) = tag.data {
            Font::from_font_file(
                gc_context,
                descriptor,
                // TODO remove when https://github.com/rust-lang/rust-clippy/issues/15252 is fixed
                #[expect(clippy::unnecessary_to_owned)]
                FontFileData::new(bytes.to_vec()),
                0,
                FontType::EmbeddedCFF,
            )
        } else {
            Ok(Self::empty_font(
                gc_context,
                &name,
                tag.is_bold,
                tag.is_italic,
                FontType::EmbeddedCFF,
            ))
        }
    }

    pub fn from_renderer(
        gc_context: &Mutation<'gc>,
        descriptor: FontDescriptor,
        font_renderer: Box<dyn FontRenderer>,
    ) -> Self {
        let metrics = font_renderer.get_font_metrics();
        Font(Gc::new(
            gc_context,
            FontData {
                glyphs: GlyphSource::ExternalRenderer {
                    glyph_cache: RefCell::new(fnv::FnvHashMap::default()),
                    kerning_cache: RefCell::new(fnv::FnvHashMap::default()),
                    font_renderer,
                },

                metrics,
                descriptor,
                font_type: FontType::Device,
                has_layout: true,
            },
        ))
    }

    pub fn empty_font(
        gc_context: &Mutation<'gc>,
        name: &str,
        is_bold: bool,
        is_italic: bool,
        font_type: FontType,
    ) -> Font<'gc> {
        let descriptor = FontDescriptor::from_parts(name, is_bold, is_italic);

        Font(Gc::new(
            gc_context,
            FontData {
                metrics: FontMetrics {
                    scale: 1.0,
                    ascent: 0,
                    descent: 0,
                    leading: 0,
                },
                glyphs: GlyphSource::Empty,
                descriptor,
                font_type,
                has_layout: true,
            },
        ))
    }

    /// Returns whether this font contains glyph shapes.
    /// If not, this font should be rendered as a device font.
    pub fn has_glyphs(self) -> bool {
        !matches!(self.0.glyphs, GlyphSource::Empty)
    }

    /// Returns a glyph entry by index.
    /// Used by `Text` display objects.
    pub fn get_glyph(&self, i: usize) -> Option<GlyphRef<'_>> {
        self.0.glyphs.get_by_index(i)
    }

    /// Returns a glyph entry by character.
    /// Used by `EditText` display objects.
    pub fn get_glyph_for_char(&self, c: char) -> Option<GlyphRef<'_>> {
        self.0.glyphs.get_by_code_point(c)
    }

    /// Determine if this font contains all the glyphs within a given string.
    pub fn has_glyphs_for_str(self, target_str: &WStr) -> bool {
        for character in target_str.chars() {
            let c = character.unwrap_or(char::REPLACEMENT_CHARACTER);
            if self.get_glyph_for_char(c).is_none() {
                return false;
            }
        }

        true
    }

    pub fn descriptor(&self) -> &FontDescriptor {
        &self.0.descriptor
    }

    pub fn has_layout(self) -> bool {
        self.0.has_layout
    }
}

impl<'gc> FontLike<'gc> for Font<'gc> {
    fn resolve_glyph(&self, c: char) -> Option<GlyphResolution<'_, 'gc>> {
        self.get_glyph_for_char(c)
            .map(|glyph| GlyphResolution::new(glyph, *self))
    }

    fn has_kerning_info(&self) -> bool {
        self.0.glyphs.has_kerning_info()
    }

    fn get_kerning_offset(&self, left: char, right: char) -> Twips {
        self.0.glyphs.get_kerning_offset(left, right)
    }

    fn get_leading_for_height(&self, height: Twips) -> Twips {
        let scale = height.get() as f32 / self.scale();

        Twips::new((self.0.metrics.leading as f32 * scale) as i32)
    }

    fn get_baseline_for_height(&self, height: Twips) -> Twips {
        let scale = height.get() as f32 / self.scale();

        Twips::new((self.0.metrics.ascent as f32 * scale) as i32)
    }

    fn get_descent_for_height(&self, height: Twips) -> Twips {
        let scale = height.get() as f32 / self.scale();

        Twips::new((self.0.metrics.descent as f32 * scale) as i32)
    }

    fn scale(&self) -> f32 {
        self.0.metrics.scale
    }

    fn font_type(&self) -> FontType {
        self.0.font_type
    }
}

pub trait FontLike<'gc> {
    /// Resolve a glyph for a char.
    ///
    /// The resolution contains information about the glyph and the font that
    /// provided the glyph.
    fn resolve_glyph(&self, c: char) -> Option<GlyphResolution<'_, 'gc>>;

    /// Returns whether this font contains kerning information.
    fn has_kerning_info(&self) -> bool;

    /// Given a pair of characters, applies the offset that should be applied
    /// to the advance value between these two characters.
    /// Returns 0 twips if no kerning offset exists between these two characters.
    fn get_kerning_offset(&self, left: char, right: char) -> Twips;

    /// Return the leading for this font at a given height.
    #[allow(dead_code)] // TODO Do we need this method at all?
    fn get_leading_for_height(&self, height: Twips) -> Twips;

    /// Get the baseline from the top of the glyph at a given height.
    fn get_baseline_for_height(&self, height: Twips) -> Twips;

    /// Get the descent from the baseline to the bottom of the glyph at a given height.
    fn get_descent_for_height(&self, height: Twips) -> Twips;

    fn scale(&self) -> f32;

    fn font_type(&self) -> FontType;

    /// Evaluate this font against a particular string on a glyph-by-glyph
    /// basis.
    ///
    /// This function takes the text string to evaluate against, the base
    /// transform to start from, the height of each glyph, and produces a list
    /// of transforms and glyphs which will be consumed by the `glyph_func`
    /// closure. This corresponds to the series of drawing operations necessary
    /// to render the text on a single horizontal line.
    ///
    /// It's guaranteed that this function will iterate over all characters
    /// from the text, irrespectively of whether they have a glyph or not.
    fn evaluate(
        &self,
        text: &WStr, // TODO: take an `IntoIterator<Item=char>`, to not depend on string representation?
        mut transform: Transform,
        params: EvalParameters,
        glyph_func: &mut dyn FnMut(usize, &Transform, GlyphRef, Twips, Twips),
    ) {
        let baseline = self.get_baseline_for_height(params.height);

        // TODO [KJ] I'm not sure whether we should iterate over characters here or over code units.
        //   I suspect Flash Player does not support full UTF-16 when displaying and laying out text.
        let mut char_indices = text
            .char_indices()
            .map(|(pos, c)| (pos, c.unwrap_or(char::REPLACEMENT_CHARACTER)))
            .peekable();

        let kerning_enabled =
            self.has_kerning_info() && (self.font_type().is_device() || params.kerning);

        let mut x = Twips::ZERO;
        while let Some((pos, c)) = char_indices.next() {
            if let Some(resolution) = self.resolve_glyph(c) {
                let glyph = resolution.glyph;
                let scale = params.height.get() as f32 / resolution.font.scale();
                let mut advance = glyph.advance();
                if kerning_enabled {
                    let next_char = char_indices.peek().map(|(_, ch)| *ch);
                    let kerning = next_char
                        .map(|ch| self.get_kerning_offset(c, ch))
                        .unwrap_or_default();
                    advance += kerning;
                }
                let twips_advance = if self.font_type() == FontType::Device {
                    let unspaced_advance =
                        round_to_pixel(Twips::new((advance.get() as f32 * scale) as i32));
                    let spaced_advance =
                        unspaced_advance + params.letter_spacing.round_to_pixel_ties_even();
                    if spaced_advance > Twips::ZERO {
                        spaced_advance
                    } else {
                        unspaced_advance
                    }
                } else {
                    Twips::new((advance.get() as f32 * scale) as i32) + params.letter_spacing
                };

                transform.matrix.a = scale;
                transform.matrix.d = scale;
                transform.matrix.ty = if glyph.rendered_at_baseline() {
                    baseline
                } else {
                    Twips::ZERO
                };

                glyph_func(pos, &transform, glyph, twips_advance, x);

                // Step horizontally.
                transform.matrix.tx += twips_advance;
                x += twips_advance;
            } else {
                // No glyph, zero advance.  This makes it possible to use this method for purposes
                // other than rendering the font, e.g. measurement, iterating over characters.
                glyph_func(pos, &transform, Glyph::empty(c).as_ref(), Twips::ZERO, x);
            }
        }
    }

    /// Measure a particular string's width.
    fn measure(&self, text: &WStr, params: EvalParameters) -> Twips {
        let mut width = Twips::ZERO;

        self.evaluate(
            text,
            Default::default(),
            params,
            &mut |_pos, _transform, _glyph, advance, x| {
                width = width.max(x + advance);
            },
        );

        width
    }
}

#[derive(Debug, Clone)]
enum SwfGlyphOrShape {
    Glyph(swf::Glyph),
    Shape {
        shape: swf::Shape,
        // Handle to registered shape, loaded lazily on first render of this glyph.
        handle: Option<ShapeHandle>,
    },
    Poisoned,
}

impl SwfGlyphOrShape {
    fn shape(&mut self) -> (&mut swf::Shape, &mut Option<ShapeHandle>) {
        if let Self::Glyph(_) = self
            && let Self::Glyph(glyph) = core::mem::replace(self, Self::Poisoned)
        {
            *self = Self::Shape {
                shape: ruffle_render::shape_utils::swf_glyph_to_shape(glyph),
                handle: None,
            };
        }

        match self {
            SwfGlyphOrShape::Shape { shape, handle } => (shape, handle),
            _ => unreachable!(),
        }
    }
}

#[derive(Clone, Debug)]
pub enum GlyphRenderData {
    Shape(ShapeHandle),
    Bitmap { handle: BitmapHandle, tx: Twips },
}

impl GlyphRenderData {
    pub fn from_shape(shape_handle: ShapeHandle) -> Self {
        Self::Shape(shape_handle)
    }

    pub fn from_bitmap(bitmap_handle: BitmapHandle, tx: Twips) -> Self {
        Self::Bitmap {
            handle: bitmap_handle,
            tx,
        }
    }
}

#[derive(Debug, Clone)]
enum GlyphShape {
    Swf(Box<RefCell<SwfGlyphOrShape>>),
    Drawing(Box<Drawing>),
    Bitmap(Rc<GlyphBitmap<'static>>),
    None,
}

impl GlyphShape {
    pub fn hit_test(&self, point: Point<Twips>, local_matrix: &Matrix) -> bool {
        match self {
            GlyphShape::Swf(glyph) => {
                let mut glyph = glyph.borrow_mut();
                let (shape, _) = glyph.shape();
                shape.shape_bounds.contains(point)
                    && ruffle_render::shape_utils::shape_hit_test(shape, point, local_matrix)
            }
            GlyphShape::Drawing(drawing) => drawing.hit_test(point, local_matrix),
            GlyphShape::Bitmap(_) => {
                // TODO Implement this.
                true
            }
            GlyphShape::None => false,
        }
    }

    pub fn register(&self, renderer: &mut dyn RenderBackend) -> Option<GlyphRenderData> {
        match self {
            GlyphShape::Swf(glyph) => {
                let mut glyph = glyph.borrow_mut();
                let (shape, handle) = glyph.shape();
                handle.get_or_insert_with(|| {
                    renderer.register_shape((&*shape).into(), &NullBitmapSource)
                });
                handle.clone().map(GlyphRenderData::from_shape)
            }
            GlyphShape::Drawing(drawing) => drawing
                .register_or_replace(renderer)
                .map(GlyphRenderData::from_shape),
            GlyphShape::Bitmap(bitmap) => bitmap
                .get_handle_or_register(renderer)
                .as_ref()
                .inspect_err(|err| {
                    tracing::error!(
                        "Failed to register glyph as a bitmap: {err}, glyphs will be missing"
                    )
                })
                .ok()
                .cloned()
                .map(|handle| GlyphRenderData::from_bitmap(handle, bitmap.tx)),
            GlyphShape::None => None,
        }
    }
}

/// A Bitmap that can be registered to a RenderBackend.
struct GlyphBitmap<'a> {
    bitmap: Cell<Option<Bitmap<'a>>>,
    handle: OnceCell<Result<BitmapHandle, Error>>,

    /// Translation in x to be applied before rendering the glyph.
    tx: Twips,
}

impl<'a> std::fmt::Debug for GlyphBitmap<'a> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.debug_struct("GlyphBitmap")
            .field("handle", &self.handle)
            .finish()
    }
}

impl<'a> GlyphBitmap<'a> {
    pub fn new(bitmap: Bitmap<'a>, tx: Twips) -> Self {
        Self {
            bitmap: Cell::new(Some(bitmap)),
            handle: OnceCell::new(),
            tx,
        }
    }

    pub fn get_handle_or_register(
        &self,
        renderer: &mut dyn RenderBackend,
    ) -> &Result<BitmapHandle, Error> {
        self.handle.get_or_init(|| {
            renderer.register_bitmap(
                self.bitmap
                    .take()
                    .expect("Bitmap should be available before registering"),
            )
        })
    }
}

#[derive(Debug, Clone)]
pub struct Glyph {
    shape: GlyphShape,
    advance: Twips,

    // The character this glyph represents.
    character: char,
}

impl Glyph {
    /// Returns an empty glyph with zero advance.
    pub fn empty(character: char) -> Self {
        Self {
            shape: GlyphShape::None,
            advance: Twips::ZERO,
            character,
        }
    }

    pub fn from_bitmap(
        character: char,
        bitmap: Bitmap<'static>,
        advance: Twips,
        tx: Twips,
    ) -> Self {
        Self {
            shape: GlyphShape::Bitmap(Rc::new(GlyphBitmap::new(bitmap, tx))),
            advance,
            character,
        }
    }

    pub fn glyph_render_data(&self, renderer: &mut dyn RenderBackend) -> Option<GlyphRenderData> {
        self.shape.register(renderer)
    }

    pub fn hit_test(&self, point: Point<Twips>, local_matrix: &Matrix) -> bool {
        self.shape.hit_test(point, local_matrix)
    }

    pub fn advance(&self) -> Twips {
        self.advance
    }

    pub fn character(&self) -> char {
        self.character
    }

    pub fn as_ref(&self) -> GlyphRef<'_> {
        GlyphRef::Direct(self)
    }

    pub fn rendered_at_baseline(&self) -> bool {
        match self.shape {
            GlyphShape::Swf(_) => true,
            GlyphShape::Drawing(_) => true,
            GlyphShape::Bitmap(_) => false,
            GlyphShape::None => false,
        }
    }

    pub fn renderable<'gc>(&self, context: &mut RenderContext<'_, 'gc>) -> bool {
        self.glyph_render_data(context.renderer).is_some()
    }

    pub fn render<'gc>(&self, context: &mut RenderContext<'_, 'gc>) {
        use ruffle_render::commands::CommandHandler;

        let Some(render_data) = self.glyph_render_data(context.renderer) else {
            return;
        };

        match render_data {
            GlyphRenderData::Shape(shape_handle) => {
                context
                    .commands
                    .render_shape(shape_handle, context.transform_stack.transform());
            }
            GlyphRenderData::Bitmap { handle, tx } => {
                context.transform_stack.push(&Transform {
                    matrix: Matrix::translate(tx, Twips::ZERO),
                    ..Default::default()
                });

                context.commands.render_bitmap(
                    handle,
                    context.transform_stack.transform(),
                    true,
                    ruffle_render::bitmap::PixelSnapping::Auto,
                );

                context.transform_stack.pop();
            }
        }
    }
}

pub struct GlyphResolution<'a, 'gc> {
    pub glyph: GlyphRef<'a>,
    pub font: Font<'gc>,
}

impl<'a, 'gc> GlyphResolution<'a, 'gc> {
    fn new(glyph: GlyphRef<'a>, font: Font<'gc>) -> Self {
        Self { glyph, font }
    }
}

/// Structure which identifies a particular font by name and properties.
#[derive(Debug, Clone, Ord, PartialOrd, Collect)]
#[collect(require_static)]
pub struct FontDescriptor {
    /// The name of the font.
    /// This is set by the author of the SWF and does not correlate to any opentype names.
    name: String,

    // All name comparisons ignore case, so this is for easy comparisons.
    lowercase_name: String,

    is_bold: bool,
    is_italic: bool,
}

impl PartialEq for FontDescriptor {
    fn eq(&self, other: &Self) -> bool {
        self.lowercase_name == other.lowercase_name
            && self.is_italic == other.is_italic
            && self.is_bold == other.is_bold
    }
}

impl Eq for FontDescriptor {}

impl Hash for FontDescriptor {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.lowercase_name.hash(state);
        self.is_bold.hash(state);
        self.is_italic.hash(state);
    }
}

impl FontDescriptor {
    /// Obtain a font descriptor from a SWF font tag.
    pub fn from_swf_tag(val: &swf::Font, encoding: &'static swf::Encoding) -> Self {
        let name = val.name.to_string_lossy(encoding);
        let lowercase_name = name.to_lowercase();

        Self {
            name,
            lowercase_name,
            is_bold: val.flags.contains(swf::FontFlag::IS_BOLD),
            is_italic: val.flags.contains(swf::FontFlag::IS_ITALIC),
        }
    }

    /// Obtain a font descriptor from a name/bold/italic triplet.
    pub fn from_parts(name: &str, is_bold: bool, is_italic: bool) -> Self {
        let mut name = name.to_string();

        if let Some(first_null) = name.find('\0') {
            name.truncate(first_null);
        };
        let lowercase_name = name.to_lowercase();

        Self {
            name,
            lowercase_name,
            is_bold,
            is_italic,
        }
    }

    /// Get the name of the font this descriptor identifies.
    pub fn name(&self) -> &str {
        &self.name
    }

    // Get the lowercase name.
    pub fn lowercase_name(&self) -> &str {
        &self.lowercase_name
    }

    /// Get the boldness of the described font.
    pub fn bold(&self) -> bool {
        self.is_bold
    }

    /// Get the italic-ness of the described font.
    pub fn italic(&self) -> bool {
        self.is_italic
    }
}

/// The text rendering engine that a text field should use.
/// This is controlled by the "Anti-alias" setting in the Flash IDE.
/// Using "Anti-alias for readability" switches to the "Advanced" text
/// rendering engine.
#[derive(Debug, PartialEq, Clone, Copy)]
pub enum TextRenderSettings {
    /// This text should render with the standard rendering engine.
    /// Set via "Anti-alias for animation" in the Flash IDE.
    ///
    /// The `grid_fit`, `thickness`, and `sharpness` parameters are present
    /// because they are retained when switching from `Advanced` to `Normal`
    /// rendering and vice versa. They are not used in Normal rendering.
    Normal {
        grid_fit: TextGridFit,
        thickness: f32,
        sharpness: f32,
    },

    /// This text should render with the advanced rendering engine.
    /// Set via "Anti-alias for readability" in the Flash IDE.
    /// The parameters are set via the CSMTextSettings SWF tag.
    /// Ruffle does not support this currently, but this also affects
    /// hit-testing behavior.
    Advanced {
        grid_fit: TextGridFit,
        thickness: f32,
        sharpness: f32,
    },
}

impl TextRenderSettings {
    pub fn is_advanced(&self) -> bool {
        matches!(self, TextRenderSettings::Advanced { .. })
    }

    pub fn with_advanced_rendering(self) -> Self {
        match self {
            TextRenderSettings::Advanced { .. } => self,
            TextRenderSettings::Normal {
                grid_fit,
                thickness,
                sharpness,
            } => TextRenderSettings::Advanced {
                grid_fit,
                thickness,
                sharpness,
            },
        }
    }

    pub fn with_normal_rendering(self) -> Self {
        match self {
            TextRenderSettings::Normal { .. } => self,
            TextRenderSettings::Advanced {
                grid_fit,
                thickness,
                sharpness,
            } => TextRenderSettings::Normal {
                grid_fit,
                thickness,
                sharpness,
            },
        }
    }

    pub fn sharpness(&self) -> f32 {
        match self {
            TextRenderSettings::Normal { sharpness, .. } => *sharpness,
            TextRenderSettings::Advanced { sharpness, .. } => *sharpness,
        }
    }

    pub fn with_sharpness(self, sharpness: f32) -> Self {
        match self {
            TextRenderSettings::Normal {
                grid_fit,
                thickness,
                sharpness: _,
            } => TextRenderSettings::Normal {
                grid_fit,
                thickness,
                sharpness,
            },
            TextRenderSettings::Advanced {
                grid_fit,
                thickness,
                sharpness: _,
            } => TextRenderSettings::Advanced {
                grid_fit,
                thickness,
                sharpness,
            },
        }
    }

    pub fn thickness(&self) -> f32 {
        match self {
            TextRenderSettings::Normal { thickness, .. } => *thickness,
            TextRenderSettings::Advanced { thickness, .. } => *thickness,
        }
    }

    pub fn with_thickness(self, thickness: f32) -> Self {
        match self {
            TextRenderSettings::Normal {
                grid_fit,
                thickness: _,
                sharpness,
            } => TextRenderSettings::Normal {
                grid_fit,
                thickness,
                sharpness,
            },
            TextRenderSettings::Advanced {
                grid_fit,
                thickness: _,
                sharpness,
            } => TextRenderSettings::Advanced {
                grid_fit,
                thickness,
                sharpness,
            },
        }
    }

    pub fn grid_fit(&self) -> swf::TextGridFit {
        match self {
            TextRenderSettings::Normal { grid_fit, .. } => *grid_fit,
            TextRenderSettings::Advanced { grid_fit, .. } => *grid_fit,
        }
    }

    pub fn with_grid_fit(self, grid_fit: TextGridFit) -> Self {
        match self {
            TextRenderSettings::Normal {
                grid_fit: _,
                thickness,
                sharpness,
            } => TextRenderSettings::Normal {
                grid_fit,
                thickness,
                sharpness,
            },
            TextRenderSettings::Advanced {
                grid_fit: _,
                thickness,
                sharpness,
            } => TextRenderSettings::Advanced {
                grid_fit,
                thickness,
                sharpness,
            },
        }
    }
}

impl From<swf::CsmTextSettings> for TextRenderSettings {
    fn from(settings: swf::CsmTextSettings) -> Self {
        if settings.use_advanced_rendering {
            TextRenderSettings::Advanced {
                grid_fit: settings.grid_fit,
                thickness: settings.thickness,
                sharpness: settings.sharpness,
            }
        } else {
            TextRenderSettings::default()
        }
    }
}

impl Default for TextRenderSettings {
    fn default() -> Self {
        Self::Normal {
            grid_fit: TextGridFit::Pixel,
            thickness: 0.0,
            sharpness: 0.0,
        }
    }
}

/// Font set contains a set of fonts used to render text.
///
/// It always contains at least one font—the main font. It may also contain
/// fallback fonts, which will be used in case glyphs are missing from the main
/// font. Fallback fonts are always used in order.
///
/// TODO [KJ] We don't know what's the exact behavior when data like kerning,
///   leading, etc. does not match between main and fallback fonts.
#[derive(Debug, Clone, Collect, Copy)]
#[collect(no_drop)]
pub struct FontSet<'gc>(Gc<'gc, FontSetData<'gc>>);

#[derive(Debug, Collect)]
#[collect(no_drop)]
struct FontSetData<'gc> {
    main_font: Font<'gc>,
    fallback_fonts: Vec<Font<'gc>>,
}

impl<'gc> FontSet<'gc> {
    /// Creates a font set from a sorted list of fonts.
    ///
    /// The first font is the main font, the rest are fallbacks.
    ///
    /// Returns None when the list is empty.
    pub fn from_fonts(mc: &Mutation<'gc>, fonts: &[Font<'gc>]) -> Option<Self> {
        let (&main_font, fallback_fonts) = fonts.split_first()?;
        Some(Self(Gc::new(
            mc,
            FontSetData {
                main_font,
                fallback_fonts: fallback_fonts.to_vec(),
            },
        )))
    }

    /// Creates a font set from one font only.
    pub fn from_one_font(mc: &Mutation<'gc>, font: Font<'gc>) -> Self {
        Self(Gc::new(
            mc,
            FontSetData {
                main_font: font,
                fallback_fonts: vec![],
            },
        ))
    }

    pub fn main_font(self) -> Font<'gc> {
        self.0.main_font
    }

    pub fn fallback_fonts(&self) -> &[Font<'gc>] {
        &self.0.fallback_fonts
    }
}

impl<'gc> FontLike<'gc> for FontSet<'gc> {
    fn resolve_glyph(&self, c: char) -> Option<GlyphResolution<'_, 'gc>> {
        if let Some(glyph) = self.0.main_font.get_glyph_for_char(c) {
            return Some(GlyphResolution::new(glyph, self.0.main_font));
        }

        for fallback_font in &self.0.fallback_fonts {
            if let Some(glyph) = fallback_font.get_glyph_for_char(c) {
                return Some(GlyphResolution::new(glyph, *fallback_font));
            }
        }

        None
    }

    fn has_kerning_info(&self) -> bool {
        self.0.main_font.has_kerning_info()
    }

    fn get_kerning_offset(&self, left: char, right: char) -> Twips {
        self.0.main_font.get_kerning_offset(left, right)
    }

    fn get_leading_for_height(&self, height: Twips) -> Twips {
        self.0.main_font.get_leading_for_height(height)
    }

    fn get_baseline_for_height(&self, height: Twips) -> Twips {
        self.0.main_font.get_baseline_for_height(height)
    }

    fn get_descent_for_height(&self, height: Twips) -> Twips {
        self.0.main_font.get_descent_for_height(height)
    }

    fn scale(&self) -> f32 {
        self.0.main_font.scale()
    }

    fn font_type(&self) -> FontType {
        self.0.main_font.font_type()
    }
}
