// SPDX-License-Identifier: MIT
// Copyright (C) 2018-present iced project and contributors

use crate::iced_constants::IcedConstants;
use crate::iced_error::IcedError;
use core::iter::{ExactSizeIterator, FusedIterator, Iterator};
use core::ops::{Add, AddAssign, Sub, SubAssign};
use core::{fmt, mem};

#[cfg(feature = "instr_info")]
pub use crate::register::info::*;

#[cfg(feature = "instr_info")]
mod info {
	use crate::iced_constants::IcedConstants;
	use crate::Register;

	#[rustfmt::skip]
	#[allow(deprecated)]
	pub(super) static REGISTER_INFOS: &[RegisterInfo; IcedConstants::REGISTER_ENUM_COUNT] = &[
		// GENERATOR-BEGIN: RegisterInfoTable
		// ⚠️This was generated by GENERATOR!🦹‍♂️
		RegisterInfo { register: Register::None, base: Register::None, full_register32: Register::None, full_register: Register::None, size: 0 },
		RegisterInfo { register: Register::AL, base: Register::AL, full_register32: Register::EAX, full_register: Register::RAX, size: 1 },
		RegisterInfo { register: Register::CL, base: Register::AL, full_register32: Register::ECX, full_register: Register::RCX, size: 1 },
		RegisterInfo { register: Register::DL, base: Register::AL, full_register32: Register::EDX, full_register: Register::RDX, size: 1 },
		RegisterInfo { register: Register::BL, base: Register::AL, full_register32: Register::EBX, full_register: Register::RBX, size: 1 },
		RegisterInfo { register: Register::AH, base: Register::AL, full_register32: Register::EAX, full_register: Register::RAX, size: 1 },
		RegisterInfo { register: Register::CH, base: Register::AL, full_register32: Register::ECX, full_register: Register::RCX, size: 1 },
		RegisterInfo { register: Register::DH, base: Register::AL, full_register32: Register::EDX, full_register: Register::RDX, size: 1 },
		RegisterInfo { register: Register::BH, base: Register::AL, full_register32: Register::EBX, full_register: Register::RBX, size: 1 },
		RegisterInfo { register: Register::SPL, base: Register::AL, full_register32: Register::ESP, full_register: Register::RSP, size: 1 },
		RegisterInfo { register: Register::BPL, base: Register::AL, full_register32: Register::EBP, full_register: Register::RBP, size: 1 },
		RegisterInfo { register: Register::SIL, base: Register::AL, full_register32: Register::ESI, full_register: Register::RSI, size: 1 },
		RegisterInfo { register: Register::DIL, base: Register::AL, full_register32: Register::EDI, full_register: Register::RDI, size: 1 },
		RegisterInfo { register: Register::R8L, base: Register::AL, full_register32: Register::R8D, full_register: Register::R8, size: 1 },
		RegisterInfo { register: Register::R9L, base: Register::AL, full_register32: Register::R9D, full_register: Register::R9, size: 1 },
		RegisterInfo { register: Register::R10L, base: Register::AL, full_register32: Register::R10D, full_register: Register::R10, size: 1 },
		RegisterInfo { register: Register::R11L, base: Register::AL, full_register32: Register::R11D, full_register: Register::R11, size: 1 },
		RegisterInfo { register: Register::R12L, base: Register::AL, full_register32: Register::R12D, full_register: Register::R12, size: 1 },
		RegisterInfo { register: Register::R13L, base: Register::AL, full_register32: Register::R13D, full_register: Register::R13, size: 1 },
		RegisterInfo { register: Register::R14L, base: Register::AL, full_register32: Register::R14D, full_register: Register::R14, size: 1 },
		RegisterInfo { register: Register::R15L, base: Register::AL, full_register32: Register::R15D, full_register: Register::R15, size: 1 },
		RegisterInfo { register: Register::AX, base: Register::AX, full_register32: Register::EAX, full_register: Register::RAX, size: 2 },
		RegisterInfo { register: Register::CX, base: Register::AX, full_register32: Register::ECX, full_register: Register::RCX, size: 2 },
		RegisterInfo { register: Register::DX, base: Register::AX, full_register32: Register::EDX, full_register: Register::RDX, size: 2 },
		RegisterInfo { register: Register::BX, base: Register::AX, full_register32: Register::EBX, full_register: Register::RBX, size: 2 },
		RegisterInfo { register: Register::SP, base: Register::AX, full_register32: Register::ESP, full_register: Register::RSP, size: 2 },
		RegisterInfo { register: Register::BP, base: Register::AX, full_register32: Register::EBP, full_register: Register::RBP, size: 2 },
		RegisterInfo { register: Register::SI, base: Register::AX, full_register32: Register::ESI, full_register: Register::RSI, size: 2 },
		RegisterInfo { register: Register::DI, base: Register::AX, full_register32: Register::EDI, full_register: Register::RDI, size: 2 },
		RegisterInfo { register: Register::R8W, base: Register::AX, full_register32: Register::R8D, full_register: Register::R8, size: 2 },
		RegisterInfo { register: Register::R9W, base: Register::AX, full_register32: Register::R9D, full_register: Register::R9, size: 2 },
		RegisterInfo { register: Register::R10W, base: Register::AX, full_register32: Register::R10D, full_register: Register::R10, size: 2 },
		RegisterInfo { register: Register::R11W, base: Register::AX, full_register32: Register::R11D, full_register: Register::R11, size: 2 },
		RegisterInfo { register: Register::R12W, base: Register::AX, full_register32: Register::R12D, full_register: Register::R12, size: 2 },
		RegisterInfo { register: Register::R13W, base: Register::AX, full_register32: Register::R13D, full_register: Register::R13, size: 2 },
		RegisterInfo { register: Register::R14W, base: Register::AX, full_register32: Register::R14D, full_register: Register::R14, size: 2 },
		RegisterInfo { register: Register::R15W, base: Register::AX, full_register32: Register::R15D, full_register: Register::R15, size: 2 },
		RegisterInfo { register: Register::EAX, base: Register::EAX, full_register32: Register::EAX, full_register: Register::RAX, size: 4 },
		RegisterInfo { register: Register::ECX, base: Register::EAX, full_register32: Register::ECX, full_register: Register::RCX, size: 4 },
		RegisterInfo { register: Register::EDX, base: Register::EAX, full_register32: Register::EDX, full_register: Register::RDX, size: 4 },
		RegisterInfo { register: Register::EBX, base: Register::EAX, full_register32: Register::EBX, full_register: Register::RBX, size: 4 },
		RegisterInfo { register: Register::ESP, base: Register::EAX, full_register32: Register::ESP, full_register: Register::RSP, size: 4 },
		RegisterInfo { register: Register::EBP, base: Register::EAX, full_register32: Register::EBP, full_register: Register::RBP, size: 4 },
		RegisterInfo { register: Register::ESI, base: Register::EAX, full_register32: Register::ESI, full_register: Register::RSI, size: 4 },
		RegisterInfo { register: Register::EDI, base: Register::EAX, full_register32: Register::EDI, full_register: Register::RDI, size: 4 },
		RegisterInfo { register: Register::R8D, base: Register::EAX, full_register32: Register::R8D, full_register: Register::R8, size: 4 },
		RegisterInfo { register: Register::R9D, base: Register::EAX, full_register32: Register::R9D, full_register: Register::R9, size: 4 },
		RegisterInfo { register: Register::R10D, base: Register::EAX, full_register32: Register::R10D, full_register: Register::R10, size: 4 },
		RegisterInfo { register: Register::R11D, base: Register::EAX, full_register32: Register::R11D, full_register: Register::R11, size: 4 },
		RegisterInfo { register: Register::R12D, base: Register::EAX, full_register32: Register::R12D, full_register: Register::R12, size: 4 },
		RegisterInfo { register: Register::R13D, base: Register::EAX, full_register32: Register::R13D, full_register: Register::R13, size: 4 },
		RegisterInfo { register: Register::R14D, base: Register::EAX, full_register32: Register::R14D, full_register: Register::R14, size: 4 },
		RegisterInfo { register: Register::R15D, base: Register::EAX, full_register32: Register::R15D, full_register: Register::R15, size: 4 },
		RegisterInfo { register: Register::RAX, base: Register::RAX, full_register32: Register::EAX, full_register: Register::RAX, size: 8 },
		RegisterInfo { register: Register::RCX, base: Register::RAX, full_register32: Register::ECX, full_register: Register::RCX, size: 8 },
		RegisterInfo { register: Register::RDX, base: Register::RAX, full_register32: Register::EDX, full_register: Register::RDX, size: 8 },
		RegisterInfo { register: Register::RBX, base: Register::RAX, full_register32: Register::EBX, full_register: Register::RBX, size: 8 },
		RegisterInfo { register: Register::RSP, base: Register::RAX, full_register32: Register::ESP, full_register: Register::RSP, size: 8 },
		RegisterInfo { register: Register::RBP, base: Register::RAX, full_register32: Register::EBP, full_register: Register::RBP, size: 8 },
		RegisterInfo { register: Register::RSI, base: Register::RAX, full_register32: Register::ESI, full_register: Register::RSI, size: 8 },
		RegisterInfo { register: Register::RDI, base: Register::RAX, full_register32: Register::EDI, full_register: Register::RDI, size: 8 },
		RegisterInfo { register: Register::R8, base: Register::RAX, full_register32: Register::R8D, full_register: Register::R8, size: 8 },
		RegisterInfo { register: Register::R9, base: Register::RAX, full_register32: Register::R9D, full_register: Register::R9, size: 8 },
		RegisterInfo { register: Register::R10, base: Register::RAX, full_register32: Register::R10D, full_register: Register::R10, size: 8 },
		RegisterInfo { register: Register::R11, base: Register::RAX, full_register32: Register::R11D, full_register: Register::R11, size: 8 },
		RegisterInfo { register: Register::R12, base: Register::RAX, full_register32: Register::R12D, full_register: Register::R12, size: 8 },
		RegisterInfo { register: Register::R13, base: Register::RAX, full_register32: Register::R13D, full_register: Register::R13, size: 8 },
		RegisterInfo { register: Register::R14, base: Register::RAX, full_register32: Register::R14D, full_register: Register::R14, size: 8 },
		RegisterInfo { register: Register::R15, base: Register::RAX, full_register32: Register::R15D, full_register: Register::R15, size: 8 },
		RegisterInfo { register: Register::EIP, base: Register::EIP, full_register32: Register::RIP, full_register: Register::RIP, size: 4 },
		RegisterInfo { register: Register::RIP, base: Register::EIP, full_register32: Register::RIP, full_register: Register::RIP, size: 8 },
		RegisterInfo { register: Register::ES, base: Register::ES, full_register32: Register::ES, full_register: Register::ES, size: 2 },
		RegisterInfo { register: Register::CS, base: Register::ES, full_register32: Register::CS, full_register: Register::CS, size: 2 },
		RegisterInfo { register: Register::SS, base: Register::ES, full_register32: Register::SS, full_register: Register::SS, size: 2 },
		RegisterInfo { register: Register::DS, base: Register::ES, full_register32: Register::DS, full_register: Register::DS, size: 2 },
		RegisterInfo { register: Register::FS, base: Register::ES, full_register32: Register::FS, full_register: Register::FS, size: 2 },
		RegisterInfo { register: Register::GS, base: Register::ES, full_register32: Register::GS, full_register: Register::GS, size: 2 },
		RegisterInfo { register: Register::XMM0, base: Register::XMM0, full_register32: Register::ZMM0, full_register: Register::ZMM0, size: 16 },
		RegisterInfo { register: Register::XMM1, base: Register::XMM0, full_register32: Register::ZMM1, full_register: Register::ZMM1, size: 16 },
		RegisterInfo { register: Register::XMM2, base: Register::XMM0, full_register32: Register::ZMM2, full_register: Register::ZMM2, size: 16 },
		RegisterInfo { register: Register::XMM3, base: Register::XMM0, full_register32: Register::ZMM3, full_register: Register::ZMM3, size: 16 },
		RegisterInfo { register: Register::XMM4, base: Register::XMM0, full_register32: Register::ZMM4, full_register: Register::ZMM4, size: 16 },
		RegisterInfo { register: Register::XMM5, base: Register::XMM0, full_register32: Register::ZMM5, full_register: Register::ZMM5, size: 16 },
		RegisterInfo { register: Register::XMM6, base: Register::XMM0, full_register32: Register::ZMM6, full_register: Register::ZMM6, size: 16 },
		RegisterInfo { register: Register::XMM7, base: Register::XMM0, full_register32: Register::ZMM7, full_register: Register::ZMM7, size: 16 },
		RegisterInfo { register: Register::XMM8, base: Register::XMM0, full_register32: Register::ZMM8, full_register: Register::ZMM8, size: 16 },
		RegisterInfo { register: Register::XMM9, base: Register::XMM0, full_register32: Register::ZMM9, full_register: Register::ZMM9, size: 16 },
		RegisterInfo { register: Register::XMM10, base: Register::XMM0, full_register32: Register::ZMM10, full_register: Register::ZMM10, size: 16 },
		RegisterInfo { register: Register::XMM11, base: Register::XMM0, full_register32: Register::ZMM11, full_register: Register::ZMM11, size: 16 },
		RegisterInfo { register: Register::XMM12, base: Register::XMM0, full_register32: Register::ZMM12, full_register: Register::ZMM12, size: 16 },
		RegisterInfo { register: Register::XMM13, base: Register::XMM0, full_register32: Register::ZMM13, full_register: Register::ZMM13, size: 16 },
		RegisterInfo { register: Register::XMM14, base: Register::XMM0, full_register32: Register::ZMM14, full_register: Register::ZMM14, size: 16 },
		RegisterInfo { register: Register::XMM15, base: Register::XMM0, full_register32: Register::ZMM15, full_register: Register::ZMM15, size: 16 },
		RegisterInfo { register: Register::XMM16, base: Register::XMM0, full_register32: Register::ZMM16, full_register: Register::ZMM16, size: 16 },
		RegisterInfo { register: Register::XMM17, base: Register::XMM0, full_register32: Register::ZMM17, full_register: Register::ZMM17, size: 16 },
		RegisterInfo { register: Register::XMM18, base: Register::XMM0, full_register32: Register::ZMM18, full_register: Register::ZMM18, size: 16 },
		RegisterInfo { register: Register::XMM19, base: Register::XMM0, full_register32: Register::ZMM19, full_register: Register::ZMM19, size: 16 },
		RegisterInfo { register: Register::XMM20, base: Register::XMM0, full_register32: Register::ZMM20, full_register: Register::ZMM20, size: 16 },
		RegisterInfo { register: Register::XMM21, base: Register::XMM0, full_register32: Register::ZMM21, full_register: Register::ZMM21, size: 16 },
		RegisterInfo { register: Register::XMM22, base: Register::XMM0, full_register32: Register::ZMM22, full_register: Register::ZMM22, size: 16 },
		RegisterInfo { register: Register::XMM23, base: Register::XMM0, full_register32: Register::ZMM23, full_register: Register::ZMM23, size: 16 },
		RegisterInfo { register: Register::XMM24, base: Register::XMM0, full_register32: Register::ZMM24, full_register: Register::ZMM24, size: 16 },
		RegisterInfo { register: Register::XMM25, base: Register::XMM0, full_register32: Register::ZMM25, full_register: Register::ZMM25, size: 16 },
		RegisterInfo { register: Register::XMM26, base: Register::XMM0, full_register32: Register::ZMM26, full_register: Register::ZMM26, size: 16 },
		RegisterInfo { register: Register::XMM27, base: Register::XMM0, full_register32: Register::ZMM27, full_register: Register::ZMM27, size: 16 },
		RegisterInfo { register: Register::XMM28, base: Register::XMM0, full_register32: Register::ZMM28, full_register: Register::ZMM28, size: 16 },
		RegisterInfo { register: Register::XMM29, base: Register::XMM0, full_register32: Register::ZMM29, full_register: Register::ZMM29, size: 16 },
		RegisterInfo { register: Register::XMM30, base: Register::XMM0, full_register32: Register::ZMM30, full_register: Register::ZMM30, size: 16 },
		RegisterInfo { register: Register::XMM31, base: Register::XMM0, full_register32: Register::ZMM31, full_register: Register::ZMM31, size: 16 },
		RegisterInfo { register: Register::YMM0, base: Register::YMM0, full_register32: Register::ZMM0, full_register: Register::ZMM0, size: 32 },
		RegisterInfo { register: Register::YMM1, base: Register::YMM0, full_register32: Register::ZMM1, full_register: Register::ZMM1, size: 32 },
		RegisterInfo { register: Register::YMM2, base: Register::YMM0, full_register32: Register::ZMM2, full_register: Register::ZMM2, size: 32 },
		RegisterInfo { register: Register::YMM3, base: Register::YMM0, full_register32: Register::ZMM3, full_register: Register::ZMM3, size: 32 },
		RegisterInfo { register: Register::YMM4, base: Register::YMM0, full_register32: Register::ZMM4, full_register: Register::ZMM4, size: 32 },
		RegisterInfo { register: Register::YMM5, base: Register::YMM0, full_register32: Register::ZMM5, full_register: Register::ZMM5, size: 32 },
		RegisterInfo { register: Register::YMM6, base: Register::YMM0, full_register32: Register::ZMM6, full_register: Register::ZMM6, size: 32 },
		RegisterInfo { register: Register::YMM7, base: Register::YMM0, full_register32: Register::ZMM7, full_register: Register::ZMM7, size: 32 },
		RegisterInfo { register: Register::YMM8, base: Register::YMM0, full_register32: Register::ZMM8, full_register: Register::ZMM8, size: 32 },
		RegisterInfo { register: Register::YMM9, base: Register::YMM0, full_register32: Register::ZMM9, full_register: Register::ZMM9, size: 32 },
		RegisterInfo { register: Register::YMM10, base: Register::YMM0, full_register32: Register::ZMM10, full_register: Register::ZMM10, size: 32 },
		RegisterInfo { register: Register::YMM11, base: Register::YMM0, full_register32: Register::ZMM11, full_register: Register::ZMM11, size: 32 },
		RegisterInfo { register: Register::YMM12, base: Register::YMM0, full_register32: Register::ZMM12, full_register: Register::ZMM12, size: 32 },
		RegisterInfo { register: Register::YMM13, base: Register::YMM0, full_register32: Register::ZMM13, full_register: Register::ZMM13, size: 32 },
		RegisterInfo { register: Register::YMM14, base: Register::YMM0, full_register32: Register::ZMM14, full_register: Register::ZMM14, size: 32 },
		RegisterInfo { register: Register::YMM15, base: Register::YMM0, full_register32: Register::ZMM15, full_register: Register::ZMM15, size: 32 },
		RegisterInfo { register: Register::YMM16, base: Register::YMM0, full_register32: Register::ZMM16, full_register: Register::ZMM16, size: 32 },
		RegisterInfo { register: Register::YMM17, base: Register::YMM0, full_register32: Register::ZMM17, full_register: Register::ZMM17, size: 32 },
		RegisterInfo { register: Register::YMM18, base: Register::YMM0, full_register32: Register::ZMM18, full_register: Register::ZMM18, size: 32 },
		RegisterInfo { register: Register::YMM19, base: Register::YMM0, full_register32: Register::ZMM19, full_register: Register::ZMM19, size: 32 },
		RegisterInfo { register: Register::YMM20, base: Register::YMM0, full_register32: Register::ZMM20, full_register: Register::ZMM20, size: 32 },
		RegisterInfo { register: Register::YMM21, base: Register::YMM0, full_register32: Register::ZMM21, full_register: Register::ZMM21, size: 32 },
		RegisterInfo { register: Register::YMM22, base: Register::YMM0, full_register32: Register::ZMM22, full_register: Register::ZMM22, size: 32 },
		RegisterInfo { register: Register::YMM23, base: Register::YMM0, full_register32: Register::ZMM23, full_register: Register::ZMM23, size: 32 },
		RegisterInfo { register: Register::YMM24, base: Register::YMM0, full_register32: Register::ZMM24, full_register: Register::ZMM24, size: 32 },
		RegisterInfo { register: Register::YMM25, base: Register::YMM0, full_register32: Register::ZMM25, full_register: Register::ZMM25, size: 32 },
		RegisterInfo { register: Register::YMM26, base: Register::YMM0, full_register32: Register::ZMM26, full_register: Register::ZMM26, size: 32 },
		RegisterInfo { register: Register::YMM27, base: Register::YMM0, full_register32: Register::ZMM27, full_register: Register::ZMM27, size: 32 },
		RegisterInfo { register: Register::YMM28, base: Register::YMM0, full_register32: Register::ZMM28, full_register: Register::ZMM28, size: 32 },
		RegisterInfo { register: Register::YMM29, base: Register::YMM0, full_register32: Register::ZMM29, full_register: Register::ZMM29, size: 32 },
		RegisterInfo { register: Register::YMM30, base: Register::YMM0, full_register32: Register::ZMM30, full_register: Register::ZMM30, size: 32 },
		RegisterInfo { register: Register::YMM31, base: Register::YMM0, full_register32: Register::ZMM31, full_register: Register::ZMM31, size: 32 },
		RegisterInfo { register: Register::ZMM0, base: Register::ZMM0, full_register32: Register::ZMM0, full_register: Register::ZMM0, size: 64 },
		RegisterInfo { register: Register::ZMM1, base: Register::ZMM0, full_register32: Register::ZMM1, full_register: Register::ZMM1, size: 64 },
		RegisterInfo { register: Register::ZMM2, base: Register::ZMM0, full_register32: Register::ZMM2, full_register: Register::ZMM2, size: 64 },
		RegisterInfo { register: Register::ZMM3, base: Register::ZMM0, full_register32: Register::ZMM3, full_register: Register::ZMM3, size: 64 },
		RegisterInfo { register: Register::ZMM4, base: Register::ZMM0, full_register32: Register::ZMM4, full_register: Register::ZMM4, size: 64 },
		RegisterInfo { register: Register::ZMM5, base: Register::ZMM0, full_register32: Register::ZMM5, full_register: Register::ZMM5, size: 64 },
		RegisterInfo { register: Register::ZMM6, base: Register::ZMM0, full_register32: Register::ZMM6, full_register: Register::ZMM6, size: 64 },
		RegisterInfo { register: Register::ZMM7, base: Register::ZMM0, full_register32: Register::ZMM7, full_register: Register::ZMM7, size: 64 },
		RegisterInfo { register: Register::ZMM8, base: Register::ZMM0, full_register32: Register::ZMM8, full_register: Register::ZMM8, size: 64 },
		RegisterInfo { register: Register::ZMM9, base: Register::ZMM0, full_register32: Register::ZMM9, full_register: Register::ZMM9, size: 64 },
		RegisterInfo { register: Register::ZMM10, base: Register::ZMM0, full_register32: Register::ZMM10, full_register: Register::ZMM10, size: 64 },
		RegisterInfo { register: Register::ZMM11, base: Register::ZMM0, full_register32: Register::ZMM11, full_register: Register::ZMM11, size: 64 },
		RegisterInfo { register: Register::ZMM12, base: Register::ZMM0, full_register32: Register::ZMM12, full_register: Register::ZMM12, size: 64 },
		RegisterInfo { register: Register::ZMM13, base: Register::ZMM0, full_register32: Register::ZMM13, full_register: Register::ZMM13, size: 64 },
		RegisterInfo { register: Register::ZMM14, base: Register::ZMM0, full_register32: Register::ZMM14, full_register: Register::ZMM14, size: 64 },
		RegisterInfo { register: Register::ZMM15, base: Register::ZMM0, full_register32: Register::ZMM15, full_register: Register::ZMM15, size: 64 },
		RegisterInfo { register: Register::ZMM16, base: Register::ZMM0, full_register32: Register::ZMM16, full_register: Register::ZMM16, size: 64 },
		RegisterInfo { register: Register::ZMM17, base: Register::ZMM0, full_register32: Register::ZMM17, full_register: Register::ZMM17, size: 64 },
		RegisterInfo { register: Register::ZMM18, base: Register::ZMM0, full_register32: Register::ZMM18, full_register: Register::ZMM18, size: 64 },
		RegisterInfo { register: Register::ZMM19, base: Register::ZMM0, full_register32: Register::ZMM19, full_register: Register::ZMM19, size: 64 },
		RegisterInfo { register: Register::ZMM20, base: Register::ZMM0, full_register32: Register::ZMM20, full_register: Register::ZMM20, size: 64 },
		RegisterInfo { register: Register::ZMM21, base: Register::ZMM0, full_register32: Register::ZMM21, full_register: Register::ZMM21, size: 64 },
		RegisterInfo { register: Register::ZMM22, base: Register::ZMM0, full_register32: Register::ZMM22, full_register: Register::ZMM22, size: 64 },
		RegisterInfo { register: Register::ZMM23, base: Register::ZMM0, full_register32: Register::ZMM23, full_register: Register::ZMM23, size: 64 },
		RegisterInfo { register: Register::ZMM24, base: Register::ZMM0, full_register32: Register::ZMM24, full_register: Register::ZMM24, size: 64 },
		RegisterInfo { register: Register::ZMM25, base: Register::ZMM0, full_register32: Register::ZMM25, full_register: Register::ZMM25, size: 64 },
		RegisterInfo { register: Register::ZMM26, base: Register::ZMM0, full_register32: Register::ZMM26, full_register: Register::ZMM26, size: 64 },
		RegisterInfo { register: Register::ZMM27, base: Register::ZMM0, full_register32: Register::ZMM27, full_register: Register::ZMM27, size: 64 },
		RegisterInfo { register: Register::ZMM28, base: Register::ZMM0, full_register32: Register::ZMM28, full_register: Register::ZMM28, size: 64 },
		RegisterInfo { register: Register::ZMM29, base: Register::ZMM0, full_register32: Register::ZMM29, full_register: Register::ZMM29, size: 64 },
		RegisterInfo { register: Register::ZMM30, base: Register::ZMM0, full_register32: Register::ZMM30, full_register: Register::ZMM30, size: 64 },
		RegisterInfo { register: Register::ZMM31, base: Register::ZMM0, full_register32: Register::ZMM31, full_register: Register::ZMM31, size: 64 },
		RegisterInfo { register: Register::K0, base: Register::K0, full_register32: Register::K0, full_register: Register::K0, size: 8 },
		RegisterInfo { register: Register::K1, base: Register::K0, full_register32: Register::K1, full_register: Register::K1, size: 8 },
		RegisterInfo { register: Register::K2, base: Register::K0, full_register32: Register::K2, full_register: Register::K2, size: 8 },
		RegisterInfo { register: Register::K3, base: Register::K0, full_register32: Register::K3, full_register: Register::K3, size: 8 },
		RegisterInfo { register: Register::K4, base: Register::K0, full_register32: Register::K4, full_register: Register::K4, size: 8 },
		RegisterInfo { register: Register::K5, base: Register::K0, full_register32: Register::K5, full_register: Register::K5, size: 8 },
		RegisterInfo { register: Register::K6, base: Register::K0, full_register32: Register::K6, full_register: Register::K6, size: 8 },
		RegisterInfo { register: Register::K7, base: Register::K0, full_register32: Register::K7, full_register: Register::K7, size: 8 },
		RegisterInfo { register: Register::BND0, base: Register::BND0, full_register32: Register::BND0, full_register: Register::BND0, size: 16 },
		RegisterInfo { register: Register::BND1, base: Register::BND0, full_register32: Register::BND1, full_register: Register::BND1, size: 16 },
		RegisterInfo { register: Register::BND2, base: Register::BND0, full_register32: Register::BND2, full_register: Register::BND2, size: 16 },
		RegisterInfo { register: Register::BND3, base: Register::BND0, full_register32: Register::BND3, full_register: Register::BND3, size: 16 },
		RegisterInfo { register: Register::CR0, base: Register::CR0, full_register32: Register::CR0, full_register: Register::CR0, size: 8 },
		RegisterInfo { register: Register::CR1, base: Register::CR0, full_register32: Register::CR1, full_register: Register::CR1, size: 8 },
		RegisterInfo { register: Register::CR2, base: Register::CR0, full_register32: Register::CR2, full_register: Register::CR2, size: 8 },
		RegisterInfo { register: Register::CR3, base: Register::CR0, full_register32: Register::CR3, full_register: Register::CR3, size: 8 },
		RegisterInfo { register: Register::CR4, base: Register::CR0, full_register32: Register::CR4, full_register: Register::CR4, size: 8 },
		RegisterInfo { register: Register::CR5, base: Register::CR0, full_register32: Register::CR5, full_register: Register::CR5, size: 8 },
		RegisterInfo { register: Register::CR6, base: Register::CR0, full_register32: Register::CR6, full_register: Register::CR6, size: 8 },
		RegisterInfo { register: Register::CR7, base: Register::CR0, full_register32: Register::CR7, full_register: Register::CR7, size: 8 },
		RegisterInfo { register: Register::CR8, base: Register::CR0, full_register32: Register::CR8, full_register: Register::CR8, size: 8 },
		RegisterInfo { register: Register::CR9, base: Register::CR0, full_register32: Register::CR9, full_register: Register::CR9, size: 8 },
		RegisterInfo { register: Register::CR10, base: Register::CR0, full_register32: Register::CR10, full_register: Register::CR10, size: 8 },
		RegisterInfo { register: Register::CR11, base: Register::CR0, full_register32: Register::CR11, full_register: Register::CR11, size: 8 },
		RegisterInfo { register: Register::CR12, base: Register::CR0, full_register32: Register::CR12, full_register: Register::CR12, size: 8 },
		RegisterInfo { register: Register::CR13, base: Register::CR0, full_register32: Register::CR13, full_register: Register::CR13, size: 8 },
		RegisterInfo { register: Register::CR14, base: Register::CR0, full_register32: Register::CR14, full_register: Register::CR14, size: 8 },
		RegisterInfo { register: Register::CR15, base: Register::CR0, full_register32: Register::CR15, full_register: Register::CR15, size: 8 },
		RegisterInfo { register: Register::DR0, base: Register::DR0, full_register32: Register::DR0, full_register: Register::DR0, size: 8 },
		RegisterInfo { register: Register::DR1, base: Register::DR0, full_register32: Register::DR1, full_register: Register::DR1, size: 8 },
		RegisterInfo { register: Register::DR2, base: Register::DR0, full_register32: Register::DR2, full_register: Register::DR2, size: 8 },
		RegisterInfo { register: Register::DR3, base: Register::DR0, full_register32: Register::DR3, full_register: Register::DR3, size: 8 },
		RegisterInfo { register: Register::DR4, base: Register::DR0, full_register32: Register::DR4, full_register: Register::DR4, size: 8 },
		RegisterInfo { register: Register::DR5, base: Register::DR0, full_register32: Register::DR5, full_register: Register::DR5, size: 8 },
		RegisterInfo { register: Register::DR6, base: Register::DR0, full_register32: Register::DR6, full_register: Register::DR6, size: 8 },
		RegisterInfo { register: Register::DR7, base: Register::DR0, full_register32: Register::DR7, full_register: Register::DR7, size: 8 },
		RegisterInfo { register: Register::DR8, base: Register::DR0, full_register32: Register::DR8, full_register: Register::DR8, size: 8 },
		RegisterInfo { register: Register::DR9, base: Register::DR0, full_register32: Register::DR9, full_register: Register::DR9, size: 8 },
		RegisterInfo { register: Register::DR10, base: Register::DR0, full_register32: Register::DR10, full_register: Register::DR10, size: 8 },
		RegisterInfo { register: Register::DR11, base: Register::DR0, full_register32: Register::DR11, full_register: Register::DR11, size: 8 },
		RegisterInfo { register: Register::DR12, base: Register::DR0, full_register32: Register::DR12, full_register: Register::DR12, size: 8 },
		RegisterInfo { register: Register::DR13, base: Register::DR0, full_register32: Register::DR13, full_register: Register::DR13, size: 8 },
		RegisterInfo { register: Register::DR14, base: Register::DR0, full_register32: Register::DR14, full_register: Register::DR14, size: 8 },
		RegisterInfo { register: Register::DR15, base: Register::DR0, full_register32: Register::DR15, full_register: Register::DR15, size: 8 },
		RegisterInfo { register: Register::ST0, base: Register::ST0, full_register32: Register::ST0, full_register: Register::ST0, size: 10 },
		RegisterInfo { register: Register::ST1, base: Register::ST0, full_register32: Register::ST1, full_register: Register::ST1, size: 10 },
		RegisterInfo { register: Register::ST2, base: Register::ST0, full_register32: Register::ST2, full_register: Register::ST2, size: 10 },
		RegisterInfo { register: Register::ST3, base: Register::ST0, full_register32: Register::ST3, full_register: Register::ST3, size: 10 },
		RegisterInfo { register: Register::ST4, base: Register::ST0, full_register32: Register::ST4, full_register: Register::ST4, size: 10 },
		RegisterInfo { register: Register::ST5, base: Register::ST0, full_register32: Register::ST5, full_register: Register::ST5, size: 10 },
		RegisterInfo { register: Register::ST6, base: Register::ST0, full_register32: Register::ST6, full_register: Register::ST6, size: 10 },
		RegisterInfo { register: Register::ST7, base: Register::ST0, full_register32: Register::ST7, full_register: Register::ST7, size: 10 },
		RegisterInfo { register: Register::MM0, base: Register::MM0, full_register32: Register::MM0, full_register: Register::MM0, size: 8 },
		RegisterInfo { register: Register::MM1, base: Register::MM0, full_register32: Register::MM1, full_register: Register::MM1, size: 8 },
		RegisterInfo { register: Register::MM2, base: Register::MM0, full_register32: Register::MM2, full_register: Register::MM2, size: 8 },
		RegisterInfo { register: Register::MM3, base: Register::MM0, full_register32: Register::MM3, full_register: Register::MM3, size: 8 },
		RegisterInfo { register: Register::MM4, base: Register::MM0, full_register32: Register::MM4, full_register: Register::MM4, size: 8 },
		RegisterInfo { register: Register::MM5, base: Register::MM0, full_register32: Register::MM5, full_register: Register::MM5, size: 8 },
		RegisterInfo { register: Register::MM6, base: Register::MM0, full_register32: Register::MM6, full_register: Register::MM6, size: 8 },
		RegisterInfo { register: Register::MM7, base: Register::MM0, full_register32: Register::MM7, full_register: Register::MM7, size: 8 },
		RegisterInfo { register: Register::TR0, base: Register::TR0, full_register32: Register::TR0, full_register: Register::TR0, size: 4 },
		RegisterInfo { register: Register::TR1, base: Register::TR0, full_register32: Register::TR1, full_register: Register::TR1, size: 4 },
		RegisterInfo { register: Register::TR2, base: Register::TR0, full_register32: Register::TR2, full_register: Register::TR2, size: 4 },
		RegisterInfo { register: Register::TR3, base: Register::TR0, full_register32: Register::TR3, full_register: Register::TR3, size: 4 },
		RegisterInfo { register: Register::TR4, base: Register::TR0, full_register32: Register::TR4, full_register: Register::TR4, size: 4 },
		RegisterInfo { register: Register::TR5, base: Register::TR0, full_register32: Register::TR5, full_register: Register::TR5, size: 4 },
		RegisterInfo { register: Register::TR6, base: Register::TR0, full_register32: Register::TR6, full_register: Register::TR6, size: 4 },
		RegisterInfo { register: Register::TR7, base: Register::TR0, full_register32: Register::TR7, full_register: Register::TR7, size: 4 },
		RegisterInfo { register: Register::TMM0, base: Register::TMM0, full_register32: Register::TMM0, full_register: Register::TMM0, size: 1024 },
		RegisterInfo { register: Register::TMM1, base: Register::TMM0, full_register32: Register::TMM1, full_register: Register::TMM1, size: 1024 },
		RegisterInfo { register: Register::TMM2, base: Register::TMM0, full_register32: Register::TMM2, full_register: Register::TMM2, size: 1024 },
		RegisterInfo { register: Register::TMM3, base: Register::TMM0, full_register32: Register::TMM3, full_register: Register::TMM3, size: 1024 },
		RegisterInfo { register: Register::TMM4, base: Register::TMM0, full_register32: Register::TMM4, full_register: Register::TMM4, size: 1024 },
		RegisterInfo { register: Register::TMM5, base: Register::TMM0, full_register32: Register::TMM5, full_register: Register::TMM5, size: 1024 },
		RegisterInfo { register: Register::TMM6, base: Register::TMM0, full_register32: Register::TMM6, full_register: Register::TMM6, size: 1024 },
		RegisterInfo { register: Register::TMM7, base: Register::TMM0, full_register32: Register::TMM7, full_register: Register::TMM7, size: 1024 },
		RegisterInfo { register: Register::DontUse0, base: Register::DontUse0, full_register32: Register::DontUse0, full_register: Register::DontUse0, size: 0 },
		RegisterInfo { register: Register::DontUseFA, base: Register::DontUseFA, full_register32: Register::DontUseFA, full_register: Register::DontUseFA, size: 0 },
		RegisterInfo { register: Register::DontUseFB, base: Register::DontUseFA, full_register32: Register::DontUseFB, full_register: Register::DontUseFB, size: 0 },
		RegisterInfo { register: Register::DontUseFC, base: Register::DontUseFA, full_register32: Register::DontUseFC, full_register: Register::DontUseFC, size: 0 },
		RegisterInfo { register: Register::DontUseFD, base: Register::DontUseFA, full_register32: Register::DontUseFD, full_register: Register::DontUseFD, size: 0 },
		RegisterInfo { register: Register::DontUseFE, base: Register::DontUseFA, full_register32: Register::DontUseFE, full_register: Register::DontUseFE, size: 0 },
		RegisterInfo { register: Register::DontUseFF, base: Register::DontUseFA, full_register32: Register::DontUseFF, full_register: Register::DontUseFF, size: 0 },
		// GENERATOR-END: RegisterInfoTable
	];

	/// [`Register`] information
	///
	/// [`Register`]: enum.Register.html
	#[derive(Debug, Copy, Clone, Eq, PartialEq, Hash)]
	pub struct RegisterInfo {
		register: Register,
		base: Register,
		full_register: Register,
		full_register32: Register,
		size: u16,
	}

	impl RegisterInfo {
		/// Gets the register
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = Register::EAX.info();
		/// assert_eq!(info.register(), Register::EAX);
		/// ```
		#[must_use]
		#[inline]
		pub const fn register(&self) -> Register {
			self.register
		}

		/// Gets the base register, eg. `AL`, `AX`, `EAX`, `RAX`, `MM0`, `XMM0`, `YMM0`, `ZMM0`, `ES`
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = Register::GS.info();
		/// assert_eq!(info.base(), Register::ES);
		/// let info = Register::RDX.info();
		/// assert_eq!(info.base(), Register::RAX);
		/// let info = Register::XMM13.info();
		/// assert_eq!(info.base(), Register::XMM0);
		/// let info = Register::YMM13.info();
		/// assert_eq!(info.base(), Register::YMM0);
		/// let info = Register::ZMM13.info();
		/// assert_eq!(info.base(), Register::ZMM0);
		/// ```
		#[must_use]
		#[inline]
		pub const fn base(&self) -> Register {
			self.base
		}

		/// The register number (index) relative to [`base()`], eg. 0-15, or 0-31, or if 8-bit GPR, 0-19
		///
		/// [`base()`]: #method.base
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = Register::GS.info();
		/// assert_eq!(info.number(), 5);
		/// let info = Register::RDX.info();
		/// assert_eq!(info.number(), 2);
		/// let info = Register::XMM13.info();
		/// assert_eq!(info.number(), 13);
		/// let info = Register::YMM13.info();
		/// assert_eq!(info.number(), 13);
		/// let info = Register::ZMM13.info();
		/// assert_eq!(info.number(), 13);
		/// ```
		#[must_use]
		#[inline]
		pub const fn number(&self) -> usize {
			self.register as usize - self.base as usize
		}

		/// The full register that this one is a part of, eg. `CL`/`CH`/`CX`/`ECX`/`RCX` -> `RCX`, `XMM11`/`YMM11`/`ZMM11` -> `ZMM11`
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = Register::GS.info();
		/// assert_eq!(info.full_register(), Register::GS);
		/// let info = Register::BH.info();
		/// assert_eq!(info.full_register(), Register::RBX);
		/// let info = Register::DX.info();
		/// assert_eq!(info.full_register(), Register::RDX);
		/// let info = Register::ESP.info();
		/// assert_eq!(info.full_register(), Register::RSP);
		/// let info = Register::RCX.info();
		/// assert_eq!(info.full_register(), Register::RCX);
		/// let info = Register::XMM3.info();
		/// assert_eq!(info.full_register(), Register::ZMM3);
		/// let info = Register::YMM3.info();
		/// assert_eq!(info.full_register(), Register::ZMM3);
		/// let info = Register::ZMM3.info();
		/// assert_eq!(info.full_register(), Register::ZMM3);
		/// ```
		#[must_use]
		#[inline]
		pub const fn full_register(&self) -> Register {
			self.full_register
		}

		/// Gets the full register that this one is a part of, except if it's a GPR in which case the 32-bit register is returned,
		/// eg. `CL`/`CH`/`CX`/`ECX`/`RCX` -> `ECX`, `XMM11`/`YMM11`/`ZMM11` -> `ZMM11`
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = Register::GS.info();
		/// assert_eq!(info.full_register32(), Register::GS);
		/// let info = Register::BH.info();
		/// assert_eq!(info.full_register32(), Register::EBX);
		/// let info = Register::DX.info();
		/// assert_eq!(info.full_register32(), Register::EDX);
		/// let info = Register::ESP.info();
		/// assert_eq!(info.full_register32(), Register::ESP);
		/// let info = Register::RCX.info();
		/// assert_eq!(info.full_register32(), Register::ECX);
		/// let info = Register::XMM3.info();
		/// assert_eq!(info.full_register32(), Register::ZMM3);
		/// let info = Register::YMM3.info();
		/// assert_eq!(info.full_register32(), Register::ZMM3);
		/// let info = Register::ZMM3.info();
		/// assert_eq!(info.full_register32(), Register::ZMM3);
		/// ```
		#[must_use]
		#[inline]
		pub const fn full_register32(&self) -> Register {
			self.full_register32
		}

		/// Size of the register in bytes
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = Register::GS.info();
		/// assert_eq!(info.size(), 2);
		/// let info = Register::BH.info();
		/// assert_eq!(info.size(), 1);
		/// let info = Register::DX.info();
		/// assert_eq!(info.size(), 2);
		/// let info = Register::ESP.info();
		/// assert_eq!(info.size(), 4);
		/// let info = Register::RCX.info();
		/// assert_eq!(info.size(), 8);
		/// let info = Register::XMM3.info();
		/// assert_eq!(info.size(), 16);
		/// let info = Register::YMM3.info();
		/// assert_eq!(info.size(), 32);
		/// let info = Register::ZMM3.info();
		/// assert_eq!(info.size(), 64);
		/// ```
		#[must_use]
		#[inline]
		pub const fn size(&self) -> usize {
			self.size as usize
		}
	}
}

#[cfg(feature = "instr_info")]
impl Register {
	/// Gets register info
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// let info = Register::EAX.info();
	/// assert_eq!(info.size(), 4);
	/// ```
	#[must_use]
	#[inline]
	pub fn info(self) -> &'static RegisterInfo {
		&REGISTER_INFOS[self as usize]
	}

	/// Gets the base register, eg. `AL`, `AX`, `EAX`, `RAX`, `MM0`, `XMM0`, `YMM0`, `ZMM0`, `ES`
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert_eq!(Register::GS.base(), Register::ES);
	/// assert_eq!(Register::SIL.base(), Register::AL);
	/// assert_eq!(Register::SP.base(), Register::AX);
	/// assert_eq!(Register::R13D.base(), Register::EAX);
	/// assert_eq!(Register::RBP.base(), Register::RAX);
	/// assert_eq!(Register::MM6.base(), Register::MM0);
	/// assert_eq!(Register::XMM28.base(), Register::XMM0);
	/// assert_eq!(Register::YMM12.base(), Register::YMM0);
	/// assert_eq!(Register::ZMM31.base(), Register::ZMM0);
	/// assert_eq!(Register::K3.base(), Register::K0);
	/// assert_eq!(Register::BND1.base(), Register::BND0);
	/// assert_eq!(Register::ST7.base(), Register::ST0);
	/// assert_eq!(Register::CR8.base(), Register::CR0);
	/// assert_eq!(Register::DR6.base(), Register::DR0);
	/// assert_eq!(Register::TR3.base(), Register::TR0);
	/// assert_eq!(Register::RIP.base(), Register::EIP);
	/// ```
	#[must_use]
	#[inline]
	pub fn base(self) -> Self {
		self.info().base()
	}

	/// The register number (index) relative to [`base()`], eg. 0-15, or 0-31, or if 8-bit GPR, 0-19
	///
	/// [`base()`]: #method.base
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert_eq!(Register::GS.number(), 5);
	/// assert_eq!(Register::SIL.number(), 10);
	/// assert_eq!(Register::SP.number(), 4);
	/// assert_eq!(Register::R13D.number(), 13);
	/// assert_eq!(Register::RBP.number(), 5);
	/// assert_eq!(Register::MM6.number(), 6);
	/// assert_eq!(Register::XMM28.number(), 28);
	/// assert_eq!(Register::YMM12.number(), 12);
	/// assert_eq!(Register::ZMM31.number(), 31);
	/// assert_eq!(Register::K3.number(), 3);
	/// assert_eq!(Register::BND1.number(), 1);
	/// assert_eq!(Register::ST7.number(), 7);
	/// assert_eq!(Register::CR8.number(), 8);
	/// assert_eq!(Register::DR6.number(), 6);
	/// assert_eq!(Register::TR3.number(), 3);
	/// assert_eq!(Register::RIP.number(), 1);
	/// ```
	#[must_use]
	#[inline]
	pub fn number(self) -> usize {
		self.info().number()
	}

	/// Gets the full register that this one is a part of, eg. `CL`/`CH`/`CX`/`ECX`/`RCX` -> `RCX`, `XMM11`/`YMM11`/`ZMM11` -> `ZMM11`
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert_eq!(Register::GS.full_register(), Register::GS);
	/// assert_eq!(Register::SIL.full_register(), Register::RSI);
	/// assert_eq!(Register::SP.full_register(), Register::RSP);
	/// assert_eq!(Register::R13D.full_register(), Register::R13);
	/// assert_eq!(Register::RBP.full_register(), Register::RBP);
	/// assert_eq!(Register::MM6.full_register(), Register::MM6);
	/// assert_eq!(Register::XMM10.full_register(), Register::ZMM10);
	/// assert_eq!(Register::YMM10.full_register(), Register::ZMM10);
	/// assert_eq!(Register::ZMM10.full_register(), Register::ZMM10);
	/// assert_eq!(Register::K3.full_register(), Register::K3);
	/// assert_eq!(Register::BND1.full_register(), Register::BND1);
	/// assert_eq!(Register::ST7.full_register(), Register::ST7);
	/// assert_eq!(Register::CR8.full_register(), Register::CR8);
	/// assert_eq!(Register::DR6.full_register(), Register::DR6);
	/// assert_eq!(Register::TR3.full_register(), Register::TR3);
	/// assert_eq!(Register::RIP.full_register(), Register::RIP);
	/// ```
	#[must_use]
	#[inline]
	pub fn full_register(self) -> Self {
		self.info().full_register()
	}

	/// Gets the full register that this one is a part of, except if it's a GPR in which case the 32-bit register is returned,
	/// eg. `CL`/`CH`/`CX`/`ECX`/`RCX` -> `ECX`, `XMM11`/`YMM11`/`ZMM11` -> `ZMM11`
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert_eq!(Register::GS.full_register32(), Register::GS);
	/// assert_eq!(Register::SIL.full_register32(), Register::ESI);
	/// assert_eq!(Register::SP.full_register32(), Register::ESP);
	/// assert_eq!(Register::R13D.full_register32(), Register::R13D);
	/// assert_eq!(Register::RBP.full_register32(), Register::EBP);
	/// assert_eq!(Register::MM6.full_register32(), Register::MM6);
	/// assert_eq!(Register::XMM10.full_register32(), Register::ZMM10);
	/// assert_eq!(Register::YMM10.full_register32(), Register::ZMM10);
	/// assert_eq!(Register::ZMM10.full_register32(), Register::ZMM10);
	/// assert_eq!(Register::K3.full_register32(), Register::K3);
	/// assert_eq!(Register::BND1.full_register32(), Register::BND1);
	/// assert_eq!(Register::ST7.full_register32(), Register::ST7);
	/// assert_eq!(Register::CR8.full_register32(), Register::CR8);
	/// assert_eq!(Register::DR6.full_register32(), Register::DR6);
	/// assert_eq!(Register::TR3.full_register32(), Register::TR3);
	/// assert_eq!(Register::RIP.full_register32(), Register::RIP);
	/// ```
	#[must_use]
	#[inline]
	pub fn full_register32(self) -> Self {
		self.info().full_register32()
	}

	/// Gets the size of the register in bytes
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert_eq!(Register::GS.size(), 2);
	/// assert_eq!(Register::SIL.size(), 1);
	/// assert_eq!(Register::SP.size(), 2);
	/// assert_eq!(Register::R13D.size(), 4);
	/// assert_eq!(Register::RBP.size(), 8);
	/// assert_eq!(Register::MM6.size(), 8);
	/// assert_eq!(Register::XMM10.size(), 16);
	/// assert_eq!(Register::YMM10.size(), 32);
	/// assert_eq!(Register::ZMM10.size(), 64);
	/// assert_eq!(Register::K3.size(), 8);
	/// assert_eq!(Register::BND1.size(), 16);
	/// assert_eq!(Register::ST7.size(), 10);
	/// assert_eq!(Register::CR8.size(), 8);
	/// assert_eq!(Register::DR6.size(), 8);
	/// assert_eq!(Register::TR3.size(), 4);
	/// assert_eq!(Register::RIP.size(), 8);
	/// ```
	#[must_use]
	#[inline]
	pub fn size(self) -> usize {
		self.info().size()
	}
}

#[cfg(any(feature = "instr_info", feature = "encoder"))]
impl Register {
	/// Checks if it's a segment register (`ES`, `CS`, `SS`, `DS`, `FS`, `GS`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(Register::GS.is_segment_register());
	/// assert!(!Register::RCX.is_segment_register());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_segment_register(self) -> bool {
		Register::ES <= self && self <= Register::GS
	}

	/// Checks if it's a general purpose register (`AL`-`R15L`, `AX`-`R15W`, `EAX`-`R15D`, `RAX`-`R15`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::GS.is_gpr());
	/// assert!(Register::CH.is_gpr());
	/// assert!(Register::DX.is_gpr());
	/// assert!(Register::R13D.is_gpr());
	/// assert!(Register::RSP.is_gpr());
	/// assert!(!Register::XMM0.is_gpr());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_gpr(self) -> bool {
		Register::AL <= self && self <= Register::R15
	}

	/// Checks if it's an 8-bit general purpose register (`AL`-`R15L`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::GS.is_gpr8());
	/// assert!(Register::CH.is_gpr8());
	/// assert!(!Register::DX.is_gpr8());
	/// assert!(!Register::R13D.is_gpr8());
	/// assert!(!Register::RSP.is_gpr8());
	/// assert!(!Register::XMM0.is_gpr8());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_gpr8(self) -> bool {
		Register::AL <= self && self <= Register::R15L
	}

	/// Checks if it's a 16-bit general purpose register (`AX`-`R15W`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::GS.is_gpr16());
	/// assert!(!Register::CH.is_gpr16());
	/// assert!(Register::DX.is_gpr16());
	/// assert!(!Register::R13D.is_gpr16());
	/// assert!(!Register::RSP.is_gpr16());
	/// assert!(!Register::XMM0.is_gpr16());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_gpr16(self) -> bool {
		Register::AX <= self && self <= Register::R15W
	}

	/// Checks if it's a 32-bit general purpose register (`EAX`-`R15D`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::GS.is_gpr32());
	/// assert!(!Register::CH.is_gpr32());
	/// assert!(!Register::DX.is_gpr32());
	/// assert!(Register::R13D.is_gpr32());
	/// assert!(!Register::RSP.is_gpr32());
	/// assert!(!Register::XMM0.is_gpr32());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_gpr32(self) -> bool {
		Register::EAX <= self && self <= Register::R15D
	}

	/// Checks if it's a 64-bit general purpose register (`RAX`-`R15`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::GS.is_gpr64());
	/// assert!(!Register::CH.is_gpr64());
	/// assert!(!Register::DX.is_gpr64());
	/// assert!(!Register::R13D.is_gpr64());
	/// assert!(Register::RSP.is_gpr64());
	/// assert!(!Register::XMM0.is_gpr64());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_gpr64(self) -> bool {
		Register::RAX <= self && self <= Register::R15
	}

	/// Checks if it's a 128-bit vector register (`XMM0`-`XMM31`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::R13D.is_xmm());
	/// assert!(!Register::RSP.is_xmm());
	/// assert!(Register::XMM0.is_xmm());
	/// assert!(!Register::YMM0.is_xmm());
	/// assert!(!Register::ZMM0.is_xmm());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_xmm(self) -> bool {
		Register::XMM0 <= self && self <= IcedConstants::XMM_LAST
	}

	/// Checks if it's a 256-bit vector register (`YMM0`-`YMM31`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::R13D.is_ymm());
	/// assert!(!Register::RSP.is_ymm());
	/// assert!(!Register::XMM0.is_ymm());
	/// assert!(Register::YMM0.is_ymm());
	/// assert!(!Register::ZMM0.is_ymm());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_ymm(self) -> bool {
		Register::YMM0 <= self && self <= IcedConstants::YMM_LAST
	}

	/// Checks if it's a 512-bit vector register (`ZMM0`-`ZMM31`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::R13D.is_zmm());
	/// assert!(!Register::RSP.is_zmm());
	/// assert!(!Register::XMM0.is_zmm());
	/// assert!(!Register::YMM0.is_zmm());
	/// assert!(Register::ZMM0.is_zmm());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_zmm(self) -> bool {
		Register::ZMM0 <= self && self <= IcedConstants::ZMM_LAST
	}

	/// Checks if it's an `XMM`, `YMM` or `ZMM` register
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::R13D.is_vector_register());
	/// assert!(!Register::RSP.is_vector_register());
	/// assert!(Register::XMM0.is_vector_register());
	/// assert!(Register::YMM0.is_vector_register());
	/// assert!(Register::ZMM0.is_vector_register());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_vector_register(self) -> bool {
		Register::XMM0 <= self && self <= IcedConstants::VMM_LAST
	}

	/// Checks if it's `EIP`/`RIP`
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(Register::EIP.is_ip());
	/// assert!(Register::RIP.is_ip());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_ip(self) -> bool {
		self == Register::EIP || self == Register::RIP
	}

	/// Checks if it's an opmask register (`K0`-`K7`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::R13D.is_k());
	/// assert!(Register::K3.is_k());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_k(self) -> bool {
		Register::K0 <= self && self <= Register::K7
	}

	/// Checks if it's a control register (`CR0`-`CR15`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::R13D.is_cr());
	/// assert!(Register::CR3.is_cr());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_cr(self) -> bool {
		Register::CR0 <= self && self <= Register::CR15
	}

	/// Checks if it's a debug register (`DR0`-`DR15`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::R13D.is_dr());
	/// assert!(Register::DR3.is_dr());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_dr(self) -> bool {
		Register::DR0 <= self && self <= Register::DR15
	}

	/// Checks if it's a test register (`TR0`-`TR7`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::R13D.is_tr());
	/// assert!(Register::TR3.is_tr());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_tr(self) -> bool {
		Register::TR0 <= self && self <= Register::TR7
	}

	/// Checks if it's an FPU stack register (`ST0`-`ST7`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::R13D.is_st());
	/// assert!(Register::ST3.is_st());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_st(self) -> bool {
		Register::ST0 <= self && self <= Register::ST7
	}

	/// Checks if it's a bound register (`BND0`-`BND3`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::R13D.is_bnd());
	/// assert!(Register::BND3.is_bnd());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_bnd(self) -> bool {
		Register::BND0 <= self && self <= Register::BND3
	}

	/// Checks if it's an MMX register (`MM0`-`MM7`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::R13D.is_mm());
	/// assert!(Register::MM3.is_mm());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_mm(self) -> bool {
		Register::MM0 <= self && self <= Register::MM7
	}

	/// Checks if it's a tile register (`TMM0`-`TMM7`)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!Register::R13D.is_tmm());
	/// assert!(Register::TMM3.is_tmm());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_tmm(self) -> bool {
		Register::TMM0 <= self && self <= IcedConstants::TMM_LAST
	}
}

// GENERATOR-BEGIN: Register
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// A register
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
#[cfg_attr(not(feature = "exhaustive_enums"), non_exhaustive)]
#[allow(missing_docs)]
pub enum Register {
	None = 0,
	AL = 1,
	CL = 2,
	DL = 3,
	BL = 4,
	AH = 5,
	CH = 6,
	DH = 7,
	BH = 8,
	SPL = 9,
	BPL = 10,
	SIL = 11,
	DIL = 12,
	R8L = 13,
	R9L = 14,
	R10L = 15,
	R11L = 16,
	R12L = 17,
	R13L = 18,
	R14L = 19,
	R15L = 20,
	AX = 21,
	CX = 22,
	DX = 23,
	BX = 24,
	SP = 25,
	BP = 26,
	SI = 27,
	DI = 28,
	R8W = 29,
	R9W = 30,
	R10W = 31,
	R11W = 32,
	R12W = 33,
	R13W = 34,
	R14W = 35,
	R15W = 36,
	EAX = 37,
	ECX = 38,
	EDX = 39,
	EBX = 40,
	ESP = 41,
	EBP = 42,
	ESI = 43,
	EDI = 44,
	R8D = 45,
	R9D = 46,
	R10D = 47,
	R11D = 48,
	R12D = 49,
	R13D = 50,
	R14D = 51,
	R15D = 52,
	RAX = 53,
	RCX = 54,
	RDX = 55,
	RBX = 56,
	RSP = 57,
	RBP = 58,
	RSI = 59,
	RDI = 60,
	R8 = 61,
	R9 = 62,
	R10 = 63,
	R11 = 64,
	R12 = 65,
	R13 = 66,
	R14 = 67,
	R15 = 68,
	EIP = 69,
	RIP = 70,
	ES = 71,
	CS = 72,
	SS = 73,
	DS = 74,
	FS = 75,
	GS = 76,
	XMM0 = 77,
	XMM1 = 78,
	XMM2 = 79,
	XMM3 = 80,
	XMM4 = 81,
	XMM5 = 82,
	XMM6 = 83,
	XMM7 = 84,
	XMM8 = 85,
	XMM9 = 86,
	XMM10 = 87,
	XMM11 = 88,
	XMM12 = 89,
	XMM13 = 90,
	XMM14 = 91,
	XMM15 = 92,
	XMM16 = 93,
	XMM17 = 94,
	XMM18 = 95,
	XMM19 = 96,
	XMM20 = 97,
	XMM21 = 98,
	XMM22 = 99,
	XMM23 = 100,
	XMM24 = 101,
	XMM25 = 102,
	XMM26 = 103,
	XMM27 = 104,
	XMM28 = 105,
	XMM29 = 106,
	XMM30 = 107,
	XMM31 = 108,
	YMM0 = 109,
	YMM1 = 110,
	YMM2 = 111,
	YMM3 = 112,
	YMM4 = 113,
	YMM5 = 114,
	YMM6 = 115,
	YMM7 = 116,
	YMM8 = 117,
	YMM9 = 118,
	YMM10 = 119,
	YMM11 = 120,
	YMM12 = 121,
	YMM13 = 122,
	YMM14 = 123,
	YMM15 = 124,
	YMM16 = 125,
	YMM17 = 126,
	YMM18 = 127,
	YMM19 = 128,
	YMM20 = 129,
	YMM21 = 130,
	YMM22 = 131,
	YMM23 = 132,
	YMM24 = 133,
	YMM25 = 134,
	YMM26 = 135,
	YMM27 = 136,
	YMM28 = 137,
	YMM29 = 138,
	YMM30 = 139,
	YMM31 = 140,
	ZMM0 = 141,
	ZMM1 = 142,
	ZMM2 = 143,
	ZMM3 = 144,
	ZMM4 = 145,
	ZMM5 = 146,
	ZMM6 = 147,
	ZMM7 = 148,
	ZMM8 = 149,
	ZMM9 = 150,
	ZMM10 = 151,
	ZMM11 = 152,
	ZMM12 = 153,
	ZMM13 = 154,
	ZMM14 = 155,
	ZMM15 = 156,
	ZMM16 = 157,
	ZMM17 = 158,
	ZMM18 = 159,
	ZMM19 = 160,
	ZMM20 = 161,
	ZMM21 = 162,
	ZMM22 = 163,
	ZMM23 = 164,
	ZMM24 = 165,
	ZMM25 = 166,
	ZMM26 = 167,
	ZMM27 = 168,
	ZMM28 = 169,
	ZMM29 = 170,
	ZMM30 = 171,
	ZMM31 = 172,
	K0 = 173,
	K1 = 174,
	K2 = 175,
	K3 = 176,
	K4 = 177,
	K5 = 178,
	K6 = 179,
	K7 = 180,
	BND0 = 181,
	BND1 = 182,
	BND2 = 183,
	BND3 = 184,
	CR0 = 185,
	CR1 = 186,
	CR2 = 187,
	CR3 = 188,
	CR4 = 189,
	CR5 = 190,
	CR6 = 191,
	CR7 = 192,
	CR8 = 193,
	CR9 = 194,
	CR10 = 195,
	CR11 = 196,
	CR12 = 197,
	CR13 = 198,
	CR14 = 199,
	CR15 = 200,
	DR0 = 201,
	DR1 = 202,
	DR2 = 203,
	DR3 = 204,
	DR4 = 205,
	DR5 = 206,
	DR6 = 207,
	DR7 = 208,
	DR8 = 209,
	DR9 = 210,
	DR10 = 211,
	DR11 = 212,
	DR12 = 213,
	DR13 = 214,
	DR14 = 215,
	DR15 = 216,
	ST0 = 217,
	ST1 = 218,
	ST2 = 219,
	ST3 = 220,
	ST4 = 221,
	ST5 = 222,
	ST6 = 223,
	ST7 = 224,
	MM0 = 225,
	MM1 = 226,
	MM2 = 227,
	MM3 = 228,
	MM4 = 229,
	MM5 = 230,
	MM6 = 231,
	MM7 = 232,
	TR0 = 233,
	TR1 = 234,
	TR2 = 235,
	TR3 = 236,
	TR4 = 237,
	TR5 = 238,
	TR6 = 239,
	TR7 = 240,
	TMM0 = 241,
	TMM1 = 242,
	TMM2 = 243,
	TMM3 = 244,
	TMM4 = 245,
	TMM5 = 246,
	TMM6 = 247,
	TMM7 = 248,
	/// Don't use it!
	#[deprecated(since = "1.12.0", note = "Not part of the public API")]
	DontUse0 = 249,
	/// Don't use it!
	#[deprecated(since = "1.12.0", note = "Not part of the public API")]
	DontUseFA = 250,
	/// Don't use it!
	#[deprecated(since = "1.12.0", note = "Not part of the public API")]
	DontUseFB = 251,
	/// Don't use it!
	#[deprecated(since = "1.12.0", note = "Not part of the public API")]
	DontUseFC = 252,
	/// Don't use it!
	#[deprecated(since = "1.12.0", note = "Not part of the public API")]
	DontUseFD = 253,
	/// Don't use it!
	#[deprecated(since = "1.12.0", note = "Not part of the public API")]
	DontUseFE = 254,
	/// Don't use it!
	#[deprecated(since = "1.12.0", note = "Not part of the public API")]
	DontUseFF = 255,
}
#[rustfmt::skip]
static GEN_DEBUG_REGISTER: [&str; 256] = [
	"None",
	"AL",
	"CL",
	"DL",
	"BL",
	"AH",
	"CH",
	"DH",
	"BH",
	"SPL",
	"BPL",
	"SIL",
	"DIL",
	"R8L",
	"R9L",
	"R10L",
	"R11L",
	"R12L",
	"R13L",
	"R14L",
	"R15L",
	"AX",
	"CX",
	"DX",
	"BX",
	"SP",
	"BP",
	"SI",
	"DI",
	"R8W",
	"R9W",
	"R10W",
	"R11W",
	"R12W",
	"R13W",
	"R14W",
	"R15W",
	"EAX",
	"ECX",
	"EDX",
	"EBX",
	"ESP",
	"EBP",
	"ESI",
	"EDI",
	"R8D",
	"R9D",
	"R10D",
	"R11D",
	"R12D",
	"R13D",
	"R14D",
	"R15D",
	"RAX",
	"RCX",
	"RDX",
	"RBX",
	"RSP",
	"RBP",
	"RSI",
	"RDI",
	"R8",
	"R9",
	"R10",
	"R11",
	"R12",
	"R13",
	"R14",
	"R15",
	"EIP",
	"RIP",
	"ES",
	"CS",
	"SS",
	"DS",
	"FS",
	"GS",
	"XMM0",
	"XMM1",
	"XMM2",
	"XMM3",
	"XMM4",
	"XMM5",
	"XMM6",
	"XMM7",
	"XMM8",
	"XMM9",
	"XMM10",
	"XMM11",
	"XMM12",
	"XMM13",
	"XMM14",
	"XMM15",
	"XMM16",
	"XMM17",
	"XMM18",
	"XMM19",
	"XMM20",
	"XMM21",
	"XMM22",
	"XMM23",
	"XMM24",
	"XMM25",
	"XMM26",
	"XMM27",
	"XMM28",
	"XMM29",
	"XMM30",
	"XMM31",
	"YMM0",
	"YMM1",
	"YMM2",
	"YMM3",
	"YMM4",
	"YMM5",
	"YMM6",
	"YMM7",
	"YMM8",
	"YMM9",
	"YMM10",
	"YMM11",
	"YMM12",
	"YMM13",
	"YMM14",
	"YMM15",
	"YMM16",
	"YMM17",
	"YMM18",
	"YMM19",
	"YMM20",
	"YMM21",
	"YMM22",
	"YMM23",
	"YMM24",
	"YMM25",
	"YMM26",
	"YMM27",
	"YMM28",
	"YMM29",
	"YMM30",
	"YMM31",
	"ZMM0",
	"ZMM1",
	"ZMM2",
	"ZMM3",
	"ZMM4",
	"ZMM5",
	"ZMM6",
	"ZMM7",
	"ZMM8",
	"ZMM9",
	"ZMM10",
	"ZMM11",
	"ZMM12",
	"ZMM13",
	"ZMM14",
	"ZMM15",
	"ZMM16",
	"ZMM17",
	"ZMM18",
	"ZMM19",
	"ZMM20",
	"ZMM21",
	"ZMM22",
	"ZMM23",
	"ZMM24",
	"ZMM25",
	"ZMM26",
	"ZMM27",
	"ZMM28",
	"ZMM29",
	"ZMM30",
	"ZMM31",
	"K0",
	"K1",
	"K2",
	"K3",
	"K4",
	"K5",
	"K6",
	"K7",
	"BND0",
	"BND1",
	"BND2",
	"BND3",
	"CR0",
	"CR1",
	"CR2",
	"CR3",
	"CR4",
	"CR5",
	"CR6",
	"CR7",
	"CR8",
	"CR9",
	"CR10",
	"CR11",
	"CR12",
	"CR13",
	"CR14",
	"CR15",
	"DR0",
	"DR1",
	"DR2",
	"DR3",
	"DR4",
	"DR5",
	"DR6",
	"DR7",
	"DR8",
	"DR9",
	"DR10",
	"DR11",
	"DR12",
	"DR13",
	"DR14",
	"DR15",
	"ST0",
	"ST1",
	"ST2",
	"ST3",
	"ST4",
	"ST5",
	"ST6",
	"ST7",
	"MM0",
	"MM1",
	"MM2",
	"MM3",
	"MM4",
	"MM5",
	"MM6",
	"MM7",
	"TR0",
	"TR1",
	"TR2",
	"TR3",
	"TR4",
	"TR5",
	"TR6",
	"TR7",
	"TMM0",
	"TMM1",
	"TMM2",
	"TMM3",
	"TMM4",
	"TMM5",
	"TMM6",
	"TMM7",
	"DontUse0",
	"DontUseFA",
	"DontUseFB",
	"DontUseFC",
	"DontUseFD",
	"DontUseFE",
	"DontUseFF",
];
impl fmt::Debug for Register {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_REGISTER[*self as usize])
	}
}
impl Default for Register {
	#[must_use]
	#[inline]
	fn default() -> Self {
		Register::None
	}
}
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type RegisterUnderlyingType = u8;
#[rustfmt::skip]
impl Register {
	/// Iterates over all `Register` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = Register> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::REGISTER_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, Register>(x as u8) })
	}
}
#[test]
#[rustfmt::skip]
fn test_register_values() {
	let mut iter = Register::values();
	assert_eq!(iter.size_hint(), (IcedConstants::REGISTER_ENUM_COUNT, Some(IcedConstants::REGISTER_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::REGISTER_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::REGISTER_ENUM_COUNT - 1, Some(IcedConstants::REGISTER_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::REGISTER_ENUM_COUNT - 1);

	let values: Vec<Register> = Register::values().collect();
	assert_eq!(values.len(), IcedConstants::REGISTER_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<Register> = Register::values().collect();
	let mut values2: Vec<Register> = Register::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[rustfmt::skip]
impl TryFrom<usize> for Register {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::REGISTER_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid Register value"))
		}
	}
}
#[test]
#[rustfmt::skip]
fn test_register_try_from_usize() {
	for value in Register::values() {
		let converted = <Register as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<Register as TryFrom<usize>>::try_from(IcedConstants::REGISTER_ENUM_COUNT).is_err());
	assert!(<Register as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
#[cfg(feature = "serde")]
#[rustfmt::skip]
#[allow(clippy::zero_sized_map_values)]
const _: () = {
	use core::marker::PhantomData;
	use serde::de;
	use serde::{Deserialize, Deserializer, Serialize, Serializer};
	type EnumType = Register;
	impl Serialize for EnumType {
		#[inline]
		fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
		where
			S: Serializer,
		{
			serializer.serialize_u8(*self as u8)
		}
	}
	impl<'de> Deserialize<'de> for EnumType {
		#[inline]
		fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
		where
			D: Deserializer<'de>,
		{
			struct Visitor<'de> {
				marker: PhantomData<EnumType>,
				lifetime: PhantomData<&'de ()>,
			}
			impl<'de> de::Visitor<'de> for Visitor<'de> {
				type Value = EnumType;
				#[inline]
				fn expecting(&self, formatter: &mut fmt::Formatter<'_>) -> fmt::Result {
					formatter.write_str("enum Register")
				}
				#[inline]
				fn visit_u64<E>(self, v: u64) -> Result<Self::Value, E>
				where
					E: de::Error,
				{
					if let Ok(v) = <usize as TryFrom<_>>::try_from(v) {
						if let Ok(value) = <EnumType as TryFrom<_>>::try_from(v) {
							return Ok(value);
						}
					}
					Err(de::Error::invalid_value(de::Unexpected::Unsigned(v), &"a valid Register variant value"))
				}
			}
			deserializer.deserialize_u8(Visitor { marker: PhantomData::<EnumType>, lifetime: PhantomData })
		}
	}
};
// GENERATOR-END: Register

impl Register {
	#[allow(clippy::unwrap_used)]
	pub(crate) fn from_u8(value: u8) -> Register {
		const _: () = assert!(IcedConstants::REGISTER_ENUM_COUNT >= 0x100);
		// This can't panic since all `u8` values are Register variants
		Register::try_from(value as usize).unwrap()
	}
}

impl Register {
	#[must_use]
	fn add(self, rhs: u32) -> Self {
		let result = (self as u32).wrapping_add(rhs);
		iced_assert!(result < IcedConstants::REGISTER_ENUM_COUNT as u32);
		// SAFETY: verified by the assert above. All values 0-max are valid enum values
		unsafe { mem::transmute(result as RegisterUnderlyingType) }
	}
	#[must_use]
	fn sub(self, rhs: u32) -> Self {
		let result = (self as u32).wrapping_sub(rhs);
		iced_assert!(result < IcedConstants::REGISTER_ENUM_COUNT as u32);
		// SAFETY: verified by the assert above. All values 0-max are valid enum values
		unsafe { mem::transmute(result as RegisterUnderlyingType) }
	}
}
// i32 + Register
impl Add<Register> for i32 {
	type Output = Register;

	#[must_use]
	#[inline]
	fn add(self, rhs: Register) -> Self::Output {
		rhs.add(self as u32)
	}
}
// u32 + Register
impl Add<Register> for u32 {
	type Output = Register;

	#[must_use]
	#[inline]
	fn add(self, rhs: Register) -> Self::Output {
		rhs.add(self)
	}
}
// Register + i32
impl Add<i32> for Register {
	type Output = Self;

	#[must_use]
	#[inline]
	fn add(self, rhs: i32) -> Self::Output {
		self.add(rhs as u32)
	}
}
// Register + u32
impl Add<u32> for Register {
	type Output = Self;

	#[must_use]
	#[inline]
	fn add(self, rhs: u32) -> Self::Output {
		self.add(rhs)
	}
}
// Register += i32
impl AddAssign<i32> for Register {
	#[inline]
	fn add_assign(&mut self, rhs: i32) {
		*self = self.add(rhs as u32)
	}
}
// Register += u32
impl AddAssign<u32> for Register {
	#[inline]
	fn add_assign(&mut self, rhs: u32) {
		*self = self.add(rhs)
	}
}
// Register - i32
impl Sub<i32> for Register {
	type Output = Self;

	#[must_use]
	#[inline]
	fn sub(self, rhs: i32) -> Self::Output {
		self.sub(rhs as u32)
	}
}
// Register - u32
impl Sub<u32> for Register {
	type Output = Self;

	#[must_use]
	#[inline]
	fn sub(self, rhs: u32) -> Self::Output {
		self.sub(rhs)
	}
}
// Register -= i32
impl SubAssign<i32> for Register {
	#[inline]
	fn sub_assign(&mut self, rhs: i32) {
		*self = self.sub(rhs as u32)
	}
}
// Register -= u32
impl SubAssign<u32> for Register {
	#[inline]
	fn sub_assign(&mut self, rhs: u32) {
		*self = self.sub(rhs)
	}
}
