#!/usr/bin/env python3
from __future__ import annotations

import contextlib
import json
import hashlib
import pathlib
import urllib.request


TEMPLATE = '''\
# This file is generated by devscripts/update_ejs.py. DO NOT MODIFY!

VERSION = {version!r}
HASHES = {{
{hash_mapping}
}}
'''
PREFIX = '    "yt-dlp-ejs=='
BASE_PATH = pathlib.Path(__file__).parent.parent
PYPROJECT_PATH = BASE_PATH / 'pyproject.toml'
PACKAGE_PATH = BASE_PATH / 'yt_dlp/extractor/youtube/jsc/_builtin/vendor'
RELEASE_URL = 'https://api.github.com/repos/yt-dlp/ejs/releases/latest'
ASSETS = {
    'yt.solver.lib.js': False,
    'yt.solver.lib.min.js': False,
    'yt.solver.deno.lib.js': True,
    'yt.solver.bun.lib.js': True,
    'yt.solver.core.min.js': False,
    'yt.solver.core.js': True,
}


def request(url: str):
    return contextlib.closing(urllib.request.urlopen(url))


def main():
    current_version = None
    with PYPROJECT_PATH.open() as file:
        for line in file:
            if not line.startswith(PREFIX):
                continue
            current_version, _, _ = line.removeprefix(PREFIX).partition('"')

    if not current_version:
        print('yt-dlp-ejs dependency line could not be found')
        return

    with request(RELEASE_URL) as resp:
        info = json.load(resp)

    version = info['tag_name']
    if version == current_version:
        print(f'yt-dlp-ejs is up to date! ({version})')
        return

    print(f'Updating yt-dlp-ejs from {current_version} to {version}')
    hashes = []
    for asset in info['assets']:
        name = asset['name']
        if name not in ASSETS:
            continue
        with request(asset['browser_download_url']) as resp:
            data = resp.read()

        # verify digest from github
        digest = asset['digest']
        algo, _, expected = digest.partition(':')
        hexdigest = hashlib.new(algo, data).hexdigest()
        assert hexdigest == expected, f'downloaded attest mismatch ({hexdigest!r} != {expected!r})'

        # calculate sha3-512 digest
        asset_hash = hashlib.sha3_512(data).hexdigest()
        hashes.append(f'    {name!r}: {asset_hash!r},')

        if ASSETS[name]:
            (PACKAGE_PATH / name).write_bytes(data)

    (PACKAGE_PATH / '_info.py').write_text(TEMPLATE.format(
        version=version,
        hash_mapping='\n'.join(hashes),
    ))

    content = PYPROJECT_PATH.read_text()
    updated = content.replace(PREFIX + current_version, PREFIX + version)
    PYPROJECT_PATH.write_text(updated)


if __name__ == '__main__':
    main()
