/*
 * Decompiled with CFR 0.152.
 */
package jdplus.toolkit.base.core.arima;

import jdplus.toolkit.base.api.dstats.ContinuousDistribution;
import jdplus.toolkit.base.api.dstats.RandomNumberGenerator;
import jdplus.toolkit.base.core.arima.ArimaException;
import jdplus.toolkit.base.core.arima.AutoCovarianceFunction;
import jdplus.toolkit.base.core.arima.IArimaModel;
import jdplus.toolkit.base.core.arima.StationaryTransformation;
import jdplus.toolkit.base.core.data.DataBlock;
import jdplus.toolkit.base.core.dstats.Normal;
import jdplus.toolkit.base.core.math.linearfilters.BackFilter;
import jdplus.toolkit.base.core.math.linearfilters.RationalBackFilter;
import jdplus.toolkit.base.core.math.matrices.FastMatrix;
import jdplus.toolkit.base.core.math.matrices.LowerTriangularMatrix;
import jdplus.toolkit.base.core.math.matrices.SymmetricMatrix;
import jdplus.toolkit.base.core.math.polynomials.Polynomial;
import jdplus.toolkit.base.core.random.XorshiftRNG;
import lombok.NonNull;

public final class ArimaSeriesGenerator {
    private final int initialdrop;
    private final double startMean;
    private final double startStdev;
    private final RandomNumberGenerator rng;
    private final ContinuousDistribution distribution;

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(@NonNull RandomNumberGenerator rng) {
        if (rng == null) {
            throw new NullPointerException("rng is marked non-null but is null");
        }
        return new Builder(rng);
    }

    public ArimaSeriesGenerator() {
        this(new Builder());
    }

    private ArimaSeriesGenerator(Builder builder) {
        this.initialdrop = builder.ndrop;
        this.startMean = builder.startMean;
        this.startStdev = builder.startStdev;
        this.rng = builder.rng.synchronize();
        this.distribution = builder.dist;
    }

    public double[] generate(IArimaModel arima, int n) {
        return this.generate(arima, 0.0, n);
    }

    public double[] generate(IArimaModel arima, double mean, int n) {
        try {
            double[] w;
            StationaryTransformation stm = arima.stationaryTransformation();
            double[] tmp = this.generateStationary((IArimaModel)stm.getStationaryModel(), mean, n + this.initialdrop);
            if (this.initialdrop == 0) {
                w = tmp;
            } else {
                w = new double[n];
                System.arraycopy(tmp, this.initialdrop, w, 0, n);
            }
            if (stm.getUnitRoots().isIdentity()) {
                return w;
            }
            Polynomial P = stm.getUnitRoots().asPolynomial();
            double[] yprev = new double[P.degree()];
            if (this.startStdev != 0.0) {
                Normal normal = new Normal(this.startMean, this.startStdev);
                for (int i = 0; i < yprev.length; ++i) {
                    yprev[i] = normal.random(this.rng);
                }
            } else if (this.startMean != 0.0) {
                for (int i = 0; i < yprev.length; ++i) {
                    yprev[i] = this.startMean;
                }
            }
            for (int i = 0; i < n; ++i) {
                int j;
                double y = w[i];
                for (j = 1; j <= P.degree(); ++j) {
                    y -= yprev[j - 1] * P.get(j);
                }
                w[i] = y;
                for (j = yprev.length - 1; j > 0; --j) {
                    yprev[j] = yprev[j - 1];
                }
                if (yprev.length <= 0) continue;
                yprev[0] = y;
            }
            return w;
        }
        catch (ArimaException ex) {
            return null;
        }
    }

    public double[] generateStationary(IArimaModel starima, int n) {
        return this.generateStationary(starima, 0.0, n);
    }

    public double[] generateStationary(IArimaModel starima, double mean, int n) {
        BackFilter ar = starima.getAr();
        BackFilter ma = starima.getMa();
        int p = ar.getDegree();
        int q = ma.getDegree();
        double[] y = new double[p];
        double[] e = new double[q];
        if (p == 0) {
            for (int i = 0; i < q; ++i) {
                e[i] = this.distribution.random(this.rng);
            }
        } else {
            FastMatrix ac = FastMatrix.square(p + q);
            AutoCovarianceFunction acf = starima.getAutoCovarianceFunction();
            acf.prepare(p);
            FastMatrix pm = ac.extract(0, p, 0, p);
            pm.diagonal().set(acf.get(0));
            for (int i = 1; i < p; ++i) {
                pm.subDiagonal(-i).set(acf.get(i));
            }
            if (q > 0) {
                FastMatrix qm = ac.extract(p, q, p, q);
                qm.diagonal().set(starima.getInnovationVariance());
                FastMatrix qp = ac.extract(p, q, 0, p);
                RationalBackFilter psi = starima.getPsiWeights();
                int nw = Math.min(q, p);
                psi.prepare(q);
                DataBlock w = DataBlock.of(psi.getWeights(q));
                for (int i = 0; i < nw; ++i) {
                    qp.column(i).drop(i, 0).copy(w.drop(0, i));
                }
                qp.mul(starima.getInnovationVariance());
            }
            SymmetricMatrix.fromLower(ac);
            SymmetricMatrix.lcholesky(ac, 1.0E-6);
            double[] x = new double[p + q];
            for (int i = 0; i < x.length; ++i) {
                x[i] = this.distribution.random(this.rng);
            }
            LowerTriangularMatrix.Lx(ac, DataBlock.of(x));
            System.arraycopy(x, 0, y, 0, p);
            if (q > 0) {
                System.arraycopy(x, p, e, 0, q);
            }
        }
        double[] z = new double[n];
        double std = Math.sqrt(starima.getInnovationVariance());
        Polynomial theta = ma.asPolynomial();
        Polynomial phi = ar.asPolynomial();
        for (int i = 0; i < n; ++i) {
            int j;
            double u = this.distribution.random(this.rng) * std;
            double t = mean + u * theta.get(0);
            for (j = 1; j <= q; ++j) {
                t += e[j - 1] * theta.get(j);
            }
            for (j = 1; j <= p; ++j) {
                t -= y[j - 1] * phi.get(j);
            }
            z[i] = t /= phi.get(0);
            if (e.length > 0) {
                for (j = e.length - 1; j > 0; --j) {
                    e[j] = e[j - 1];
                }
                e[0] = u;
            }
            if (y.length <= 0) continue;
            for (j = y.length - 1; j > 0; --j) {
                y[j] = y[j - 1];
            }
            y[0] = t;
        }
        return z;
    }

    public static double[] generate(IArimaModel model, int n, double[] initial, ContinuousDistribution distribution, int warmup) {
        Polynomial phi = model.getAr().asPolynomial();
        int p = phi.degree();
        if (initial.length < p) {
            throw new IllegalArgumentException();
        }
        Polynomial theta = model.getMa().asPolynomial();
        int q = theta.degree();
        XorshiftRNG rng = XorshiftRNG.fromSystemNanoTime();
        double[] z = new double[n];
        double[] e = new double[q];
        double[] y = new double[p];
        int i = 0;
        int j = initial.length - 1;
        while (i < p) {
            y[i] = initial[j];
            ++i;
            --j;
        }
        for (i = 0; i < n + warmup; ++i) {
            int j2;
            double u = distribution.random((RandomNumberGenerator)rng);
            double t = u * theta.get(0);
            for (j2 = 1; j2 <= q; ++j2) {
                t += e[j2 - 1] * theta.get(j2);
            }
            for (j2 = 1; j2 <= p; ++j2) {
                t -= y[j2 - 1] * phi.get(j2);
            }
            t /= phi.get(0);
            if (q > 0) {
                for (j2 = q - 1; j2 > 0; --j2) {
                    e[j2] = e[j2 - 1];
                }
                e[0] = u;
            }
            if (p > 0) {
                for (j2 = p - 1; j2 > 0; --j2) {
                    y[j2] = y[j2 - 1];
                }
                y[0] = t;
            }
            if (i < warmup) continue;
            z[i - warmup] = t;
        }
        return z;
    }

    public static class Builder {
        private int ndrop = 0;
        private double startMean = 100.0;
        private double startStdev = 10.0;
        private final RandomNumberGenerator rng;
        private ContinuousDistribution dist = new Normal();

        private Builder() {
            this.rng = XorshiftRNG.fromSystemNanoTime();
        }

        private Builder(RandomNumberGenerator rng) {
            this.rng = rng;
        }

        public Builder initialWarmUp(int n) {
            this.ndrop = n;
            return this;
        }

        public Builder distribution(ContinuousDistribution distribution) {
            this.dist = distribution;
            return this;
        }

        public Builder startMean(double mean) {
            this.startMean = mean;
            return this;
        }

        public Builder startStdev(double stdev) {
            this.startStdev = stdev;
            return this;
        }

        public ArimaSeriesGenerator build() {
            return new ArimaSeriesGenerator(this);
        }
    }
}

