## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup--------------------------------------------------------------------
library(rLifting)

if (!requireNamespace("dplyr", quietly = TRUE) ||
    !requireNamespace("ggplot2", quietly = TRUE) ||
    !requireNamespace("knitr", quietly = TRUE)) {
  knitr::opts_chunk$set(eval = FALSE)
  message("Required packages 'dplyr', 'ggplot2' or 'knitr' are missing. Vignette code will not run.")
} else {
  library(ggplot2)
  library(dplyr)
  library(knitr)
}

data("benchmark_offline", package = "rLifting")

## ----speed_table--------------------------------------------------------------
df_speed = benchmark_offline |>
  filter(!is.na(Time)) |>
  group_by(Pkg) |>
  summarise(
    Median_ms = median(Time) * 1000,
    Mean_ms   = mean(Time)   * 1000,
    .groups = "drop"
  ) |>
  arrange(Median_ms) |>
  mutate(
    Paradigm = case_when(
      Pkg == "wavethresh" ~ "DWT",
      Pkg %in% c("adlift", "nlt") ~ "Adaptive Lifting",
      TRUE ~ "Lifting"
    )
  ) |>
  select(Pkg, Paradigm, Median_ms, Mean_ms)

kable(df_speed,
      col.names = c("Package", "Paradigm", "Median (ms)", "Mean (ms)"),
      digits = 2,
      caption = "Execution time for the full denoising pipeline (1024 points, Haar).")

## ----mse_plot, fig.width=7, fig.height=4.5------------------------------------
df_mse = benchmark_offline |> filter(!is.na(MSE))

ggplot(df_mse, aes(x = reorder(Pkg, MSE), y = MSE, fill = Pkg)) +
  geom_boxplot() +
  labs(
    title = "Reconstruction error (MSE)",
    subtitle = "Doppler signal, n = 1024, \u03c3 = 0.3 (lower is better)",
    x = "Package",
    y = "MSE"
  ) +
  theme_minimal() +
  theme(legend.position = "none")

