## ----echo=FALSE, message=FALSE, warning=FALSE---------------------------------
# Ensure the temporary library from R CMD check is visible (esp. on Windows)
libdir <- Sys.getenv("R_LIBS")
if (nzchar(libdir)) {
  parts <- strsplit(libdir, .Platform$path.sep, fixed = TRUE)[[1]]
  .libPaths(unique(c(parts, .libPaths())))
}

# now load your package
suppressPackageStartupMessages(library(ecotourism))


## ----echo=TRUE, eval=TRUE, message=FALSE, warning=FALSE-----------------------
library(dplyr)
library(ecotourism)
data("manta_rays")
manta_rays |> glimpse()


## ----echo=TRUE, fig.width=6, fig.height=4, eval=TRUE--------------------------
library(ggplot2)
library(ggthemes)

manta_rays |> 
  ggplot() +
    geom_sf(data = oz_lga) +
    geom_point(aes(x = obs_lon, y = obs_lat), color = "red") +
    theme_map()


## ----echo=TRUE, fig.width=6, fig.height=4, eval=TRUE--------------------------

week_order <- c("Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Sunday")

manta_rays |> 
  ggplot(aes(x = factor(weekday, levels = week_order))) +
    geom_bar() +
    labs(x = "Weekday", y = "Number of Records") +
    theme_minimal()


## ----echo=TRUE, fig.width=6, fig.height=4, eval=TRUE, message=FALSE, warning=FALSE----
library(lubridate)
manta_rays |>
    dplyr::mutate(month =month(month, label = TRUE, abbr = TRUE)) |>
    ggplot(aes(x = factor(month))) +
    geom_bar() +
    labs(x = "Month", y = "Number of Records") +
    theme_minimal()


## ----echo=TRUE, fig.width=6, fig.height=4, eval=TRUE--------------------------
manta_rays |>
    ggplot(aes(x = factor(year))) +
    geom_bar() +
    labs(x = "Year", y = "Number of Records") +
    theme_minimal()


## ----echo=TRUE, eval=TRUE, message=FALSE, warning=FALSE, fig.width=6, fig.height=4----
library(ggbeeswarm)

# Prepare manta_rays occurrence counts per day
manta_rays_daily <- manta_rays |>
  group_by(ws_id, date) |>
  summarise(occurrence = n(), .groups = "drop")

# Join with weather data for precipitation
manta_rays_weather <- manta_rays_daily |>
  left_join(weather |> select(ws_id, date, prcp), 
            by = c("ws_id", "date"))

# Simple plot: rainy day vs manta_rays occurrence
manta_rays_weather |>
  filter(!is.na(prcp)) |>
  mutate(rain = if_else(prcp > 5, "yes", "no")) |>
  ggplot(aes(x = rain, y = occurrence)) +
  geom_quasirandom(alpha = 0.6) +
  ylim(c(0, 15)) +
  labs(
    title = "Relationship between rainy day and Manta Rays occurrence",
    x = "Rainy",
    y = "Number of Manta Rays records"
  ) +
  theme_minimal()



## ----echo=TRUE, eval=TRUE, message=FALSE, warning=FALSE, fig.width=6, fig.height=4----

manta_rays_weather <- manta_rays_daily |> 
  left_join(
    weather |> select(ws_id, date, temp, prcp),
    by = c("ws_id", "date")
  )


ggplot(manta_rays_weather, aes(temp, occurrence, color = prcp)) +
  geom_point(alpha = 0.5) +
  scale_color_viridis_c() +
  labs(
    title = "Manta Rays occurrence vs temperature, colored by precipitation",
    x = "Mean daily temperature (°C)",
    y = "Occurrences",
    color = "Precipitation (mm)"
  ) +
  theme_minimal()



