% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gpPolar.R, R/gpPolarTemp.R
\name{gpPolar}
\alias{gpPolar}
\alias{gpPolarHigh}
\title{Bayesian single-index regression with Gaussian process link and one-to-one polar transformation}
\usage{
gpPolar(
  x,
  y,
  prior = list(index = list(psi = list(alpha = NULL)), link = list(kappa = list(min_kappa
    = 0.5, max_kappa = 4, grid.width = 0.1)), sigma2 = list(shape = 2, rate = 0.01)),
  init = list(index = list(psi = NULL), link = list(kappa = 2), sigma2 = 0.01),
  sampling = TRUE,
  fitted = TRUE,
  monitors2 = NULL,
  niter = 10000,
  nburnin = 1000,
  thin = 1,
  thin2 = NULL,
  nchain = 1,
  setSeed = FALSE
)

gpPolarHigh(
  x,
  y,
  prior = list(index = list(psi = list(alpha = NULL)), link = list(kappa = list(min_kappa
    = 0.5, max_kappa = 4, grid.width = 0.1)), sigma2 = list(shape = 2, rate = 0.01)),
  init = list(index = list(psi = NULL), link = list(kappa = 2), sigma2 = 0.01),
  sampling = TRUE,
  fitted = TRUE,
  monitors2 = NULL,
  niter = 10000,
  nburnin = 1000,
  thin = 1,
  thin2 = NULL,
  nchain = 1,
  setSeed = FALSE
)
}
\arguments{
\item{x}{Numeric data.frame/matrix of predictors. Each row is an observation.}

\item{y}{Numeric response numeric vector/matrix. Other types  are not available.}

\item{prior}{Optional named list of prior settings with sublists:
\describe{
\item{\code{index}}{ \code{psi} is polar angle and rescaled Beta distribution on \eqn{[0, \pi]} is assigned.
Only shape parameter \code{alpha} of \eqn{p-1} dimension vector is needed since rate parameters is computed to satisfy \eqn{\theta_{j, \text{MAP}}}.
By default, the shape parameter for each element of the polar vector is set to \code{5000}.}

\item{\code{link}}{{Prior for the smoothness parameter \code{kappa} in the Gaussian process kernel: Prior for \eqn{\kappa} is discrete uniform of equally spaced grid points
in \eqn{[\kappa_{\text{min}}, \kappa_{\text{max}}}].
\code{min_kappa} is minimum value of kappa (default \code{0.5}), \code{max_kappa} is maximum value of kappa (default \code{4}),
and \code{grid.width} is space (default \code{0.1}).}}
\item{\code{sigma2}}{Error-variance prior hyperparameters. An Inverse-Gamma prior is assigned to \eqn{\sigma^2}
where \code{shape} is shape parameter and \code{rate} is rate parameter of inverse gamma distribution.
(default \code{shape = 2, rate = 0.01})}

}}

\item{init}{Optional named list of initial values. If the values are not assigned, they are randomly sampled from prior.
\describe{
\item{\code{index}}{Initial vector of polar angle \code{psi} with \eqn{p-1} dimension. Each element of angle is between 0 and \eqn{\pi}.}
\item{\code{link}}{Initial scalar scale parameter of covariance kernel \code{kappa}. (default: \code{2})}
\item{\code{sigma2}}{Initial scalar error variance. (default: \code{0.01})}
}}

\item{sampling}{Logical. If \code{TRUE} (default), run MCMC; otherwise return prepared nimble model objects without sampling.}

\item{fitted}{Logical. If \code{TRUE} (default), fitted values drawn from posterior distribution are included in the output and \code{c("linkFunction", "kappa", "Xlin")} is monitored for prediction.}

\item{monitors2}{Optional character vector of additional monitor nodes. To check the names of the nodes, set \code{fit <- gpPolar(x, y, sampling = FALSE)} and then inspect the variable names stored in the model object using \code{fit$model$getVarNames()}.}

\item{niter}{Integer. Total MCMC iterations (default \code{10000}).}

\item{nburnin}{Integer. Burn-in iterations (default \code{1000}).}

\item{thin}{Integer. Thinning for monitors1 (default \code{1}).}

\item{thin2}{Integer. Optional thinning for \code{monitors2} (default \code{1}).}

\item{nchain}{Integer. Number of MCMC chains (default \code{1}).}

\item{setSeed}{Logical or numeric argument.  Further details are provided in \link[nimble]{runMCMC}.}
}
\value{
A \code{list} typically containing:
\describe{
\item{\code{model}}{Nimble model}
\item{\code{sampler}}{Nimble sampler}
\item{\code{sampling}}{Posterior draws of \eqn{\theta}, \eqn{\sigma^2}, and nodes for fitted values by default. Variables specified in \code{monitors2} will be added if provided.}
\item{\code{fitted}}{If \code{fitted = TRUE}, in-sample fitted values is given.}
\item{\code{input}}{List of data,input values for prior and initial values, and computation time without compiling.}
}
}
\description{
Fits a single–index model \eqn{Y_i \sim \mathcal{N}(f(X_i'\theta), \sigma^2), i = 1,\cdots,n} where
the index \eqn{\theta} is specified and computed via a one-to-one polar
transformation, and the link \eqn{f(\cdot)} is represented with a Gaussian
process.
}
\details{
\strong{Model} The single–index model is specified as \eqn{Y_i = f(X_i'{\theta}) + \epsilon_i},
where the index vector \eqn{\theta} lies on the unit sphere with (\eqn{\|\theta\|_2=1}) with non-zero first component
to ensure identifiability and is parameterized via a one-to-one polar transformation with angle \eqn{\psi_1,...,\psi_{p-1}}.
Sampling is  performed on the angular parameters \eqn{\theta} defining
the index vector. The link function \eqn{f(\cdot)} is modeled by a Gaussian process
prior with zero mean and an Ornstein–Uhlenbeck (OU) covariance kernel
\eqn{\exp(-\kappa |t_i - t_j|), i, j = 1,\ldots, N}, where \eqn{\kappa} is a bandwidth (smoothness)
parameter and \eqn{t_i, t_j} is index value(\eqn{t_i = X_i'\theta}).

\strong{Priors}
\itemize{
\item Index vector: Uniform prior on the unit sphere (\eqn{\|\theta\|_2=1}).
\item Bandwidth parameter \eqn{\kappa}: discrete uniform prior over a fixed grid.
\item Error variance \eqn{\sigma^2}: Inverse–Gamma prior.

}

\strong{Sampling} For \code{gpPolar()}, \eqn{\theta} is sampled by Metropolis-Hastings and updated with \eqn{f},
\eqn{\kappa} is chosen by grid search method that maximizes likelihood,
\eqn{\sigma^2} are sampled from their full conditional
distributions using Gibbs sampling.
Since \eqn{\kappa} is sampled by grid search, more than 5 dimension of variables \code{gpPolarHigh()} is recommended.
For \code{gpPolarHigh()}, all sampling parameters' samplers are assigned by nimble.
}
\examples{
\donttest{
library(MASS)
N <- 100    # Sample Size
p <- 3
mu <- c(0,0,0)
rho <- 0
Cx <- rbind(c(1,rho,rho), c(rho,1,rho), c(rho, rho,1))
X <- mvrnorm(n = N, mu=mu, Sigma=Cx, tol=1e-8)
alpha <- c(1,1,1)
alpha <- alpha/sqrt(sum(alpha^2))
z <- matrix(0,N)
z <- X \%*\% alpha
z <- z[,1]
sigma <- 0.3
f <- exp(z)
y <- f + rnorm(N, 0, sd=sigma) # adding noise
y <- y-mean(y)
f_all <- f
x_all <- X
y_all <- y
data_frame <- cbind(x_all, y, f)
colnames(data_frame) = c('x1', 'x2', 'x3', 'y','f')

# One-call version
fit1 <- gpPolar(X, y)
fit2 <- gpPolarHigh(X, y)

# Split version
models1 <- gpPolar(X, y, sampling = FALSE)
models2 <- gpPolarHigh(X, y, sampling = FALSE)
Ccompile1 <- compileModelAndMCMC(models1)
Ccompile2 <- compileModelAndMCMC(models2)
mcmc.out1 <- runMCMC(Ccompile1$mcmc, niter = 5000, nburnin = 1000, thin = 1,
                    nchains = 1, setSeed = TRUE, init = models1$input$init,
                    summary = TRUE, samplesAsCodaMCMC = TRUE)
mcmc.out2 <- runMCMC(Ccompile2$mcmc, niter = 5000, nburnin = 1000, thin = 1,
                    nchains = 1, setSeed = TRUE, init = models2$input$init,
                    summary = TRUE, samplesAsCodaMCMC = TRUE)
}

}
\references{
Dhara, K., Lipsitz, S., Pati, D., & Sinha, D. (2019). A new Bayesian single index model with or without covariates missing at random.
\emph{Bayesian analysis}, 15(3), 759.
}
