##-------------------------------
## Martingale type residual (MT)
##-------------------------------

resMT <- function(theta,y,X,cc,family="ST"){
  n     <- length(y)
  p     <- ncol(X)
  resm  <- c()
  resmt <- c()
  S     <- c()

  if(family=="SN")
  {
    betas  <- theta[1:p]
    sigma2 <- theta[(p+1)]
    lambda <- theta[(p+2)]

    Delta <- sqrt(sigma2)*(lambda/sqrt(lambda^2 + 1))
    mu    <- X%*%betas - sqrt(2/pi)*Delta
    S     <- 1 - cdfSNI(y, mu, sigma2, lambda, nu = NULL, type = family)
  }
  if(family=="ST")
  {
    betas  <- theta[1:p]
    sigma2 <- theta[(p+1)]
    lambda <- theta[(p+2)]
    nu     <- theta[(p+3)]

    Delta  <- sqrt(sigma2)*(lambda/sqrt(lambda^2 + 1))
    k1     <- sqrt(nu/2)*(gamma((nu-1)/2)/gamma(nu/2))
    mu     <- X%*%betas - sqrt(2/pi)*k1*Delta
    S      <- 1 - cdfSNI(y, mu, sigma2, lambda, nu, type = family)
  }
  if(family=="SCN")
  {
    betas  <- theta[1:p]
    sigma2 <- theta[(p+1)]
    lambda <- theta[(p+2)]
    nu1    <- theta[(p+3)]
    nu2    <- theta[(p+3)]
    nu     <- c(nu1, nu2)

    Delta  <- sqrt(sigma2)*(lambda/sqrt(lambda^2 + 1))
    k1     <- (nu1/(nu2^(1/2))) + 1-nu1
    mu     <- X%*%betas - sqrt(2/pi)*k1*Delta
    S      <- 1 - AcumSCN(y, mu, sigma2, lambda, nu)
  }

  for (i in 1:n)
  {
    resm[i]  <- 1 - cc[i] + log(S[i])
    resmt[i] <- sqrt(-2*((1 - cc[i])*log(1 - cc[i] - resm[i]) + resm[i]))*sign(resm[i])
  }
  if (any(is.na(resmt))){
    nas <- which(is.na(resmt))
    logg <- c()
    logg[nas] <- log(1 - cc[nas] - resm[nas])
    if(any(is.infinite(logg))){
      naas <- which(is.infinite(logg))
      logg[naas] <- -.Machine$double.xmin
    }
    resmt[nas] <- sqrt(-2*((1 - cc[nas])*logg[nas] + resm[nas]))*sign(resm[nas])
  }

  return(list(resm=resm , resmt=resmt))
}


############### -----------------------------------------------------------------
############### Envelopes of the MT residuals for the SMSN-CR models with ggplot2
############### -----------------------------------------------------------------

EnvelopeRMT_ggp <- function(theta, y, X, cc, type = "ST", family = "ST") {
  res <- resMT(theta, y, X, cc, family = family)

  title <- switch(type,
                  "SN" = "SN-CR",
                  "ST" = "ST-CR",
                  "SCN" = "SCN-CR",
                  "N" = "N-CR",
                  "T" = "T-CR",
                  "CN" = "CN-CR",
                  "ICR"
  )

  envelope <- 0.95
  x <- res$resmt
  good <- !is.na(x)
  ord <- order(x[good])
  ord.x <- x[good][ord]

  n <- length(ord.x)
  P <- ppoints(n)
  z <- qnorm(P)

  Q.x <- quantile(ord.x, c(0.25, 0.75))
  Q.z <- qnorm(c(0.25, 0.75))
  b <- (Q.x[2] - Q.x[1]) / (Q.z[2] - Q.z[1])
  a <- Q.x[1] - b * Q.z[1]
  zz <- qnorm(1 - (1 - envelope) / 2)
  SE <- b / dnorm(z) * sqrt(P * (1 - P) / n)

  fit.value <- a + b * z
  upper <- fit.value + zz * SE
  lower <- fit.value - zz * SE

  p <- ggplot2::ggplot(data = data.frame(z = z, ord.x = ord.x), ggplot2::aes(x = z, y = ord.x)) +
    ggplot2::scale_color_manual(values = c("#00A1D5FF", "#79AF97FF")) +
    ggplot2::geom_point(colour = "#00A1D5FF", shape = 19, size = 3) +
    ggplot2::geom_abline(intercept = a, slope = b, col = "black", linewidth = 1) +
    ggplot2::geom_ribbon(ggplot2::aes(ymin = lower, ymax = upper), fill = "#A6BDDB", alpha = 0.5) +
    ggplot2::geom_line(ggplot2::aes(y = upper), linetype = "solid", linewidth = 0.5, col = "#A6BDDB") +
    ggplot2::geom_line(ggplot2::aes(y = lower), linetype = "solid", linewidth = 0.5, col = "#A6BDDB") +
    ggplot2::labs(x = "Standard normal quantile", y = "Martingale-type residuals", title = title) +
    ggplot2::theme(
      plot.title = ggplot2::element_text(hjust = 0.5, size = 16),
      legend.box.margin = ggplot2::margin(0, 0, 0, 0),
      panel.border = ggplot2::element_rect(color = "black", fill = NA),
      panel.grid.major = ggplot2::element_line(linewidth = 1.1),
      panel.background = ggplot2::element_rect(fill = "transparent"),
      panel.grid.minor = ggplot2::element_line(linewidth = 0.4),
      axis.title = ggplot2::element_text(size = 13.5),
      legend.text = ggplot2::element_text(size = 13.5),
      legend.title = ggplot2::element_text(size = 12.5),
      axis.text = ggplot2::element_text(size = 12.5),
      axis.text.y = ggplot2::element_text(size = 12.5, angle = 90, hjust = 0.5)
    ) +
    ggplot2::guides(color = ggplot2::guide_legend(title = NULL)) +
    ggplot2::theme(panel.grid = ggplot2::element_line(color = "grey89"))

  return(p)
}

