# JapanAPIs - Access Japanese Data via Public APIs and Curated Datasets
# Version 0.1.1
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_country_info_jp

library(testthat)


test_that("get_country_info_jp() returns a tibble with correct structure", {
  skip_on_cran()
  skip_if_offline()

  result <- get_country_info_jp()

  # Skip test if API is unavailable
  skip_if(is.null(result), "API is unavailable")

  # Check that result is a tibble
  expect_s3_class(result, "tbl_df")

  # Check number of columns
  expect_equal(ncol(result), 8)

  # Check column names
  expected_names <- c("name_common", "name_official", "region", "subregion",
                      "capital", "area", "population", "languages")
  expect_named(result, expected_names)
})

test_that("get_country_info_jp() returns correct data types", {
  skip_on_cran()
  skip_if_offline()

  result <- get_country_info_jp()
  skip_if(is.null(result), "API is unavailable")

  # Check column types
  expect_type(result$name_common, "character")
  expect_type(result$name_official, "character")
  expect_type(result$region, "character")
  expect_type(result$subregion, "character")
  expect_type(result$capital, "character")
  expect_type(result$area, "double")
  expect_type(result$population, "integer")
  expect_type(result$languages, "character")
})

test_that("get_country_info_jp() returns expected values for Japan", {
  skip_on_cran()
  skip_if_offline()

  result <- get_country_info_jp()
  skip_if(is.null(result), "API is unavailable")

  # Check that we get data about Japan
  expect_equal(result$name_common, "Japan")
  expect_equal(result$region, "Asia")
  expect_equal(result$subregion, "Eastern Asia")
  expect_equal(result$capital, "Tokyo")

  # Check that numeric values are reasonable
  expect_gt(result$area, 0)
  expect_gt(result$population, 0)

  # Check that languages field is not empty
  expect_true(nchar(result$languages) > 0)
})

test_that("get_country_info_jp() returns a single row", {
  skip_on_cran()
  skip_if_offline()

  result <- get_country_info_jp()
  skip_if(is.null(result), "API is unavailable")

  # Should return exactly one row for Japan
  expect_equal(nrow(result), 1)
})

test_that("get_country_info_jp() handles API errors gracefully", {
  skip_on_cran()

  # This test just ensures the function doesn't crash
  # Result can be NULL or a tibble, both are acceptable
  expect_no_error({
    result <- get_country_info_jp()
    expect_true(is.null(result) || inherits(result, "tbl_df"))
  })
})

test_that("get_country_info_jp() returns NULL or tibble (never crashes)", {
  skip_on_cran()

  result <- get_country_info_jp()

  # Result must be either NULL or a tibble
  expect_true(is.null(result) || is.data.frame(result))

  # If it's a tibble, it should have the correct structure
  if (!is.null(result)) {
    expect_s3_class(result, "tbl_df")
    expect_equal(ncol(result), 8)
  }
})
