% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/priors.R
\name{cbc_priors}
\alias{cbc_priors}
\title{Create prior specifications for CBC models}
\usage{
cbc_priors(
  profiles,
  no_choice = NULL,
  n_draws = 100,
  draw_type = "halton",
  interactions = NULL,
  ...
)
}
\arguments{
\item{profiles}{A data frame of profiles created by \code{cbc_profiles()}}

\item{no_choice}{Prior specification for no-choice alternative. Can be:
\itemize{
\item A single numeric value for fixed no-choice utility
\item A \code{rand_spec()} object for random no-choice utility
\item \code{NULL} if no no-choice option (default)
}}

\item{n_draws}{Number of draws for DB-error calculation if using Bayesian
priors. Defaults to \code{100}}

\item{draw_type}{Specify the draw type as a character: \code{"halton"}
(the default) or \code{"sobol"} (recommended for models with more than 5
random parameters).}

\item{interactions}{A list of interaction specifications created by \code{int_spec()}.
Only interactions between fixed (non-random) parameters are supported.
Each interaction must specify the appropriate level(s) for categorical variables.
Defaults to \code{NULL} (no interactions).}

\item{...}{Named arguments specifying priors for each attribute:
\itemize{
\item For fixed parameters:
\itemize{
\item Continuous variables: provide a single numeric value
\item Categorical variables: provide either:
\itemize{
\item An unnamed vector of values one less than the number of levels (dummy coding)
\item A named vector mapping levels to coefficients (remaining level becomes reference)
}
}
\item For random parameters: use \code{rand_spec()} to specify distribution, parameters, and correlations
}}
}
\value{
A structured prior specification object including parameter draws for
random coefficients and interaction terms. This object contains:
\itemize{
\item \code{pars}: Vector of mean parameter values
\item \code{par_draws}: Matrix of parameter draws (if random parameters specified)
\item \code{correlation}: Correlation matrix for random parameters (if applicable)
\item \code{interactions}: List of interaction specifications
\item \code{attrs}: Detailed attribute information
\item Additional metadata for validation and compatibility checking
}
}
\description{
Creates a standardized prior specification object for use in CBC analysis
functions like \code{cbc_choices()} and \code{cbc_design()}. Supports both fixed and random
parameters, with flexible specification of categorical variable levels and
interaction terms between fixed parameters.
}
\details{
\subsection{Fixed vs Random Parameters}{

\strong{Fixed parameters} assume all respondents have the same preference coefficients.
Specify these as simple numeric values.

\strong{Random parameters} assume preference coefficients vary across respondents
according to a specified distribution. Use \code{rand_spec()} to define the
distribution type, mean, and standard deviation.
}

\subsection{Categorical Variable Specification}{

For categorical variables, you can specify priors in two ways:
\enumerate{
\item \strong{Unnamed vector}: Provide coefficients for all levels except the first
(which becomes the reference level). Order matters and should match the
natural order of levels.
\item \strong{Named vector}: Explicitly map coefficient values to specific levels.
Any level not specified becomes the reference level.
}
}

\subsection{Interaction Terms}{

Use the \code{interactions} parameter with \code{int_spec()} to include interaction
effects between attributes. Only interactions between fixed parameters are
supported. For categorical variables involved in interactions, you must
specify the relevant levels.
}

\subsection{No-Choice Options}{

When including a no-choice alternative, provide a \code{no_choice} parameter.
This can be either a fixed numeric value or a \code{rand_spec()} for random
no-choice utility.
}
}
\examples{
library(cbcTools)

# Create profiles for examples
profiles <- cbc_profiles(
  price = c(1, 1.5, 2, 2.5, 3),
  type = c('Fuji', 'Gala', 'Honeycrisp'),
  freshness = c('Poor', 'Average', 'Excellent')
)

# Example 1: Simple fixed priors
priors_fixed <- cbc_priors(
  profiles = profiles,
  price = -0.25, # Negative = prefer lower prices
  type = c(0.5, 1.0), # "Fuji" is reference level
  freshness = c(0.6, 1.2) # "Poor" reference level
)

# Example 2: Named categorical priors (more explicit)
priors_named <- cbc_priors(
  profiles = profiles,
  price = -0.25,
  type = c("Gala" = 0.5, "Honeycrisp" = 1.0),  # "Fuji" is reference
  freshness = c("Average" = 0.6, "Excellent" = 1.2)  # "Poor" is reference
)

# Example 3: Random parameters - normal distributions for "price" and "freshness"
priors_random <- cbc_priors(
  profiles = profiles,
  price = rand_spec(
    dist = "n",
    mean = -0.25,
    sd = 0.1
  ),
  type = c(0.5, 1.0),
  freshness = rand_spec(
    dist = "n",
    mean = c(0.6, 1.2),
    sd = c(0.1, 0.1)
  )
)

# Example 4: Correlated random parameters
priors_correlated <- cbc_priors(
  profiles = profiles,
  price = rand_spec(
    dist = "n",
    mean = -0.1,
    sd = 0.05,
    correlations = list(
      cor_spec(
        with = "type",
        with_level = "Honeycrisp",
        value = 0.3
      )
    )
  ),
  type = rand_spec(
    dist = "n",
    mean = c("Gala" = 0.1, "Honeycrisp" = 0.2),
    sd = c("Gala" = 0.05, "Honeycrisp" = 0.1)
  ),
  freshness = c(0.1, 0.2)
)

# Example 5: With interaction terms
priors_interactions <- cbc_priors(
  profiles = profiles,
  price = -0.25,
  type = c("Fuji" = 0.5, "Honeycrisp" = 1.0),
  freshness = c("Average" = 0.6, "Excellent" = 1.2),
  interactions = list(
    # Price sensitivity varies by apple type
    int_spec(
      between = c("price", "type"),
      with_level = "Fuji",
      value = 0.1
    ),
    int_spec(
      between = c("price", "type"),
      with_level = "Honeycrisp",
      value = 0.2
    ),
    # Type preferences vary by freshness
    int_spec(
      between = c("type", "freshness"),
      level = "Honeycrisp",
      with_level = "Excellent",
      value = 0.3
    )
  )
)

# Example 6: Including no-choice option
priors_nochoice_fixed <- cbc_priors(
  profiles = profiles,
  price = -0.25,
  type = c(0.5, 1.0),
  freshness = c(0.6, 1.2),
  no_choice = -0.5 # Negative values make no-choice less attractive
)

# Example 7: Random no-choice
priors_nochoice_random <- cbc_priors(
  profiles = profiles,
  price = -0.25,
  type = c(0.5, 1.0),
  freshness = c(0.6, 1.2),
  no_choice = rand_spec(dist = "n", mean = -0.5, sd = 0.2)
)

# View the priors
priors_fixed
priors_random
}
