% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils_api-plan.R, R/utils_api-plan-with.R,
%   R/utils_api-tweak.R
\name{plan}
\alias{plan}
\alias{with.FutureStrategyList}
\alias{tweak}
\title{Plan how to resolve a future}
\usage{
plan(
  strategy = NULL,
  ...,
  substitute = TRUE,
  .skip = FALSE,
  .call = TRUE,
  .cleanup = NA,
  .init = TRUE
)

\method{with}{FutureStrategyList}(data, expr, ..., local = FALSE, envir = parent.frame(), .cleanup = NA)

tweak(strategy, ..., penvir = parent.frame())
}
\arguments{
\item{strategy}{A future backend or the name of one.}

\item{substitute}{If \code{TRUE}, the \code{strategy} expression is
\code{substitute()}:d, otherwise not.}

\item{.skip}{(internal) If \code{TRUE}, then attempts to set a future backend
that is the same as what is currently in use, will be skipped.}

\item{.call}{(internal) Used for recording the call to this function.}

\item{.cleanup}{(internal) Used to stop implicitly started clusters.}

\item{.init}{(internal) Used to initiate workers.}

\item{data}{The future plan to use temporarily, e.g. \code{plan(multisession)}.}

\item{expr}{The R expression to be evaluated.}

\item{local}{If TRUE, then the future plan specified by \code{data}
is applied temporarily in the calling frame. Argument \code{expr} must
not be specified if \code{local = TRUE}.}

\item{envir}{The environment where the future plan should be set and the
expression evaluated.}

\item{penvir}{The environment used when searching for a future
function by its name.}

\item{\ldots}{Additional arguments overriding the default arguments
of the evaluation function.  Which additional arguments are supported
depends on which future backend is used, e.g. several support
argument \code{workers} but not all. For details, see the individual
backends of which some are linked to below.}
}
\value{
\code{plan()} returns a the previous plan invisibly if a new future backend
is chosen, otherwise it returns the current one visibly.

The value of the expression evaluated (invisibly).

a future function.
}
\description{
This function allows \emph{the user} to plan the future, more specifically,
it specifies how \code{\link[=future]{future()}}:s are resolved,
e.g. sequentially or in parallel.
}
\details{
The default backend is \code{\link{sequential}}, but another one can be set
using \code{plan()}, e.g. \code{plan(multisession)} will launch parallel workers
running in the background, which then will be used to resolve future.
To shut down background workers launched this way, call \code{plan(sequential)}.
}
\section{Built-in evaluation strategies}{

The \pkg{future} package provides the following built-in backends:

\describe{
\item{\code{\link{sequential}}:}{
Resolves futures sequentially in the current \R process, e.g.
\code{plan(sequential)}.
}
\item{\code{\link{multisession}}:}{
Resolves futures asynchronously (in parallel) in separate
\R sessions running in the background on the same machine, e.g.
\code{plan(multisession)} and \code{plan(multisession, workers = 2)}.
}
\item{\code{\link{multicore}}:}{
Resolves futures asynchronously (in parallel) in separate
\emph{forked} \R processes running in the background on
the same machine, e.g.
\code{plan(multicore)} and \code{plan(multicore, workers = 2)}.
This backend is not supported on Windows.
}
\item{\code{\link{cluster}}:}{
Resolves futures asynchronously (in parallel) in separate
\R sessions running typically on one or more machines, e.g.
\code{plan(cluster)}, \code{plan(cluster, workers = 2)}, and
\code{plan(cluster, workers = c("n1", "n1", "n2", "server.remote.org"))}.
}
}
}

\section{Other evaluation strategies available}{


In addition to the built-in ones, additional parallel backends are
implemented in future-backend packages \pkg{future.callr} and
\pkg{future.mirai} that leverage R package \pkg{callr} and
\pkg{mirai}:

\describe{
\item{\code{callr}:}{
Similar to \code{multisession}, this resolved futures in parallel in
background \R sessions on the local machine via the \pkg{callr}
package, e.g. \code{plan(future.callr::callr)} and
\code{plan(future.callr::callr, workers = 2)}. The difference is that
each future is processed in a fresh parallel R worker, which is
automatically shut down as soon as the future is resolved.
This can help decrease the overall memory. Moreover, contrary
to \code{multisession}, \code{callr} does not rely on socket connections,
which means it is not limited by the number of connections that
\R can have open at any time.
}

\item{\code{mirai_multisession}:}{
Similar to \code{multisession}, this resolved futures in parallel in
background \R sessions on the local machine via the \pkg{mirai}
package, e.g. \code{plan(future.mirai::mirai_multisession)} and
\code{plan(future.mirai::mirai_multisession, workers = 2)}.
}

\item{\code{mirai_cluster}:}{
Similar to \code{cluster}, this resolved futures in parallel via
pre-configured \R \pkg{mirai} daemon processes, e.g.
\code{plan(future.mirai::mirai_cluster)}.
}
}

Another example is the \pkg{future.batchtools} package, which leverages
\pkg{batchtools} package, to resolve futures via high-performance compute
(HPC) job schedulers, e.g. LSF, Slurm, TORQUE/PBS, Grid Engine, and
OpenLava;

\describe{
\item{\code{batchtools_slurm}:}{
The backend resolved futures via the Slurm scheduler, e.g.
\code{plan(future.batchtools::batchtools_slurm)}.
}

\item{\code{batchtools_torque}:}{
The backend resolved futures via the TORQUE/PBS scheduler, e.g.
\code{plan(future.batchtools::batchtools_torque)}.
}

\item{\code{batchtools_sge}:}{
The backend resolved futures via the Grid Engine (SGE, AGE) scheduler,
e.g. \code{plan(future.batchtools::batchtools_sge)}.
}

\item{\code{batchtools_lsf}:}{
The backend resolved futures via the Load Sharing Facility (LSF)
scheduler, e.g. \code{plan(future.batchtools::batchtools_lsf)}.
}

\item{\code{batchtools_openlava}:}{
The backend resolved futures via the OpenLava scheduler, e.g.
\code{plan(future.batchtools::batchtools_openlava)}.
}
}
}

\section{For package developers}{


Please refrain from modifying the future backend inside your packages /
functions, i.e. do not call \code{plan()} in your code. Instead, leave
the control on what backend to use to the end user. This idea is part of
the core philosophy of the future framework---as a developer you can never
know what future backends the user have access to. Moreover, by not making
any assumptions about what backends are available, your code will also work
automatically with any new backends developed after you wrote your code.

If you think it is necessary to modify the future backend within a
function, then make sure to undo the changes when exiting the function.
This can be achieved by using \code{with(plan(...), local = TRUE)}, e.g.

\preformatted{
  my_fcn <- function(x) {
    with(plan(multisession), local = TRUE)
    y <- analyze(x)
    summarize(y)
  }
}

This is important because the end-user might have already set the future
strategy elsewhere for other purposes and will most likely not known that
calling your function will break their setup.
\emph{Remember, your package and its functions might be used in a greater
context where multiple packages and functions are involved and those might
also rely on the future framework, so it is important to avoid stepping on
others' toes.}
}

\section{Using plan() in scripts and vignettes}{


When writing scripts or vignettes that use futures, try to place any
call to \code{plan()} as far up (i.e. as early on) in the code as possible.
This will help users to quickly identify where the future plan is set up
and allow them to modify it to their computational resources.
Even better is to leave it to the user to set the \code{plan()} prior to
\code{source()}:ing the script or running the vignette.
If a \file{\link{.future.R}} exists in the current directory and / or in
the user's home directory, it is sourced when the \pkg{future} package is
\emph{loaded}. Because of this, the \file{.future.R} file provides a
convenient place for users to set the \code{plan()}.
This behavior can be controlled via an \R option---see
\link[=future.options]{future options} for more details.
}

\examples{
a <- b <- c <- NA_real_

# An sequential future
plan(sequential)
f <- future({
  a <- 7
  b <- 3
  c <- 2
  a * b * c
})
y <- value(f)
print(y)
str(list(a = a, b = b, c = c)) ## All NAs


# A sequential future with lazy evaluation
plan(sequential)
f <- future({
  a <- 7
  b <- 3
  c <- 2
  a * b * c
}, lazy = TRUE)
y <- value(f)
print(y)
str(list(a = a, b = b, c = c)) ## All NAs


# A multicore future (specified as a string)
plan("multicore")
f <- future({
  a <- 7
  b <- 3
  c <- 2
  a * b * c
})
y <- value(f)
print(y)
str(list(a = a, b = b, c = c)) ## All NAs

## Multisession futures gives an error on R CMD check on
## Windows (but not Linux or macOS) for unknown reasons.
## The same code works in package tests.
\donttest{

# A multisession future (specified via a string variable)
plan("future::multisession")
f <- future({
  a <- 7
  b <- 3
  c <- 2
  a * b * c
})
y <- value(f)
print(y)
str(list(a = a, b = b, c = c)) ## All NAs

}


## Explicitly specifying number of workers
## (default is parallelly::availableCores())
plan(multicore, workers = 2)
message("Number of parallel workers: ", nbrOfWorkers())


## Explicitly close multisession workers by switching plan
plan(sequential)
# Evaluate a future using the 'multisession' plan
with(plan(multisession, workers = 2), {
  f <- future(Sys.getpid())
  w_pid <- value(f)
})
print(c(main = Sys.getpid(), worker = w_pid))



# Evaluate a future locally using the 'multisession' plan
local({
  with(plan(multisession, workers = 2), local = TRUE)

  f <- future(Sys.getpid())
  w_pid <- value(f)
  print(c(main = Sys.getpid(), worker = w_pid))
})


}
\seealso{
Use \code{\link[=plan]{plan()}} to set a future to become the
new default strategy.
}
