% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smap.R
\name{smap2}
\alias{smap2}
\alias{smap2_vec}
\alias{smap2_lgl}
\alias{smap2_int}
\alias{smap2_dbl}
\alias{smap2_chr}
\alias{smap2_structure}
\title{Map Functions Over Two Glycan Structure Vectors}
\usage{
smap2(.x, .y, .f, ..., .parallel = FALSE)

smap2_vec(.x, .y, .f, ..., .ptype = NULL, .parallel = FALSE)

smap2_lgl(.x, .y, .f, ..., .parallel = FALSE)

smap2_int(.x, .y, .f, ..., .parallel = FALSE)

smap2_dbl(.x, .y, .f, ..., .parallel = FALSE)

smap2_chr(.x, .y, .f, ..., .parallel = FALSE)

smap2_structure(.x, .y, .f, ..., .parallel = FALSE)
}
\arguments{
\item{.x}{A glycan structure vector (glyrepr_structure).}

\item{.y}{A vector of the same length as \code{.x}, or length 1 (will be recycled).}

\item{.f}{A function that takes an igraph object (from \code{.x}) and a value (from \code{.y}) and returns a result.
Can be a function, purrr-style lambda (\code{~ .x + .y}), or a character string naming a function.}

\item{...}{Additional arguments passed to \code{.f}.}

\item{.parallel}{Logical; whether to use parallel processing. If \code{FALSE} (default),
parallel processing is disabled. Set to \code{TRUE} to enable parallel processing.
See examples in \code{\link{smap}} for how to set up and use parallel processing.}

\item{.ptype}{A prototype for the return type (for \code{smap2_vec}).}
}
\value{
\itemize{
\item \code{smap2()}: A list
\item \code{smap2_vec()}: An atomic vector of type specified by \code{.ptype}
\item \code{smap2_lgl/int/dbl/chr()}: Atomic vectors of the corresponding type
\item \code{smap2_structure()}: A new glyrepr_structure object
}
}
\description{
These functions apply a function to each unique structure combination in two glycan structure vectors,
taking advantage of hash-based deduplication to avoid redundant computation.
Similar to purrr map2 functions, but optimized for glycan structure vectors.
}
\details{
These functions only compute \code{.f} once for each unique combination of structure and corresponding
\code{.y} value, then map the results back to the original vector positions. This is much more efficient
than applying \code{.f} to each element pair individually when there are duplicate structure-value combinations.

\strong{Return Types:}
\itemize{
\item \code{smap2()}: Returns a list with the same length as \code{.x}
\item \code{smap2_vec()}: Returns an atomic vector with the same length as \code{.x}
\item \code{smap2_lgl()}: Returns a logical vector
\item \code{smap2_int()}: Returns an integer vector
\item \code{smap2_dbl()}: Returns a double vector
\item \code{smap2_chr()}: Returns a character vector
\item \code{smap2_structure()}: Returns a new glycan structure vector (\code{.f} must return igraph objects)
}
}
\examples{
# Create structure vectors with duplicates
core1 <- o_glycan_core_1()
core2 <- n_glycan_core()
structures <- glycan_structure(core1, core2, core1)  # core1 appears twice
weights <- c(1.0, 2.0, 1.0)  # corresponding weights

# Map a function that uses both structure and weight
smap2_dbl(structures, weights, function(g, w) igraph::vcount(g) * w)

# Use purrr-style lambda functions  
smap2_dbl(structures, weights, ~ igraph::vcount(.x) * .y)

# Test with recycling (single weight for all structures)
smap2_dbl(structures, 2.5, ~ igraph::vcount(.x) * .y)

# Map a function that modifies structure based on second argument
# This example adds a graph attribute instead of modifying topology
add_weight_attr <- function(g, weight) {
  igraph::set_graph_attr(g, "weight", weight)
}
weights_to_add <- c(1.5, 2.5, 1.5)
smap2_structure(structures, weights_to_add, add_weight_attr)

}
