/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Pmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5Oprivate.h"  
#include "H5Ppkg.h"      

#define H5O_CPY_OPTION_SIZE sizeof(unsigned)
#define H5O_CPY_OPTION_DEF  0
#define H5O_CPY_OPTION_ENC  H5P__encode_unsigned
#define H5O_CPY_OPTION_DEC  H5P__decode_unsigned

#define H5O_CPY_MERGE_COMM_DT_LIST_SIZE  sizeof(H5O_copy_dtype_merge_list_t *)
#define H5O_CPY_MERGE_COMM_DT_LIST_DEF   NULL
#define H5O_CPY_MERGE_COMM_DT_LIST_SET   H5P__ocpy_merge_comm_dt_list_set
#define H5O_CPY_MERGE_COMM_DT_LIST_GET   H5P__ocpy_merge_comm_dt_list_get
#define H5O_CPY_MERGE_COMM_DT_LIST_ENC   H5P__ocpy_merge_comm_dt_list_enc
#define H5O_CPY_MERGE_COMM_DT_LIST_DEC   H5P__ocpy_merge_comm_dt_list_dec
#define H5O_CPY_MERGE_COMM_DT_LIST_DEL   H5P__ocpy_merge_comm_dt_list_del
#define H5O_CPY_MERGE_COMM_DT_LIST_COPY  H5P__ocpy_merge_comm_dt_list_copy
#define H5O_CPY_MERGE_COMM_DT_LIST_CMP   H5P__ocpy_merge_comm_dt_list_cmp
#define H5O_CPY_MERGE_COMM_DT_LIST_CLOSE H5P__ocpy_merge_comm_dt_list_close

#define H5O_CPY_MCDT_SEARCH_CB_SIZE sizeof(H5O_mcdt_cb_info_t)
#define H5O_CPY_MCDT_SEARCH_CB_DEF                                                                           \
    {                                                                                                        \
        NULL, NULL                                                                                           \
    }

static H5O_copy_dtype_merge_list_t *H5P__free_merge_comm_dtype_list(H5O_copy_dtype_merge_list_t *dt_list);
static herr_t                       H5P__copy_merge_comm_dt_list(H5O_copy_dtype_merge_list_t **value);

static herr_t H5P__ocpy_reg_prop(H5P_genclass_t *pclass);

static herr_t H5P__ocpy_merge_comm_dt_list_set(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__ocpy_merge_comm_dt_list_get(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__ocpy_merge_comm_dt_list_enc(const void *value, void **_pp, size_t *size);
static herr_t H5P__ocpy_merge_comm_dt_list_dec(const void **_pp, void *value);
static herr_t H5P__ocpy_merge_comm_dt_list_del(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__ocpy_merge_comm_dt_list_copy(const char *name, size_t size, void *value);
static int    H5P__ocpy_merge_comm_dt_list_cmp(const void *value1, const void *value2, size_t size);
static herr_t H5P__ocpy_merge_comm_dt_list_close(const char *name, size_t size, void *value);

const H5P_libclass_t H5P_CLS_OCPY[1] = {{
    "object copy",        
    H5P_TYPE_OBJECT_COPY, 

    &H5P_CLS_ROOT_g,           
    &H5P_CLS_OBJECT_COPY_g,    
    &H5P_CLS_OBJECT_COPY_ID_g, 
    &H5P_LST_OBJECT_COPY_ID_g, 
    H5P__ocpy_reg_prop,        

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

static const unsigned H5O_def_ocpy_option_g = H5O_CPY_OPTION_DEF; 
static const H5O_copy_dtype_merge_list_t *H5O_def_merge_comm_dtype_list_g =
    H5O_CPY_MERGE_COMM_DT_LIST_DEF; 
static const H5O_mcdt_cb_info_t H5O_def_mcdt_cb_g =
    H5O_CPY_MCDT_SEARCH_CB_DEF; 

H5FL_DEFINE(H5O_copy_dtype_merge_list_t);

static herr_t
H5P__ocpy_reg_prop(H5P_genclass_t *pclass)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5P__register_real(pclass, H5O_CPY_OPTION_NAME, H5O_CPY_OPTION_SIZE, &H5O_def_ocpy_option_g, NULL,
                           NULL, NULL, H5O_CPY_OPTION_ENC, H5O_CPY_OPTION_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5O_CPY_MERGE_COMM_DT_LIST_NAME, H5O_CPY_MERGE_COMM_DT_LIST_SIZE,
                           &H5O_def_merge_comm_dtype_list_g, NULL, H5O_CPY_MERGE_COMM_DT_LIST_SET,
                           H5O_CPY_MERGE_COMM_DT_LIST_GET, H5O_CPY_MERGE_COMM_DT_LIST_ENC,
                           H5O_CPY_MERGE_COMM_DT_LIST_DEC, H5O_CPY_MERGE_COMM_DT_LIST_DEL,
                           H5O_CPY_MERGE_COMM_DT_LIST_COPY, H5O_CPY_MERGE_COMM_DT_LIST_CMP,
                           H5O_CPY_MERGE_COMM_DT_LIST_CLOSE) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5O_CPY_MCDT_SEARCH_CB_NAME, H5O_CPY_MCDT_SEARCH_CB_SIZE,
                           &H5O_def_mcdt_cb_g, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5O_copy_dtype_merge_list_t *
H5P__free_merge_comm_dtype_list(H5O_copy_dtype_merge_list_t *dt_list)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    while (dt_list) {
        H5O_copy_dtype_merge_list_t *tmp_node;

        tmp_node = dt_list->next;

        (void)H5MM_xfree(dt_list->path);
        (void)H5FL_FREE(H5O_copy_dtype_merge_list_t, dt_list);

        dt_list = tmp_node;
    } 

    FUNC_LEAVE_NOAPI(NULL)
} 

static herr_t
H5P__copy_merge_comm_dt_list(H5O_copy_dtype_merge_list_t **value)
{
    const H5O_copy_dtype_merge_list_t *src_dt_list;             
    H5O_copy_dtype_merge_list_t       *dst_dt_list      = NULL; 
    H5O_copy_dtype_merge_list_t       *dst_dt_list_tail = NULL,
                                *tmp_dt_list            = NULL; 
    herr_t ret_value                                    = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    src_dt_list = *value;
    while (src_dt_list) {
        
        if (NULL == (tmp_dt_list = H5FL_CALLOC(H5O_copy_dtype_merge_list_t)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed");
        if (NULL == (tmp_dt_list->path = H5MM_strdup(src_dt_list->path)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed");

        
        if (dst_dt_list_tail) {
            dst_dt_list_tail->next = tmp_dt_list;
            dst_dt_list_tail       = tmp_dt_list;
        } 
        else {
            dst_dt_list      = tmp_dt_list;
            dst_dt_list_tail = tmp_dt_list;
        } 
        tmp_dt_list = NULL;

        
        src_dt_list = src_dt_list->next;
    } 

    
    *value = dst_dt_list;

done:
    if (ret_value < 0) {
        dst_dt_list = H5P__free_merge_comm_dtype_list(dst_dt_list);
        if (tmp_dt_list) {
            tmp_dt_list->path = (char *)H5MM_xfree(tmp_dt_list->path);
            tmp_dt_list       = H5FL_FREE(H5O_copy_dtype_merge_list_t, tmp_dt_list);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__ocpy_merge_comm_dt_list_set(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                                 size_t H5_ATTR_UNUSED size, void *value)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (H5P__copy_merge_comm_dt_list((H5O_copy_dtype_merge_list_t **)value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy merge committed dtype list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__ocpy_merge_comm_dt_list_get(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                                 size_t H5_ATTR_UNUSED size, void *value)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (H5P__copy_merge_comm_dt_list((H5O_copy_dtype_merge_list_t **)value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy merge committed dtype list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__ocpy_merge_comm_dt_list_enc(const void *value, void **_pp, size_t *size)
{
    const H5O_copy_dtype_merge_list_t *const *dt_list_ptr = (const H5O_copy_dtype_merge_list_t *const *)value;
    uint8_t                                 **pp          = (uint8_t **)_pp;
    const H5O_copy_dtype_merge_list_t        *dt_list; 
    size_t                                    len;     

    FUNC_ENTER_PACKAGE_NOERR

    assert(dt_list_ptr);
    assert(size);

    
    dt_list = *dt_list_ptr;
    while (dt_list) {
        
        len = strlen(dt_list->path) + 1;

        
        if (*pp) {
            H5MM_memcpy(*(char **)pp, dt_list->path, len);
            *pp += len;
        } 

        
        *size += len;

        
        dt_list = dt_list->next;
    } 

    
    if (*pp)
        *(*pp)++ = (uint8_t)'\0';

    
    *size += 1;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__ocpy_merge_comm_dt_list_dec(const void **_pp, void *_value)
{
    H5O_copy_dtype_merge_list_t **dt_list =
        (H5O_copy_dtype_merge_list_t **)_value; 
    const uint8_t              **pp           = (const uint8_t **)_pp;
    H5O_copy_dtype_merge_list_t *dt_list_tail = NULL,
                                *tmp_dt_list  = NULL; 
    size_t len;                                       
    herr_t ret_value = SUCCEED;                       

    FUNC_ENTER_PACKAGE

    
    assert(pp);
    assert(*pp);
    assert(dt_list);

    
    *dt_list = NULL;

    
    len = strlen(*(const char **)pp);
    while (len > 0) {
        
        if (NULL == (tmp_dt_list = H5FL_CALLOC(H5O_copy_dtype_merge_list_t)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed");
        if (NULL == (tmp_dt_list->path = H5MM_strdup(*(const char **)pp)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed");
        *pp += len + 1;
        assert(len == strlen(tmp_dt_list->path));

        
        if (dt_list_tail) {
            dt_list_tail->next = tmp_dt_list;
            dt_list_tail       = tmp_dt_list;
        } 
        else {
            *dt_list     = tmp_dt_list;
            dt_list_tail = tmp_dt_list;
        } 
        tmp_dt_list = NULL;

        
        len = strlen(*(const char **)pp);
    } 

    
    *pp += 1;

done:
    if (ret_value < 0) {
        *dt_list = H5P__free_merge_comm_dtype_list(*dt_list);
        if (tmp_dt_list) {
            tmp_dt_list->path = (char *)H5MM_xfree(tmp_dt_list->path);
            tmp_dt_list       = H5FL_FREE(H5O_copy_dtype_merge_list_t, tmp_dt_list);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__ocpy_merge_comm_dt_list_del(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                                 size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(value);

    
    H5P__free_merge_comm_dtype_list(*(H5O_copy_dtype_merge_list_t **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__ocpy_merge_comm_dt_list_copy(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (H5P__copy_merge_comm_dt_list((H5O_copy_dtype_merge_list_t **)value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy merge committed dtype list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5P__ocpy_merge_comm_dt_list_cmp(const void *_dt_list1, const void *_dt_list2, size_t H5_ATTR_UNUSED size)
{
    const H5O_copy_dtype_merge_list_t *dt_list1 = *(H5O_copy_dtype_merge_list_t *const *)
                                                      _dt_list1, 
        *dt_list2    = *(H5O_copy_dtype_merge_list_t *const *)_dt_list2;
    herr_t ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(_dt_list1);
    assert(_dt_list2);
    assert(size == sizeof(H5O_copy_dtype_merge_list_t *));

    
    while (dt_list1 && dt_list2) {
        assert(dt_list1->path);
        assert(dt_list2->path);

        
        ret_value = strcmp(dt_list1->path, dt_list2->path);
        if (ret_value != 0)
            HGOTO_DONE(ret_value);

        
        dt_list1 = dt_list1->next;
        dt_list2 = dt_list2->next;
    } 

    
    if (dt_list1)
        HGOTO_DONE(1);
    if (dt_list2)
        HGOTO_DONE(-1);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__ocpy_merge_comm_dt_list_close(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(value);

    
    H5P__free_merge_comm_dtype_list(*(H5O_copy_dtype_merge_list_t **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5Pset_copy_object(hid_t plist_id, unsigned cpy_option)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (cpy_option & ~H5O_COPY_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unknown option specified");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_OBJECT_COPY, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5O_CPY_OPTION_NAME, &cpy_option) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set copy object flag");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_copy_object(hid_t plist_id, unsigned *cpy_option )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_OBJECT_COPY, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (cpy_option)
        if (H5P_get(plist, H5O_CPY_OPTION_NAME, cpy_option) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get object copy flag");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Padd_merge_committed_dtype_path(hid_t plist_id, const char *path)
{
    H5P_genplist_t              *plist;               
    H5O_copy_dtype_merge_list_t *old_list;            
    H5O_copy_dtype_merge_list_t *new_obj   = NULL;    
    herr_t                       ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!path)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no path specified");
    if (path[0] == '\0')
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "path is empty string");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_OBJECT_COPY, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5O_CPY_MERGE_COMM_DT_LIST_NAME, &old_list) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get merge named dtype list");

    
    if (NULL == (new_obj = H5FL_CALLOC(H5O_copy_dtype_merge_list_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
    if (NULL == (new_obj->path = H5MM_strdup(path)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
    new_obj->next = old_list;

    
    if (H5P_poke(plist, H5O_CPY_MERGE_COMM_DT_LIST_NAME, &new_obj) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set merge named dtype list");

done:
    if (ret_value < 0)
        if (new_obj) {
            new_obj->path = (char *)H5MM_xfree(new_obj->path);
            new_obj       = H5FL_FREE(H5O_copy_dtype_merge_list_t, new_obj);
        } 

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pfree_merge_committed_dtype_paths(hid_t plist_id)
{
    H5P_genplist_t              *plist;               
    H5O_copy_dtype_merge_list_t *dt_list;             
    herr_t                       ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_OBJECT_COPY, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5O_CPY_MERGE_COMM_DT_LIST_NAME, &dt_list) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get merge committed dtype list");

    
    dt_list = H5P__free_merge_comm_dtype_list(dt_list);

    
    if (H5P_poke(plist, H5O_CPY_MERGE_COMM_DT_LIST_NAME, &dt_list) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set merge committed dtype list");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_mcdt_search_cb(hid_t plist_id, H5O_mcdt_search_cb_t func, void *op_data)
{
    H5P_genplist_t    *plist;               
    H5O_mcdt_cb_info_t cb_info;             
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!func && op_data)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "callback is NULL while user data is not");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_OBJECT_COPY, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    cb_info.func      = func;
    cb_info.user_data = op_data;

    
    if (H5P_set(plist, H5O_CPY_MCDT_SEARCH_CB_NAME, &cb_info) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set callback info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_mcdt_search_cb(hid_t plist_id, H5O_mcdt_search_cb_t *func , void **op_data )
{
    H5P_genplist_t    *plist;               
    H5O_mcdt_cb_info_t cb_info;             
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_OBJECT_COPY, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_get(plist, H5O_CPY_MCDT_SEARCH_CB_NAME, &cb_info) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get callback info");

    if (func)
        *func = cb_info.func;

    if (op_data)
        *op_data = cb_info.user_data;

done:
    FUNC_LEAVE_API(ret_value)
} 
