% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/attribute_lifetable.R
\name{attribute_lifetable}
\alias{attribute_lifetable}
\title{Attribute premature deaths or YLL to an environmental stressor using a life table approach}
\usage{
attribute_lifetable(
  age_group,
  sex,
  bhd_central,
  bhd_lower = NULL,
  bhd_upper = NULL,
  population,
  health_outcome = NULL,
  min_age = NULL,
  max_age = NULL,
  approach_exposure = "single_year",
  approach_newborns = "without_newborns",
  year_of_analysis,
  time_horizon = NULL,
  exp_central = NULL,
  exp_lower = NULL,
  exp_upper = NULL,
  cutoff_central = 0,
  cutoff_lower = NULL,
  cutoff_upper = NULL,
  erf_eq_central = NULL,
  erf_eq_lower = NULL,
  erf_eq_upper = NULL,
  rr_central = NULL,
  rr_lower = NULL,
  rr_upper = NULL,
  rr_increment = NULL,
  erf_shape = NULL,
  prop_pop_exp = 1,
  geo_id_micro = "a",
  geo_id_macro = NULL,
  info = NULL
)
}
\arguments{
\item{age_group}{\code{Numeric vector} or \code{string vector} providing the \strong{age groups} considered in the assessment. In case of use in \code{attribute_lifetable)()}, it must be a \code{numeric} and contain single year age groups. See Details for more info. \emph{Optional argument for \code{attribute_health()}; needed for \code{attribute_lifetable()}.}}

\item{sex}{\code{Numeric vector} or \code{string vector} specifying the \strong{sex} of the groups considered in the assessment.\emph{Optional argument.}}

\item{bhd_central, bhd_lower, bhd_upper}{\code{Numeric value} or \code{numeric vector} providing the \strong{baseline health data} of the health outcome of interest in the study population and (optionally) the corresponding lower bound and the upper 95\% confidence interval bounds. See Details for more info. \emph{Only applicable in RR pathways; always required.}}

\item{population}{\code{Numeric vector} \strong{\code{For attribute_lifetable()}}, it is an \emph{obligatory argument} specifying the \strong{mid-year populations} per age (i.e. age group size = 1 year) for the (first) year of analysis.
\strong{\code{For attribute_health()}} it is an \emph{optional argument} which specifies the \strong{population used to calculate attributable impacts rate} per 100 000 population. See Details for more info.}

\item{health_outcome}{\code{String} specifying the desired result of the life table assessment. Options: \code{"deaths"} (premature deaths), \code{"yll"} (years of life lost).}

\item{min_age, max_age}{\code{Numberic value} specifying the minimum and maximum age for which the exposure will affect the exposed population, respectively. Default \code{min_age}: 30. Default \code{max_age}: none. See Details for more info.}

\item{approach_exposure}{\code{String} specifying whether exposure is constant or only in one year. Options: \code{"single_year"} (default), \code{"constant"}.}

\item{approach_newborns}{\code{String} specifying whether newborns are to be considered in the years after the year of analysis or not. Options: \code{"without_newborns"} (default), \code{"with_newborns"}. See Details for more info.}

\item{year_of_analysis}{\code{Numeric value} providing the first with exposure to the environmental stressor.}

\item{time_horizon}{\code{Numeric value} specifying the time horizon (number of years) for which the attributable YLL or premature deaths are to be considered. See Details for more info. \emph{Optional argument.}}

\item{exp_central, exp_lower, exp_upper}{\code{Numeric value} or \code{numeric vector} specifying the \strong{exposure level(s)} to the environmental stressor and (optionally) the corresponding lower and upper bound of the 95\% confidence interval. See Details for more info.}

\item{cutoff_central, cutoff_lower, cutoff_upper}{\code{Numeric value} specifying the \strong{exposure cut-off value} and (optionally) the corresponding lower and upper 95\% confidence interval bounds. Default: 0. See Details for more info.}

\item{erf_eq_central, erf_eq_lower, erf_eq_upper}{\code{String} or \code{function} specifying the \strong{exposure-response function} and (optionally) the corresponding lower and upper 95\% confidence interval functions. See Details for more info. \emph{Required in AR pathways; in RR pathways required only if \code{rr_...} argument(s) not specified.}}

\item{rr_central, rr_lower, rr_upper}{\code{Numeric value} specifying the \strong{central relative risk} estimate and (optionally) the corresponding lower and upper 95\% confidence interval bounds. \emph{Only applicable in RR pathways; not required if \code{erf_eq_...} argument(s) already specified.}}

\item{rr_increment}{\code{Numeric value} specifying the \strong{exposure increment} for which the provided relative risk is valid. See Details for more info. \emph{Only applicable in RR pathways; not required if \code{erf_eq_...} argument(s) already specified.}}

\item{erf_shape}{\code{String value} specifying the \strong{exposure-response function shape} to be assumed. Options (no default): \code{"linear"}, \code{log_linear}", \code{"linear_log"}, \code{"log_log"}. \emph{Only applicable in RR pathways; not required if \code{erf_eq_...} argument(s) already specified.}}

\item{prop_pop_exp}{\code{Numeric value} or \code{numeric vector} specifying the \strong{population fraction(s) exposed} for each exposure (category). Default: 1. See Details for more info. \emph{Only applicable in RR pathways.}}

\item{geo_id_micro, geo_id_macro}{\code{Numeric vector} or \code{string vector} providing \strong{unique IDs of the geographic area} considered in the assessment (\code{geo_id_micro}) and (optionally) providing higher-level IDs (\code{geo_id_macro}) to aggregate the geographic areas at. See Details for more info. \emph{Only applicable in assessments with multiple geographic units.}}

\item{info}{\code{String}, \code{data frame} or \code{tibble} providing \strong{information about the assessment}. See Details for more info. \emph{Optional argument.}}
}
\value{
This function returns a \code{list} containing:

1) \code{health_main} (\code{tibble}) containing the main results;
\itemize{
 \item \code{impact} (\code{numeric} column) attributable health burden/impact
 \item \code{pop_fraction} (\code{numeric} column) population attributable fraction; only applicable in relative risk assessments
 \item And many more
 }

2) \code{health_detailed} (\code{list}) containing detailed (and interim) results.
\itemize{
 \item \code{results_raw} (\code{tibble}) containing results for each combination of input uncertainty
 \item \code{results_by_geo_id_micro} (\code{tibble}) containing results for each geographic unit under analysis (specified in \code{geo_id_micro} argument)
 \item \code{results_by_year} (\code{tibble}) containing results by year
 \item \code{results_by_sex} (\code{tibble}) containing results by sex
 \item \code{results_by_age_group} (\code{tibble}) containing results by age group
 \item \code{intermediate_calculations} (\code{tibble}) containing intermediate results, among others population projections (for both the exposed and unexposed scenarios) and impact by age and year stored in nested \code{tibbles}
 \item \code{input_table} (\code{tibble}) containing the inputs to each relevant argument
 \item \code{input_args} (\code{list}) containing all the argument inputs used in the background
 }
}
\description{
This function assesses premature deaths or years of life lost (YLL) attributable to exposure to an environmental stressor using a life table approach.
}
\details{
\strong{Function arguments}

\code{age_group}

The numeric values must refer to 1 year age groups, e.g. \code{c(0:99)}. To convert multi-year/larger age groups to 1 year age groups use the function \code{prepare_lifetable()} (see its function documentation for more info).

\code{bhd_central,bhd_lower,bhd_upper}

Deaths per age must be inputted with 1 value per age (i.e. age group size = 1 year). There must be greater than or equal to 1 deaths per age to avoid issues during the calculation of survival probabilities.

\code{population}

The population data must be inputted with 1 value per age (i.e. age group size = 1 year). The values must be greater than or equal to 1 per age to avoid issues during the calculation of survival probabilities.

Mid-year population of year x can be approximated as the mean of either end-year populations of years x-1 and x or start-of-year populations of years x and x+1. For each age, the inputted values must be greater than or equal to 1 to avoid issues during the calculation of survival probabilities.

\code{approach_newborns}

If \code{"with_newborns"} is selected, it is assumed that for each year after the year of analysis n babies (population aged 0) are born.

\code{time_horizon}

Applicable for the following cases:
#' \itemize{
 \item YLL: \code{single_year} or \code{constant} exposure
 \item premature deaths: \code{constant} exposure
}
For example, if 10 is entered one is interested in the impacts of exposure during the year of analysis and the next 9 years (= 10 years in total). Default value: length of the numeric vector specified in the \code{age_group} argument.

\code{min_age}, \code{max_age}
The \code{min_age} default value 30 implies that all adults aged 30 or older will be affected by the exposure; \code{max_age} analogeously specifies the age above which no health effects of the exposure are considered.

\strong{Conversion of multi-year to single year age groups}

To convert multi-year/larger age groups to 1 year age groups use the function \code{prepare_lifetable()} and see its function documentation for more info.

\strong{Life table methodology}

The life table methodology of \code{attribute_lifetable()} follows that of the WHO tool AirQ+, and is described in more detail by Miller & Hurley (2003, https://doi.org/10.1136/jech.57.3.200).

In short, two scenarios are compared: 1) a scenario with the exposure level specified in the function ("exposed scenario") and 2) a scenario with no exposure ("unexposed scenario"). First, the entry and mid-year populations of the (first) year of analysis in the unexposed scenario is determined using modified survival probabilities. Second, age-specific population projections using scenario-specific survival probabilities are done for both scenarios. Third, by subtracting the populations in the unexposed scenario from the populations in the exposed scenario the premature deaths/years of life lost attributable to the exposure are determined.

An expansive life table case study by Miller (2010) is available here: https://cleanair.london/app/uploads/CAL-098-Mayors-health-study-report-June-2010-1.pdf.

\emph{\strong{Determination of populations in the (first) year of analysis}}

The entry (i.e. start of year) populations in both scenarios is determined as follows:
\deqn{entry\_population_{year_1} = midyear\_population_{year_1} + \frac{deaths_{year_1}}{2}}

\emph{\strong{Exposed scenario}} The survival probabilities in the exposed scenario from start of year i to start of year i+1  are calculated as follows:
\deqn{prob\_survival = \frac{midyear\_population_i - \frac{deaths_i}{2}}{midyear\_population_i + \frac{deaths_i}{2}}}
Analogously, the probability of survival from start of year i to mid-year i:
\deqn{prob\_survival\_until\_midyear = 1 - \frac{1 - prob\_survival}{2}}

\emph{\strong{Unexposed scenario}} The survival probabilities in the unexposed scenario are calculated as follows:

First, the age-group specific hazard rate in the exposed scenario is calculated using the inputted age-specific mid-year populations and deaths.
\deqn{hazard\_rate = \frac{deaths}{mid\_year\_population}}
Second, the hazard rate is multiplied with the modification factor (\eqn{= 1 - PAF}) to obtain the age-specific hazard rate in the unexposed scenario.
\deqn{hazard\_rate\_mod = hazard\_rate \times modification\_factor}
Third, the the age-specific survival probabilities (from the start until the end in a given age group) in the unexposed scenario are calculated as follows (cf. Miller & Hurley 2003):
\deqn{prob\_survival\_mod = \frac{2-hazard\_rate\_mod}{2+hazard\_rate\_mod}}
Then the mid-year populations of the (first) year of analysis (year_1) in the unexposed scenario are determined as follows:

First, the survival probabilities from start of year i to mid-year i in the unexposed scenario is calculated as:
\deqn{prob\_survival\_until\_midyear\_{mod} = 1 - \frac{1 - prob\_survival\_mod}{2}}
Second, the mid-year populations of the (first) year of analysis (year_1) in the unexposed scenario is calculated:
\deqn{midyear\_population\_unexposed_{year_1} = entry\_population_{year_1} \times prob\_survival\_until\_midyear_{mod}}

\emph{\strong{Population projection}}

Using the age group-specific and scenario-specific survival probabilities calculated above, future populations of each age-group under each scenario are calculated.

\emph{\strong{Unexposed scenario}} The entry and mid-year population projections of in the exposed scenario is done as follows:

First, the entry population of year i+1 is calculated (which is the same as the end of year population of year i) by multiplying the entry population of year i and the modified survival probabilities.
\deqn{entry\_population_{i+1} = entry\_population_i \times prob\_survival\_mod}
Second, the mid-year population of year i+1 is calculated.
\deqn{midyear\_population_{i+1} = entry\_population_{i+1} \times prob\_survival\_until\_midyear}

\emph{\strong{Exposed scenario}} The population projections for the two possible options of \code{approach_exposure} (\code{"single_year"} and \code{"constant"}) for the unexposed scenario are different. In the case of \code{"single_year"} exposure, the population projection for the years after the year of exposure is the same as in the unexposed scenario.

In the case of \code{"constant"} the population projection is done as follows:

First, the entry population of year i+1 is calculated (which is the same as the end of year population of year i) using the entry population of year i.
\deqn{entry\_population_{i+1} = entry\_population_i \times prob\_survival}
Second, the mid-year population of year i+1 is calculated.
\deqn{midyear\_population_{i+1} = entry\_population_{i+1} \times prob\_survival\_until\_midyear}

\strong{Conversion of alternative risk measures to relative risks}

For conversion of hazard ratios and/or odds ratios to relative risks refer to https://doi.org/10.1111/biom.13197 and/or use the conversion tool for hazard ratios (https://ebm-helper.cn/en/Conv/HR_RR.html) and/or odds ratios (https://ebm-helper.cn/en/Conv/OR_RR.html).
}
\examples{
# Goal: determine YLL attributable to air pollution exposure during one year
# using the life table approach
results <- attribute_lifetable(
  health_outcome = "yll",
  approach_exposure = "single_year",
  approach_newborns = "without_newborns",
  exp_central = 8.85,
  prop_pop_exp = 1,
  cutoff_central = 5,
  rr_central =  1.118,
  rr_increment = 10,
  erf_shape = "log_linear",
  age_group = exdat_lifetable$age_group,
  sex = exdat_lifetable$sex,
  bhd_central = exdat_lifetable$deaths,
  population = exdat_lifetable$midyear_population,
  year_of_analysis = 2019,
  min_age = 20
)
results$health_main$impact # Attributable YLL

# Goal: determine attributable premature deaths due to air pollution exposure
# during one year using the life table approach
results_pm_deaths <- attribute_lifetable(
  health_outcome = "deaths",
  approach_exposure = "single_year",
  exp_central = 8.85,
  prop_pop_exp = 1,
  cutoff_central = 5,
  rr_central =  1.118,
  rr_increment = 10,
  erf_shape = "log_linear",
  age_group = exdat_lifetable$age_group,
  sex = exdat_lifetable$sex,
  bhd_central = exdat_lifetable$deaths,
  population = exdat_lifetable$midyear_population,
  year_of_analysis = 2019,
  min_age = 20
)
results_pm_deaths$health_main$impact # Attributable premature deaths

# Goal: determine YLL attributable to air pollution exposure (exposure distribution)
# during one year using the life table approach
results <- attribute_lifetable(
  health_outcome = "yll",
  exp_central = rep(c(8, 9, 10), each = 100*2), # each = length of sex or age_group vector
  prop_pop_exp = rep(c(0.2, 0.3, 0.5), each = 100*2), # each = length of sex or age_group vector
  cutoff_central = 5,
  rr_central = 1.118,
  rr_lower = 1.06,
  rr_upper = 1.179,
  rr_increment = 10,
  erf_shape = "log_linear",
  age_group = rep(
    exdat_lifetable$age_group,
    times = 3), # times = number of exposure categories
  sex = rep(
    exdat_lifetable$sex,
    times = 3), # times = number of exposure categories
  population = rep(
    exdat_lifetable$midyear_population,
    times = 3), # times = number of exposure categories
  bhd_central = rep(
    exdat_lifetable$deaths,
    times = 3), # times = number of exposure categories
  year_of_analysis = 2019,
  min_age = 20
)
results$health_main$impact_rounded # Attributable YLL
}
\author{
Alberto Castro & Axel Luyten
}
