% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/api-snic.R
\name{snic_animation}
\alias{snic_animation}
\title{Animated visualization of SNIC seeding and segmentation}
\usage{
snic_animation(
  x,
  seeds,
  file_path,
  max_frames = 100L,
  delay = 10,
  progress = getOption("snic.progress", FALSE),
  ...,
  snic_args = list(compactness = 0.5),
  device_args = list(res = 96, bg = "white")
)
}
\arguments{
\item{x}{A \code{\link[terra:SpatRaster-class]{SpatRaster}}
representing the image to segment. Dimensions and coordinate reference
are inferred automatically.}

\item{seeds}{A two-column object specifying seed coordinates. If \code{x}
has a CRS, use columns \code{lat} and \code{lon} (in \code{EPSG:4326});
otherwise use pixel indices \code{(r, c)}. Typically created with
\code{\link{snic_grid}} or interactively with
\code{\link{snic_grid_manual}}.}

\item{file_path}{Path where the resulting GIF is saved. The file must not
already exist and the parent directory must be writable.}

\item{max_frames}{Maximum number of frames to render. If there are more
seeds than \code{max_frames}, only the first \code{max_frames} seeds are
used.}

\item{delay}{Per-frame delay in centiseconds (1/100 s). Passed to
\code{magick::image_animate()}. Default is 10 (0.1 s per frame).}

\item{progress}{Logical scalar; if \code{TRUE}, show the textual progress
bar while generating frames.}

\item{...}{Additional arguments forwarded to \code{\link{snic_plot}} when
drawing each frame (e.g., RGB band indices or palette options).}

\item{snic_args}{Named list of extra arguments passed to \code{\link{snic}}
on every iteration (e.g., \code{compactness}). Arguments \code{x} and
\code{seeds} are reserved and cannot be overridden.}

\item{device_args}{Named list of arguments passed to
\code{grDevices::png()} when rendering frames. Defaults to
\code{list(res = 96, bg = "white")}. Values such as \code{width},
\code{height}, and \code{filename} are managed automatically.}
}
\value{
Invisibly, the file path of the generated GIF.
}
\description{
Generate an animated GIF illustrating how SNIC segmentation evolves
as seeds are progressively added. This function runs a sequence of
SNIC segmentations using incremental subsets of the provided seeds
and compiles the results into an animation.
}
\details{
For each iteration, the function adds one seed to the current set and
re-runs \code{\link{snic}}. The segmentation and seed locations are drawn
using \code{\link{snic_plot}}, saved as PNGs, and then combined into an
animated GIF using the \pkg{magick} package. This is intended for
exploratory and didactic use to illustrate the influence of seed placement
and parameters such as \code{compactness}.
}
\examples{
if (requireNamespace("terra", quietly = TRUE) &&
    requireNamespace("magick", quietly = TRUE)) {
    tif_dir <- system.file("demo-geotiff", package = "snic", mustWork = TRUE)
    files <- file.path(
        tif_dir,
        c(
            "S2_20LMR_B02_20220630.tif",
            "S2_20LMR_B04_20220630.tif",
            "S2_20LMR_B08_20220630.tif",
            "S2_20LMR_B12_20220630.tif"
        )
    )
    s2 <- terra::aggregate(terra::rast(files), fact = 8)

    set.seed(42)
    seeds <- snic_grid(s2, type = "random", spacing = 10L, padding = 0L)

    gif_file <- snic_animation(
        s2,
        seeds = seeds,
        file_path = tempfile("snic-demo", fileext = ".gif"),
        max_frames = 20L,
        snic_args = list(compactness = 0.1),
        r = 4, g = 3, b = 1,
        device_args = list(height = 192, width = 256)
    )
    gif_file
}
}
\seealso{
\code{\link{snic}}, \code{\link{snic_plot}},
  \code{\link{snic_grid}}, \code{\link{snic_grid_manual}}.
}
