% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model.R
\docType{class}
\name{Model}
\alias{Model}
\title{Unified Machine Learning Interface using R6}
\description{
Provides a consistent interface for various machine learning models in R,
with automatic detection of formula vs matrix interfaces, built-in
cross-validation, model interpretability, and visualization.

An R6 class that provides a unified interface for regression and classification
models with automatic interface detection, cross-validation, and interpretability
features. The task type (regression vs classification) is automatically detected
from the response variable type.
}
\examples{
\donttest{
# Regression example with glmnet
library(glmnet)
X <- matrix(rnorm(100), ncol = 4)
y <- 2*X[,1] - 1.5*X[,2] + rnorm(25)  # numeric -> regression

mod <- Model$new(glmnet::glmnet)
mod$fit(X, y, alpha = 0, lambda = 0.1)
mod$summary()
predictions <- mod$predict(X)

# Classification example  
data(iris)
iris_binary <- iris[iris$Species \%in\% c("setosa", "versicolor"), ]
X_class <- as.matrix(iris_binary[, 1:4])
y_class <- iris_binary$Species  # factor -> classification

mod2 <- Model$new(e1071::svm)
mod2$fit(X_class, y_class, kernel = "radial")
mod2$summary()

# Cross-validation
cv_scores <- cross_val_score(mod, X, y, cv = 5)
}

}
\author{
Your Name
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{model_fn}}{The modeling function (e.g., glmnet::glmnet, randomForest::randomForest)}

\item{\code{fitted}}{The fitted model object}

\item{\code{task}}{Type of task: "regression" or "classification" (automatically detected)}

\item{\code{X_train}}{Training features matrix}

\item{\code{y_train}}{Training target vector}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Model-new}{\code{Model$new()}}
\item \href{#method-Model-fit}{\code{Model$fit()}}
\item \href{#method-Model-predict}{\code{Model$predict()}}
\item \href{#method-Model-print}{\code{Model$print()}}
\item \href{#method-Model-summary}{\code{Model$summary()}}
\item \href{#method-Model-plot}{\code{Model$plot()}}
\item \href{#method-Model-clone_model}{\code{Model$clone_model()}}
\item \href{#method-Model-clone}{\code{Model$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-new"></a>}}
\if{latex}{\out{\hypertarget{method-Model-new}{}}}
\subsection{Method \code{new()}}{
Initialize a new Model
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$new(model_fn)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{model_fn}}{A modeling function (e.g., glmnet, randomForest, svm)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new Model object
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-fit"></a>}}
\if{latex}{\out{\hypertarget{method-Model-fit}{}}}
\subsection{Method \code{fit()}}{
Fit the model to training data

Automatically detects task type (regression vs classification) based on
the type of the response variable y. Numeric y -> regression,
factor y -> classification.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$fit(X, y, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{X}}{Feature matrix or data.frame}

\item{\code{y}}{Target vector (numeric for regression, factor for classification)}

\item{\code{...}}{Additional arguments passed to the model function}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
self (invisible) for method chaining
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-predict"></a>}}
\if{latex}{\out{\hypertarget{method-Model-predict}{}}}
\subsection{Method \code{predict()}}{
Generate predictions from fitted model
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$predict(X, type = NULL, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{X}}{Feature matrix for prediction}

\item{\code{type}}{Type of prediction ("response", "class", "probabilities")}

\item{\code{...}}{Additional arguments passed to predict function}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Vector of predictions
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-print"></a>}}
\if{latex}{\out{\hypertarget{method-Model-print}{}}}
\subsection{Method \code{print()}}{
Print model information
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$print()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
self (invisible) for method chaining
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-summary"></a>}}
\if{latex}{\out{\hypertarget{method-Model-summary}{}}}
\subsection{Method \code{summary()}}{
Compute numerical derivatives and statistical significance

Uses finite differences to compute approximate partial derivatives
for each feature, providing model-agnostic interpretability.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$summary(h = 0.01, alpha = 0.05)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{h}}{Step size for finite differences (default: 0.01)}

\item{\code{alpha}}{Significance level for p-values (default: 0.05)}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
The method computes numerical derivatives using central differences.

Statistical significance is assessed using t-tests on the derivative
estimates across samples.
}

\subsection{Returns}{
A data.frame with derivative statistics (invisible)
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-plot"></a>}}
\if{latex}{\out{\hypertarget{method-Model-plot}{}}}
\subsection{Method \code{plot()}}{
Create partial dependence plot for a feature

Visualizes the relationship between a feature and the predicted outcome
while holding other features at their mean values.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$plot(feature = 1, n_points = 100)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{feature}}{Index or name of feature to plot}

\item{\code{n_points}}{Number of points for the grid (default: 100)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
self (invisible) for method chaining
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-clone_model"></a>}}
\if{latex}{\out{\hypertarget{method-Model-clone_model}{}}}
\subsection{Method \code{clone_model()}}{
Create a deep copy of the model

Useful for cross-validation and parallel processing where
multiple independent model instances are needed.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$clone_model()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A new Model object with same configuration
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-clone"></a>}}
\if{latex}{\out{\hypertarget{method-Model-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
