#' Create raster representing one or more random walks
#'
#' This function creates a raster where the cell values represent the number of times one or more random "walkers" traverse the cell. If you simulate multiple random walkers, you can do computation in parallel, which can be controlled by allowing **fasterRaster** to use multiple cores and more memory using the "cores" and "memory" arguments in the [faster()] function.
#'
#' This function needs the **GRASS** addon `r.random.walk`. If it is not installed, it will try to install it.#'
#' @param x A `GRaster` to serve as a template.
#'
#' @param n Numeric: Number of walkers. Default  is 1.
#'
#' @param steps Numeric: Number of steps taken by each walker. Default is 100000.
#'
#' @param directions Either 4 or 8: Directions in which a walker can turn at any point. If 4, then walks are confined to north/south/east/west directions (Rook's case). If 8, then the cardinal and subcardinal directions are allowed (Queen's case).
#'
#' @param avoid Logical: If `FALSE` (default), then walkers can traverse their own walks. If `TRUE`, walkers avoid their own trails. A self-avoiding random walk can take much longer to compute.
#'
#' @param sameStart Logical: If `FALSE` (default), walkers can begin anywhere. If `TRUE`, then walkers start from the same place.
#'
#' @param seed Integer or `NULL` (default): If `NULL`, then a random seed is generated by the function. If numeric, results will be deterministic. In either case, the value will be rounded to the nearest integer.
#'
#' @param check Logical: If `TRUE` (default), function will check to see if the addon `r.random.walk` has been installed. If it has not, it will attempt to install it.
#'
#' @returns A `GRaster` with cell values representing the number of times one or more walkers traversed the cell.
#'
#' @seealso [rNormRast()], [rUnifRast()], [rSpatialDepRast()], [fractalRast()]

#' @example man/examples/ex_randRast.r
#'
#' @aliases rWalkRast
#' @rdname rWalkRast
#' @exportMethod rWalkRast
methods::setMethod(
	f = "rWalkRast",
	signature = c(x = "GRaster"),
	function(x, n = 1, steps = 100000, directions = 8, avoid = FALSE, sameStart = FALSE, seed = NULL, check = TRUE) {

	if (!(directions %in% c(4, 8))) stop("The `directions` argument must be 4 or 8.")
	directions <- as.character(directions)
	
	if (!is.null(seed)) seed <- round(seed)

	if (check) .addons("r.random.walk")

    .locationRestore(x)
    .region(x)

	src <- .makeSourceName("rRandWalk_r_random_walk", "raster")
	args <- list(
		cmd = "r.random.walk",
		output = src,
		steps = steps,
		directions = directions,
		memory = faster("memory"),
		nprocs = faster("cores"),
		nwalkers = n,
		flags = c(.quiet(), "overwrite")
	)

	if (!is.null(seed)) {
		args <- c(args, seed = seed)
	} else if (is.null(seed)) {
		args$flags <- c(args$flags, "s")
	}
	if (avoid) args$flags <- c(args$flags, "a")
	if (sameStart) args$flags <- c(args$flags, "t")

	do.call(rgrass::execGRASS, args = args)

	# srcIn <- src
	# src <- .makeSourceName("rRandWalk_r_mapcalc", "raster")
	# ex <- paste0(src, " = int(", srcIn, ")")
	# rgrass::execGRASS("r.mapcalc", expression = ex, flags = c(.quiet(), "overwrite"))

	.makeGRaster(src, "randomWalk")

	} # EOF
)
