% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/amfewma.R
\name{AMFEWMA_PhaseI}
\alias{AMFEWMA_PhaseI}
\title{Adaptive Multivariate Functional EWMA control chart - Phase I}
\usage{
AMFEWMA_PhaseI(
  mfdobj,
  mfdobj_tuning,
  lambda = NULL,
  k = NULL,
  ARL0 = 200,
  bootstrap_pars = list(n_seq = 200, l_seq = 2000),
  optimization_pars = list(lambda_grid = c(0.1, 0.2, 0.3, 0.5, 1), k_grid = c(1, 2, 3,
    4), epsilon = 0.1, sd_small = 0.25, sd_big = 2),
  discrete_grid_length = 25,
  score_function = "huber",
  fev = 0.9,
  n_skip = 100
)
}
\arguments{
\item{mfdobj}{An object of class \code{mfd} containing the Phase I multivariate
functional data set, to be used to train the multivariate functional
principal component analysis model.}

\item{mfdobj_tuning}{An object of class \code{mfd} containing the Phase I multivariate
functional data set, to be used as tuning data set to estimate the
AMFEWMA control chart limit.}

\item{lambda}{lambda parameter to be used in the score function.
See Equation (7) or (8) of Capezza et al. (2024).
If it is provided, it must be a number between zero and one.
If NULL, it is chosen through the selected according to the
optimization procedure presented in Section 2.4 of Capezza et al. (2024).
In this case, it is chosen among the values of
\code{optimization_pars$lambda_grid}.
Default value is NULL.}

\item{k}{k parameter to be used in the score function.
See Equation (7) or (8) of Capezza et al. (2024).
If it is provided, it must be a number greater than zero.
If NULL, it is chosen through the selected according to the
optimization procedure presented in Section 2.4 of Capezza et al. (2024).
In this case, it is chosen among the values of
\code{optimization_pars$k_grid}.
Default value is NULL.}

\item{ARL0}{The nominal in-control average run length.
Default value is 200.}

\item{bootstrap_pars}{Parameters of the bootstrap procedure described in
Section 2.4 of Capezza et al. (2024) for the estimation of the
control chart limit.
It must be a list with two arguments.
\code{n_seq} is the number of bootstrap sequences to be generated.
\code{l_seq} is the length of each bootstrap sequence, i.e., the number
of observations to be sampled with replacement from the tuning set.
Default value is \code{list(n_seq = 200, l_seq = 2000)}.}

\item{optimization_pars}{Parameters to be used in the optimization procedure described in Section
2.4 of Capezza et al. (2024) for the selection of the parameters
lambda and k.
It must be a list of the following parameters.
\code{lambda_grid} contains the possible values of
the parameter \code{lambda}.
\code{k_grid} contains the possible values of the parameter \code{k}.
\code{epsilon} is the parameter used in Equation (10) of
Capezza et al. (2024).
When performing the parameter optimization,
first the parameters lambda and k are selected to minimize the ARL
with respect to a large shift,
then the same parameters are chosen to minimize the ARL
with respect to a small shift, given that the resulting ARL with respect
to the previous large shift does not increase, in percentage,
more than \code{epsilon}*100.
Default value is 0.1.
\code{sd_small} is a positive constant that multiplies the
standard deviation function to define the small shift delta_1 in Section
2.4 of Capezza et al. (2024). In fact, the small shift is defined as
delta_1(t) = mu_0(t) + \code{sd_small} * sigma(t), where
mu_0(t) is the estimated in-control mean function and
sigma(t) is the estimated standard deviation function.
Default value is 0.25.
\code{sd_big} is a positive constant that multiplies the
standard deviation function to define the large shift delta_2 in Section
2.4 of Capezza et al. (2024). In fact, the large shift is defined as
delta_2(t) = mu_0(t) + \code{sd_large} * sigma(t), where
mu_0(t) is the estimated in-control mean function and
sigma(t) is the estimated standard deviation function.
Default value is 2.}

\item{discrete_grid_length}{The number of equally spaced argument values at which the \code{mfd}
objects are discretized.
Default value is 25.}

\item{score_function}{Score function to be used in Equation (7) or (8) of Capezza et al. (2024),
to calculate the weighting parameter of the EWMA statistic
for each observation of the sequence.
Two values are possible.
If "huber", it uses the score function (7)
inspired by the Huber's function.
If "tukey", it uses the score function (8)
inspired by the Tukey's bisquare function.}

\item{fev}{Number between 0 and 1 denoting the fraction
of variability that must be explained by the
principal components to be selected after
applying multivariate functional principal component analysis
on \code{mfdobj}. Default is 0.9.}

\item{n_skip}{The upper control limit of the AMFEWMA control chart is set
to achieve a desired in-control ARL, evaluated after the
monitoring statistic has reached steady state.
A monitoring statistic is in a steady state
if the process has been in control long enough
for the effect of the starting value to become negligible
(Lucas and Saccucci, 1990).
In this regard, the first \code{n_skip} observations
are excluded from the calculation of the run length.
Default value is 100.}
}
\value{
A list with the following elements.
\code{lambda} is the selected lambda parameter.
\code{k} is the selected k parameter.
\code{mod_1} contains the estimated Phase I model. It is a list with
the following elements.
\itemize{
\item \code{mfdobj} the \code{mfdobj} object passed as input to this function,
\item \code{mfdobj_tuning} the \code{mfdobj_tuning} object
passed as input to this function,
\item \code{inv_sigmaY_reg}: the matrix containing the discretized
version of the function K^*(s,t) defined in Equation (9) of
Capezza et al. (2024),
\item \code{mean_mfdobj}: the estimated mean function,
\item \code{h}: the calculated upper control limit of the AMFEWMA control chart,
\item \code{ARL0}: the estimated in-control ARL, which should be close to the
nominal value passed as input to this function,
\item \code{lambda}: the lambda parameter selected by the optimization
procedure described in Section 2.4 of Capezza et al. (2024).
\item \code{k}: The function C_j(t)=k sigma_j(t) appearing in the score
functions (7) and (8) of Capezza et al. (2024).
\item \code{grid_points}: the grid containing the points over which
the functional data are discretized before computing the AMFEWMA monitoring
statistic and estimating all the model parameters.
\item \code{V2_mat}: the \code{n_seq}X\code{l_seq} matrix containing,
in each column, the AMFEWMA monitoring statistic values of each
bootstrap sequence.
This matrix is used to set the control chart limit \code{h} to
ensure that the desired average run length is achieved.
\item \code{n_skip}: the \code{n_skip} input parameter passed to this function,
\item \code{huber}: if the input parameter \code{score_function} is
\code{"huber"}, this is TRUE, else is FALSE,
\item \code{vectors}: the discretized eigenfunctions psi_l(t) of
the covariance function, appearing in Equation (9) of Capezza et al. (2024).
\item \code{values}: the eigenvalues rho_l of
the covariance function, appearing in Equation (9) of Capezza et al. (2024).
}
}
\description{
This function performs Phase I of the
Adaptive Multivariate Functional EWMA (AMFEWMA) control chart proposed
by Capezza et al. (2024)
}
\examples{
\dontrun{set.seed(0)
library(funcharts)
dat_I <- simulate_mfd(nobs = 1000,
                      correlation_type_x = c("Bessel", "Bessel", "Bessel"),
                      sd_x = c(0.3, 0.3, 0.3))
dat_tun <- simulate_mfd(nobs = 1000,
                        correlation_type_x = c("Bessel", "Bessel", "Bessel"),
                        sd_x = c(0.3, 0.3, 0.3))
dat_II <- simulate_mfd(nobs = 200,
                       correlation_type_x = c("Bessel", "Bessel", "Bessel"),
                       shift_type_x = c("C", "C", "C"),
                       d_x = c(2, 2, 2),
                       sd_x = c(0.3, 0.3, 0.3))
mfdobj_I <- get_mfd_list(dat_I$X_list)
mfdobj_tun <- get_mfd_list(dat_tun$X_list)
mfdobj_II <- get_mfd_list(dat_II$X_list)

p <- plot_mfd(mfdobj_I[1:100])
lines_mfd(p, mfdobj_II, col = "red")

mod <- AMFEWMA_PhaseI(mfdobj = mfdobj_I, mfdobj_tuning = mfdobj_tun)
print(mod$k)
cc <- AMFEWMA_PhaseII(mfdobj_2 = rbind_mfd(mfdobj_I[1:100], mfdobj_II),
                      mod_1 = mod)
plot_control_charts(cc$cc, nobsI = 100)
}


}
\references{
Capezza, C., Capizzi, G., Centofanti, F., Lepore, A., Palumbo, B. (2025)
An Adaptive Multivariate Functional EWMA Control Chart.
\emph{Journal of Quality Technology},  57(1):1--15,
doi:https://doi.org/10.1080/00224065.2024.2383674.

Lucas, J. M., Saccucci, M. S. (1990)
Exponentially weighted moving average control schemes:
properties and enhancements. \emph{Technometrics}, 32(1), 1-12.
}
\author{
C. Capezza, F. Centofanti
}
