/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Opkg.h"      

static void *H5O__cont_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags, size_t p_size,
                              const uint8_t *p);
static herr_t H5O__cont_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                               const void *_mesg);
static size_t H5O__cont_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__cont_free(void *mesg);
static herr_t H5O__cont_delete(H5F_t *f, H5O_t *open_oh, void *_mesg);
static herr_t H5O__cont_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_CONT[1] = {{
    H5O_CONT_ID,        
    "hdr continuation", 
    sizeof(H5O_cont_t), 
    0,                  
    H5O__cont_decode,   
    H5O__cont_encode,   
    NULL,               
    H5O__cont_size,     
    NULL,               
    H5O__cont_free,     
    H5O__cont_delete,   
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    H5O__cont_debug     
}};

H5FL_DEFINE(H5O_cont_t);

static void *
H5O__cont_decode(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, unsigned H5_ATTR_UNUSED mesg_flags,
                 unsigned H5_ATTR_UNUSED *ioflags, size_t p_size, const uint8_t *p)
{
    H5O_cont_t    *cont      = NULL;
    const uint8_t *p_end     = p + p_size - 1;
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    
    if (NULL == (cont = H5FL_MALLOC(H5O_cont_t)))
        HGOTO_ERROR(H5E_OHDR, H5E_NOSPACE, NULL, "memory allocation failed");

    

    cont->chunkno = 0;

    if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    H5F_addr_decode(f, &p, &(cont->addr));

    if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_size(f), p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    H5F_DECODE_LENGTH(f, p, cont->size);
    if (cont->size == 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "invalid continuation chunk size (0)");

    
    ret_value = cont;

done:
    if (NULL == ret_value && NULL != cont)
        H5FL_FREE(H5O_cont_t, cont);
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__cont_encode(H5F_t *f, bool H5_ATTR_UNUSED disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                 const void *_mesg)
{
    const H5O_cont_t *cont = (const H5O_cont_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(cont);
    assert(H5_addr_defined(cont->addr));
    assert(cont->size > 0);

    
    H5F_addr_encode(f, &p, cont->addr);
    H5F_ENCODE_LENGTH(f, p, cont->size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static size_t
H5O__cont_size(const H5F_t *f, bool H5_ATTR_UNUSED disable_shared, const void H5_ATTR_UNUSED *_mesg)
{
    size_t ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    ret_value = (size_t)(H5F_SIZEOF_ADDR(f) + 
                         H5F_SIZEOF_SIZE(f)); 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__cont_free(void *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(mesg);

    mesg = H5FL_FREE(H5O_cont_t, mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__cont_delete(H5F_t *f, H5O_t *open_oh, void *_mesg)
{
    H5O_cont_t *mesg      = (H5O_cont_t *)_mesg;
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(mesg);

    
    
    if (H5O__chunk_delete(f, open_oh, mesg->chunkno) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, FAIL, "unable to remove chunk from cache");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__cont_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_cont_t *cont = (const H5O_cont_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(cont);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth, "Continuation address:", cont->addr);

    Rfprintf(stream, "%*s%-*s %lu\n", indent, "", fwidth,
            "Continuation size in bytes:", (unsigned long)(cont->size));
    Rfprintf(stream, "%*s%-*s %d\n", indent, "", fwidth, "Points to chunk number:", (int)(cont->chunkno));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
