/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5ESmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5ESpkg.h"     
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5RSprivate.h" 
#include "H5VLprivate.h" 

typedef struct H5ES_get_requests_ctx_t {
    hid_t *connector_ids; 
    void **requests;      
    size_t array_len;     
    size_t i;             
} H5ES_get_requests_ctx_t;

typedef struct H5ES_wait_ctx_t {
    H5ES_t  *es;              
    uint64_t timeout;         
    size_t  *num_in_progress; 
    bool    *op_failed;       
} H5ES_wait_ctx_t;

typedef struct H5ES_cancel_ctx_t {
    H5ES_t *es;               
    size_t *num_not_canceled; 
    bool   *op_failed;        
} H5ES_cancel_ctx_t;

typedef struct H5ES_gei_ctx_t {
    H5ES_t          *es;            
    size_t           num_err_info;  
    size_t           curr_err;      
    H5ES_err_info_t *curr_err_info; 
} H5ES_gei_ctx_t;

static herr_t H5ES__close(H5ES_t *es);
static herr_t H5ES__close_cb(void *es, void **request_token);
static herr_t H5ES__insert(H5ES_t *es, H5VL_connector_t *connector, void *request_token, const char *app_file,
                           const char *app_func, unsigned app_line, const char *caller, const char *api_args);
static int    H5ES__get_requests_cb(H5ES_event_t *ev, void *_ctx);
static herr_t H5ES__handle_fail(H5ES_t *es, H5ES_event_t *ev);
static herr_t H5ES__op_complete(H5ES_t *es, H5ES_event_t *ev, H5VL_request_status_t ev_status);
static int    H5ES__wait_cb(H5ES_event_t *ev, void *_ctx);
static int    H5ES__cancel_cb(H5ES_event_t *ev, void *_ctx);
static int    H5ES__get_err_info_cb(H5ES_event_t *ev, void *_ctx);
static int    H5ES__close_failed_cb(H5ES_event_t *ev, void *_ctx);

bool H5_PKG_INIT_VAR = false;

static const H5I_class_t H5I_EVENTSET_CLS[1] = {{
    H5I_EVENTSET,              
    0,                         
    0,                         
    (H5I_free_t)H5ES__close_cb 
}};

H5FL_DEFINE_STATIC(H5ES_t);

herr_t
H5ES__init_package(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5I_register_type(H5I_EVENTSET_CLS) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTINIT, FAIL, "unable to initialize interface");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5ES_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR) {
        
        n += (H5I_dec_type_ref(H5I_EVENTSET) > 0);

        
        if (0 == n)
            H5_PKG_INIT_VAR = false;
    } 

    FUNC_LEAVE_NOAPI(n)
} 

static herr_t
H5ES__close_cb(void *_es, void H5_ATTR_UNUSED **rt)
{
    H5ES_t *es        = (H5ES_t *)_es; 
    herr_t  ret_value = SUCCEED;       

    FUNC_ENTER_PACKAGE

    
    assert(es);

    
    if (H5ES__close(es) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CLOSEERROR, FAIL, "unable to close event set");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5ES_t *
H5ES__create(void)
{
    H5ES_t *es        = NULL; 
    H5ES_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (es = H5FL_CALLOC(H5ES_t)))
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTALLOC, NULL, "can't allocate event set object");

    
    ret_value = es;

done:
    if (!ret_value)
        if (es && H5ES__close(es) < 0)
            HDONE_ERROR(H5E_EVENTSET, H5E_CANTRELEASE, NULL, "unable to free event set");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5ES__insert(H5ES_t *es, H5VL_connector_t *connector, void *request_token, const char *app_file,
             const char *app_func, unsigned app_line, const char *caller, const char *api_args)
{
    H5ES_event_t *ev          = NULL;    
    bool          ev_inserted = false;   
    herr_t        ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(es);

    
    if (NULL == (ev = H5ES__event_new(connector, request_token)))
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTCREATE, FAIL, "can't create event object");

    
    
    ev->op_info.app_file_name = app_file;
    ev->op_info.app_func_name = app_func;
    ev->op_info.app_line_num  = app_line;

    
    ev->op_info.op_ins_count = es->op_counter++;

    
    ev->op_info.op_ins_ts    = H5_now_usec();
    ev->op_info.op_exec_ts   = UINT64_MAX;
    ev->op_info.op_exec_time = UINT64_MAX;

    
    
    ev->op_info.api_name = caller;
    assert(ev->op_info.api_args == NULL);
    if (api_args && NULL == (ev->op_info.api_args = H5MM_xstrdup(api_args)))
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTALLOC, FAIL, "can't copy API routine arguments");

    
    H5ES__list_append(&es->active, ev);
    ev_inserted = true;

    
    if (es->ins_func) {
        int status = -1;

        
        H5_BEFORE_USER_CB(FAIL)
            {
                status = (es->ins_func)(&ev->op_info, es->ins_ctx);
            }
        H5_AFTER_USER_CB(FAIL)
        if (status < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CALLBACK, FAIL, "'insert' callback for event set failed");
    }

done:
    
    if (ret_value < 0)
        if (ev) {
            if (ev_inserted)
                H5ES__list_remove(&es->active, ev);
            if (H5ES__event_free(ev) < 0)
                HDONE_ERROR(H5E_EVENTSET, H5E_CANTRELEASE, FAIL, "unable to release event");
        }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5ES_insert(hid_t es_id, H5VL_connector_t *connector, void *token, const char *caller,
            const char *caller_args, ...)
{
    H5ES_t     *es = NULL;             
    const char *app_file;              
    const char *app_func;              
    unsigned    app_line;              
    H5RS_str_t *rs = NULL;             
    const char *api_args;              
    va_list     ap;                    
    bool        arg_started = false;   
    herr_t      ret_value   = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(connector);
    assert(token);
    assert(caller);
    assert(caller_args);

    
    if (NULL == (es = (H5ES_t *)H5I_object_verify(es_id, H5I_EVENTSET)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an event set");

    
    if (es->err_occurred)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTINSERT, FAIL, "event set has failed operations");

    
    va_start(ap, caller_args);
    arg_started = true;

    
    (void)va_arg(ap, char *); 
    app_file = va_arg(ap, char *);
    (void)va_arg(ap, char *); 
    app_func = va_arg(ap, char *);
    (void)va_arg(ap, char *); 
    app_line = va_arg(ap, unsigned);

    
    if (NULL == (rs = H5RS_create(NULL)))
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTALLOC, FAIL, "can't allocate ref-counted string");

    
    
    assert(0 == strncmp(caller_args, "*s*sIu", 6));
    if (H5_trace_args(rs, caller_args + 6, ap) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTSET, FAIL, "can't create formatted API arguments");
    if (NULL == (api_args = H5RS_get_str(rs)))
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTGET, FAIL, "can't get pointer to formatted API arguments");

    
    if (H5ES__insert(es, connector, token, app_file, app_func, app_line, caller, api_args) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTINSERT, FAIL, "event set has failed operations");

done:
    
    if (arg_started)
        va_end(ap);
    if (rs)
        H5RS_decr(rs);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5ES__insert_request(H5ES_t *es, H5VL_connector_t *connector, void *token)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(es);
    assert(connector);
    assert(token);

    
    if (H5ES__insert(es, connector, token, NULL, NULL, 0, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTINSERT, FAIL, "event set has failed operations");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5ES__get_requests_cb(H5ES_event_t *ev, void *_ctx)
{
    H5ES_get_requests_ctx_t *ctx       = (H5ES_get_requests_ctx_t *)_ctx; 
    int                      ret_value = H5_ITER_CONT;                    

    FUNC_ENTER_PACKAGE

    
    assert(ev);
    assert(ctx);
    assert(ctx->i < ctx->array_len);

    
    if (ctx->connector_ids)
        if ((ctx->connector_ids[ctx->i] = H5VL_conn_register(H5VL_OBJ_CONNECTOR(ev->request))) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTREGISTER, H5_ITER_ERROR, "unable to register VOL connector ID");

    
    if (ctx->requests)
        ctx->requests[ctx->i] = H5VL_OBJ_DATA(ev->request);

    
    if (++ctx->i == ctx->array_len)
        ret_value = H5_ITER_STOP;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5ES__get_requests(H5ES_t *es, H5_iter_order_t order, hid_t *connector_ids, void **requests, size_t array_len)
{
    H5ES_get_requests_ctx_t ctx;                 
    herr_t                  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(es);
    assert(array_len > 0);
    assert(requests || connector_ids);

    
    ctx.connector_ids = connector_ids;
    ctx.requests      = requests;
    ctx.array_len     = array_len;
    ctx.i             = 0;

    
    if (H5ES__list_iterate(&es->active, order, H5ES__get_requests_cb, &ctx) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_BADITER, FAIL, "iteration failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5ES__handle_fail(H5ES_t *es, H5ES_event_t *ev)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(es);
    assert(es->active.head);
    assert(ev);

    
    es->err_occurred = true;

    
    H5ES__list_remove(&es->active, ev);

    
    H5ES__list_append(&es->failed, ev);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5ES__op_complete(H5ES_t *es, H5ES_event_t *ev, H5VL_request_status_t ev_status)
{
    H5VL_request_specific_args_t vol_cb_args;                    
    hid_t                        err_stack_id = H5I_INVALID_HID; 
    herr_t                       ret_value    = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    assert(es);
    assert(ev);
    assert(H5VL_REQUEST_STATUS_SUCCEED == ev_status || H5VL_REQUEST_STATUS_FAIL == ev_status ||
           H5VL_REQUEST_STATUS_CANCELED == ev_status);

    
    if (H5VL_REQUEST_STATUS_SUCCEED == ev_status || H5VL_REQUEST_STATUS_CANCELED == ev_status) {
        
        if (es->comp_func) {
            H5ES_status_t op_status; 
            int           status = -1;

            
            if (H5VL_REQUEST_STATUS_SUCCEED == ev_status) {
                
                op_status = H5ES_STATUS_SUCCEED;

                
                vol_cb_args.op_type                      = H5VL_REQUEST_GET_EXEC_TIME;
                vol_cb_args.args.get_exec_time.exec_ts   = &ev->op_info.op_exec_ts;
                vol_cb_args.args.get_exec_time.exec_time = &ev->op_info.op_exec_time;

                
                if (H5VL_request_specific(ev->request, &vol_cb_args) < 0)
                    HGOTO_ERROR(H5E_EVENTSET, H5E_CANTGET, FAIL,
                                "unable to retrieve execution time info for operation");
            }
            else
                
                op_status = H5ES_STATUS_CANCELED;

            
            H5_BEFORE_USER_CB(FAIL)
                {
                    status = (es->comp_func)(&ev->op_info, op_status, H5I_INVALID_HID, es->comp_ctx);
                }
            H5_AFTER_USER_CB(FAIL)
            if (status < 0)
                HGOTO_ERROR(H5E_EVENTSET, H5E_CALLBACK, FAIL, "'complete' callback for event set failed");
        } 

        
        if (H5ES__event_completed(ev, &es->active) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTRELEASE, FAIL, "unable to release completed event");
    } 
    else if (H5VL_REQUEST_STATUS_FAIL == ev_status) {
        
        if (es->comp_func) {
            
            vol_cb_args.op_type                         = H5VL_REQUEST_GET_ERR_STACK;
            vol_cb_args.args.get_err_stack.err_stack_id = H5I_INVALID_HID;
            int status                                  = -1;

            
            if (H5VL_request_specific(ev->request, &vol_cb_args) < 0)
                HGOTO_ERROR(H5E_EVENTSET, H5E_CANTGET, FAIL, "unable to retrieve error stack for operation");

            
            err_stack_id = vol_cb_args.args.get_err_stack.err_stack_id;

            
            H5_BEFORE_USER_CB(FAIL)
                {
                    status = (es->comp_func)(&ev->op_info, H5ES_STATUS_FAIL, err_stack_id, es->comp_ctx);
                }
            H5_AFTER_USER_CB(FAIL)
            if (status < 0)
                HGOTO_ERROR(H5E_EVENTSET, H5E_CALLBACK, FAIL, "'complete' callback for event set failed");
        } 

        
        if (H5ES__handle_fail(es, ev) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTSET, FAIL, "unable to handle failed event");
    } 
    else
        HGOTO_ERROR(H5E_EVENTSET, H5E_BADVALUE, FAIL, "unknown event status?!?");

done:
    
    if (H5I_INVALID_HID != err_stack_id)
        if (H5I_dec_ref(err_stack_id) < 0)
            HDONE_ERROR(H5E_EVENTSET, H5E_CANTDEC, FAIL,
                        "unable to decrement ref count on error stack for failed operation")

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5ES__wait_cb(H5ES_event_t *ev, void *_ctx)
{
    H5ES_wait_ctx_t      *ctx       = (H5ES_wait_ctx_t *)_ctx;     
    H5VL_request_status_t ev_status = H5VL_REQUEST_STATUS_SUCCEED; 
    uint64_t start_time = 0, elapsed_time = 0; 
    int      ret_value = H5_ITER_CONT;         

    FUNC_ENTER_PACKAGE

    
    assert(ev);
    assert(ctx);

    
    if (ctx->timeout != H5ES_WAIT_NONE && ctx->timeout != H5ES_WAIT_FOREVER)
        start_time = H5_now_usec();
    if (H5VL_request_wait(ev->request, ctx->timeout, &ev_status) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTWAIT, H5_ITER_ERROR, "unable to test operation");
    if (ctx->timeout != H5ES_WAIT_NONE && ctx->timeout != H5ES_WAIT_FOREVER)
        elapsed_time = H5_now_usec() - start_time;

    
    if (ev_status == H5VL_REQUEST_STATUS_FAIL) {
        
        if (H5ES__op_complete(ctx->es, ev, ev_status) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTRELEASE, H5_ITER_ERROR, "unable to release completed event");

        
        *ctx->op_failed = true;

        
        ret_value = H5_ITER_STOP;
    } 
    else if (ev_status == H5VL_REQUEST_STATUS_SUCCEED || ev_status == H5VL_REQUEST_STATUS_CANCELED) {
        
        if (H5ES__op_complete(ctx->es, ev, ev_status) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTRELEASE, H5_ITER_ERROR, "unable to release completed event");
    } 
    else if (ev_status == H5VL_REQUEST_STATUS_CANT_CANCEL)
        
        HGOTO_ERROR(H5E_EVENTSET, H5E_BADVALUE, H5_ITER_ERROR,
                    "received \"can't cancel\" status for operation");
    else {
        
        assert(ev_status == H5VL_REQUEST_STATUS_IN_PROGRESS);

        
        (*ctx->num_in_progress)++;
    } 

    
    if (ctx->timeout != H5ES_WAIT_NONE && ctx->timeout != H5ES_WAIT_FOREVER) {
        
        if ((elapsed_time * 1000) > ctx->timeout)
            ctx->timeout = H5ES_WAIT_NONE;
        else
            ctx->timeout -= (elapsed_time * 1000); 
    }                                              

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5ES__wait(H5ES_t *es, uint64_t timeout, size_t *num_in_progress, bool *op_failed)
{
    H5ES_wait_ctx_t ctx;                 
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(es);
    assert(num_in_progress);
    assert(op_failed);

    
    *num_in_progress = 0;
    *op_failed       = false;

    
    ctx.es              = es;
    ctx.timeout         = timeout;
    ctx.num_in_progress = num_in_progress;
    ctx.op_failed       = op_failed;

    
    if (H5ES__list_iterate(&es->active, H5_ITER_NATIVE, H5ES__wait_cb, &ctx) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_BADITER, FAIL, "iteration failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5ES__cancel_cb(H5ES_event_t *ev, void *_ctx)
{
    H5ES_cancel_ctx_t    *ctx       = (H5ES_cancel_ctx_t *)_ctx;   
    H5VL_request_status_t ev_status = H5VL_REQUEST_STATUS_SUCCEED; 
    int                   ret_value = H5_ITER_CONT;                

    FUNC_ENTER_PACKAGE

    
    assert(ev);
    assert(ctx);

    
    if (H5VL_request_cancel(ev->request, &ev_status) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTCANCEL, H5_ITER_ERROR, "unable to cancel operation");

    
    if (ev_status == H5VL_REQUEST_STATUS_FAIL) {
        
        if (H5ES__op_complete(ctx->es, ev, ev_status) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTSET, H5_ITER_ERROR, "unable to handle failed event");

        
        *ctx->op_failed = true;

        
        ret_value = H5_ITER_STOP;
    } 
    else if (ev_status == H5VL_REQUEST_STATUS_SUCCEED) {
        
        (*ctx->num_not_canceled)++;

        
        if (H5ES__op_complete(ctx->es, ev, ev_status) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTRELEASE, H5_ITER_ERROR, "unable to release completed event");
    } 
    else if (ev_status == H5VL_REQUEST_STATUS_CANT_CANCEL || ev_status == H5VL_REQUEST_STATUS_IN_PROGRESS) {
        
        (*ctx->num_not_canceled)++;
    } 
    else {
        
        assert(ev_status == H5VL_REQUEST_STATUS_CANCELED);

        
        if (H5ES__op_complete(ctx->es, ev, ev_status) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTRELEASE, H5_ITER_ERROR, "unable to release completed event");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5ES__cancel(H5ES_t *es, size_t *num_not_canceled, bool *op_failed)
{
    H5ES_cancel_ctx_t ctx;                 
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(es);
    assert(num_not_canceled);
    assert(op_failed);

    
    *num_not_canceled = 0;
    *op_failed        = false;

    
    ctx.es               = es;
    ctx.num_not_canceled = num_not_canceled;
    ctx.op_failed        = op_failed;

    
    if (H5ES__list_iterate(&es->active, H5_ITER_NATIVE, H5ES__cancel_cb, &ctx) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_BADITER, FAIL, "iteration failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5ES__get_err_info_cb(H5ES_event_t *ev, void *_ctx)
{
    H5VL_request_specific_args_t vol_cb_args;                        
    H5ES_gei_ctx_t              *ctx       = (H5ES_gei_ctx_t *)_ctx; 
    int                          ret_value = H5_ITER_CONT;           

    FUNC_ENTER_PACKAGE

    
    assert(ev);
    assert(ctx);

    
    
    if (NULL == (ctx->curr_err_info->api_name = H5MM_xstrdup(ev->op_info.api_name)))
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTALLOC, H5_ITER_ERROR, "can't copy HDF5 API routine name");
    if (NULL == (ctx->curr_err_info->api_args = H5MM_xstrdup(ev->op_info.api_args)))
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTALLOC, H5_ITER_ERROR, "can't copy HDF5 API routine arguments");
    if (NULL == (ctx->curr_err_info->app_file_name = H5MM_xstrdup(ev->op_info.app_file_name)))
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTALLOC, H5_ITER_ERROR, "can't copy HDF5 application file name");
    if (NULL == (ctx->curr_err_info->app_func_name = H5MM_xstrdup(ev->op_info.app_func_name)))
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTALLOC, H5_ITER_ERROR, "can't copy HDF5 application function name");
    ctx->curr_err_info->app_line_num = ev->op_info.app_line_num;
    ctx->curr_err_info->op_ins_count = ev->op_info.op_ins_count;
    ctx->curr_err_info->op_ins_ts    = ev->op_info.op_ins_ts;
    ctx->curr_err_info->op_exec_ts   = ev->op_info.op_exec_ts;
    ctx->curr_err_info->op_exec_time = ev->op_info.op_exec_time;

    
    vol_cb_args.op_type                         = H5VL_REQUEST_GET_ERR_STACK;
    vol_cb_args.args.get_err_stack.err_stack_id = H5I_INVALID_HID;

    
    if (H5VL_request_specific(ev->request, &vol_cb_args) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTGET, H5_ITER_ERROR, "unable to retrieve error stack for operation");

    
    ctx->curr_err_info->err_stack_id = vol_cb_args.args.get_err_stack.err_stack_id;

    
    H5ES__list_remove(&ctx->es->failed, ev);

    
    if (H5ES__event_free(ev) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTRELEASE, H5_ITER_ERROR, "unable to release failed event");

    
    ctx->curr_err++;
    ctx->curr_err_info++;

    
    if (ctx->curr_err == ctx->num_err_info)
        ret_value = H5_ITER_STOP;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5ES__get_err_info(H5ES_t *es, size_t num_err_info, H5ES_err_info_t err_info[], size_t *num_cleared)
{
    H5ES_gei_ctx_t ctx;                 
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(es);
    assert(num_err_info);
    assert(err_info);
    assert(num_cleared);

    
    ctx.es            = es;
    ctx.num_err_info  = num_err_info;
    ctx.curr_err      = 0;
    ctx.curr_err_info = &err_info[0];

    
    if (H5ES__list_iterate(&es->failed, H5_ITER_NATIVE, H5ES__get_err_info_cb, &ctx) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_BADITER, FAIL, "iteration failed");

    
    *num_cleared = ctx.curr_err;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5ES__close_failed_cb(H5ES_event_t *ev, void *_ctx)
{
    H5ES_t *es        = (H5ES_t *)_ctx; 
    int     ret_value = H5_ITER_CONT;   

    FUNC_ENTER_PACKAGE

    
    assert(ev);
    assert(es);

    
    H5ES__list_remove(&es->failed, ev);

    
    if (H5ES__event_free(ev) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTRELEASE, H5_ITER_ERROR, "unable to release failed event");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5ES__close(H5ES_t *es)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(es);

    
    if (H5ES__list_count(&es->active) > 0)
        HGOTO_ERROR(
            H5E_EVENTSET, H5E_CANTCLOSEOBJ, FAIL,
            "can't close event set while unfinished operations are present (i.e. wait on event set first)");

    
    if (H5ES__list_iterate(&es->failed, H5_ITER_NATIVE, H5ES__close_failed_cb, (void *)es) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_BADITER, FAIL, "iteration failed");

    
    es = H5FL_FREE(H5ES_t, es);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
