% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/latent.fit.R
\name{latent.fit}
\alias{latent.fit}
\title{Fitting procedure of the latent variable model}
\usage{
latent.fit(
  y,
  sites,
  niter,
  nburn,
  nthin,
  quiet,
  trace,
  gev.vary,
  spatial.covariates,
  log.scale,
  correlation,
  mcmc.init,
  mcmc.prior,
  mcmc.jumps
)
}
\arguments{
\item{y}{A matrix of observed block maxima. Each column corresponds to a site position.}

\item{sites}{The coordinates of the sites where the data are observed. Each row corresponds to a site position.}

\item{niter}{The number of MCMC iterations.}

\item{nburn}{The number of first MCMC iterations that are discarded. Zero by default.}

\item{nthin}{The size of the MCMC thinning. One by default (i.e. no thinning).}

\item{quiet}{A logical indicating if the progression of the routine should be displayed. TRUE by default.}

\item{trace}{If \code{quiet} is FALSE, the log-likelihood of the model is displayed each block of \code{trace} MCMC steps to observe fitting progression.}

\item{gev.vary}{A logical vector of size three indicating if the GEV parameters (respectively the location, the scale and the shape) are spatially-varying. If not (by default for the shape), the parameter is the same at each position.}

\item{spatial.covariates}{A numerical matrix of spatial covariates. Each row corresponds to a site position. See details.}

\item{log.scale}{A logical value indicating if the GEV scale parameter \eqn{\sigma} is modelled by its log. FALSE by default. See details.}

\item{correlation}{A character string indicating the form of the correlation function associated to the latent Gaussian processes that describes the marginal parameters. Must be one of \code{"expo"}, \code{"gauss"}, \code{"mat32"} (By default) and \code{"mat52"}, respectively corresponding to the exponential, Gaussian, Matern-3/2 and Matern-5/2 correlation functions.}

\item{mcmc.init}{A named list indicating the initial states of the chains. See details.}

\item{mcmc.prior}{A named list indicating the hyperparameters of the prior distributions. See details.}

\item{mcmc.jumps}{A named list indicating the amplitude of the jumps to propose the MCMC candidates. See details.}
}
\value{
A named list with following elements (less elements if \code{fit.margins} is \code{FALSE}):
\itemize{
\item{\code{GEV}: the Markov chains associated to the GEV parameters. The dimensions of the array correspond respectively to the sites positions, the three GEV parameters and the states of the Markov chains.}
\item{\code{llik}: the log-likelihood of the model for each step of the algorithm.}
\item{\code{time}: time (in sec) spent for the fit.}
\item{\code{spatial}: a named list with four elements linked to the GEV spatially-varying parameters:
\itemize{
\item{\code{vary}: the argument \code{gev.vary}.}
\item{\code{beta}: the \eqn{\beta} parameters for each GEV parameter. The dimensions correspond respectively to the steps of the Markov chains, the \eqn{p} spatial covariates and the GEV parameters}
\item{\code{sills}: the Markov chains associated to the sills in the correlation functions of the latent Gaussian processes.}
\item{\code{ranges}: the Markov chains associated to the ranges in the correlation functions of the latent Gaussian processes.}}}
\item{\code{data}: the data fitted.}
\item{\code{sites}: the sites where the data are observed.}
\item{\code{spatial.covariates}: the spatial covariates.}
\item{\code{correlation}: the type of correlation function for the marginal latent processes.}
\item{\code{nstep}: the number of steps at the end of the routine after burn-in and thinning.}
\item{\code{log.scale}: a boolean indicating if the scale parameter has been modelled via its logarithm.}
\item{\code{fit.type}: "latent" character string to specify the type of fit.}}
}
\description{
Metropolis-within-Gibbs algorithm that returns posterior distribution (as Markov chains) for the marginal GEV parameters of an observed spatial process.
This function is close to \code{hkevp.fit} but with less parameters since conditional independence is assumed and only the margins are estimated.
In \code{SpatialExtremes} library, a similar function can be found under the name \code{latent}.
}
\details{
Details of the MCMC procedure are presented in \cite{Davison et al. (2012)}. This function follows the indications and the choices of the authors, with the exception of several small changes:
\itemize{
\item{The scale parameter \eqn{\sigma} can be modelled like the two other marginal parameters as in \cite{Davison et al. (2012)} or by its logarithm as in \cite{Reich and Shaby (2012)}. For this, use the argument \code{log.scale}, set to FALSE by default.}

\item{The Inverse-Gamma prior distributions defined for the bandwith parameter \eqn{\tau} and for the ranges \eqn{\lambda} of the latent processes are replaced by a Beta distribution over the interval \eqn{[0,2D_{max}]}, where \eqn{D_{max}} stands for the maximum distance between two sites.}}

If the the parameters are assumed spatially-varying, the user can provide spatial covariates to fit the mean of the latent Gaussian processes. Recall for instance for the GEV location parameter that:
\deqn{\mu(s) = \beta_{0,\mu} + \beta_{1,\mu} c_1(s) + ... + \beta_{p,\mu} c_p(s) ~.}
The given matrix \code{spatial.covariates} that represents the \eqn{c_i(s)} elements should have the first column filled with ones to account for the intercept \eqn{\beta_0}.

The arguments \code{mcmc.init}, \code{mcmc.prior} and \code{mcmc.jumps} are named list that have default values. The user can make point changes in these arguments, by setting \code{mcmc.init = list(loc = .5)} for instance, but must respect the constraints of each element:
\itemize{
\item{\code{mcmc.init}: all elements are of length one. The possible elements are:
\itemize{
\item{\code{loc}, \code{scale} and \code{shape} (GEV parameters).}
\item{\code{range} and \code{sill} of the correlation functions.}}}
\item{mcmc.prior: the possible elements are:
\itemize{
\item{\code{constant.gev}: a \eqn{2 \times 3} matrix of normal parameters for spatially-constant \eqn{\mu}, \eqn{\sigma} and \eqn{\xi}. The first row are the means, the second are the standard deviations.}
\item{\code{beta.sd}: the normal sd prior of all \eqn{\beta} parameters (a single value).}
\item{\code{range}: the two Beta parameters.}
\item{\code{sill}: the two Inverse-Gamma parameters.}}}
\item{mcmc.jumps: the possible elements are \code{gev} and \code{range}, vectors of length 3 (for each GEV parameter).}}
}
\examples{
\donttest{
# Simulation of HKEVP:
sites <- as.matrix(expand.grid(1:4,1:4))
loc <- sites[,1]*10
scale <- 3
shape <- .2
alpha <- 1
tau <- 2
ysim <- hkevp.rand(15, sites, sites, loc, scale, shape, alpha, tau)

# Latent Variable Model fit:
set.seed(1)
fit <- latent.fit(ysim, sites, niter = 10000, nburn = 5000, nthin = 5)
mcmc.plot(fit, TRUE)
par(mfrow = c(2,2))
apply(fit$GEV[,1,], 1, acf)
}




}
\references{
Reich, B. J., & Shaby, B. A. (2012). A hierarchical max-stable spatial model for extreme precipitation. The annals of applied statistics, 6(4), 1430. <DOI:10.1214/12-AOAS591>

Davison, A. C., Padoan, S. A., & Ribatet, M. (2012). Statistical modeling of spatial extremes. Statistical Science, 27(2), 161-186. <DOI:10.1214/11-STS376>
}
\seealso{
hkevp.fit
}
\author{
Quentin Sebille
}
