% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_BKP.R, R/simulate_DKP.R
\name{simulate}
\alias{simulate}
\alias{simulate.BKP}
\alias{simulate.DKP}
\title{Simulate from a Fitted BKP or DKP Model}
\usage{
\method{simulate}{BKP}(object, Xnew, n_sim = 1, threshold = NULL, seed = NULL, ...)

\method{simulate}{DKP}(object, Xnew, n_sim = 1, seed = NULL, ...)
}
\arguments{
\item{object}{An object of class \code{"BKP"} or \code{"DKP"}, typically
returned by \code{\link{fit.BKP}} or \code{\link{fit.DKP}}.}

\item{Xnew}{A numeric matrix or vector of new input locations for simulation.}

\item{n_sim}{Number of posterior samples to generate (default = \code{1}).}

\item{threshold}{Classification threshold for binary output (only used for
BKP). If specified, the output will include binary class labels with values
above the threshold classified as 1 (default is \code{NULL}).}

\item{seed}{Optional integer seed for reproducibility.}

\item{...}{Additional arguments (currently unused).}
}
\value{
A list with the following components:
\describe{
\item{\code{sims}}{
For \strong{BKP}: A numeric matrix of dimension \code{nrow(Xnew) × n_sim}, containing simulated success probabilities.\cr
For \strong{DKP}: A numeric array of dimension \code{n_sim × q × nrow(Xnew)}, containing simulated class probabilities
from Dirichlet posteriors, where \code{q} is the number of classes.
}

\item{\code{mean}}{
For \strong{BKP}: A numeric vector of posterior mean success probabilities at each \code{Xnew}.\cr
For \strong{DKP}: A numeric matrix of dimension \code{nrow(Xnew) × q}, containing posterior mean class probabilities.
}

\item{\code{class}}{
For \strong{BKP}: A binary matrix of dimension \code{nrow(Xnew) × n_sim} indicating simulated class labels (0/1),
returned if \code{threshold} is specified.\cr
For \strong{DKP}: A numeric matrix of dimension \code{nrow(Xnew) × n_sim} containing MAP-predicted class labels,
returned only when training data is single-label (i.e., each row of \code{Y} sums to 1).
}
}
}
\description{
Generates random samples from the posterior predictive
distribution of a fitted BKP or DKP model at new input locations.

For BKP models, posterior samples are drawn from Beta distributions
representing success probabilities, with optional binary class labels
determined by a threshold.

For DKP models, posterior samples are drawn from Dirichlet distributions
representing class probabilities, with optional class labels determined by
the maximum a posteriori (MAP) rule if training responses are one-hot
encoded.
}
\examples{
## -------------------- BKP Simulation Example --------------------
set.seed(123)

# Define true success probability function
true_pi_fun <- function(x) {
  (1 + exp(-x^2) * cos(10 * (1 - exp(-x)) / (1 + exp(-x)))) / 2
}

n <- 30
Xbounds <- matrix(c(-2,2), nrow=1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)
y <- rbinom(n, size = m, prob = true_pi)

# Fit BKP model
model <- fit.BKP(X, y, m, Xbounds=Xbounds)

# Simulate 5 posterior draws of success probabilities
Xnew <- matrix(seq(-2, 2, length.out = 100), ncol = 1)
simulate(model, Xnew, n_sim = 5)

# Simulate binary classifications (threshold = 0.5)
simulate(model, Xnew, n_sim = 5, threshold = 0.5)

## -------------------- DKP Simulation Example --------------------
set.seed(123)

# Define true class probability function (3-class)
true_pi_fun <- function(X) {
  p <- (1 + exp(-X^2) * cos(10 * (1 - exp(-X)) / (1 + exp(-X)))) / 2
  return(matrix(c(p/2, p/2, 1 - p), nrow = length(p)))
}

n <- 30
Xbounds <- matrix(c(-2, 2), nrow = 1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model <- fit.DKP(X, Y, Xbounds = Xbounds)

# Simulate 5 draws from posterior Dirichlet distributions at new point
Xnew <- matrix(seq(-2, 2, length.out = 100), ncol = 1)
simulate(model, Xnew = Xnew, n_sim = 5)

}
\seealso{
\code{\link{fit.BKP}}, \code{\link{fit.DKP}},
\code{\link{predict.BKP}}, \code{\link{predict.DKP}}
}
\keyword{BKP}
\keyword{DKP}
