% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/S4-BuyseTest-get.R
\docType{methods}
\name{getPairScore}
\alias{getPairScore}
\alias{getPairScore,S4BuyseTest-method}
\title{Extract the Score of Each Pair}
\usage{
\S4method{getPairScore}{S4BuyseTest}(
  object,
  endpoint = NULL,
  strata = NULL,
  cumulative = FALSE,
  rm.withinStrata = TRUE,
  rm.strata = is.na(object@strata),
  rm.indexPair = TRUE,
  rm.weight = FALSE,
  rm.corrected = (object@correction.uninf == 0),
  unlist = TRUE,
  trace = 1
)
}
\arguments{
\item{object}{an \R object of class \code{\linkS4class{S4BuyseTest}}, i.e., output of \code{\link{BuyseTest}}}

\item{endpoint}{[integer/character vector] the endpoint for which the scores should be output.}

\item{strata}{[character vector] the strata relative to which the score should be output.}

\item{cumulative}{[logical] should the scores be cumulated over endpoints?}

\item{rm.withinStrata}{[logical] should the columns indicating the position of each member of the pair
within each treatment group be removed?}

\item{rm.strata}{[logical] should the column containing the level of the strata variable be removed from the output?}

\item{rm.indexPair}{[logical] should the column containing the number associated to each pair be removed from the output?}

\item{rm.weight}{[logical] should the column weight be removed from the output?}

\item{rm.corrected}{[logical] should the columns corresponding to the scores after weighting be removed from the output?}

\item{unlist}{[logical] should the structure of the output be simplified when possible?}

\item{trace}{[logical] should a message be printed to explain what happened
when the function returned \code{NULL}?}
}
\description{
Extract the score of each pair.
}
\details{
The maximal output (i.e. with all columns) contains for each endpoint, a data.table with:
\itemize{
\item \code{"strata"}: the name of the strata to which the pair belongs.
\item \code{"index.T"}: the index of the treatment observation in the pair relative to the original dataset.
\item \code{"index.C"}: the index of the control observation in the pair relative to the original dataset.
\item \code{"indexWithinStrata.T"}: the index of the treatment observation in the pair relative to the treatment group and the strata.
\item \code{"indexWithinStrata.C"}: the index of the control observation in the pair relative to the control group and the strata.
\item \code{"favorable"}: the probability that the endpoint is better in the treatment arm vs. in the control arm.
\item \code{"unfavorable"}: the probability that the endpoint is worse in the treatment arm vs. in the control arm.
\item \code{"neutral"}: the probability that the endpoint is no different in the treatment arm vs. in the control arm.
\item \code{"uninformative"}: the weight of the pair that cannot be attributed to favorable/unfavorable/neutral.
\item \code{"weight"}: the residual weight of the pair to be analyzed at the current outcome. Each pair starts with a weight of 1.
\item \code{"favorable.corrected"}: same as \code{"favorable"}  after weighting.
\item \code{"unfavorable.corrected"}: same as \code{"favorable"} after weighting.
\item \code{"neutral.corrected"}: same as \code{"favorable"} after weighting.
\item \code{"uninformative.corrected"}: same as \code{"favorable"} after weighting.
}
Note that the \code{.T} and \code{.C} may change since they correspond of the label of the treatment and control arms.
The first weighting consists in multiplying the probability by the residual weight of the pair
(i.e. the weight of the pair that was not informative at the previous endpoint). This is always performed.
For time to event endpoint an additional weighting may be performed to avoid a possible bias in presence of censoring.
}
\examples{
library(data.table)
library(prodlim)

## run BuyseTest
library(survival) ## import veteran

BT.keep <- BuyseTest(trt ~ tte(time, threshold = 20, status = "status") + cont(karno),
                     data = veteran, keep.pairScore = TRUE, 
                     trace = 0, method.inference = "none")

## Extract scores
pScore <- getPairScore(BT.keep, endpoint = 1)

## look at one pair
indexPair <- intersect(which(pScore$index.1 == 22),
                       which(pScore$index.2 == 71))
pScore[indexPair]

## retrive pair in the original dataset
pVeteran <- veteran[pScore[indexPair,c(index.1,index.2)],]
pVeteran

## the observation from the control group is censored at 97
## the observation from the treatment group has an event at 112
## since the threshold is 20, and (112-20)<97
## we know that the pair is not in favor of the treatment

## the formula for probability in favor of the control is
## Sc(97)/Sc(112+20)
## where Sc(t) is the survival at time t in the control arm.

## we first estimate the survival in each arm
e.KM <- prodlim(Hist(time,status)~trt, data = veteran)

## and compute the survival
iSurv <- predict(e.KM, times =  c(97,112+20),
                 newdata = data.frame(trt = 1, stringsAsFactors = FALSE))[[1]]

## the probability in favor of the control is then
pUF <- iSurv[2]/iSurv[1]
pUF
## and the complement to one of that is the probability of being neutral
pN <- 1 - pUF
pN

if(require(testthat)){
   testthat::expect_equal(pUF, pScore[indexPair, unfavorable])
   testthat::expect_equal(pN, pScore[indexPair, neutral])
}
}
\author{
Brice Ozenne
}
\keyword{S4BuyseTest-method}
\keyword{get}
