% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_FadingCorr.R
\name{calc_FadingCorr}
\alias{calc_FadingCorr}
\title{Fading Correction after Huntley & Lamothe (2001)}
\usage{
calc_FadingCorr(
  age.faded,
  g_value,
  tc,
  tc.g_value = tc,
  n.MC = 10000,
  seed = NULL,
  interval = c(0.01, 500),
  txtProgressBar = TRUE,
  verbose = TRUE
)
}
\arguments{
\item{age.faded}{\link{numeric} \link{vector} (\strong{required}):
vector of length 2 containing the uncorrected age and the error in ka
(see example).}

\item{g_value}{\link{vector} or \linkS4class{RLum.Results} (\strong{required}):
either a vector of length 2 containing the g-value and error obtained from
separate fading measurements (see example), or an \linkS4class{RLum.Results}
object produced by \link{analyse_FadingMeasurement}. If the latter, the \code{tc}
argument is set automatically.}

\item{tc}{\link{numeric} (\strong{required}):
time in seconds between irradiation and the prompt measurement (cf. Huntley & Lamothe 2001).
The argument is ignored when \code{g_value} is an \linkS4class{RLum.Results} object.}

\item{tc.g_value}{\link{numeric} (\emph{with default}):
time in seconds between irradiation and the prompt measurement used in the
estimation of the g-value. If the g-value was normalised, the normalisation
time (in seconds) should be given, e.g., for a g-value normalised to 2 days,
the value 172800 should be used. If nothing is provided the time is set to
\code{tc}, which is usual case for g-values obtained using the
SAR method and \eqn{g}-values that have been not normalised to 2 days.}

\item{n.MC}{\link{integer} (\emph{with default}):
number of Monte Carlo simulation runs for error estimation.
If \code{n.MC = 'auto'} is used the function tries to find a 'stable' error for
the age. See details for further information.
\strong{Note:} This may take a while!}

\item{seed}{\link{integer} (\emph{optional}):
sets the seed for the random number generator in R using \link{set.seed}}

\item{interval}{\link{numeric} (\emph{with default}):
a vector containing the end-points (age interval) of the interval to be searched for the root in 'ka'.
This argument is passed to the function \link[stats:uniroot]{stats::uniroot} used for solving the equation.}

\item{txtProgressBar}{\link{logical} (\emph{with default}):
enable/disable the progress bar.}

\item{verbose}{\link{logical} (\emph{with default}):
enable/disable output to the terminal.}
}
\value{
Returns an S4 object of type \linkS4class{RLum.Results}.\cr

Slot: \strong{\verb{@data}}\cr
\tabular{lll}{
\strong{Object} \tab \strong{Type} \tab \strong{Comment} \cr
\code{age.corr} \tab \link{data.frame} \tab Corrected age \cr
\code{age.corr.MC} \tab \link{numeric} \tab MC simulation results with all possible ages from that simulation \cr
}

Slot: \strong{\verb{@info}}\cr

\tabular{lll}{
\strong{Object} \tab \strong{Type} \tab \strong{Comment} \cr
\code{info} \tab \link{character} \tab the original function call
}
}
\description{
Apply a fading correction according to Huntley & Lamothe (2001) for a given
\eqn{g}-value and a given \eqn{t_{c}}
}
\details{
This function solves the equation used for correcting the fading affected age
including the error for a given \eqn{g}-value according to Huntley & Lamothe (2001):

\deqn{
\frac{A_{f}}{A} = 1 - \kappa * \Big[ln(\frac{A}{t_c}) - 1\Big]
}

with \eqn{\kappa} defined as

\deqn{
\kappa = \frac{\frac{\mathrm{g\_value}}{ln(10)}}{100}
}

\eqn{A} and \eqn{A_{f}} are given in ka. \eqn{t_c} is given in s, however, it
is internally recalculated to ka.

As the \eqn{g}-value slightly depends on the time \eqn{t_{c}} between
irradiation and the prompt measurement, a value for \code{tc} must always be
provided.
If the \eqn{g}-value was normalised to a distinct
time or evaluated with a different tc value (e.g., external irradiation), also
the \eqn{t_{c}} value for the \eqn{g}-value needs to be provided (argument \code{tc.g_value}
and then the \eqn{g}-value is recalculated
to \eqn{t_{c}} of the measurement used for estimating the age applying the
following equation:

\deqn{\kappa_{tc} = \kappa_{tc.g} / (1 - \kappa_{tc.g} * ln(tc/tc.g))}

where

\deqn{\kappa_{tc.g} = g / 100 / ln(10)}

The error of the fading-corrected age is determined using a Monte Carlo
simulation approach. Solving of the equation is performed using
\link{uniroot}. Large values for \code{n.MC} will significantly
increase the computation time.\cr

\strong{\code{n.MC = 'auto'}}

The error estimation based on a stochastic process, i.e. for a small number of
MC runs the calculated error varies considerably every time the function is called,
even with the same input values.
The argument option \code{n.MC = 'auto'} tries to find a stable value for the standard error, i.e.
the standard deviation of values calculated during the MC runs (\code{age.corr.MC}),
within a given precision (2 digits) by increasing the number of MC runs stepwise and
calculating the corresponding error.

If the determined error does not differ from the 9 values calculated previously
within a precision of (here) 3 digits the calculation is stopped as it is assumed
that the error is stable. Please note that (a) the duration depends on the input
values as well as on the provided computation resources and it may take a while,
(b) the length (size) of the output
vector \code{age.corr.MC}, where all the single values produced during the MC runs
are stored, equals the number of MC runs (here termed observations).

To avoid an endless loop the calculation is stopped if the number of observations
exceeds 10^7.
This limitation can be overwritten by setting the number of MC runs manually,
e.g. \code{n.MC = 10000001}. Note: For this case the function is not checking whether the calculated
error is stable.\cr

\strong{\code{seed}}

This option allows to recreate previously calculated results by setting the seed
for the R random number generator (see \link{set.seed} for details). This option
should not be mixed up with the option \strong{\code{n.MC = 'auto'}}. The results may
appear similar, but they are not comparable!\cr

\strong{FAQ}\cr

\strong{Q}: Which \eqn{t_{c}} value is expected?\cr

\strong{A}: \eqn{t_{c}} is the time in seconds between irradiation and the prompt measurement
applied during your \eqn{D_{e}} measurement. However, this \eqn{t_{c}} might
differ from the \eqn{t_{c}} used for estimating the \eqn{g}-value. In the
case of an SAR measurement \eqn{t_{c}} should be similar, however,
if it differs, you have to provide this
\eqn{t_{c}}  value (the one used for estimating the \eqn{g}-value) using
the argument \code{tc.g_value}.\cr

\strong{Q}: The function could not find a solution, what should I do?\cr

\strong{A}: This usually happens for model parameters exceeding the boundaries of the
fading correction model (e.g., very high \eqn{g}-value). Please check
whether another fading correction model might be more appropriate.
}
\note{
Special thanks to Sébastien Huot for his support and clarification via e-mail.
}
\section{Function version}{
 0.4.4
}

\examples{

##run the examples given in the appendix of Huntley and Lamothe, 2001

##(1) faded age: 100 a
results <- calc_FadingCorr(
   age.faded = c(0.1,0),
   g_value = c(5.0, 1.0),
   tc = 2592000,
   tc.g_value = 172800,
   n.MC = 100)

##(2) faded age: 1 ka
results <- calc_FadingCorr(
   age.faded = c(1,0),
   g_value = c(5.0, 1.0),
   tc = 2592000,
   tc.g_value = 172800,
   n.MC = 100)

##(3) faded age: 10.0 ka
results <- calc_FadingCorr(
   age.faded = c(10,0),
   g_value = c(5.0, 1.0),
   tc = 2592000,
   tc.g_value = 172800,
   n.MC = 100)

##access the last output
get_RLum(results)

} 

\section{How to cite}{
Kreutzer, S., 2025. calc_FadingCorr(): Fading Correction after Huntley & Lamothe (2001). Function version 0.4.4. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., Mercier, N., Philippe, A., Riedesel, S., Autzen, M., Mittelstrass, D., Gray, H.J., Galharret, J., Colombo, M., Steinbuch, L., Boer, A.d., 2025. Luminescence: Comprehensive Luminescence Dating Data Analysis. R package version 1.1.0. https://r-lum.github.io/Luminescence/
}

\references{
Huntley, D.J., Lamothe, M., 2001. Ubiquity of anomalous fading
in K-feldspars and the measurement and correction for it in optical dating.
Canadian Journal of Earth Sciences, 38, 1093-1106.
}
\seealso{
\linkS4class{RLum.Results}, \link{analyse_FadingMeasurement}, \link{get_RLum}, \link{uniroot}
}
\author{
Sebastian Kreutzer, Institute of Geography, Heidelberg University (Germany)
, RLum Developer Team}
\keyword{datagen}
