% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convert_Wavelength2Energy.R
\name{convert_Wavelength2Energy}
\alias{convert_Wavelength2Energy}
\title{Emission Spectra Conversion from Wavelength to Energy Scales (Jacobian Conversion)}
\usage{
convert_Wavelength2Energy(object, digits = 3L, order = FALSE)
}
\arguments{
\item{object}{\linkS4class{RLum.Data.Spectrum}, \link{data.frame}, \link{matrix} (\strong{required}): input object to be converted.
If the input is not an \linkS4class{RLum.Data.Spectrum}, the first column is always treated as the wavelength
column. The function supports a list of allowed input objects.}

\item{digits}{\link{integer} (\emph{with default}): set the number of digits on the returned energy axis}

\item{order}{\link{logical} (\emph{with default}): enable/disable sorting of the
values in ascending energy order. After the conversion, the longest
wavelength has the lowest energy value and the shortest wavelength the
highest. While this is correct, some R functions expect increasing x-values.}
}
\value{
The same object class as provided as input is returned.
}
\description{
The function provides a convenient and fast way to convert emission spectra wavelength
to energy scales. The function works on \linkS4class{RLum.Data.Spectrum}, \link{data.frame} and \link{matrix} and
a \link{list} of such objects. The function was written to smooth the workflow while analysing
emission spectra data. This is in particular useful if you want to further treat your data
and apply, e.g., a signal deconvolution.
}
\details{
The intensity of the spectrum is re-calculated using the following approach to recalculate
wavelength and corresponding intensity values
(e.g., Appendix 4 in Blasse and Grabmaier, 1994; Mooney and Kambhampati, 2013):

\deqn{\phi_{E} = \phi_{\lambda} * \lambda^2 / (hc)}

with
\eqn{\phi_{E}} the intensity per interval of energy \eqn{E} (1/eV),
\eqn{\phi_{\lambda}} the intensity per interval of wavelength \eqn{\lambda}
(1/nm) and
\eqn{h} (eV * s) the Planck constant and \eqn{c} (nm/s) the velocity of light.

For transforming the wavelength axis (x-values) the equation as follow is used

\deqn{E = hc/\lambda}
}
\note{
This conversion works solely for emission spectra. In case of absorption spectra only
the x-axis has to be converted.
}
\section{Function version}{
 0.1.1
}

\examples{

##=====================##
##(1) Literature example after Mooney et al. (2013)
##(1.1) create matrix
m <- matrix(
  data = c(seq(400, 800, 50), rep(1, 9)), ncol = 2)

##(1.2) set plot function to reproduce the
##literature figure
p <- function(m) {
 plot(x = m[, 1], y = m[, 2])
 polygon(
 x = c(m[, 1], rev(m[, 1])),
 y = c(m[, 2], rep(0, nrow(m))))
 for (i in 1:nrow(m)) {
  lines(x = rep(m[i, 1], 2), y = c(0, m[i, 2]))
 }
}

##(1.3) plot curves
par(mfrow = c(1,2))
p(m)
p(convert_Wavelength2Energy(m))

##=====================##
##(2) Another example using density curves
##create dataset
xy <- density(
 c(rnorm(n = 100, mean = 500, sd = 20),
 rnorm(n = 100, mean = 800, sd = 20)))
xy <- data.frame(xy$x, xy$y)

##plot
par(mfrow = c(1,2))
plot(
 xy,
 type = "l",
 xlim = c(150, 1000),
 xlab = "Wavelength [nm]",
 ylab = "Luminescence [a.u.]"
)
plot(
 convert_Wavelength2Energy(xy),
 xy$y,
 type = "l",
 xlim = c(1.23, 8.3),
 xlab = "Energy [eV]",
 ylab = "Luminescence [a.u.]"
)

} 

\section{How to cite}{
Kreutzer, S., 2025. convert_Wavelength2Energy(): Emission Spectra Conversion from Wavelength to Energy Scales (Jacobian Conversion). Function version 0.1.1. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., Mercier, N., Philippe, A., Riedesel, S., Autzen, M., Mittelstrass, D., Gray, H.J., Galharret, J., Colombo, M., Steinbuch, L., Boer, A.d., 2025. Luminescence: Comprehensive Luminescence Dating Data Analysis. R package version 1.1.0. https://r-lum.github.io/Luminescence/
}

\references{
Blasse, G., Grabmaier, B.C., 1994. Luminescent Materials. Springer.

Mooney, J., Kambhampati, P., 2013. Get the Basics Right: Jacobian Conversion of Wavelength and
Energy Scales for Quantitative Analysis of Emission Spectra. J. Phys. Chem. Lett. 4, 3316–3318.
\doi{10.1021/jz401508t}

Mooney, J., Kambhampati, P., 2013. Correction to “Get the Basics Right: Jacobian Conversion of
Wavelength and Energy Scales for Quantitative Analysis of Emission Spectra.” J. Phys. Chem. Lett. 4,
3316–3318. \doi{10.1021/jz401508t}

\strong{Further reading}

Angulo, G., Grampp, G., Rosspeintner, A., 2006. Recalling the appropriate representation of
electronic spectra. Spectrochimica Acta Part A: Molecular and Biomolecular Spectroscopy 65,
727–731. \doi{10.1016/j.saa.2006.01.007}

Wang, Y., Townsend, P.D., 2013. Potential problems in collection and data processing of
luminescence signals. Journal of Luminescence 142, 202–211. \doi{10.1016/j.jlumin.2013.03.052}
}
\seealso{
\linkS4class{RLum.Data.Spectrum}, \link{plot_RLum}
}
\author{
Sebastian Kreutzer, Institute of Geography, Heidelberg University (Germany)
, RLum Developer Team}
\keyword{IO}
