% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/source_compete.R
\name{bsm_bc}
\alias{bsm_bc}
\title{Bayesian spline model with a Box-Cox transformation}
\usage{
bsm_bc(
  y,
  x = NULL,
  x_test = NULL,
  psi = NULL,
  lambda = NULL,
  sample_lambda = TRUE,
  nsave = 1000,
  nburn = 1000,
  nskip = 0,
  verbose = TRUE
)
}
\arguments{
\item{y}{\code{n x 1} vector of observed counts}

\item{x}{\code{n x 1} vector of observation points; if NULL, assume equally-spaced on [0,1]}

\item{x_test}{\code{n_test x 1} vector of testing points; if NULL, assume equal to \code{x}}

\item{psi}{prior variance (inverse smoothing parameter); if NULL,
sample this parameter}

\item{lambda}{Box-Cox transformation; if NULL, estimate this parameter}

\item{sample_lambda}{logical; if TRUE, sample lambda, otherwise
use the fixed value of lambda above or the MLE (if lambda unspecified)}

\item{nsave}{number of MCMC iterations to save}

\item{nburn}{number of MCMC iterations to discard}

\item{nskip}{number of MCMC iterations to skip between saving iterations,
i.e., save every (nskip + 1)th draw}

\item{verbose}{logical; if TRUE, print time remaining}
}
\value{
a list with the following elements:
\itemize{
\item \code{coefficients} the posterior mean of the regression coefficients
\item \code{fitted.values} the posterior predictive mean at the test points \code{x_test}
\item \code{post_theta}: \code{nsave x p} samples from the posterior distribution
of the regression coefficients
\item \code{post_ypred}: \code{nsave x n_test} samples
from the posterior predictive distribution at \code{x_test}
\item \code{post_g}: \code{nsave} posterior samples of the transformation
evaluated at the unique \code{y} values
\item \code{post_lambda} \code{nsave} posterior samples of lambda
\item \code{model}: the model fit (here, \code{sbsm_bc})
}
as well as the arguments passed in.
}
\description{
MCMC sampling for Bayesian spline regression with a
(known or unknown) Box-Cox transformation.
}
\details{
This function provides fully Bayesian inference for a
transformed spline model via MCMC sampling. The transformation is
parametric from the Box-Cox family, which has one parameter \code{lambda}.
That parameter may be fixed in advanced or learned from the data.
}
\note{
Box-Cox transformations may be useful in some cases, but
in general we recommend the nonparametric transformation (with
Monte Carlo, not MCMC sampling) in \code{\link{sbsm}}.
}
\examples{
# Simulate some data:
n = 100 # sample size
x = sort(runif(n)) # observation points

# Transform a noisy, periodic function:
y = g_inv_bc(
  sin(2*pi*x) + sin(4*pi*x) + rnorm(n, sd = .5),
             lambda = .5) # Signed square-root transformation

# Fit the Bayesian spline model with a Box-Cox transformation:
fit = bsm_bc(y = y, x = x)
names(fit) # what is returned
round(quantile(fit$post_lambda), 3) # summary of unknown Box-Cox parameter

# Plot the model predictions (point and interval estimates):
pi_y = t(apply(fit$post_ypred, 2, quantile, c(0.05, .95))) # 90\% PI
plot(x, y, type='n', ylim = range(pi_y,y),
     xlab = 'x', ylab = 'y', main = paste('Fitted values and prediction intervals'))
polygon(c(x, rev(x)),c(pi_y[,2], rev(pi_y[,1])),col='gray', border=NA)
lines(x, y, type='p')
lines(x, fitted(fit), lwd = 3)

}
