% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_extreme_event.R
\name{derive_extreme_event}
\alias{derive_extreme_event}
\title{Add the Worst or Best Observation for Each By Group as New Records}
\usage{
derive_extreme_event(
  dataset = NULL,
  by_vars,
  events,
  tmp_event_nr_var = NULL,
  order,
  mode,
  source_datasets = NULL,
  check_type = "warning",
  set_values_to = NULL,
  keep_source_vars = exprs(everything())
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by the \code{by_vars} and \code{order} arguments are expected to be in the dataset.}

\item{by_vars}{Grouping variables

\emph{Default}: \code{NULL}

\emph{Permitted Values}: list of variables created by \code{exprs()}
e.g. \code{exprs(USUBJID, VISIT)}}

\item{events}{Conditions and new values defining events

A list of \code{event()} or \code{event_joined()} objects is expected. Only
observations listed in the \code{events} are considered for deriving extreme
event. If multiple records meet the filter \code{condition}, take the first
record sorted by \code{order}. The data is grouped by \code{by_vars}, i.e., summary
functions like \code{all()} or \code{any()} can be used in \code{condition}.

For \code{event_joined()} events the observations are selected by calling
\code{filter_joined()}. The \code{condition} field is passed to the \code{filter_join} argument.}

\item{tmp_event_nr_var}{Temporary event number variable

The specified variable is added to all source datasets and is set to the
number of the event before selecting the records of the event.

It can be used in \code{order} to determine which record should be used if
records from more than one event are selected.

The variable is not included in the output dataset.}

\item{order}{Sort order

If a particular event from \code{events} has more than one observation, within
the event and by group, the records are ordered by the specified order.

For handling of \code{NA}s in sorting variables see \href{../articles/generic.html#sort_order}{Sort Order}.

\emph{Permitted Values:} list of expressions created by \code{exprs()}, e.g.,
\code{exprs(ADT, desc(AVAL))}}

\item{mode}{Selection mode (first or last)

If a particular event from \code{events} has more than one observation,
\code{"first"}/\code{"last"} is used to select the first/last record of this type of
event sorting by \code{order}.

\emph{Permitted Values:} \code{"first"}, \code{"last"}}

\item{source_datasets}{Source datasets

A named list of datasets is expected. The \code{dataset_name} field of \code{event()}
and \code{event_joined()} refers to the dataset provided in the list.}

\item{check_type}{Check uniqueness?

If \code{"warning"} or \code{"error"} is specified, the specified message is issued
if the observations of the input dataset are not unique with respect to the
by variables and the order.

\emph{Default:} \code{"warning"}

\emph{Permitted Values:} \code{"none"}, \code{"warning"}, \code{"error"}}

\item{set_values_to}{Variables to be set

The specified variables are set to the specified values for the new
observations.

Set a list of variables to some specified value for the new records
\itemize{
\item LHS refer to a variable.
\item RHS refers to the values to set to the variable. This can be a string, a
symbol, a numeric value, an expression or NA. If summary functions are
used, the values are summarized by the variables specified for \code{by_vars}.
}

For example:

\if{html}{\out{<div class="sourceCode">}}\preformatted{  set_values_to = exprs(
    AVAL = sum(AVAL),
    DTYPE = "AVERAGE",
  )
}\if{html}{\out{</div>}}}

\item{keep_source_vars}{Variables to keep from the source dataset

For each event the specified variables are kept from the selected
observations. The variables specified for \code{by_vars} and created by
\code{set_values_to} are always kept. The \code{keep_source_vars} field of
the event will take precedence over the value of the \code{keep_source_vars}
argument.

\emph{Permitted Values}: A list of expressions where each element is
a symbol or a tidyselect expression, e.g., \code{exprs(VISIT, VISITNUM, starts_with("RS"))}.}
}
\value{
The input dataset with the best or worst observation of each by group
added as new observations.
}
\description{
Add the first available record from \code{events} for each by group as new
records, all variables of the selected observation are kept. It can be used
for selecting the extreme observation from a series of user-defined events.
This distinguishes \code{derive_extreme_event()} from \code{derive_extreme_records()},
where extreme records are derived based on certain order of existing
variables.
}
\details{
\enumerate{
\item For each event select the observations to consider:
\enumerate{
\item If the event is of class \code{event}, the observations of the source dataset
are restricted by \code{condition} and then the first or last (\code{mode})
observation per by group (\code{by_vars}) is selected.

If the event is of class \code{event_joined}, \code{filter_joined()} is called to
select the observations.
\item The variables specified by the \code{set_values_to} field of the event
are added to the selected observations.
\item The variable specified for \code{tmp_event_nr_var} is added and set to
the number of the event.
\item Only the variables specified for the \code{keep_source_vars} field of the
event, and the by variables (\code{by_vars}) and the variables created by
\code{set_values_to} are kept. If \code{keep_source_vars = NULL} is used for an event
in \code{derive_extreme_event()} the value of the \code{keep_source_vars} argument of
\code{derive_extreme_event()} is used.
}
\item All selected observations are bound together.
\item For each group (with respect to the variables specified for the
\code{by_vars} parameter) the first or last observation (with respect to the
order specified for the \code{order} parameter and the mode specified for the
\code{mode} parameter) is selected.
\item The variables specified by the \code{set_values_to} parameter are added to
the selected observations.
\item The observations are added to input dataset.
}

\strong{Note:} This function creates temporary datasets which may be much bigger
than the input datasets. If this causes memory issues, please try setting
the admiral option \code{save_memory} to \code{TRUE} (see \code{set_admiral_options()}).
This reduces the memory consumption but increases the run-time.
}
\examples{
library(tibble)
library(dplyr)
library(lubridate)

adqs <- tribble(
  ~USUBJID, ~PARAMCD,       ~AVALC,        ~ADY,
  "1",      "NO SLEEP",     "N",              1,
  "1",      "WAKE UP",      "N",              2,
  "1",      "FALL ASLEEP",  "N",              3,
  "2",      "NO SLEEP",     "N",              1,
  "2",      "WAKE UP",      "Y",              2,
  "2",      "WAKE UP",      "Y",              3,
  "2",      "FALL ASLEEP",  "N",              4,
  "3",      "NO SLEEP",     NA_character_,    1
)

# Add a new record for each USUBJID storing the the worst sleeping problem.
derive_extreme_event(
  adqs,
  by_vars = exprs(USUBJID),
  events = list(
    event(
      condition = PARAMCD == "NO SLEEP" & AVALC == "Y",
      set_values_to = exprs(AVALC = "No sleep", AVAL = 1)
    ),
    event(
      condition = PARAMCD == "WAKE UP" & AVALC == "Y",
      set_values_to = exprs(AVALC = "Waking up more than three times", AVAL = 2)
    ),
    event(
      condition = PARAMCD == "FALL ASLEEP" & AVALC == "Y",
      set_values_to = exprs(AVALC = "More than 30 mins to fall asleep", AVAL = 3)
    ),
    event(
      condition = all(AVALC == "N"),
      set_values_to = exprs(
        AVALC = "No sleeping problems", AVAL = 4
      )
    ),
    event(
      condition = TRUE,
      set_values_to = exprs(AVALC = "Missing", AVAL = 99)
    )
  ),
  tmp_event_nr_var = event_nr,
  order = exprs(event_nr, desc(ADY)),
  mode = "first",
  set_values_to = exprs(
    PARAMCD = "WSP",
    PARAM = "Worst Sleeping Problems"
  )
)

# Use different mode by event
adhy <- tribble(
  ~USUBJID, ~AVISITN, ~CRIT1FL,
  "1",             1, "Y",
  "1",             2, "Y",
  "2",             1, "Y",
  "2",             2, NA_character_,
  "2",             3, "Y",
  "2",             4, NA_character_
) \%>\%
  mutate(
    PARAMCD = "ALKPH",
    PARAM = "Alkaline Phosphatase (U/L)"
  )

derive_extreme_event(
  adhy,
  by_vars = exprs(USUBJID),
  events = list(
    event(
      condition = is.na(CRIT1FL),
      set_values_to = exprs(AVALC = "N")
    ),
    event(
      condition = CRIT1FL == "Y",
      mode = "last",
      set_values_to = exprs(AVALC = "Y")
    )
  ),
  tmp_event_nr_var = event_nr,
  order = exprs(event_nr, AVISITN),
  mode = "first",
  keep_source_vars = exprs(AVISITN),
  set_values_to = exprs(
    PARAMCD = "ALK2",
    PARAM = "ALKPH <= 2 times ULN"
  )
)

# Derive confirmed best overall response (using event_joined())
# CR - complete response, PR - partial response, SD - stable disease
# NE - not evaluable, PD - progressive disease
adsl <- tribble(
  ~USUBJID, ~TRTSDTC,
  "1",      "2020-01-01",
  "2",      "2019-12-12",
  "3",      "2019-11-11",
  "4",      "2019-12-30",
  "5",      "2020-01-01",
  "6",      "2020-02-02",
  "7",      "2020-02-02",
  "8",      "2020-02-01"
) \%>\%
  mutate(TRTSDT = ymd(TRTSDTC))

adrs <- tribble(
  ~USUBJID, ~ADTC,        ~AVALC,
  "1",      "2020-01-01", "PR",
  "1",      "2020-02-01", "CR",
  "1",      "2020-02-16", "NE",
  "1",      "2020-03-01", "CR",
  "1",      "2020-04-01", "SD",
  "2",      "2020-01-01", "SD",
  "2",      "2020-02-01", "PR",
  "2",      "2020-03-01", "SD",
  "2",      "2020-03-13", "CR",
  "4",      "2020-01-01", "PR",
  "4",      "2020-03-01", "NE",
  "4",      "2020-04-01", "NE",
  "4",      "2020-05-01", "PR",
  "5",      "2020-01-01", "PR",
  "5",      "2020-01-10", "PR",
  "5",      "2020-01-20", "PR",
  "6",      "2020-02-06", "PR",
  "6",      "2020-02-16", "CR",
  "6",      "2020-03-30", "PR",
  "7",      "2020-02-06", "PR",
  "7",      "2020-02-16", "CR",
  "7",      "2020-04-01", "NE",
  "8",      "2020-02-16", "PD"
) \%>\%
  mutate(
    ADT = ymd(ADTC),
    PARAMCD = "OVR",
    PARAM = "Overall Response by Investigator"
  ) \%>\%
  derive_vars_merged(
    dataset_add = adsl,
    by_vars = exprs(USUBJID),
    new_vars = exprs(TRTSDT)
  )

derive_extreme_event(
  adrs,
  by_vars = exprs(USUBJID),
  tmp_event_nr_var = event_nr,
  order = exprs(event_nr, ADT),
  mode = "first",
  source_datasets = list(adsl = adsl),
  events = list(
    event_joined(
      description = paste(
        "CR needs to be confirmed by a second CR at least 28 days later",
        "at most one NE is acceptable between the two assessments"
      ),
      join_vars = exprs(AVALC, ADT),
      join_type = "after",
      first_cond_upper = AVALC.join == "CR" &
        ADT.join >= ADT + 28,
      condition = AVALC == "CR" &
        all(AVALC.join \%in\% c("CR", "NE")) &
        count_vals(var = AVALC.join, val = "NE") <= 1,
      set_values_to = exprs(
        AVALC = "CR"
      )
    ),
    event_joined(
      description = paste(
        "PR needs to be confirmed by a second CR or PR at least 28 days later,",
        "at most one NE is acceptable between the two assessments"
      ),
      join_vars = exprs(AVALC, ADT),
      join_type = "after",
      first_cond_upper = AVALC.join \%in\% c("CR", "PR") &
        ADT.join >= ADT + 28,
      condition = AVALC == "PR" &
        all(AVALC.join \%in\% c("CR", "PR", "NE")) &
        count_vals(var = AVALC.join, val = "NE") <= 1,
      set_values_to = exprs(
        AVALC = "PR"
      )
    ),
    event(
      description = paste(
        "CR, PR, or SD are considered as SD if occurring at least 28",
        "after treatment start"
      ),
      condition = AVALC \%in\% c("CR", "PR", "SD") & ADT >= TRTSDT + 28,
      set_values_to = exprs(
        AVALC = "SD"
      )
    ),
    event(
      condition = AVALC == "PD",
      set_values_to = exprs(
        AVALC = "PD"
      )
    ),
    event(
      condition = AVALC \%in\% c("CR", "PR", "SD", "NE"),
      set_values_to = exprs(
        AVALC = "NE"
      )
    ),
    event(
      description = "set response to MISSING for patients without records in ADRS",
      dataset_name = "adsl",
      condition = TRUE,
      set_values_to = exprs(
        AVALC = "MISSING"
      ),
      keep_source_vars = exprs(TRTSDT)
    )
  ),
  set_values_to = exprs(
    PARAMCD = "CBOR",
    PARAM = "Best Confirmed Overall Response by Investigator"
  )
) \%>\%
  filter(PARAMCD == "CBOR")

}
\seealso{
\code{\link[=event]{event()}}, \code{\link[=event_joined]{event_joined()}}, \code{\link[=derive_vars_extreme_event]{derive_vars_extreme_event()}}

BDS-Findings Functions for adding Parameters/Records: 
\code{\link{default_qtc_paramcd}()},
\code{\link{derive_expected_records}()},
\code{\link{derive_extreme_records}()},
\code{\link{derive_locf_records}()},
\code{\link{derive_param_bmi}()},
\code{\link{derive_param_bsa}()},
\code{\link{derive_param_computed}()},
\code{\link{derive_param_doseint}()},
\code{\link{derive_param_exist_flag}()},
\code{\link{derive_param_exposure}()},
\code{\link{derive_param_framingham}()},
\code{\link{derive_param_map}()},
\code{\link{derive_param_qtc}()},
\code{\link{derive_param_rr}()},
\code{\link{derive_param_wbc_abs}()},
\code{\link{derive_summary_records}()}
}
\concept{der_prm_bds_findings}
\keyword{der_prm_bds_findings}
