#' Calculates the consistency ratio of each decision-maker
#'
#' @author Frankie Cho
#'
#' @description The `ahp.cr` function calculates the consistency ratio of each decision-maker, defined by the following equation:
#'
#' \deqn{CR = (\lambda-n)/((n-1)(RI))}
#'
#' Where \eqn{\lambda} is the maximum eigenvalue of the pairwise comparison matrix, \eqn{n} is the number of attributes, and RI is the random index. Following \insertCite{Saaty2007fuzzy;textual}{ahpsurvey}, the RI is a function of \eqn{n} and is the consistency ratio of randomly generated pairwise comparison matrices.
#'
#' @param ahpmat A list of pairwise comparison matrices of each decision maker generated by `ahp.mat`.
#' @param atts a list of attributes in the correct order. The RI is asymptotic as it approaches n=15, thus it is set to be equal to 1.6 if the number of attributes exceeds 16.
#' @param ri A user-supplied random index value, probably user generated using `ahp.ri`.
#'
#' @return A `list` of consistency ratios of each decision-maker.
#'
#' @include ahp_mat.R
#'
#' @examples
#'
#' data(city200)
#' atts <- c('cult', 'fam', 'house', 'jobs', 'trans')
#'
#' cityahp <- ahp.mat(df = city200, atts = atts, negconvert = TRUE) 
#' ahp.cr(cityahp, atts)
#'
#'@references
#'\insertAllCited{}
#'
#'@export

ahp.cr <- function(ahpmat, atts, ri = NULL) {
    respmat <- ahpmat
    maxeigen <- vector()
    CI <- vector()
    CR <- vector()
    RI <- c(0.0000000, 0.0000000, 0.5251686, 0.8836651, 1.1081014, 1.2492774, 1.3415514, 1.4048466, 1.4507197, 1.4857266, 1.5141022,
            1.5356638, 1.5545925, 1.5703498, 1.5839958, 1.5955704, 1.6053208, 1.6140648, 1.6218900, 1.6288505, 1.6355145, 1.6410749,
            1.6462439, 1.6509834, 1.6554325, 1.6592237, 1.6631050, 1.6662368, 1.6696396, 1.6723214, 1.6751007, 1.6778474, 1.6801459,
            1.6824754, 1.6844494, 1.6865981, 1.6884438, 1.6901943, 1.6918461, 1.6935071, 1.6950605, 1.6965334, 1.6979425, 1.6992006,
            1.7004654, 1.7016392, 1.7027780, 1.7038778, 1.7050314, 1.7060381, rep(1.7060381,1000))
    
    if (length(atts) > 50 & is.null(ri)) {
      warning("CR may not be accurate for more than 50 dimensions by default. Using `ahp.ri` to generate your own RI with your number of dimensions is recommended.")
    }
    
    ## Computing the maximum eigenvalue of the matrix and using it for calculation for the
    ## CR
    for (i in 1:length(respmat)) {
        eigenvalues <- eigen(respmat[[i]])$values
        maxeigen[i] <- max(Re(eigenvalues[abs(Im(eigenvalues)) < 0.000001]))
        CI[i] <- (maxeigen[i] - length(atts))/(length(atts) - 1)
        
        if (is.null(ri)){
          CR[i] <- CI[i]/RI[length(atts)]
          ## As a general rule, CR should be lower than 0.1.
        } else {
          CR[i] <- CI[i]/ri
        }
    }
    return(CR)
}
