% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dists.R
\name{dodgr_distances}
\alias{dodgr_distances}
\title{Calculate matrix of pair-wise distances between points.}
\usage{
dodgr_distances(
  graph,
  from = NULL,
  to = NULL,
  shortest = TRUE,
  pairwise = FALSE,
  heap = "BHeap",
  parallel = TRUE,
  quiet = TRUE
)
}
\arguments{
\item{graph}{\code{data.frame} or equivalent object representing the network
graph (see Notes). For \code{dodgr} street networks, this may be a network derived
from either \pkg{sf} or \pkg{silicate} ("sc") data, generated with
\link{weight_streetnet}.

The \code{from} and \code{to} columns of \code{graph} may be either single
columns of numeric or character values specifying the numbers or names of
graph vertices, or combinations to two columns specifying geographical
(longitude and latitude,) coordinates. In the latter case, almost any sensible
combination of names will be accepted (for example, \verb{fromx, fromy},
\verb{from_x, from_y}, or \verb{fr_lat, fr_lon}.)

Note that longitude and latitude values are always interpreted in 'dodgr' to
be in EPSG:4326 / WSG84 coordinates. Any other kinds of coordinates should
first be reprojected to EPSG:4326 before submitting to any 'dodgr' routines.

See further information in Details.}

\item{from}{Vector or matrix of points \strong{from} which route distances are to
be calculated, specified as one of the following:
\itemize{
\item Single character vector precisely matching node numbers or names
given in \code{graph$from} or \code{graph$to}.
\item Single vector of integer-ish values, in which case these will be
presumed to specify indices into \link{dodgr_vertices}, and NOT to
correspond to values in the 'from' or 'to' columns of the graph. See the
example below for a demonstration.
\item Matrix or equivalent of longitude and latitude coordinates, in which
case these will be matched on to the nearest coordinates of 'from' and 'to'
points in the graph.
}}

\item{to}{Vector or matrix of points \strong{to} which route distances are to be
calculated. If \code{to} is \code{NULL}, pairwise distances will be calculated from
all \code{from} points to all other nodes in \code{graph}. If both \code{from} and \code{to} are
\code{NULL}, pairwise distances are calculated between all nodes in \code{graph}.}

\item{shortest}{If \code{FALSE}, calculate distances along the \emph{fastest}
rather than shortest routes. For street networks produced with
\link{weight_streetnet}, distances may also be calculated along the
\emph{fastest} routes with the \code{shortest = FALSE} option. Graphs must in
this case have columns of \code{time} and \code{time_weighted}. Note that the fastest
routes will only be approximate when derived from \pkg{sf}-format data
generated with the \pkg{osmdata} function \code{osmdata_sf()}, and will be much
more accurate when derived from \code{sc}-format data generated with
\code{osmdata_sc()}. See \link{weight_streetnet} for details.}

\item{pairwise}{If \code{TRUE}, calculate distances only between the ordered
pairs of \code{from} and \code{to}.}

\item{heap}{Type of heap to use in priority queue. Options include
Fibonacci Heap (default; \code{FHeap}), Binary Heap (\code{BHeap}),
\verb{Trinomial Heap (}TriHeap\verb{), Extended Trinomial Heap (}TriHeapExt\verb{, and 2-3 Heap (}Heap23`).}

\item{parallel}{If \code{TRUE}, perform routing calculation in parallel.
Calculations in parallel ought very generally be advantageous. For small
graphs, calculating distances in parallel is likely to offer relatively
little gain in speed, but increases from parallel computation will generally
markedly increase with increasing graph sizes. By default, parallel
computation uses the maximal number of available cores or threads. This
number can be reduced by specifying a value via
\verb{RcppParallel::setThreadOptions (numThreads = <desired_number>)}. Parallel
calculations are, however, not able to be interrupted (for example, by
\code{Ctrl-C}), and can only be stopped by killing the R process.}

\item{quiet}{If \code{FALSE}, display progress messages on screen.}
}
\value{
square matrix of distances between nodes
}
\description{
Alias for \link{dodgr_dists}
}
\details{
\code{graph} must minimally contain three columns of \code{from},
\code{to}, \code{dist}. If an additional column named \code{weight} or
\code{wt} is present, shortest paths are calculated according to values
specified in that column; otherwise according to \code{dist} values. Either
way, final distances between \code{from} and \code{to} points are calculated
by default according to values of \code{dist}. That is, paths between any pair of
points will be calculated according to the minimal total sum of \code{weight}
values (if present), while reported distances will be total sums of \code{dist}
values.
}
\examples{
# A simple graph
graph <- data.frame (
    from = c ("A", "B", "B", "B", "C", "C", "D", "D"),
    to = c ("B", "A", "C", "D", "B", "D", "C", "A"),
    d = c (1, 2, 1, 3, 2, 1, 2, 1)
)
dodgr_dists (graph)

# Example of "from" and "to" as integer-ish values, in which case they are
# interpreted to index into "dodgr_vertices()":
graph <- data.frame (
    from = c (1, 3, 2, 2, 3, 3, 4, 4),
    to = c (2, 1, 3, 4, 2, 4, 3, 1),
    d = c (1, 2, 1, 3, 2, 1, 2, 1)
)
dodgr_dists (graph, from = 1, to = 2)
# That then gives distance from "1" to "3" because the vertices are built
# sequentially along "graph$from":
dodgr_vertices (graph)
# And vertex$id [2] is "3"

# A larger example from the included [hampi()] data.
graph <- weight_streetnet (hampi)
from <- sample (graph$from_id, size = 100)
to <- sample (graph$to_id, size = 50)
d <- dodgr_dists (graph, from = from, to = to)
# d is a 100-by-50 matrix of distances between `from` and `to`

\dontrun{
# a more complex street network example, thanks to @chrijo; see
# https://github.com/UrbanAnalyst/dodgr/issues/47

xy <- rbind (
    c (7.005994, 51.45774), # limbeckerplatz 1 essen germany
    c (7.012874, 51.45041)
) # hauptbahnhof essen germany
xy <- data.frame (lon = xy [, 1], lat = xy [, 2])
essen <- dodgr_streetnet (pts = xy, expand = 0.2, quiet = FALSE)
graph <- weight_streetnet (essen, wt_profile = "foot")
d <- dodgr_dists (graph, from = xy, to = xy)
# First reason why this does not work is because the graph has multiple,
# disconnected components.
table (graph$component)
# reduce to largest connected component, which is always number 1
graph <- graph [which (graph$component == 1), ]
d <- dodgr_dists (graph, from = xy, to = xy)
# should work, but even then note that
table (essen$level)
# There are parts of the network on different building levels (because of
# shopping malls and the like). These may or may not be connected, so it may
# be necessary to filter out particular levels
index <- which (!(essen$level == "-1" | essen$level == "1")) # for example
library (sf) # needed for following sub-select operation
essen <- essen [index, ]
graph <- weight_streetnet (essen, wt_profile = "foot")
graph <- graph [which (graph$component == 1), ]
d <- dodgr_dists (graph, from = xy, to = xy)
}
}
\seealso{
Other distances: 
\code{\link{dodgr_dists}()},
\code{\link{dodgr_dists_categorical}()},
\code{\link{dodgr_dists_nearest}()},
\code{\link{dodgr_flows_aggregate}()},
\code{\link{dodgr_flows_disperse}()},
\code{\link{dodgr_flows_si}()},
\code{\link{dodgr_isochrones}()},
\code{\link{dodgr_isodists}()},
\code{\link{dodgr_isoverts}()},
\code{\link{dodgr_paths}()},
\code{\link{dodgr_times}()}
}
\concept{distances}
