% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getquantiles.R
\name{PDFtoQuantiles}
\alias{PDFtoQuantiles}
\alias{getquantiles}
\alias{getRTquantiles}
\alias{RTDensityToQuantiles}
\alias{CDFtoQuantiles}
\title{Get Quantiles from vectors of PDF or CDF values}
\usage{
PDFtoQuantiles(pdf_df, p = c(0.1, 0.3, 0.5, 0.7, 0.9), agg_over = NULL,
  scaled = FALSE)

CDFtoQuantiles(cdf, x = NULL, p)
}
\arguments{
\item{pdf_df}{dataframe. Should have at least two columns:
\itemize{
\item \code{rt} (for reaction times) or \code{x} for the support values of the pdf
\item \code{dens} or \code{pdf} for the pdf values
\item All other columns will be used as grouping factors, for which separate quantiles will be returned.
}}

\item{p}{numeric vector. Probabilities for returned quantiles. Default:
c(.1, .3, .5, .7, .9).}

\item{agg_over}{character. Names of columns in \code{pdf_df} to aggregate over (using the mean of
densities, which is valid only, if groups occur with equal probabilities) before
computing the quantiles.}

\item{scaled}{logical. Indicating whether the pdf values are from a proper probability
distribution. Non-scaled pdfs will scaled to 1. If \code{scaled} is TRUE, this may cause
problems with high probabilities. In any case we strongly recommend to cover the most
probability mass with the values in the support vector.}

\item{cdf}{numeric. A increasing vector of the same length as \code{x} giving the CDF for respective x-Values.
Dataframe inputs are accepted. If a column \code{x} is available there, this will be used as support values.}

\item{x}{numeric. A increasing vector of same length as \code{cdf}. Can also be specified as column of \code{cdf}.}
}
\value{
\code{PDFtoQuantiles} returns a \code{tibble} with columns p and q indicating
probabilities and respective quantiles. Furthermore, the output has grouping columns
identical to the additional columns in the input (without \code{rt}/\code{x}, \code{dens} and \code{densscaled}),
but without the ones in the \code{agg_over} argument. \code{CDFtoQuantiles}
returns only a \code{data.frame} with columns \code{p} and \code{q}.
}
\description{
\code{CDFtoQuantiles} computes quantiles for a given CDF.
\code{PDFtoQuantiles} computes the quantiles for given PDF values within
groups of other variables, if available.
}
\details{
For a reasonable accuracy the number of steps in the support column (\code{rt}/\code{x})
should be high, i.e. the distance between values small.
We recommend, to ensure that the support vector in the input to be equidistant,
i.e. the difference between consecutive support values should be constant, though
this is not required.
If both column names \code{x} and \code{rt} are present in \code{pdf_df}, \code{rt} will be preferred.
Attention should be given to the columns of \code{pdf_df} other than \code{rt}/\code{x}
and \code{dens}/\code{pdf}.

The column for the pdf may be scaled to integrate to 1 but do not have to.
\subsection{Quantile computation in the \code{dynConfiR} package}{

As argument \code{pdf_df}, the outputs of \code{predictRT} and \code{predictRTModels} from the
\code{dynConfiR} package can be used. In the context of confidence models grouping factors
often used are conditions, correct/incorrect answers and confidence ratings.
}
}
\examples{
## Demonstrate PDFtoQuantiles
pred <- expand.grid(model = c("dynWEV", "PCRMt"),
                    rt =  seq(0, 15, length.out=1200),
                    condition = c(1,2,3),
                    rating = c(1,2))
pred$dens <- dchisq(pred$rt, 3) # pdf may also be used as column name
head(pred)
res <- PDFtoQuantiles(pred, p=c(0.3, 0.5, 0.7))
head(res)
nrow(res) #= 3(quantiles)*2(models)*3(conditions)*2(rating)
# Compare to true quantiles of Chi-square distribution
qchisq(p=c(0.3, 0.5, 0.7), 3)
res$q[1:3]


res2 <- PDFtoQuantiles(pred, p=c(0.3, 0.5, 0.7), agg_over = "model")
nrow(res2) #=18 because res aggregated over models

\donttest{
  pred$pdf <- dchisq(pred$rt, 3)
  head(pred)
  # following call throws a warning, because both columns pdf and dens are present
  PDFtoQuantiles(pred, p=c(0.3, 0.5, 0.7), agg_over = "model")
}

\donttest{
  pred2 <- data.frame(rt=seq(0, 7, length.out=100))
  pred2$dens <- dchisq(pred2$rt, 5)
  # following call throws a warning, because density is assumed to be scaled (scaled=TRUE), i.e.
  # integrate to 1, but the .95 quantile is not reached in the rt column
  PDFtoQuantiles(pred2, p=c(0.3, 0.5, 0.95), scaled=TRUE) # Gives a warning
}

## Demonstrate CDFtoQuantiles
X <- seq(-2, 2, length.out=300)
pdf_values <- pnorm(X)
CDFtoQuantiles(pdf_values, X, p=c(0.2, 0.5, 0.8))
qnorm(c(0.2, 0.5, 0.8))
}
\author{
Sebastian Hellmann.
}
