% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gdalraster_proc.R
\name{rasterize}
\alias{rasterize}
\title{Burn vector geometries into a raster}
\usage{
rasterize(
  src_dsn,
  dstfile,
  band = NULL,
  layer = NULL,
  where = NULL,
  sql = NULL,
  burn_value = NULL,
  burn_attr = NULL,
  invert = NULL,
  te = NULL,
  tr = NULL,
  tap = NULL,
  ts = NULL,
  dtName = NULL,
  dstnodata = NULL,
  init = NULL,
  fmt = NULL,
  co = NULL,
  add_options = NULL,
  quiet = FALSE
)
}
\arguments{
\item{src_dsn}{Data source name for the input vector layer (filename or
connection string).}

\item{dstfile}{Filename of the output raster. Must support update mode
access. This file will be created (or overwritten if it already exists -
see Note).}

\item{band}{Numeric vector. The band(s) to burn values into (for existing
\code{dstfile}). The default is to burn into band 1. Not used when creating a
new raster.}

\item{layer}{Character vector of layer names(s) from \code{src_dsn}  that will be
used for input features. At least one layer name or a \code{sql} option must be
specified.}

\item{where}{An optional SQL WHERE style query string to select features to
burn in from the input \code{layer}(s).}

\item{sql}{An SQL statement to be evaluated against \code{src_dsn} to produce a
virtual layer of features to be burned in (alternative to \code{layer}).}

\item{burn_value}{A fixed numeric value to burn into a band for all
features. A numeric vector can be supplied, one burn value per band being
written to.}

\item{burn_attr}{Character string. Name of an attribute field on the
features to be used for a burn-in value. The value will be burned into all
output bands.}

\item{invert}{Logical scalar. \code{TRUE} to invert rasterization. Burn the fixed
burn value, or the burn value associated with the first feature, into all
parts of the raster not inside the provided polygon.}

\item{te}{Numeric vector of length four. Sets the output raster extent. The
values must be expressed in georeferenced units. If not specified, the
extent of the output raster will be the extent of the vector layer.}

\item{tr}{Numeric vector of length two. Sets the target pixel resolution.
The values must be expressed in georeferenced units. Both must be positive.}

\item{tap}{Logical scalar. (target aligned pixels) Align the coordinates of
the extent of the output raster to the values of \code{tr}, such that the
aligned extent includes the minimum extent. Alignment means that
xmin / resx, ymin / resy, xmax / resx and ymax / resy are integer values.}

\item{ts}{Numeric vector of length two. Sets the output raster size in
pixels (xsize, ysize). Note that \code{ts} cannot be used with \code{tr}.}

\item{dtName}{Character name of output raster data type, e.g., \code{Byte},
\code{Int16}, \code{UInt16}, \code{Int32}, \code{UInt32}, \code{Float32}, \code{Float64}.
Defaults to \code{Float64}.}

\item{dstnodata}{Numeric scalar. Assign a nodata value to output bands.}

\item{init}{Numeric vector. Pre-initialize the output raster band(s) with
these value(s). However, it is not marked as the nodata value in the output
file. If only one value is given, the same value is used in all the bands.}

\item{fmt}{Output raster format short name (e.g., \code{"GTiff"}). Will attempt
to guess from the output filename if \code{fmt} is not specified.}

\item{co}{Optional list of format-specific creation options for the output
raster in a vector of "NAME=VALUE" pairs
(e.g., \code{options = c("TILED=YES","COMPRESS=LZW")} to set LZW compression
during creation of a tiled GTiff file).}

\item{add_options}{An optional character vector of additional command-line
options to \code{gdal_rasterize} (see the \code{gdal_rasterize} documentation at the
URL above for all available options).}

\item{quiet}{Logical scalar. If \code{TRUE}, a progress bar will not be
displayed. Defaults to \code{FALSE}.}
}
\value{
Logical indicating success (invisible \code{TRUE}).
An error is raised if the operation fails.
}
\description{
\code{rasterize()} burns vector geometries (points, lines, or polygons) into
the band(s) of a raster dataset. Vectors are read from any GDAL
OGR-supported vector format.
This function is a wrapper for the \command{gdal_rasterize} command-line
utility (\url{https://gdal.org/en/stable/programs/gdal_rasterize.html}).
}
\note{
The function creates a new target raster when any of the \code{fmt}, \code{dstnodata},
\code{init}, \code{co}, \code{te}, \code{tr}, \code{tap}, \code{ts}, or \code{dtName} arguments are used. The
resolution or size must be specified using the \code{tr} or \code{ts} argument for all
new rasters. The target raster will be overwritten if it already exists and
any of these creation-related options are used.
}
\examples{
# MTBS fire perimeters for Yellowstone National Park 1984-2022
dsn <- system.file("extdata/ynp_fires_1984_2022.gpkg", package="gdalraster")
sql <- "SELECT * FROM mtbs_perims ORDER BY mtbs_perims.ig_year"
out_file <- file.path(tempdir(), "ynp_fires_1984_2022.tif")

rasterize(src_dsn = dsn,
          dstfile = out_file,
          sql = sql,
          burn_attr = "ig_year",
          tr = c(90,90),
          tap = TRUE,
          dtName = "Int16",
          dstnodata = -9999,
          init = -9999,
          co = c("TILED=YES","COMPRESS=LZW"))

ds <- new(GDALRaster, out_file)
pal <- scales::viridis_pal(end = 0.8, direction = -1)(6)
ramp <- scales::colour_ramp(pal)
plot_raster(ds, legend = TRUE, col_map_fn = ramp, na_col = "#d9d9d9",
            main="YNP Fires 1984-2022 - Most Recent Burn Year")

ds$close()
deleteDataset(out_file)
}
\seealso{
\code{\link[=polygonize]{polygonize()}}
}
