% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/04_stat_qc.R
\name{stat_QC}
\alias{stat_QC}
\title{Produce QC Charts with ggplot Framework.}
\usage{
stat_QC(mapping = NULL, data = NULL, geom = "hline",
  position = "identity", na.rm = FALSE, show.legend = NA,
  inherit.aes = TRUE, n = NULL, method = "xBar.rBar",
  color.qc_limits = "red", color.qc_center = "blue",
  color.point = "black", color.line = "black",
  physical.limits = c(NA, NA), auto.label = FALSE,
  limit.txt.label = c("LCL", "UCL"), label.digits = 1,
  show.1n2.sigma = FALSE, ...)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link[=aes]{aes()}} or
\code{\link[=aes_]{aes_()}}. If specified and \code{inherit.aes = TRUE} (the
default), it is combined with the default mapping at the top level of the
plot. You must supply \code{mapping} if there is no plot mapping.}

\item{data}{The data to be displayed in this layer. There are three
options:

If \code{NULL}, the default, the data is inherited from the plot
data as specified in the call to \code{\link[=ggplot]{ggplot()}}.

A \code{data.frame}, or other object, will override the plot
data. All objects will be fortified to produce a data frame. See
\code{\link[=fortify]{fortify()}} for which variables will be created.

A \code{function} will be called with a single argument,
the plot data. The return value must be a \code{data.frame}, and
will be used as the layer data.}

\item{geom}{The geometric object to use display the data}

\item{position}{Position adjustment, either as a string, or the result of
a call to a position adjustment function.}

\item{na.rm}{a logical value indicating whether NA values should be
stripped before the computation proceeds.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.
It can also be a named logical vector to finely select the aesthetics to
display.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link[=borders]{borders()}}.}

\item{n}{number, for
\itemize{
  \item \bold{Studentized Charts}, used for custom or hypothetical subgroup size.
  \item \bold{np Charts}, used to specify a fixed area of opportunity.
}}

\item{method}{string, calling the following methods:
\itemize{
  \item \bold{Individuals Charts}: mR, XmR,
  \item \bold{Attribute Charts}: c, np, p, u,
  \item \bold{Studentized Charts}: xBar.rBar, xBar.rMedian, xBar.sBar, xMedian.rBar,
xMedian.rMedian
  \item \bold{Dispersion Charts}: rBar, rMedian, sBar.
}}

\item{color.qc_limits}{color, used to colorize the plot's upper and lower control limits.}

\item{color.qc_center}{color, used to colorize the plot's center line.}

\item{color.point}{color, used to colorize points in studentized plots. You will need geom_point() for C, P, U, NP, and XmR charts.}

\item{color.line}{color, used to colorize lines connecting points in studentized plots. You will need geom_line() for C, P, U, NP, and XmR charts.}

\item{physical.limits}{vector, specify lower physical boundary and upper physical boundary}

\item{auto.label}{boolean setting, if T labels graph with control limits.}

\item{limit.txt.label}{vector, provides option for naming or not showing the limit text labels (e.g., UCL, LCL)
\itemize{
\item \bold{limit.txt.label = c("LCL", "UCL")}: default
\item \bold{limit.txt.label = c("Low", "High")}: changes the label text to low and high
\item \bold{limit.txt.label = NA}: does not show label text.
}}

\item{label.digits}{integer, number of decimal places to display.}

\item{show.1n2.sigma}{boolean setting, if T labels graph 1 and 2 sigma lines. Line color is set by color.qc_limits}

\item{...}{Other arguments passed on to \code{\link[=layer]{layer()}}. These are
often aesthetics, used to set an aesthetic to a fixed value, like
\code{colour = "red"} or \code{size = 3}. They may also be parameters
to the paired geom/stat.}
}
\value{
ggplot control charts.
}
\description{
Produce QC charts with ggplot framework. Support for faceting and
layering of multiple QC chart lines on a single plot. Charts supported (see method argument for call):
\itemize{
\item \bold{Individuals Charts}: mR, XmR,
\item \bold{Attribute Charts}: c, np, p, u,
\item \bold{Studentized Charts}: xBar.rBar, xBar.rMedian, xBar.sBar, xMedian.rBar,
xMedian.rMedian,
\item \bold{Dispersion Charts}: rBar, rMedian, sBar.
}
To label chart lines see \link[ggQC]{stat_QC_labels}
}
\examples{
# Load Libraries ----------------------------------------------------------
 require(ggQC)
 require(ggplot2)

# Setup Data --------------------------------------------------------------
 set.seed(5555)
 Process1 <- data.frame(processID = as.factor(rep(1,100)),
                        metric_value = rnorm(100,0,1),
                        subgroup_sample = rep(1:20, each=5),
                        Process_run_id = 1:100)
 set.seed(5556)
 Process2 <- data.frame(processID = as.factor(rep(2,100)),
                        metric_value = rnorm(100,5, 1),
                        subgroup_sample = rep(1:10, each=10),
                        Process_run_id = 101:200)

 Both_Processes <- rbind(Process1, Process2)

#############################
#  Example 1:  XmR Chart    #
#############################


EX1.1 <- ggplot(Both_Processes, aes(x=Process_run_id, y = metric_value)) +
 geom_point() + geom_line() + stat_QC(method="XmR") +
 stat_QC_labels(method="XmR", digits = 2) +
 facet_grid(.~processID, scales = "free_x")
#EX1.1

EX1.2 <- ggplot(Both_Processes, aes(x=Process_run_id, y = metric_value)) +
 stat_mR() + ylab("Moving Range") +
 stat_QC_labels(method="mR", digits = 2) +
 facet_grid(.~processID, scales = "free_x")
#EX1.2

#############################
#  Example 2:  XbarR Chart  #
#############################

EX2.1 <- ggplot(Both_Processes, aes(x = subgroup_sample,
                          y = metric_value,
                          group = processID)) +
 stat_summary(fun.y = "mean", color = "blue", geom = c("point")) +
 stat_summary(fun.y = "mean", color = "blue", geom = c("line")) +
 stat_QC(method = "xBar.rBar") + facet_grid(.~processID, scales = "free_x")
#EX2.1

EX2.2 <- ggplot(Both_Processes, aes(x = subgroup_sample,
                          y = metric_value,
                          group = processID)) +
 stat_summary(fun.y = "QCrange", color = "blue", geom = "point") +
 stat_summary(fun.y = "QCrange", color = "blue", geom = "line") +
 stat_QC(method = "rBar") +
 ylab("Range") +
 facet_grid(.~processID, scales = "free_x")
 #EX2.2

#############################
#  Example 3:  p Chart      #
#############################
# p chart Setup -----------------------------------------------------------
 set.seed(5556)
 bin_data <- data.frame(
   trial=1:30,
   Num_Incomplete_Items = rpois(30, lambda = 30),
   Num_Items_in_Set = runif(n = 30, min = 50, max = 100))
   bin_data$Proportion_Incomplete <- bin_data$Num_Incomplete_Items/bin_data$Num_Items_in_Set

# Plot p chart ------------------------------------------------------------
EX3.1 <- ggplot(data = bin_data, aes(x=trial,
                           y=Proportion_Incomplete,
                           n=Num_Items_in_Set)) +
 geom_point() + geom_line() +
 stat_QC(method = "p")
 #EX3.1

#############################
#  Example 4:  u Chart      #
#############################
# u chart Setup -----------------------------------------------------------
 set.seed(5555)
 bin_data <- data.frame(
   trial=1:30,
   Num_of_Blemishes = rpois(30, lambda = 30),
   Num_Items_Inspected = runif(n = 30, min = 50, max = 100)
   )
   bin_data$Blemish_Rate <- bin_data$Num_of_Blemishes/bin_data$Num_Items_Inspected

# Plot u chart ------------------------------------------------------------
EX4.1 <- ggplot(data = bin_data, aes(x=trial,
                           y=Blemish_Rate,
                           n=Num_Items_Inspected)) +
 geom_point() + geom_line() +
 stat_QC(method = "u")
#EX4.1

#############################
#  Example 5:  np Chart     #
#############################
# np chart Setup -----------------------------------------------------------
 set.seed(5555)
 bin_data <- data.frame(
   trial=1:30,
   NumNonConforming = rbinom(30, 30, prob = .50))
 Units_Tested_Per_Batch <- 60

# Plot np chart ------------------------------------------------------------
 EX5.1 <- ggplot(data = bin_data, aes(trial, NumNonConforming)) +
  geom_point() +
  stat_QC(method = "np", n = Units_Tested_Per_Batch)
#EX5.1

#############################
#  Example 6:  c Chart     #
#############################
# c chart Setup -----------------------------------------------------------
 set.seed(5555)
 Process1 <- data.frame(Process_run_id = 1:30,
                        Counts=rpois(n = 30, lambda = 25),
                        Group = "A")
 Process2 <- data.frame(Process_run_id = 1:30,
                        Counts = rpois(n = 30, lambda = 5),
                        Group = "B")

 all_processes <- rbind(Process1, Process2)
# Plot C Chart ------------------------------------------------------------

 EX6.1 <- ggplot(all_processes, aes(x=Process_run_id, y = Counts)) +
   geom_point() + geom_line() +
   stat_QC(method = "c", auto.label = TRUE, label.digits = 2) +
   scale_x_continuous(expand =  expand_scale(mult = .25)) +
   facet_grid(.~Group)
# EX6.1
}
