#' Draw Text Annotations to a ggplot Object
#' @description
#' This function overlays text annotations onto any ggplot object. It is particularly useful for adding annotations from CSV files generated by the ggsem Shiny app but can also be used with custom annotation data.
#' @param annotations_data A data frame containing annotation information. Typically, this comes from a CSV file generated by the ggsem Shiny app. The required columns include:
#' \itemize{
#'   \item \code{text}: The text to annotate (character).
#'   \item \code{x}, \code{y}: The coordinates for the text (numeric).
#'   \item \code{font}: The font family to use (character, e.g., "serif").
#'   \item \code{size}: The size of the text (numeric).
#'   \item \code{color}: The color of the text (character, valid hex color).
#'   \item \code{angle}: The rotation angle of the text (numeric, in degrees).
#'   \item \code{alpha}: The transparency of the text (numeric, 0 to 1).
#'   \item \code{fontface}: The font style (character, e.g., "bold").
#'   \item \code{math_expression}: Logical, whether the text should be parsed as a mathematical expression.
#' }
#'
#' @param zoom_level Numeric. Adjusts the size of annotations based on the zoom level. Default is \code{1}.
#'
#' @return ggplot2 annotation layers
#' @importFrom ggplot2 annotate
#' @export
#'
#' @examples
#' library(ggplot2)
#'
#' annotations_data <- data.frame(
#' text = 'x1', x = 9.5, y = 21, font = 'sans',
#' size = 16, color = '#FFFFFF', fill = NA, angle = 0,
#' alpha = 1, fontface = 'plain', math_expression = FALSE,
#' lavaan = FALSE, network = FALSE, locked = FALSE,
#' group_label = FALSE, loop_label = FALSE, group = 1
#' )
#'
#' p <- ggplot()
#'
#' p + draw_annotations(annotations_data, zoom_level = 1.2)
draw_annotations <- function(annotations_data, zoom_level = 1) {
  if (nrow(annotations_data) > 0) {
    annotations_data$color <- sapply(annotations_data$color, valid_hex)
    annotations_data$fill <- sapply(annotations_data$fill, valid_hex)
    annotations_data$alpha <- sapply(annotations_data$alpha, valid_alpha)
    annotations_data$fontface <- sapply(annotations_data$fontface, valid_fontface)
    annotations_data$font <- sapply(annotations_data$font, valid_font)
    annotations_data$math_expression <- sapply(annotations_data$math_expression, valid_logical)
    annotations_data$locked <- sapply(annotations_data$locked, valid_logical)

    layers <- lapply(1:nrow(annotations_data), function(i) {
      annotation_text <- if (annotations_data$math_expression[i]) {
        suppressWarnings(tryCatch(parse(text = annotations_data$text[i]), error = function(e) annotations_data$text[i]))
      } else {
        annotations_data$text[i]
      }

      adjusted_size <- (annotations_data$size[i] / 3) / zoom_level

      # Add annotation to the plot
      annotate("label",
               x = annotations_data$x[i],
               y = annotations_data$y[i],
               label = annotation_text,
               size = adjusted_size,
               color = annotations_data$color[i],
               fill = annotations_data$fill[i],
               alpha = annotations_data$alpha[i],
               angle = annotations_data$angle[i],
               family = annotations_data$font[i],
               fontface = annotations_data$fontface[i],
               label.size = NA
      )
    })
    return(layers)
  } else {
    return(NULL)
  }
}
