% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{grmc}
\alias{grmc}
\title{Gradient of the Negative Log-Likelihood for the McDonald (Mc)/Beta Power Distribution}
\usage{
grmc(par, data)
}
\arguments{
\item{par}{A numeric vector of length 3 containing the distribution parameters
in the order: \code{gamma} (\eqn{\gamma > 0}), \code{delta} (\eqn{\delta \ge 0}),
\code{lambda} (\eqn{\lambda > 0}).}

\item{data}{A numeric vector of observations. All values must be strictly
between 0 and 1 (exclusive).}
}
\value{
Returns a numeric vector of length 3 containing the partial derivatives
of the negative log-likelihood function \eqn{-\ell(\theta | \mathbf{x})} with
respect to each parameter:
\eqn{(-\partial \ell/\partial \gamma, -\partial \ell/\partial \delta, -\partial \ell/\partial \lambda)}.
Returns a vector of \code{NaN} if any parameter values are invalid according
to their constraints, or if any value in \code{data} is not in the
interval (0, 1).
}
\description{
Computes the gradient vector (vector of first partial derivatives) of the
negative log-likelihood function for the McDonald (Mc) distribution (also
known as Beta Power) with parameters \code{gamma} (\eqn{\gamma}), \code{delta}
(\eqn{\delta}), and \code{lambda} (\eqn{\lambda}). This distribution is the
special case of the Generalized Kumaraswamy (GKw) distribution where
\eqn{\alpha = 1} and \eqn{\beta = 1}. The gradient is useful for optimization.
}
\details{
The components of the gradient vector of the negative log-likelihood
(\eqn{-\nabla \ell(\theta | \mathbf{x})}) for the Mc (\eqn{\alpha=1, \beta=1})
model are:

\deqn{
-\frac{\partial \ell}{\partial \gamma} = n[\psi(\gamma+\delta+1) - \psi(\gamma)] -
\lambda\sum_{i=1}^{n}\ln(x_i)
}
\deqn{
-\frac{\partial \ell}{\partial \delta} = n[\psi(\gamma+\delta+1) - \psi(\delta+1)] -
\sum_{i=1}^{n}\ln(1-x_i^{\lambda})
}
\deqn{
-\frac{\partial \ell}{\partial \lambda} = -\frac{n}{\lambda} - \gamma\sum_{i=1}^{n}\ln(x_i) +
\delta\sum_{i=1}^{n}\frac{x_i^{\lambda}\ln(x_i)}{1-x_i^{\lambda}}
}

where \eqn{\psi(\cdot)} is the digamma function (\code{\link[base]{digamma}}).
These formulas represent the derivatives of \eqn{-\ell(\theta)}, consistent with
minimizing the negative log-likelihood. They correspond to the relevant components
of the general GKw gradient (\code{\link{grgkw}}) evaluated at \eqn{\alpha=1, \beta=1}.
}
\examples{
\donttest{
# Assuming existence of rmc, llmc, grmc, hsmc functions for Mc distribution

# Generate sample data
set.seed(123)
true_par_mc <- c(gamma = 2, delta = 3, lambda = 0.5)
sample_data_mc <- rmc(100, gamma = true_par_mc[1], delta = true_par_mc[2],
                      lambda = true_par_mc[3])
hist(sample_data_mc, breaks = 20, main = "Mc(2, 3, 0.5) Sample")

# --- Find MLE estimates ---
start_par_mc <- c(1.5, 2.5, 0.8)
mle_result_mc <- stats::optim(par = start_par_mc,
                              fn = llmc,
                              gr = grmc, # Use analytical gradient for Mc
                              method = "BFGS",
                              hessian = TRUE,
                              data = sample_data_mc)

# --- Compare analytical gradient to numerical gradient ---
if (mle_result_mc$convergence == 0 &&
    requireNamespace("numDeriv", quietly = TRUE)) {

  mle_par_mc <- mle_result_mc$par
  cat("\nComparing Gradients for Mc at MLE estimates:\n")

  # Numerical gradient of llmc
  num_grad_mc <- numDeriv::grad(func = llmc, x = mle_par_mc, data = sample_data_mc)

  # Analytical gradient from grmc
  ana_grad_mc <- grmc(par = mle_par_mc, data = sample_data_mc)

  cat("Numerical Gradient (Mc):\n")
  print(num_grad_mc)
  cat("Analytical Gradient (Mc):\n")
  print(ana_grad_mc)

  # Check differences
  cat("Max absolute difference between Mc gradients:\n")
  print(max(abs(num_grad_mc - ana_grad_mc)))

} else {
  cat("\nSkipping Mc gradient comparison.\n")
}

# Example with Hessian comparison (if hsmc exists)
if (mle_result_mc$convergence == 0 &&
    requireNamespace("numDeriv", quietly = TRUE) && exists("hsmc")) {

  num_hess_mc <- numDeriv::hessian(func = llmc, x = mle_par_mc, data = sample_data_mc)
  ana_hess_mc <- hsmc(par = mle_par_mc, data = sample_data_mc)
  cat("\nMax absolute difference between Mc Hessians:\n")
  print(max(abs(num_hess_mc - ana_hess_mc)))

}

}

}
\references{
McDonald, J. B. (1984). Some generalized functions for the size distribution
of income. \emph{Econometrica}, 52(3), 647-663.

Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation},

(Note: Specific gradient formulas might be derived or sourced from additional references).
}
\seealso{
\code{\link{grgkw}} (parent distribution gradient),
\code{\link{llmc}} (negative log-likelihood for Mc),
\code{hsmc} (Hessian for Mc, if available),
\code{\link{dmc}} (density for Mc),
\code{\link[stats]{optim}},
\code{\link[numDeriv]{grad}} (for numerical gradient comparison),
\code{\link[base]{digamma}}.
}
\author{
Lopes, J. E.
}
\keyword{distribution}
\keyword{gradient}
\keyword{likelihood}
\keyword{mcdonald}
\keyword{optimize}
