/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Emodule.h" 

#include "H5private.h"   
#include "H5Epkg.h"      
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5TSprivate.h" 

#define H5E_CLS_NAME     "HDF5"
#define H5E_CLS_LIB_NAME "HDF5"

typedef struct H5E_print_t {
    FILE     *stream;
    H5E_cls_t cls;
} H5E_print_t;

#ifndef H5_NO_DEPRECATED_SYMBOLS
static herr_t H5E__walk1_cb(int n, H5E_error1_t *err_desc, void *client_data);
#endif 
static herr_t H5E__walk2_cb(unsigned n, const H5E_error2_t *err_desc, void *client_data);
static herr_t H5E__copy_stack_entry(H5E_entry_t *dst_entry, const H5E_entry_t *src_entry);
static herr_t H5E__set_stack_entry(H5E_error2_t *err_entry, const char *file, const char *func, unsigned line,
                                   hid_t cls_id, hid_t maj_id, hid_t min_id, const char *desc, va_list *ap);
static herr_t H5E__clear_entries(H5E_stack_t *estack, size_t nentries);
static herr_t H5E__unregister_class(void *cls, void **request);
static int    H5E__close_msg_cb(void *obj_ptr, hid_t obj_id, void *udata);
static void   H5E__free_msg(H5E_msg_t *msg);
static herr_t H5E__close_msg(H5E_msg_t *err, void **request);
static herr_t H5E__close_stack(H5E_stack_t *err_stack, void **request);

#ifndef H5_HAVE_THREADSAFE_API

H5E_stack_t H5E_stack_g[1];
#endif 

H5FL_DEFINE(H5E_stack_t);

H5FL_DEFINE_STATIC(H5E_cls_t);

H5FL_DEFINE_STATIC(H5E_msg_t);

hid_t H5E_ERR_CLS_g = FAIL;

#include "H5Edefin.h"

static const H5E_stack_t H5E_err_stack_def = {
    0, 
    {  
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}},
     {false, {H5I_INVALID_HID, H5I_INVALID_HID, H5I_INVALID_HID, 0, NULL, NULL, NULL}}},

#ifndef H5_NO_DEPRECATED_SYMBOLS
#ifdef H5_USE_16_API_DEFAULT
    {1, true, (H5E_auto1_t)H5Eprint1, (H5E_auto2_t)H5E__print2, (H5E_auto1_t)H5Eprint1,
     (H5E_auto2_t)H5E__print2},
#else  
    {2, true, (H5E_auto1_t)H5Eprint1, (H5E_auto2_t)H5E__print2, (H5E_auto1_t)H5Eprint1,
     (H5E_auto2_t)H5E__print2},
#endif 
#else  
    {(H5E_auto2_t)H5E__print2},
#endif 

    NULL, 
    0     
};

hid_t H5E_first_maj_id_g = H5I_INVALID_HID;
hid_t H5E_last_maj_id_g  = H5I_INVALID_HID;
hid_t H5E_first_min_id_g = H5I_INVALID_HID;
hid_t H5E_last_min_id_g  = H5I_INVALID_HID;

static const H5I_class_t H5I_ERRCLS_CLS[1] = {{
    H5I_ERROR_CLASS,                  
    0,                                
    0,                                
    (H5I_free_t)H5E__unregister_class 
}};

static const H5I_class_t H5I_ERRMSG_CLS[1] = {{
    H5I_ERROR_MSG,             
    0,                         
    0,                         
    (H5I_free_t)H5E__close_msg 
}};

static const H5I_class_t H5I_ERRSTK_CLS[1] = {{
    H5I_ERROR_STACK,             
    0,                           
    0,                           
    (H5I_free_t)H5E__close_stack 
}};

static const H5E_cls_t H5E_err_cls_s = {false, H5E_CLS_NAME, H5E_CLS_LIB_NAME, H5_VERS_STR};

#include "H5Emajdef.h"

#include "H5Emindef.h"

herr_t
H5E_init(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)
    

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5E__init_package(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5I_register_type(H5I_ERRCLS_CLS) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTINIT, FAIL, "unable to initialize ID group");

    
    if (H5I_register_type(H5I_ERRMSG_CLS) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTINIT, FAIL, "unable to initialize ID group");

    
    if (H5I_register_type(H5I_ERRSTK_CLS) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTINIT, FAIL, "unable to initialize ID group");

#ifndef H5_HAVE_THREADSAFE_API
    H5E__set_default_auto(H5E_stack_g);
#endif 

    
    if ((H5E_ERR_CLS_g = H5I_register(H5I_ERROR_CLASS, &H5E_err_cls_s, false)) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTREGISTER, FAIL, "can't register error class");

#include "H5Einit.h"

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5E_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR) {
        int64_t ncls, nmsg, nstk;

        
        ncls = H5I_nmembers(H5I_ERROR_CLASS);
        nmsg = H5I_nmembers(H5I_ERROR_MSG);
        nstk = H5I_nmembers(H5I_ERROR_STACK);

        if ((ncls + nmsg + nstk) > 0) {
            
            H5E_clear_stack();

            
            if (nstk > 0)
                (void)H5I_clear_type(H5I_ERROR_STACK, false, false);

            
            if (ncls > 0) {
                (void)H5I_clear_type(H5I_ERROR_CLASS, false, false);

                
                if (H5I_nmembers(H5I_ERROR_CLASS) == 0)
                    H5E_ERR_CLS_g = H5I_INVALID_HID;
            } 

            
            if (nmsg > 0) {
                (void)H5I_clear_type(H5I_ERROR_MSG, false, false);

                
                if (H5I_nmembers(H5I_ERROR_MSG) == 0) {

#include "H5Eterm.h"
                } 
            }     

            n++; 
        }        
        else {
            
            n += (H5I_dec_type_ref(H5I_ERROR_STACK) > 0);
            n += (H5I_dec_type_ref(H5I_ERROR_CLASS) > 0);
            n += (H5I_dec_type_ref(H5I_ERROR_MSG) > 0);

            
            if (0 == n)
                H5_PKG_INIT_VAR = false;
        } 
    }     

    FUNC_LEAVE_NOAPI(n)
} 

herr_t
H5E_user_cb_prepare(H5E_user_cb_state_t *state)
{
    H5E_stack_t *stack;               
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (NULL == (stack = H5E__get_my_stack()))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");

        
#ifndef H5_NO_DEPRECATED_SYMBOLS
    assert(1 == stack->auto_op.vers || 2 == stack->auto_op.vers);

    state->vers = stack->auto_op.vers;
    if (1 == stack->auto_op.vers)
        state->u.func1 = stack->auto_op.func1;
    else
        state->u.func2 = stack->auto_op.func2;
#else  
    state->func2 = stack->auto_op.func2;
#endif 
    state->data = stack->auto_data;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5E_user_cb_restore(const H5E_user_cb_state_t *state)
{
    H5E_stack_t *stack;               
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (NULL == (stack = H5E__get_my_stack()))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");

        
#ifndef H5_NO_DEPRECATED_SYMBOLS
    stack->auto_op.vers = state->vers;
    if (1 == state->vers)
        stack->auto_op.func1 = state->u.func1;
    else
        stack->auto_op.func2 = state->u.func2;
#else  
    stack->auto_op.func2 = state->func2;
#endif 
    stack->auto_data = state->data;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5E__free_class(H5E_cls_t *cls)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(cls);

    
    if (cls->app_cls) {
        
        cls->cls_name = H5MM_xfree_const(cls->cls_name);
        cls->lib_name = H5MM_xfree_const(cls->lib_name);
        cls->lib_vers = H5MM_xfree_const(cls->lib_vers);
        cls           = H5FL_FREE(H5E_cls_t, cls);
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

H5E_cls_t *
H5E__register_class(const char *cls_name, const char *lib_name, const char *version)
{
    H5E_cls_t *cls       = NULL; 
    H5E_cls_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(cls_name);
    assert(lib_name);
    assert(version);

    
    if (NULL == (cls = H5FL_CALLOC(H5E_cls_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    cls->app_cls = true;

    
    if (NULL == (cls->cls_name = strdup(cls_name)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    if (NULL == (cls->lib_name = strdup(lib_name)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    if (NULL == (cls->lib_vers = strdup(version)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    ret_value = cls;

done:
    if (!ret_value)
        if (cls && H5E__free_class(cls) < 0)
            HDONE_ERROR(H5E_ERROR, H5E_CANTRELEASE, NULL, "unable to free error class");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5E__unregister_class(void *_cls, void H5_ATTR_UNUSED **request)
{
    H5E_cls_t *cls = (H5E_cls_t *)_cls;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(cls);

    
    if (H5I_iterate(H5I_ERROR_MSG, H5E__close_msg_cb, cls, false) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_BADITER, FAIL, "unable to free all messages in this error class");

    
    if (H5E__free_class(cls) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTRELEASE, FAIL, "unable to free error class");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

ssize_t
H5E__get_class_name(const H5E_cls_t *cls, char *name, size_t size)
{
    ssize_t len = -1; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(cls);

    
    len = (ssize_t)strlen(cls->cls_name);

    
    if (name) {
        strncpy(name, cls->cls_name, size);
        if ((size_t)len >= size)
            name[size - 1] = '\0';
    } 

    
    FUNC_LEAVE_NOAPI(len)
} 

static int
H5E__close_msg_cb(void *obj_ptr, hid_t obj_id, void *udata)
{
    H5E_msg_t *err_msg   = (H5E_msg_t *)obj_ptr;
    H5E_cls_t *cls       = (H5E_cls_t *)udata;
    int        ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE

    
    assert(err_msg);

    
    if (err_msg->cls == cls) {
        if (H5E__close_msg(err_msg, NULL) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTCLOSEOBJ, H5_ITER_ERROR, "unable to close error message");
        if (NULL == H5I_remove(obj_id))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTREMOVE, H5_ITER_ERROR, "unable to remove error message");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void
H5E__free_msg(H5E_msg_t *msg)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(msg);
    assert(msg->app_msg);

    
    msg->msg = H5MM_xfree_const(msg->msg);
    msg      = H5FL_FREE(H5E_msg_t, msg);

    FUNC_LEAVE_NOAPI_VOID
} 

static herr_t
H5E__close_msg(H5E_msg_t *err, void H5_ATTR_UNUSED **request)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(err);

    
    if (err->app_msg)
        
        H5E__free_msg(err);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

H5E_msg_t *
H5E__create_msg(H5E_cls_t *cls, H5E_type_t msg_type, const char *msg_str)
{
    H5E_msg_t *msg       = NULL; 
    H5E_msg_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(cls);
    assert(msg_type == H5E_MAJOR || msg_type == H5E_MINOR);
    assert(msg_str);

    
    if (NULL == (msg = H5FL_CALLOC(H5E_msg_t)))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTALLOC, NULL, "memory allocation failed");

    
    msg->app_msg = true;
    msg->cls     = cls;
    msg->type    = msg_type;
    if (NULL == (msg->msg = strdup(msg_str)))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTALLOC, NULL, "memory allocation failed");

    
    ret_value = msg;

done:
    if (!ret_value)
        if (msg)
            H5E__free_msg(msg);

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5E_stack_t *
H5E__get_current_stack(void)
{
    H5E_stack_t *current_stack;      
    H5E_stack_t *estack_copy = NULL; 
    unsigned     u;                  
    H5E_stack_t *ret_value = NULL;   

    FUNC_ENTER_PACKAGE

    
    if (NULL == (current_stack = H5E__get_my_stack()))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, NULL, "can't get current error stack");

    
    if (NULL == (estack_copy = H5FL_CALLOC(H5E_stack_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    estack_copy->nused = current_stack->nused;
    for (u = 0; u < current_stack->nused; u++)
        if (H5E__copy_stack_entry(&estack_copy->entries[u], &current_stack->entries[u]) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTSET, NULL, "can't set error entry");

    
    estack_copy->auto_op   = current_stack->auto_op;
    estack_copy->auto_data = current_stack->auto_data;

    
    H5E__destroy_stack(current_stack);

    
    ret_value = estack_copy;

done:
    if (ret_value == NULL)
        if (estack_copy)
            estack_copy = H5FL_FREE(H5E_stack_t, estack_copy);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5E__set_current_stack(H5E_stack_t *estack)
{
    H5E_stack_t *current_stack;       
    unsigned     u;                   
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(estack);

    
    if (NULL == (current_stack = H5E__get_my_stack()))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");

    
    H5E__destroy_stack(current_stack);

    
    current_stack->nused = estack->nused;
    for (u = 0; u < current_stack->nused; u++)
        if (H5E__copy_stack_entry(&current_stack->entries[u], &estack->entries[u]) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTSET, FAIL, "can't set error entry");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5E__close_stack(H5E_stack_t *estack, void H5_ATTR_UNUSED **request)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(estack);

    
    H5E__destroy_stack(estack);

    
    estack = H5FL_FREE(H5E_stack_t, estack);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

ssize_t
H5E__get_num(const H5E_stack_t *estack)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(estack);

    FUNC_LEAVE_NOAPI((ssize_t)estack->nused)
} 

herr_t
H5E__print2(hid_t err_stack, FILE *stream)
{
    H5E_stack_t *estack;              
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (err_stack == H5E_DEFAULT) {
        if (NULL == (estack = H5E__get_my_stack()))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");
    } 
    else {
        
        H5E_clear_stack();

        if (NULL == (estack = (H5E_stack_t *)H5I_object_verify(err_stack, H5I_ERROR_STACK)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a error stack ID");
    } 

    
    if (H5E__print(estack, stream, false) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTLIST, FAIL, "can't display error stack");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5E__append_stack(H5E_stack_t *dst_stack, const H5E_stack_t *src_stack)
{
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dst_stack);
    assert(src_stack);

    
    for (u = 0; u < src_stack->nused; u++) {
        
        if (H5E__copy_stack_entry(&dst_stack->entries[dst_stack->nused], &src_stack->entries[u]) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTSET, FAIL, "can't set error entry");

        
        dst_stack->nused++;

        
        if (dst_stack->nused >= H5E_MAX_ENTRIES)
            break;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5E__set_default_auto(H5E_stack_t *stk)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    memcpy(stk, &H5E_err_stack_def, sizeof(H5E_err_stack_def));

    FUNC_LEAVE_NOAPI_VOID
} 

ssize_t
H5E__get_msg(const H5E_msg_t *msg, H5E_type_t *type, char *msg_str, size_t size)
{
    ssize_t len = -1; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(msg);

    
    len = (ssize_t)strlen(msg->msg);

    
    if (msg_str) {
        strncpy(msg_str, msg->msg, size);
        if ((size_t)len >= size)
            msg_str[size - 1] = '\0';
    } 

    
    if (type)
        *type = msg->type;

    
    FUNC_LEAVE_NOAPI(len)
} 

#ifndef H5_NO_DEPRECATED_SYMBOLS

static herr_t
H5E__walk1_cb(int n, H5E_error1_t *err_desc, void *client_data)
{
    H5E_print_t     *eprint = (H5E_print_t *)client_data;
    FILE            *stream;                             
    const H5E_cls_t *cls_ptr;                            
    H5E_msg_t       *maj_ptr;                            
    H5E_msg_t       *min_ptr;                            
    const char      *maj_str   = "No major description"; 
    const char      *min_str   = "No minor description"; 
    bool             have_desc = true; 
#ifdef H5_HAVE_THREADSAFE_API
    uint64_t thread_id = 0; 
#endif
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(err_desc);

    
    if (!client_data)
        stream = Rstderr;
    else
        stream = eprint->stream;

    
    maj_ptr = (H5E_msg_t *)H5I_object_verify(err_desc->maj_num, H5I_ERROR_MSG);
    min_ptr = (H5E_msg_t *)H5I_object_verify(err_desc->min_num, H5I_ERROR_MSG);

    
    if (!maj_ptr || !min_ptr)
        HGOTO_DONE(FAIL);

    if (maj_ptr->msg)
        maj_str = maj_ptr->msg;
    if (min_ptr->msg)
        min_str = min_ptr->msg;

    
    cls_ptr = maj_ptr->cls;

#ifdef H5_HAVE_THREADSAFE_API
    if (H5TS_thread_id(&thread_id) < 0)
        HGOTO_DONE(FAIL);
#endif

    
    if (eprint->cls.lib_name == NULL || strcmp(cls_ptr->lib_name, eprint->cls.lib_name) != 0) {
        
        if (cls_ptr->cls_name)
            eprint->cls.cls_name = cls_ptr->cls_name;
        if (cls_ptr->lib_name)
            eprint->cls.lib_name = cls_ptr->lib_name;
        if (cls_ptr->lib_vers)
            eprint->cls.lib_vers = cls_ptr->lib_vers;

        Rfprintf(stream, "%s-DIAG: Error detected in %s (%s)",
                (cls_ptr->cls_name ? cls_ptr->cls_name : "(null)"),
                (cls_ptr->lib_name ? cls_ptr->lib_name : "(null)"),
                (cls_ptr->lib_vers ? cls_ptr->lib_vers : "(null)"));

        
#ifdef H5_HAVE_PARALLEL
        {
            int mpi_rank, mpi_initialized, mpi_finalized;

            MPI_Initialized(&mpi_initialized);
            MPI_Finalized(&mpi_finalized);

            if (mpi_initialized && !mpi_finalized) {
                MPI_Comm_rank(MPI_COMM_WORLD, &mpi_rank);
                Rfprintf(stream, " MPI-process %d", mpi_rank);
            } 
#ifdef H5_HAVE_THREADSAFE_API
            else
                Rfprintf(stream, " thread %" PRIu64, thread_id);
#endif
        } 
#else
#ifdef H5_HAVE_THREADSAFE_API
        Rfprintf(stream, " thread %" PRIu64, thread_id);
#endif
#endif
        Rfprintf(stream, ":\n");
    } 

    
    if (err_desc->desc == NULL || strlen(err_desc->desc) == 0)
        have_desc = false;

    
    Rfprintf(stream, "%*s#%03d: %s line %u in %s()%s%s\n", H5E_INDENT, "", n, err_desc->file_name,
            err_desc->line, err_desc->func_name, (have_desc ? ": " : ""), (have_desc ? err_desc->desc : ""));
    Rfprintf(stream, "%*smajor: %s\n", (H5E_INDENT * 2), "", maj_str);
    Rfprintf(stream, "%*sminor: %s\n", (H5E_INDENT * 2), "", min_str);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

static herr_t
H5E__walk2_cb(unsigned n, const H5E_error2_t *err_desc, void *client_data)
{
    H5E_print_t *eprint = (H5E_print_t *)client_data;
    FILE        *stream;                             
    H5E_cls_t   *cls_ptr;                            
    H5E_msg_t   *maj_ptr;                            
    H5E_msg_t   *min_ptr;                            
    const char  *maj_str   = "No major description"; 
    const char  *min_str   = "No minor description"; 
    bool         have_desc = true; 
#ifdef H5_HAVE_THREADSAFE_API
    uint64_t thread_id = 0; 
#endif
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(err_desc);

    
    if (!client_data)
        stream = Rstderr;
    else
        stream = eprint->stream;

    
    maj_ptr = (H5E_msg_t *)H5I_object_verify(err_desc->maj_num, H5I_ERROR_MSG);
    min_ptr = (H5E_msg_t *)H5I_object_verify(err_desc->min_num, H5I_ERROR_MSG);

    
    if (!maj_ptr || !min_ptr)
        HGOTO_DONE(FAIL);

    if (maj_ptr->msg)
        maj_str = maj_ptr->msg;
    if (min_ptr->msg)
        min_str = min_ptr->msg;

    
    cls_ptr = (H5E_cls_t *)H5I_object_verify(err_desc->cls_id, H5I_ERROR_CLASS);

    
    if (!cls_ptr)
        HGOTO_DONE(FAIL);

#ifdef H5_HAVE_THREADSAFE_API
    if (H5TS_thread_id(&thread_id) < 0)
        HGOTO_DONE(FAIL);
#endif

    
    if (eprint->cls.lib_name == NULL || strcmp(cls_ptr->lib_name, eprint->cls.lib_name) != 0) {
        
        if (cls_ptr->cls_name)
            eprint->cls.cls_name = cls_ptr->cls_name;
        if (cls_ptr->lib_name)
            eprint->cls.lib_name = cls_ptr->lib_name;
        if (cls_ptr->lib_vers)
            eprint->cls.lib_vers = cls_ptr->lib_vers;

        Rfprintf(stream, "%s-DIAG: Error detected in %s (%s)",
                (cls_ptr->cls_name ? cls_ptr->cls_name : "(null)"),
                (cls_ptr->lib_name ? cls_ptr->lib_name : "(null)"),
                (cls_ptr->lib_vers ? cls_ptr->lib_vers : "(null)"));

        
#ifdef H5_HAVE_PARALLEL
        {
            int mpi_rank, mpi_initialized, mpi_finalized;

            MPI_Initialized(&mpi_initialized);
            MPI_Finalized(&mpi_finalized);

            if (mpi_initialized && !mpi_finalized) {
                MPI_Comm_rank(MPI_COMM_WORLD, &mpi_rank);
                Rfprintf(stream, " MPI-process %d", mpi_rank);
            } 
#ifdef H5_HAVE_THREADSAFE_API
            else
                Rfprintf(stream, " thread %" PRIu64, thread_id);
#endif
        } 
#else
#ifdef H5_HAVE_THREADSAFE_API
        Rfprintf(stream, " thread %" PRIu64, thread_id);
#endif
#endif
        Rfprintf(stream, ":\n");
    } 

    
    if (err_desc->desc == NULL || strlen(err_desc->desc) == 0)
        have_desc = false;

    
    Rfprintf(stream, "%*s#%03u: %s line %u in %s()%s%s\n", H5E_INDENT, "", n, err_desc->file_name,
            err_desc->line, err_desc->func_name, (have_desc ? ": " : ""), (have_desc ? err_desc->desc : ""));
    Rfprintf(stream, "%*smajor: %s\n", (H5E_INDENT * 2), "", maj_str);
    Rfprintf(stream, "%*sminor: %s\n", (H5E_INDENT * 2), "", min_str);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5E__print(const H5E_stack_t *estack, FILE *stream, bool bk_compatible)
{
    H5E_print_t   eprint;  
    H5E_walk_op_t walk_op; 
    herr_t        ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(estack);

    
    if (!stream)
        eprint.stream = Rstderr;
    else
        eprint.stream = stream;

    
    memset(&eprint.cls, 0, sizeof(H5E_cls_t));

    
    if (bk_compatible) {
#ifndef H5_NO_DEPRECATED_SYMBOLS
        walk_op.vers    = 1;
        walk_op.u.func1 = H5E__walk1_cb;
        if (H5E__walk(estack, H5E_WALK_DOWNWARD, &walk_op, (void *)&eprint) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTLIST, FAIL, "can't walk error stack");
#else  
        assert(0 && "version 1 error stack print without deprecated symbols!");
#endif 
    }  
    else {
        walk_op.vers    = 2;
        walk_op.u.func2 = H5E__walk2_cb;
        if (H5E__walk(estack, H5E_WALK_DOWNWARD, &walk_op, (void *)&eprint) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTLIST, FAIL, "can't walk error stack");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5E__walk(const H5E_stack_t *estack, H5E_direction_t direction, const H5E_walk_op_t *op, void *client_data)
{
    int    i;                        
    herr_t ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(estack);
    assert(op);

    
    if (direction != H5E_WALK_UPWARD && direction != H5E_WALK_DOWNWARD)
        direction = H5E_WALK_UPWARD;

    
    if (op->vers == 1) {
#ifndef H5_NO_DEPRECATED_SYMBOLS
        if (op->u.func1) {
            H5E_error1_t old_err;

            ret_value = SUCCEED;
            if (H5E_WALK_UPWARD == direction) {
                for (i = 0; i < (int)estack->nused && ret_value == H5_ITER_CONT; i++) {
                    
                    old_err.maj_num   = estack->entries[i].err.maj_num;
                    old_err.min_num   = estack->entries[i].err.min_num;
                    old_err.file_name = estack->entries[i].err.file_name;
                    old_err.func_name = estack->entries[i].err.func_name;
                    old_err.line      = estack->entries[i].err.line;
                    old_err.desc      = estack->entries[i].err.desc;

                    
                    H5_BEFORE_USER_CB_NOERR(H5_ITER_ERROR)
                        {
                            ret_value = (op->u.func1)(i, &old_err, client_data);
                        }
                    H5_AFTER_USER_CB_NOERR(H5_ITER_ERROR)
                } 
            }     
            else {
                H5_CHECK_OVERFLOW(estack->nused - 1, size_t, int);
                for (i = (int)(estack->nused - 1); i >= 0 && ret_value == H5_ITER_CONT; i--) {
                    
                    old_err.maj_num   = estack->entries[i].err.maj_num;
                    old_err.min_num   = estack->entries[i].err.min_num;
                    old_err.file_name = estack->entries[i].err.file_name;
                    old_err.func_name = estack->entries[i].err.func_name;
                    old_err.line      = estack->entries[i].err.line;
                    old_err.desc      = estack->entries[i].err.desc;

                    
                    H5_BEFORE_USER_CB_NOERR(H5_ITER_ERROR)
                        {
                            ret_value =
                                (op->u.func1)((int)(estack->nused - (size_t)(i + 1)), &old_err, client_data);
                        }
                    H5_AFTER_USER_CB_NOERR(H5_ITER_ERROR)
                } 
            }     

            if (ret_value < 0)
                HERROR(H5E_ERROR, H5E_CANTLIST, "can't walk error stack");
        } 
#else     
        assert(0 && "version 1 error stack walk without deprecated symbols!");
#endif    
    }     
    else {
        assert(op->vers == 2);
        if (op->u.func2) {
            ret_value = SUCCEED;
            if (H5E_WALK_UPWARD == direction) {
                for (i = 0; i < (int)estack->nused && ret_value == H5_ITER_CONT; i++) {
                    
                    H5_BEFORE_USER_CB_NOERR(H5_ITER_ERROR)
                        {
                            ret_value = (op->u.func2)((unsigned)i, &estack->entries[i].err, client_data);
                        }
                    H5_AFTER_USER_CB_NOERR(H5_ITER_ERROR)
                }
            } 
            else {
                H5_CHECK_OVERFLOW(estack->nused - 1, size_t, int);
                for (i = (int)(estack->nused - 1); i >= 0 && ret_value == H5_ITER_CONT; i--) {
                    
                    H5_BEFORE_USER_CB_NOERR(H5_ITER_ERROR)
                        {
                            ret_value = (op->u.func2)((unsigned)(estack->nused - (size_t)(i + 1)),
                                                      &estack->entries[i].err, client_data);
                        }
                    H5_AFTER_USER_CB_NOERR(H5_ITER_ERROR)
                }
            } 

            if (ret_value < 0)
                HERROR(H5E_ERROR, H5E_CANTLIST, "can't walk error stack");
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5E__get_auto(const H5E_stack_t *estack, H5E_auto_op_t *op, void **client_data)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(estack);

    
    if (op)
        *op = estack->auto_op;
    if (client_data)
        *client_data = estack->auto_data;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5E_get_default_auto_func(H5E_auto2_t *func)
{
    H5E_stack_t  *estack;              
    H5E_auto_op_t op        = {0};     
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (NULL == (estack = H5E__get_my_stack()))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");

    
    if (H5E__get_auto(estack, &op, NULL) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get automatic error info");

    
    *func = op.func2;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5E__set_auto(H5E_stack_t *estack, const H5E_auto_op_t *op, void *client_data)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(estack);

    
    estack->auto_op   = *op;
    estack->auto_data = client_data;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5E_printf_stack(const char *file, const char *func, unsigned line, hid_t maj_id, hid_t min_id,
                 const char *fmt, ...)
{
    H5E_stack_t *estack;               
    va_list      ap;                   
    bool         va_started = false;   
    herr_t       ret_value  = SUCCEED; 

    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(maj_id >= H5E_first_maj_id_g && maj_id <= H5E_last_maj_id_g);
    assert(min_id >= H5E_first_min_id_g && min_id <= H5E_last_min_id_g);
    assert(fmt);

    
    if (NULL == (estack = H5E__get_my_stack()))
        HGOTO_DONE(FAIL);

    
    if (!estack->paused) {
        
        va_start(ap, fmt);
        va_started = true;

        
        if (H5E__push_stack(estack, false, file, func, line, H5E_ERR_CLS_g, maj_id, min_id, fmt, &ap) < 0)
            HGOTO_DONE(FAIL);
    }

done:
    if (va_started)
        va_end(ap);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5E__push_stack(H5E_stack_t *estack, bool app_entry, const char *file, const char *func, unsigned line,
                hid_t cls_id, hid_t maj_id, hid_t min_id, const char *fmt, va_list *ap)
{
    herr_t ret_value = SUCCEED; 

    
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(cls_id > 0);
    assert(maj_id > 0);
    assert(min_id > 0);

    
    if (estack->nused < H5E_MAX_ENTRIES) {
        estack->entries[estack->nused].app_entry = app_entry;
        if (H5E__set_stack_entry(&estack->entries[estack->nused].err, file, func, line, cls_id, maj_id,
                                 min_id, fmt, ap) < 0)
            HGOTO_DONE(FAIL);
        estack->nused++;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5E__copy_stack_entry(H5E_entry_t *dst_entry, const H5E_entry_t *src_entry)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dst_entry);
    assert(src_entry);

    
    *dst_entry = *src_entry;

    
    if (dst_entry->app_entry) {
        
        if (dst_entry->err.cls_id != H5E_ERR_CLS_g)
            if (H5I_inc_ref(dst_entry->err.cls_id, false) < 0)
                HGOTO_ERROR(H5E_ERROR, H5E_CANTINC, FAIL, "unable to increment ref count on error class");
        if (dst_entry->err.maj_num < H5E_first_maj_id_g || dst_entry->err.maj_num > H5E_last_maj_id_g)
            if (H5I_inc_ref(dst_entry->err.maj_num, false) < 0)
                HGOTO_ERROR(H5E_ERROR, H5E_CANTINC, FAIL, "unable to increment ref count on error message");
        if (dst_entry->err.min_num < H5E_first_min_id_g || dst_entry->err.min_num > H5E_last_min_id_g)
            if (H5I_inc_ref(dst_entry->err.min_num, false) < 0)
                HGOTO_ERROR(H5E_ERROR, H5E_CANTINC, FAIL, "unable to increment ref count on error message");
        
        if (NULL == (dst_entry->err.file_name = strdup(src_entry->err.file_name)))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTCOPY, FAIL, "unable to duplicate file name");
        if (NULL == (dst_entry->err.func_name = strdup(src_entry->err.func_name)))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTCOPY, FAIL, "unable to duplicate function name");
    }
    if (NULL == (dst_entry->err.desc = strdup(src_entry->err.desc)))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTCOPY, FAIL, "unable to duplicate error description");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5E__set_stack_entry(H5E_error2_t *err_entry, const char *file, const char *func, unsigned line, hid_t cls_id,
                     hid_t maj_id, hid_t min_id, const char *fmt, va_list *ap)
{
    herr_t ret_value = SUCCEED; 

    
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(err_entry);
    assert(cls_id > 0);
    assert(maj_id > 0);
    assert(min_id > 0);

    
    if (!func)
        func = "Unknown_Function";
    if (!file)
        file = "Unknown_File";
    if (!fmt)
        fmt = "No description given";

    
    
    err_entry->cls_id  = cls_id;
    err_entry->maj_num = maj_id;
    err_entry->min_num = min_id;
    
    err_entry->func_name = func;
    err_entry->file_name = file;
    err_entry->line      = line;
    if (ap) {
        char *desc = NULL;

        H5_WARN_FORMAT_NONLITERAL_OFF
        if (HDvasprintf(&desc, fmt, *ap) < 0)
            HGOTO_DONE(FAIL);
        H5_WARN_FORMAT_NONLITERAL_ON

        err_entry->desc = desc;
    }
    else {
        if (NULL == (err_entry->desc = strdup(fmt)))
            HGOTO_DONE(FAIL);
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5E__clear_entries(H5E_stack_t *estack, size_t nentries)
{
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(estack);
    assert(estack->nused >= nentries);

    
    for (u = 0; nentries > 0; nentries--, u++) {
        H5E_entry_t *error; 

        error = &(estack->entries[estack->nused - (u + 1)]);

        
        
        
        if (error->err.min_num < H5E_first_min_id_g || error->err.min_num > H5E_last_min_id_g)
            if (H5I_dec_ref(error->err.min_num) < 0)
                HGOTO_ERROR(H5E_ERROR, H5E_CANTDEC, FAIL, "unable to decrement ref count on error message");
        if (error->err.maj_num < H5E_first_maj_id_g || error->err.maj_num > H5E_last_maj_id_g)
            if (H5I_dec_ref(error->err.maj_num) < 0)
                HGOTO_ERROR(H5E_ERROR, H5E_CANTDEC, FAIL, "unable to decrement ref count on error message");
        if (error->err.cls_id != H5E_ERR_CLS_g)
            if (H5I_dec_ref(error->err.cls_id) < 0)
                HGOTO_ERROR(H5E_ERROR, H5E_CANTDEC, FAIL, "unable to decrement ref count on error class");

        
        
        if (error->app_entry) {
            H5MM_xfree_const(error->err.file_name);
            H5MM_xfree_const(error->err.func_name);
        }
        error->err.file_name = NULL;
        error->err.func_name = NULL;
        error->err.desc      = (const char *)H5MM_xfree_const(error->err.desc);
    }

    
    estack->nused -= u;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5E_clear_stack(void)
{
    H5E_stack_t *estack;              
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (NULL == (estack = H5E__get_my_stack()))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");

    
    if (estack->nused)
        if (H5E__clear_entries(estack, estack->nused) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTSET, FAIL, "can't clear error stack");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5E__destroy_stack(H5E_stack_t *estack)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (estack == NULL)
        if (NULL == (estack = H5E__get_my_stack()))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");

    
    if (estack->nused)
        if (H5E__clear_entries(estack, estack->nused) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTSET, FAIL, "can't clear error stack");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5E__pop(H5E_stack_t *estack, size_t count)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(estack);
    assert(estack->nused >= count);

    
    if (H5E__clear_entries(estack, count) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTRELEASE, FAIL, "can't remove errors from stack");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5E_dump_api_stack(void)
{
    H5E_stack_t *estack    = H5E__get_my_stack();
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(estack);

#ifdef H5_NO_DEPRECATED_SYMBOLS
    if (estack->auto_op.func2) {
        
        H5_BEFORE_USER_CB_NOERR(H5_ITER_ERROR)
            {
                (void)((estack->auto_op.func2)(H5E_DEFAULT, estack->auto_data));
            }
        H5_AFTER_USER_CB_NOERR(H5_ITER_ERROR)
    }
#else  
    if (estack->auto_op.vers == 1) {
        if (estack->auto_op.func1) {
            
            H5_BEFORE_USER_CB_NOERR(H5_ITER_ERROR)
                {
                    (void)((estack->auto_op.func1)(estack->auto_data));
                }
            H5_AFTER_USER_CB_NOERR(H5_ITER_ERROR)
        }
    } 
    else {
        if (estack->auto_op.func2) {
            
            H5_BEFORE_USER_CB_NOERR(H5_ITER_ERROR)
                {
                    (void)((estack->auto_op.func2)(H5E_DEFAULT, estack->auto_data));
                }
            H5_AFTER_USER_CB_NOERR(H5_ITER_ERROR)
        }
    } 
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5E_pause_stack(void)
{
    H5E_stack_t *estack = H5E__get_my_stack();

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(estack);

    
    estack->paused++;

    FUNC_LEAVE_NOAPI_VOID
} 

void
H5E_resume_stack(void)
{
    H5E_stack_t *estack = H5E__get_my_stack();

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(estack);
    assert(estack->paused);

    
    estack->paused--;

    FUNC_LEAVE_NOAPI_VOID
} 
