#######################################################
# .hydroplotts Daily, Monthly and Annual Time Series  #
#######################################################
# It requires the function'drawxaxis' that is stored in the 'lib_Plot.R' library
# 'x'		 : daily time series of type 'zoo'
# 'x.monthly : monthly time series of type 'zoo'
# 'x.annual' : annual time series of type 'zoo'
# 'win.len1' : number of days for being used in the computation of the first moving average
# 'win.len2' : number of days for being used in the computation of the second moving average
# 'var.type' : string representing the type of variable being plotted (e.g., "Precipitation", "Temperature" or "Flow").
#              ONLY used for labelling the y axis and the title of the plot (in case it is missing)
# 'var.unit' : string representing the measurement unit of the variable being plotted ("mm" for precipitation, "C" for temperature, and "m3/s" for flow).
#              ONLY used for labelling the y axis
# 'main'     : string representing the main title of the plot
# 'pfreq'    : string indicating how many plots are desired by the user.
#              Valid values are:
#              -) 'dma': Daily, Monthly and Annual values are plotted
#              -) 'ma' : Monthly and Annual values are plotted
#              -) 'dm' : Daily and Monthly values are plotted
# 'tick.tstep': string indicating the time step that have to be used for
#               putting the ticks ont he time axis.
#               Possible values are: 'days', 'months', 'years'
# 'lab.tstep' : string indicating the time step that have to be used for
#               putting the labels ont he time axis.

.hydroplotts <- function(x, x.monthly, x.annual, win.len1=361*1, win.len2=365*3,
			 pfreq="dma", tick.tstep= "months", lab.tstep= "years", lab.fmt,
                         var.type, var.unit="units", main=NULL, xlab="Time", ylab=NULL, 
                         cex.main=1.3, cex.lab=1.3, cex.axis=1.3, col="blue", 
                         lwd=1, lty=1, ...) {

      # Checking that 'x' is a zoo object
      if (is.na(match(class(x), c("zoo"))))
            stop("Invalid argument: 'x' must be of class 'zoo'")

      # Checking that 'x.monthly' is a zoo object
      if ( length(x.monthly) > 1 ) {
        if (is.na(match(class(x.monthly), c("zoo"))))
            stop("Invalid argument: 'x.monthly' must be of class 'zoo'")
      } # IF end

      # Checking that 'x.annual' is a zoo object
      if ( length(x.annual) > 1 ) {
        if (is.na(match(class(x.annual), c("zoo"))))
            stop("Invalid argument: 'x.annual' must be of class 'zoo'")
      } # IF end

      # Checking that the user provied a valid argument for 'pfreq'
      if (is.na(match(pfreq, c("o", "dma", "ma", "dm"))))
          stop("Invalid argument: 'pfreq' must be in c('o', 'dma', 'ma', 'dm')")

      # Checking that the user provided a valid argument for 'tick.tstep'
      if (is.na(match(tick.tstep, c("days", "months", "years") ) ) )
         stop("Invalid argument: 'tick.tstep' must be in c('days', 'months', 'years')")

      # Checking that the user provided a valid argument for 'lab.tstep'
      if (is.na(match(lab.tstep, c("days", "months", "years") ) ) )
         stop("Invalid argument: 'lab.tstep' must be in c('days', 'months', 'years')")

      # Requiring the Zoo Library (Zoo's ordered observations)
      require(zoo)

      # Booleans indicating if the moving averages for the dayly and monthly
      # time series can be computed and ploted. By default they  are FALSE,
      # and only if the lenght(x) is large enough they are changed into TRUE
      d.ma1 <- FALSE
      d.ma2 <- FALSE
      m.ma1 <- FALSE
      m.ma2 <- FALSE

      # Generating a Moving Average of the Daily time series, with a window width 'win.len1'
      if (win.len1 > 0 ) { # If 'win.len1==0', the moving average is not computed
          win.len <- win.len1
          if (length(x) >= win.len) {
            d.ma1 <- TRUE
            daily.ma1 <- ma.zoo(x, win.len) }
      } # IF end

      # Generating a Moving Average of the Daily time series, with a window width 'win.len2'
      if (win.len2 > 0 ) {  # If 'win.len2==0', the moving average is not computed
          win.len <- win.len2
          if (length(x) >= win.len) {
            d.ma2 <- TRUE
            daily.ma2 <- ma.zoo(x, win.len) }
      } # IF end

      # Generating a Moving Average of the Monthly time series, with a window width 'win.len1'
      win.len <- round(win.len1/365,1)*12
      if (length(x.monthly) >= win.len) {
        m.ma1 <- TRUE
        monthly.ma1 <- ma.zoo( x.monthly, win.len )  }

      # Generating a Moving Average of the Monthly time series, with a window width 'win.len2'
      win.len <- round(win.len2/365,1)*12
      if (length(x.monthly) >= win.len) {
        m.ma2 <- TRUE
        monthly.ma2 <- ma.zoo( x.monthly, win.len ) }


      # Plotting only the original zoo object, without moving avergaes and legends
      if ( pfreq == "o") {
          # Plotting the Daily Time Series
          # xaxt = "n": is for avoiding drawing the x axis
          plot.zoo(x, xaxt = "n", type="o", 
                   main=main, xlab=xlab, ylab=ylab, 
                   cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col, 
                   lty=lty, lwd=lwd, ...)
          # Draws monthly ticks in the X axis, but labels only in years
          drawxaxis(x, tick.tstep=tick.tstep, lab.tstep=lab.tstep, lab.fmt=lab.fmt,
                    cex.lab=cex.lab, cex.axis=cex.axis, ...)
      } # IF end


      # Plotting the Daily, if needed
      if ( pfreq %in% c("dma", "dm") ) {
          # Generating the title of the Daily Time Series plot
          title <- paste("Daily", main, sep= " ")
          # Plotting the Daily Time Series
          # xaxt = "n": is for avoiding drawing the x axis
          plot.zoo(x, xaxt = "n", type="o", 
                   main=title, xlab=xlab, ylab=paste(ylab," [", var.unit,"/day]", sep=""), 
                   cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col, 
                   lty=lty, lwd=lwd, ...)
          # Draws monthly ticks in the X axis, but labels only in years
          drawxaxis(x, tick.tstep=tick.tstep, lab.tstep=lab.tstep, lab.fmt=lab.fmt,
                    cex.lab=cex.lab, cex.axis=cex.axis, ...)

          if (d.ma1) {
            # Plotting the 1st Moving Average of the Daily time series. If win.len1=365*1 => "Annual Moving Average"
            lines(daily.ma1, type="o", lty=2, lwd=1, col="green", cex = .5) }
          if (d.ma2) {
            # Plotting the 2nd Moving Average of the Daily time series. If win.len2=365*3 => "Moving Average of 3 Years"
            lines(daily.ma2, type="o", lty=3, lwd=1, col="red", cex = .5) }
          # Drawing a legend. y.intersp=0.5, is for the vertical spacin in the legend
          legend("topleft", c("Daily series", paste("MA(", round(win.len1/365,2), " years)", sep=""),
                 paste("MA(", round(win.len2/365,2), " years)", sep="") ), 
                 bty="n", cex =0.9, col= c(col,"green","red"), lwd= c(lwd,1,1), lty=c(lty,2,3) ) #bty="n" => no box around the legend
      } # IF end


      # Plotting the Monthly, if needed
      if ( pfreq %in% c("dma", "dm", "ma") ) {
        # Generating the title of the Monthly Time Series plot
        title <- paste("Monthly", main, sep= " ")
        # Plotting the Monthly time series
        plot.zoo(x.monthly, xaxt = "n", type="o",
                 main=title, xlab=xlab, ylab=paste(ylab," [", var.unit,"/month]", sep=""),
                 cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col, 
                 lty=lty, lwd=lwd, ... )


        # Draws monthly ticks in the X axis, but labels only in years
        drawxaxis(x.monthly, tick.tstep=tick.tstep, lab.tstep=lab.tstep, lab.fmt=lab.fmt,
                  cex.lab=cex.lab, cex.axis=cex.axis, ...)
        if (m.ma1) {
        # Plotting the 1st Moving Average of the Daily time series. If win.len1=365*1 => "Annual Moving Average"
        lines(monthly.ma1, type="o", lty=2, lwd=1, col="green", cex = .5) }
        if (m.ma2) {
        # Plotting the 2nd Moving Average of the Daily time series. If win.len2=365*3 => "Moving Average of 3 Years"
        lines(monthly.ma2, type="o", lty=3, lwd=1, col="red", cex = .5) }
        # Drawing a legend
        legend("topleft", c("Monthly series", paste("MA(", round(win.len1/365,1), " years)", sep=""),
             paste("MA(", round(win.len2/365,1), " years)", sep="") ), 
             bty="n", cex =0.9, col = c(col,"green","red"), lwd= c(lwd,1,1), lty=c(lty,2,3) ) #bty="n" => no box around the legend
      } # IF end

      # Plotting the Annual, if needed
      if ( pfreq %in% c("dma", "ma") ) {
          # Generating the title of the Annual Time Series plot
           title <- paste("Annual", main, sep= " ")
          # Plotting the Annual time series
          plot.zoo(x.annual, xaxt = "n", type="o", 
                   main=title, xlab="Time", ylab=paste(ylab," [", var.unit,"/year]", sep=""),
                   cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col, 
                   lty=lty, lwd=lwd, ...)
          # Draws monthly ticks in the X axis, but labels only in years
          drawxaxis(x.annual, tick.tstep="years", lab.tstep="years", lab.fmt=lab.fmt,
                    cex.lab=cex.lab, cex.axis=cex.axis, ...)
      } # IF end

} # '.hydroplotts' end



#####################################################
# BoxPlot of Daily, Monthly and Annual Time Serires #
#####################################################
# 'x'		 : daily time series of type 'zoo'
# 'x.monthly : monthly time series of type 'zoo'
# 'x.annual' : annual time series of type 'zoo'
# 'var.type' : string representing the type of variable being plotted
#              (e.g., "Precipitation", "Temperature" or "Flow").
#              ONLY used for labelling the y axis and the title of the plot (in case it is missing)
# 'var.unit' : string representing the measurement unit of the variable
#              being plotted ("mm" for precipitation, "C" for temperature, and "m3/s" for flow).
#              ONLY used for labelling the y axis
# 'main'     : string representing the main title of the plot
# 'pfreq'    : string indicating how many plots are desired by the user.
#              Valid values are:
#              -) 'dma': Daily, Monthly and Annual values are plotted
#              -) 'dm' : Daily and Monthly values are plotted
#              -) 'ma' : Monthly and Annual values are plotted
#
.hydroplotboxplot <- function(x, x.monthly, x.annual, pfreq="dma",
                              var.type, var.unit="units", 
                              main=NULL, xlab=NULL, ylab=NULL,
                              cex.main=1.3, cex.lab=1.3, cex.axis=1.3, 
                              col="lightblue", 
                              ...
			      ) {

  # Checking that 'x' is a zoo object
  if (is.na(match(class(x), c("zoo"))))
        stop("Invalid argument: 'x' must be of class 'zoo'")

  # Checking that 'x.monthly' is a zoo object
  if (is.na(match(class(x.monthly), c("zoo"))))
        stop("Invalid argument: 'x.monthly' must be of class 'zoo'")

  # Checking that 'x.annual' is a zoo object
  if (is.na(match(class(x.annual), c("zoo"))))
        stop("Invalid argument: 'x.annual' must be of class 'zoo'")

  # Checking that the user provied a valid argument for 'pfreq'
  if (is.na(match(pfreq, c("dma", "ma", "dm"))))
      stop("Invalid argument: 'pfreq' must be in c('dma', 'ma', 'dm')")

 # Requiring the Zoo Library (Z’s ordered observations)
 require(zoo)

 # Checking if the Daily ts have to be plotted
 if ( pfreq %in% c("dma", "dm") ) {
   # Generating a factor based on the year in which each daily date falls
   cyear <- format(time(x), "%Y")
   years <- factor(cyear, levels=unique(cyear), ordered=TRUE)
   # Generating the title of the Daily plot
   title <- paste("Boxplot of Daily", main, sep= " ")
   # Drawing boxplot of Daily values against Year
   boxplot( coredata(x)~years, main=title, ylab=paste(ylab," [", var.unit, "/day]", sep=""), 
            cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col, ...)
 } # IF end

 # Checking if the Monthly ts have to be plotted
 if ( pfreq %in% c("dma", "dm", "ma") ) {
   # Generating a factor based on the month in which each monthly date falls
   cmonth <- format(time(x.monthly), "%b")
   months <- factor(cmonth, levels=unique(cmonth), ordered=TRUE)
   # Generating the title of the Monthly plot
   title <- paste("Boxplot of Monthly", main, sep= " ")
   # Drawing boxplot of Monthly values against Year
   boxplot( coredata(x.monthly)~months, main=title, ylab=paste(ylab," [", var.unit,"/month]", sep=""), 
            cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col, ...)
 } # IF end

 # Checking if the Annual ts have to be plotted
 if ( pfreq %in% c("dma", "ma") ) {
   # Generating the title of the Annual plot
   title <- paste("Boxplot of Annual", main, sep= " ")
   # Drawing boxplot of Annual values against Year
   boxplot( coredata(x.annual), main=title, ylab=paste(ylab," [", var.unit, "/year]", sep=""), 
            cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col, ...)
 } # IF end

} #'.hydroplotboxplot' end


#######################################################
# Histogram of Daily, Monthly and Annual Time Serires #
#######################################################
# 'x'		 : daily time series of type 'zoo'
# 'x.monthly : monthly time series of type 'zoo'
# 'x.annual' : annual time series of type 'zoo'
# 'var.type' : string representing the type of variable being plotted (e.g., "Precipitation", "Temperature" or "Flow").
#              ONLY used for labelling the y axis and the title of the plot (in case it is missing)
# 'var.unit' : string representing the measurement unit of the variable being plotted ("mm" for precipitation, "C" for temperature, and "m3/s" for flow).
#              ONLY used for labelling the x axis
# 'main'     : string representing the main title of the plot
# 'pfreq'    : string indicating how many plots are desired by the user.
#              Valid values are:
#              -) 'dma': Daily, Monthly and Annual values are plotted
#              -) 'ma' : Monthly and Annual values are plotted
#              -) 'dm' : Daily and Monthly values are plotted
.hydroplothist <- function(x, x.monthly, x.annual, pfreq="dma",
                           var.type, var.unit="units", 
                           main=NULL, xlab=NULL, ylab=NULL, 
                           cex.main=1.3, cex.lab=1.3, cex.axis=1.3, col="lightblue", 
                           ...
			   ) {

      # Checking that 'x' is a zoo object
      if (is.na(match(class(x), c("zoo"))))
            stop("Invalid argument: 'x' must be of class 'zoo'")

      # Checking that 'x.monthly' is a zoo object
      if (is.na(match(class(x.monthly), c("zoo"))))
            stop("Invalid argument: 'x.monthly' must be of class 'zoo'")

      # Checking that 'x.annual' is a zoo object
      if (is.na(match(class(x.annual), c("zoo"))))
            stop("Invalid argument: 'x.annual' must be of class 'zoo'")

      # Checking that the user provied a valid argument for 'pfreq'
      if (is.na(match(pfreq, c("dma", "ma", "dm"))))
          stop("Invalid argument: 'pfreq' must be in c('dma', 'ma', 'dm')")

     # Requiring the Zoo Library (Z’s ordered observations)
     require(zoo)

     # Checking if the Daily ts have to be plotted
     if ( pfreq %in% c("dma", "dm") ) {
       # Generating the title of the Daily plot
       title <- paste("Histogram of Monthly", main, sep= " ")
       # Drawing an histogram of Daily Precipitation
       hist(x, br=100, freq=FALSE, main=title, xlab=paste(ylab," [", var.unit, "/day]", sep=""), 
            ylab="Pbb", cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col, ...)
     } # IF end

     # Checking if the Monthly ts have to be plotted
     if ( pfreq %in% c("dma", "dm", "ma") ) {
       # Generating the title of the Monthly plot
       title <- paste("Histogram of Monthly", main, sep= " ")
       # Drawing an histogram of Monthly Precipitation
       hist(x.monthly, br=10, freq=FALSE, main=title, xlab=paste(ylab," [", var.unit, "/month]", sep=""), 
            ylab="Pbb", cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col, ...)
     } # IF end

     # Checking if the Annual ts have to be plotted
     if ( pfreq %in% c("dma", "ma") ) {
       # Generating the title of the Annual plot
       title <- paste("Histogram of Annual", main, sep= " ")
       # Drawing an histogram of Annual Precipitation
       hist(x.annual, br=5, freq=FALSE, main=title, xlab=paste(ylab," [", var.unit, "/year]", sep=""), 
            ylab="Pbb", cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col, ...)
     } # IF end

} # '.hydroplothist' end


#########################################################################
# hydroplot: Daily, Monthly and Annual plots of hydrological time series#
#########################################################################
# 9 plots:
# 1: Line plot with Daily time series, with 2 moving averages, specified by 'win.len1' and 'win.len2'
# 2: Line plot with Monthly time series, with 2 moving averages, specified by 'win.len1' and 'win.len2'
# 3: Line plot with Annual time series
# 4: Boxplot with daily time series
# 5: Boxplot with monthly time series
# 6: Boxplot with annual time series
# 7: Histogram of the daily time series
# 8: Histogram of the monthly time series
# 9: Histogram of the annual time series

# 'x'	     : variable of type 'zoo', with daily ts
# 'same'     : Character representing the name of the meteorological station
#              ONLY used for labelling the title
# 'var.unit' : Character representing the measurement unit of the variable being plotted.
#              ONLY used for labelling the axes
#              e.g., "mm" for precipitation, "C" for temperature, and "m3/s" for flow.
# 'main'     : Character representing the main title of the plot. If the user did not provide a title, this is
#              created automatically as: main= paste(var.type, "at", st.name, sep=" "),
# 'win.len1' : number of days for being used in the computation of the first moving average
# 'win.len2' : number of days for being used in the computation of the second moving average
# 'ptype'    : Character indicating the type of plot that will be plotted. Valid values are
#              -) ptype= "ts"              => only time series
#              -) ptype= "ts+boxplot"      => only time series + boxplot
#              -) ptype= "ts+hist"         => only time series + histogram
#              -) ptype= "ts+boxplot+hist" => time series + boxplot + histogram
# 'pfreq'    : Character indicating how many plots are desired by the user.
#              Valid values are:
#              -) 'dma': Daily, Monthly and Annual values are plotted
#              -) 'ma' : Monthly and Annual values are plotted
#              -) 'dm' : Daily and Monthly values are plotted
# 'var.type' : string representing the type of variable being plotted
#              Used for determining the function used for computing the
#              Monthly and Annual values when 'FUN' is missing
#              Valid values are:
#              -) "Precipitation" => FUN = sum
#              -) "Temperature"   => FUN = mean
#              -) "Flow"          => FUN = mean
# 'FUN'      : ONLY required when 'var.type' is missing
#              Function that have to be applied for transforming from daily to monthly or annual time step
#              For precipitation FUN MUST be "sum"
#              For temperature and flow time series, FUN MUST be "mean"#
# 'na.rm'    : Logical. Should missing values be removed?
#              TRUE : the monthly and annual values  are computed considering only those values different from NA
#              FALSE: if there is AT LEAST one NA within a year, the monthly and annual values are NA
# 'tick.tstep': string indicating the time step that have to be used for
#               putting the ticks ont he time axis.
#               Possible values are: 'days', 'months', 'years'
# 'lab.tstep' : string indicating the time step that have to be used for
#               putting the labels ont he time axis.

hydroplot <- function(x, FUN, na.rm=TRUE,
                      ptype="ts+boxplot+hist",
		      pfreq="dma",                      
                      var.type,                      
                      var.unit="units",
                      main=NULL, xlab="Time", ylab=NULL,
                      win.len1=365*1,
                      win.len2=365*3,                      
                      tick.tstep="months",
                      lab.tstep="years",
                      lab.fmt,
                      cex=0.7,
                      cex.main=1.3,
                      cex.lab=1.3,
                      cex.axis=1.3,
                      col=c("blue", "lightblue", "lightblue"),
                      ...) {

     # Checking that 'x' is a zoo object
     if (is.na(match(class(x), c("zoo"))))
            stop("Invalid argument: 'x' must be of class 'zoo'")

     # Checking that the user provied a valid argument for 'ptype'
     if (is.na(match(ptype, c("ts", "ts+boxplot", "ts+hist", "ts+boxplot+hist"))))
            stop("Invalid argument: 'ptype' must be in c('ts', 'ts+boxplot', 'ts+hist', 'ts+boxplot+hist')")

     # Checking that the user provied a valid argument for 'pfreq'
     if ( sfreq(x) == "daily" ) {
       if (is.na(match(pfreq, c("o", "dma", "dm", "ma"))))
          stop("Invalid argument: 'pfreq' must be in c('o', 'dma', 'ma', 'dm')")
     } else if ( sfreq(x) == "monthly" ) {
         if ( pfreq != "ma" ) {
            message("Warning: 'x' is a monthly object, so 'pfreq' has been changed to 'ma'")
            pfreq="ma"
         }
       } # ELSE end

     if ( (pfreq == "o") & (ptype != "ts") ) {
          message(paste("Note: pfreq='o' => ptype has been changed to 'ts'" , sep="") )
          ptype <- "ts"
     } # IF end

     # Checking that the user provied a valid argument for 'var.type'
     if (missing(FUN) & (pfreq != "o") ) {
        # If the user did not provide a title for the plots, this is created automatically
        if (missing(var.type)) {
          stop("Missing argument: 'var.type' OR 'FUN' must be provided")
        } else # If 'var.type' is provided
             # Checking that the user provied a valid argument for 'var.type'
             if (is.na(match(var.type, c("Precipitation", "Temperature", "Flow") ) ) ) {
                   stop("Invalid argument: 'var.type' must be in c('Precipitation', 'Temperature', 'Flow')")
             } else {
                      if (var.type=="Precipitation") {
                          FUN <- sum
                          if (missing(var.unit)) { var.unit <- "mm"   }
                      } else if (var.type=="Temperature") {
                               FUN <- mean
                               if (missing(var.unit)) { var.unit <- "dC" }
                             } else if (var.type=="Flow") {
                                    FUN <- mean
                                    if (missing(var.unit)) { var.unit= "m3/s" }
                               }
                      } #ELSE end
     } # IF end

     # Assigning a dummy value to FUN, which is not used when pfreq="o"
     if (pfreq == "o") FUN <- mean

     def.par <- par(no.readonly = TRUE) # save default, for resetting...
     on.exit(par(def.par))

     # Requiring the Zoo Library
     require(zoo)

     # If the user didn't provided a value for 'tick.tstep' and
     # the lenght of the daily ts is less than 1 year, the ticks in
     # the 'x' axis are placed by day; if larger than a year, they are placed by month
     if ( missing(tick.tstep) ) {
       #if (length(x) <= 366) {
       if ( ( (sfreq(x) == "daily") & ( length(x) <= 366 ) ) |
          ( (sfreq(x) == "monthly") & ( length(x) <= 12 ) ) ) {
         tick.tstep <- "days"
       } else tick.tstep <- "months"
     } # IF end



     # If the user didn't provided a value for 'lab.tstep' and
     # the lenght of the daily ts is less than 1 year, the labels in
     # the 'x' axis are placed by month; if larger than a year, they are placed by year
     if ( missing(lab.tstep) ) {
       #if (length(x) <=366) {
       if ( ( (sfreq(x) == "daily") & ( length(x) <= 366 ) ) |
          ( (sfreq(x) == "monthly") & ( length(x) <= 12 ) ) ) {
         lab.tstep <- "months"
       } else lab.tstep <- "years"
     } # IF end
 
     # If the user didn't provide a value for 'lab.fmt', default values are used
     if (missing(lab.fmt)) {
       if (lab.tstep == "days") {
         lab.fmt <- "%Y-%m-%d"
       } else if (lab.tstep == "months") {
           lab.fmt <- "%b"
         } else if (lab.tstep == "years") {
           lab.fmt <- "%Y"
           }
     } # IF end

     
     if (pfreq != "o") {

       # If 'x' is too short for plotting annual values, 'pfreq' is automatically changed
       if ( ( (sfreq(x) == "daily") & ( length(x) <= 366 ) ) |
            ( (sfreq(x) == "monthly") & ( length(x) <= 12 ) ) ) {
           if ( match(pfreq, c("dma", "ma") ) ) {
             if (pfreq == "dma") pfreq <- "dm"
             if (pfreq == "ma") pfreq <- "m"
             message(paste("Warning: your ts is too short for plotting annual time series => 'pfreq'= ", pfreq, sep="") )
           }
       } # IF end

       # Computing the monthly time series
       if ( sfreq(x) == "daily" ) {
         x.monthly <- daily2monthly(x, FUN=FUN, na.rm=na.rm)
       } else if ( sfreq(x) == "monthly" ) {
          x.monthly <- x
          } else x.monthly <- NA

       # Computing the annual time series
       if ( !is.na( match( sfreq(x), c("daily", "monthly") ) ) ) {
         x.annual <- daily2annual(x, FUN=FUN, na.rm=na.rm, out.fmt="%Y-%m-%d")
       } else if ( sfreq(x) == "annual" ) {
          x.annual <- x
          } else x.annual <- NA

     } else {
       x.monthly <- NA
       x.annual  <- NA
       } # ELSE end   
     

     if (ptype=="ts") {

       # Setting up the screen with 3 rows and 3 columns
       if (pfreq == "o") { 
          par(mfcol=c(1,1))
       } else if (pfreq == "dma") { 
            par(mfcol=c(3,1))
         } else if (pfreq %in% c("dm", "ma")) { 
            par(mfcol=c(2,1))
           } # ELSE end
       # Drawing the daily, monthly and annual time series of the variable against time
       .hydroplotts(x=x, x.monthly=x.monthly, x.annual=x.annual, pfreq=pfreq,
                    win.len1=win.len1, win.len2=win.len2, var.type=var.type, 
                    var.unit=var.unit, main=main, xlab=xlab, ylab=ylab,  
                    tick.tstep= tick.tstep, lab.tstep= lab.tstep, lab.fmt, 
                    cex=cex, cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col[1], ...)
     } # IF end

     else if (ptype=="ts+boxplot") {
       # Setting up the screen with 3 rows and 3 columns
       if (pfreq == "dma") { par(mfcol=c(3,2))
       } else if (pfreq %in% c("dm", "ma")) { par(mfcol=c(2,2))
         } # ELSE end
       # Drawing the daily, monthly and annual time series of the variable against time
       .hydroplotts(x=x, x.monthly=x.monthly, x.annual=x.annual, pfreq=pfreq,
                    win.len1=win.len1, win.len2=win.len2, var.type=var.type, 
                    var.unit=var.unit, main=main, xlab=xlab, ylab=ylab,  
                    tick.tstep= tick.tstep, lab.tstep= lab.tstep, lab.fmt, 
                    cex=cex, cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col[1], ...)
       # Drawing a boxplot of the daily, monthly and annual time series of the variable
       .hydroplotboxplot(x=x, x.monthly=x.monthly, x.annual=x.annual, pfreq=pfreq,
                         var.type=var.type, var.unit=var.unit, main=main, xlab=xlab, ylab=ylab, 
                         cex=cex, cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col[2], ...)
     } # ELSE end

     else if (ptype=="ts+hist") {
       # Setting up the screen with 3 rows and 3 columns
       if (pfreq == "dma") { par(mfcol=c(3,2))
       } else if (pfreq %in% c("dm", "ma")) { par(mfcol=c(2,2))
         } # ELSE end
       # Drawing the daily, monthly and annual time series of the variable against time
       .hydroplotts(x=x, x.monthly=x.monthly, x.annual=x.annual, pfreq=pfreq,
                    win.len1=win.len1, win.len2=win.len2, var.type=var.type, 
                    var.unit=var.unit, main=main, xlab=xlab, ylab=ylab,  
                    tick.tstep= tick.tstep, lab.tstep= lab.tstep, lab.fmt, 
                    cex=cex, cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col[1], ...)
       # Drawing an histogram of the daily, monthly and annual time series of the variable
       .hydroplothist(x=x, x.monthly=x.monthly, x.annual=x.annual, pfreq=pfreq,
                      var.type=var.type, var.unit=var.unit, main=main, xlab=xlab, ylab=ylab, 
                      cex=cex, cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col[3], ...)
     } # ELSE end

     else if (ptype=="ts+boxplot+hist") {
       # Setting up the screen with 3 rows and 3 columns
       if (pfreq == "dma") { par(mfcol=c(3,3))
       } else if (pfreq %in% c("dm", "ma")) { par(mfcol=c(2,3))
         } # ELSE end
       # Drawing the daily, monthly and annual time series of the variable against time
       .hydroplotts(x=x, x.monthly=x.monthly, x.annual=x.annual, pfreq=pfreq,
                    win.len1=win.len1, win.len2=win.len2, var.type=var.type, 
                    var.unit=var.unit, main=main, xlab=xlab, ylab=ylab,  
                    tick.tstep= tick.tstep, lab.tstep= lab.tstep, lab.fmt, 
                    cex=cex, cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col[1], ...)
       # Drawing a boxplot of the daily, monthly and annual time series of the variable
       .hydroplotboxplot(x=x, x.monthly=x.monthly, x.annual=x.annual, pfreq=pfreq,
                         var.type=var.type, var.unit=var.unit, main=main, xlab=xlab, ylab=ylab, 
                         cex=cex, cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col[2], ...)
       # Drawing an histogram of the daily, monthly and annual time series of the variable
       .hydroplothist(x=x, x.monthly=x.monthly, x.annual=x.annual, pfreq=pfreq,
                      var.type=var.type, var.unit=var.unit, main=main, xlab=xlab, ylab=ylab, 
                      cex=cex, cex.main=cex.main, cex.lab=cex.lab, cex.axis=cex.axis, col=col[3], ...)
     } # ELSE end

 } # 'hydroplot end
