% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modmedmlmbrms.R
\name{modmed.mlm.brms}
\alias{modmed.mlm.brms}
\title{Custom model fitting function for a 1-1-1 (moderated) mediation for the brms code}
\usage{
modmed.mlm.brms(
  data,
  L2ID,
  X,
  Y,
  M,
  moderator = NULL,
  mod.a = FALSE,
  mod.b = FALSE,
  mod.cprime = FALSE,
  covars.m = NULL,
  covars.y = NULL,
  random.a = FALSE,
  random.b = FALSE,
  random.cprime = FALSE,
  random.mod.a = FALSE,
  random.mod.b = FALSE,
  random.mod.cprime = FALSE,
  random.mod.m = FALSE,
  random.mod.y = FALSE,
  random.covars.m = NULL,
  random.covars.y = NULL,
  returndata = FALSE,
  family = gaussian,
  iter = 7000,
  control = list(adapt_delta = 0.95),
  chains = 4,
  ...
)
}
\arguments{
\item{data}{Data frame in long format.}

\item{L2ID}{(Character) Name of column that contains grouping variable in \code{data} (e.g., \code{"SubjectID"}).}

\item{X}{(Character) Name of column that contains the X independent variable in \code{data}.}

\item{Y}{(Character) Name of column that contains the Y dependent variable in \code{data}.}

\item{M}{(Character) Name of column that contains the M mediating variable in \code{data}.}

\item{moderator}{Optional Character that contains name of column that contains the moderator variable in \code{data}}

\item{mod.a}{(Logical) Add moderator to 'a' path (i.e., SmX:W, where W is the moderator)?}

\item{mod.b}{(Logical) Add moderator to 'b' path (i.e., SyM:W, where W is the moderator)?}

\item{mod.cprime}{(Logical) Add moderator to 'c' path (i.e., SyX:W, where W is the moderator)}

\item{covars.m}{(Character vector) Optional covariates to include in the model for M. (not yet implemented for brms)}

\item{covars.y}{(Character vector) Optional covariates to include in the model for Y. (not yet implemented for brms)}

\item{random.a}{(Logical) Add random slope for 'a' path (i.e,. SmX)?}

\item{random.b}{(Logical) Add random slope for 'b' path (i.e., SyM)?}

\item{random.cprime}{(Logical) Add random slope for 'cprime' direct effect path (i.e., SyX)?}

\item{random.mod.a}{(Logical) Add random slope for 'a' path moderator?}

\item{random.mod.b}{(Logical) Add random slope for 'b' path moderator?}

\item{random.mod.cprime}{(Logical) Add random slope for 'c' path moderator?}

\item{random.mod.m}{(Logical) Add random slope for effect of moderator on M?}

\item{random.mod.y}{(Logical) Add random slope for effect of moderator on Y?}

\item{random.covars.m}{(Logical vector) Add random slopes for covariates on M? (not yet implemented for brms)}

\item{random.covars.y}{(Logical vector) Add random slopes for covariates on Y? (not yet implemented for brms)}

\item{returndata}{(Logical) Whether to save restructured data in its own slot. Defaults to \code{FALSE}.}

\item{family}{Argument passed to \code{\link[brms]{brm}} A character string naming the distribution of the response variable to be used in the model.}

\item{iter}{Argument passed to \code{\link[brms]{brm}} Number of total iterations.}

\item{control}{Argument passed to \code{\link[brms]{brm}} To decrease (or eliminate at best) the number of divergent transitions that cause a bias in the obtained posterior samples.}

\item{chains}{Argument passed to \code{\link[brms]{brm}} Set the number of chains}

\item{...}{Additional arguments to pass to \code{\link[brms]{brm}}}
}
\value{
A list with the following elements:
\itemize{
 \item{\code{model} The fitted model from \code{\link[brms]{brm}}. Use as you would a fitted model from that package.}
 \item{\code{args} Arguments used to call the function. Useful for later automating extraction of the indirect
   effect or other quantities.}
 \item{\code{conv} Whether \code{\link[brms]{brm}} finished estimation, not diagnostic of convergence.}
}
}
\description{
Custom model fitting function for a 1-1-1 (moderated) mediation for the brms code
}
\details{
Implements custom function to do (moderated) mediation with two-level multilevel models
  with Bayesian estimation via the \code{\link[brms]{brms}} package. Does not handle covariates at the moment.
  Bayesian estimation using \code{\link[brms]{brms}} was studied by Falk, Vogel, Hammami & Miočević (in press). It is
  suggested if you use this function that you also do \code{cite("brms")} to figure out how to cite that package.
}
\examples{
\donttest{

# Note: 2000 iterations is just an example so that run time is not too long.
# Pick something larger (e.g., 5000+) in practice

# Example data for 1-1-1 w/o moderation
data(BPG06dat)

# random effects for a and b paths (and intercept), no moderation
# (For moderation, note that modmed.mlm syntax is typically the same)
fit<-modmed.mlm.brms(BPG06dat,"id", "x", "y" , "m", cores=2,
                     random.a=TRUE, random.b=TRUE,
                     iter = 2000, control = list(adapt_delta=0.95),
                     seed = 1234)

# Examine model results and some diagnostics
summary(fit$model)

# Potential scale reduction (PSR) or Rhat guidelines vary but the largest
#  should be close to 1 ( < 1.1, < 1.05, < 1.01).
# It is also possible to extract all of them.
max(brms::rhat(fit$model)) # largest rhat

# Fit (loo and WAIC)
brms::loo(fit$model)
brms::waic(fit$model)

# Point and interval estimates, diagnostics, for quantities of interest

# Traceplots: TODO, list conversions for how brms represents parameters with
# How these are colloquially referred to in mediation literature.
plot(fit$model, variable="b_SmX") # this is traceplot for one parameter

# Example of extracting/computing intervals for particular quantities
res.indirect <- extract.modmed.mlm.brms(fit, "indirect")
res.a <- extract.modmed.mlm.brms(fit, "a")
res.b <- extract.modmed.mlm.brms(fit, "b")
res.cprime <- extract.modmed.mlm.brms(fit, "cprime")

# Summary of results is in CI slot, example:
res.indirect$CI

# 99\% CI
res.indirect <- extract.modmed.mlm.brms(fit, "indirect", ci.conf = .99)
}

}
\references{
Falk, C. F., Vogel, T., Hammami, S., & Miočević, M. (in press). Multilevel mediation analysis in R: A comparison of bootstrap and Bayesian approaches. Behavior Research Methods. \doi{10.3758/s13428-023-02079-4}  Preprint: \doi{10.31234/osf.io/ync34}

Paul-Christian Bürkner (2017). brms: An R Package for Bayesian Multilevel Models Using Stan. Journal of Statistical Software, 80(1), 1-28. doi:10.18637/jss.v080.i01
}
