% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute_dmod.R
\name{compute_dmod_npar}
\alias{compute_dmod_npar}
\title{Function for computing non-parametric \mjeqn{d_{Mod}}{d_Mod} effect sizes for a single focal group}
\usage{
compute_dmod_npar(
  referent_int,
  referent_slope,
  focal_int,
  focal_slope,
  focal_x,
  referent_sd_y
)
}
\arguments{
\item{referent_int}{Referent group's intercept.}

\item{referent_slope}{Referent group's slope.}

\item{focal_int}{Focal group's intercept.}

\item{focal_slope}{Focal group's slope.}

\item{focal_x}{Focal group's vector of predictor scores.}

\item{referent_sd_y}{Referent group's criterion standard deviation.}
}
\value{
A vector of effect sizes (\mjeqn{d_{Mod_{Signed}}}{d_Mod_Signed},
    \mjeqn{d_{Mod_{Unsigned}}}{d_Mod_Unsigned}, \mjeqn{d_{Mod_{Under}}}{d_Mod_Under},
    \mjeqn{d_{Mod_{Over}}}{d_Mod_Over}), proportions of under- and over-predicted criterion scores,
    minimum and maximum differences (i.e., \mjeqn{d_{Mod_{Under}}}{d_Mod_Under} and \mjeqn{d_{Mod_{Over}}}{d_Mod_Over}),
    and the scores associated with minimum and maximum differences.
}
\description{
\loadmathjax
This function computes non-parametric \mjeqn{d_{Mod}}{d_Mod} effect sizes from user-defined descriptive statistics
and regression coefficients, using a distribution of observed scores as weights.
This non-parametric function is best used when the assumption of normally distributed predictor
scores is not reasonable and/or the distribution of scores observed in a sample is likely to represent the
distribution of scores in the population of interest.
If one has access to the full raw data set, the \code{dMod} function may be used
as a wrapper to this function so that the regression equations and descriptive statistics can
be computed automatically within the program.
}
\details{
The \mjeqn{d_{Mod_{Signed}}}{d_Mod_Signed} effect size (i.e., the average of differences in prediction over
the range of predictor scores) is computed as
\mjdeqn{d_{Mod_{Signed}}=\frac{\sum_{i=1}^{m}n_{i}\left[X_{i}\left(b_{1_{1}}-b_{1_{2}}\right)+b_{0_{1}}-b_{0_{2}}\right]}{SD_{Y_{1}}\sum_{i=1}^{m}n_{i}},}{d_Mod_Signed = sum(n_i * [X_i * (b_1_1 - b_1_2) + b_0_1 - b_0_2]) / (SD_Y_1 * sum(n_i)),}
where
  \itemize{
    \item {\mjeqn{SD_{Y_{1}}}{SD_Y_1} is the referent group's criterion standard deviation;}
    \item {\mjeqn{m}{m} is the number of unique scores in the distribution of focal-group predictor scores;}
    \item {\mjeqn{X}{X} is the vector of unique focal-group predictor scores, indexed \mjseqn{i=1} through \mjseqn{m};}
    \item {\mjeqn{X_{i}}{X_i} is the \mjeqn{i^{th}}{ith} unique score value;}
    \item {\mjeqn{n}{n} is the vector of frequencies associated with the elements of \mjeqn{X}{X};}
    \item {\mjeqn{n_{i}}{n_i} is the number of cases with a score equal to \mjeqn{X_{i}}{X_i};}
    \item {\mjeqn{b_{1_{1}}}{b_1_1} and \mjeqn{b_{1_{2}}}{b_1_2} are the slopes of the regression of \mjeqn{Y}{Y} on \mjeqn{X}{X} for the referent and focal groups, respectively; and}
    \item {\mjeqn{b_{0_{1}}}{b_0_1} and \mjeqn{b_{0_{2}}}{b_0_2} are the intercepts of the regression of \mjeqn{Y}{Y} on \mjeqn{X}{X} for the referent and focal groups, respectively.}
  }

The \mjeqn{d_{Mod_{Under}}}{d_Mod_Under} and \mjeqn{d_{Mod_{Over}}}{d_Mod_Over} effect sizes are computed
using the same equation as \mjeqn{d_{Mod_{Signed}}}{d_Mod_Signed}, but \mjeqn{d_{Mod_{Under}}}{d_Mod_Under} is
the weighted average of all scores in the area of underprediction (i.e., the differences in prediction with
negative signs) and \mjeqn{d_{Mod_{Over}}}{d_Mod_Over} is the weighted average of all scores in the area of
overprediction (i.e., the differences in prediction with negative signs).

The \mjeqn{d_{Mod_{Unsigned}}}{d_Mod_Unsigned} effect size (i.e., the average of absolute differences in prediction over
the range of predictor scores) is computed as
\mjdeqn{d_{Mod_{Unsigned}}=\frac{\sum_{i=1}^{m}n_{i}\left|X_{i}\left(b_{1_{1}}-b_{1_{2}}\right)+b_{0_{1}}-b_{0_{2}}\right|}{SD_{Y_{1}}\sum_{i=1}^{m}n_{i}}.}{d_Mod_Unsigned = sum(n_i * |X_i * (b_1_1 - b_1_2) + b_0_1 - b_0_2]| / (SD_Y_1 * sum(n_i)).}

The \mjeqn{d_{Min}}{d_Min} effect size (i.e., the smallest absolute difference in prediction observed over the
range of predictor scores) is computed as
\mjdeqn{d_{Min}=\frac{1}{SD_{Y_{1}}}Min\left[\left|X\left(b_{1_{1}}-b_{1_{2}}\right)+b_{0_{1}}-b_{0_{2}}\right|\right].}{d_Min = 1/SD_Y_1 * Min[X * (b_1_1 - b_1_2) + b_0_1 - b_0_2].}

The \mjeqn{d_{Max}}{d_Max} effect size (i.e., the largest absolute difference in prediction observed over the
range of predictor scores)is computed as
\mjdeqn{d_{Max}=\frac{1}{SD_{Y_{1}}}Max\left[\left|X\left(b_{1_{1}}-b_{1_{2}}\right)+b_{0_{1}}-b_{0_{2}}\right|\right].}{d_Max = 1/SD_Y_1 * Max[X * (b_1_1 - b_1_2) + b_0_1 - b_0_2].}
\emph{Note}: When \mjeqn{d_{Min}}{d_Min} and \mjeqn{d_{Max}}{d_Max} are computed in this package, the output will display the
signs of the differences (rather than the absolute values of the differences) to aid in interpretation.
}
\examples{
# Generate some hypothetical data for a referent group and three focal groups:
set.seed(10)
refDat <- MASS::mvrnorm(n = 1000, mu = c(.5, .2),
                        Sigma = matrix(c(1, .5, .5, 1), 2, 2), empirical = TRUE)
foc1Dat <- MASS::mvrnorm(n = 1000, mu = c(-.5, -.2),
                         Sigma = matrix(c(1, .5, .5, 1), 2, 2), empirical = TRUE)
foc2Dat <- MASS::mvrnorm(n = 1000, mu = c(0, 0),
                         Sigma = matrix(c(1, .3, .3, 1), 2, 2), empirical = TRUE)
foc3Dat <- MASS::mvrnorm(n = 1000, mu = c(-.5, -.2),
                         Sigma = matrix(c(1, .3, .3, 1), 2, 2), empirical = TRUE)
colnames(refDat) <- colnames(foc1Dat) <- colnames(foc2Dat) <- colnames(foc3Dat) <- c("X", "Y")

# Compute a regression model for each group:
refRegMod <- lm(Y ~ X, data.frame(refDat))$coef
foc1RegMod <- lm(Y ~ X, data.frame(foc1Dat))$coef
foc2RegMod <- lm(Y ~ X, data.frame(foc2Dat))$coef
foc3RegMod <- lm(Y ~ X, data.frame(foc3Dat))$coef

# Use the subgroup regression models to compute d_mod for each referent-focal pairing:

# Focal group #1:
compute_dmod_npar(referent_int = refRegMod[1], referent_slope = refRegMod[2],
             focal_int = foc1RegMod[1], focal_slope = foc1RegMod[2],
             focal_x = foc1Dat[,"X"], referent_sd_y = 1)

# Focal group #2:
compute_dmod_npar(referent_int = refRegMod[1], referent_slope = refRegMod[2],
             focal_int = foc2RegMod[1], focal_slope = foc1RegMod[2],
             focal_x = foc2Dat[,"X"], referent_sd_y = 1)

# Focal group #3:
compute_dmod_npar(referent_int = refRegMod[1], referent_slope = refRegMod[2],
             focal_int = foc3RegMod[1], focal_slope = foc3RegMod[2],
             focal_x = foc3Dat[,"X"], referent_sd_y = 1)
}
