% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rbiom_objects.r
\name{rbiom_objects}
\alias{rbiom_objects}
\title{Working with rbiom Objects.}
\description{
Rbiom objects make it easy to access and manipulate your BIOM data, ensuring
all the disparate components remain in sync. These objects behave largely
like lists, in that you can access and assign to them using the \code{$}
operator. The sections below list all the fields which can be read and/or
written, and the helper functions for common tasks like rarefying and
subsetting. To create an rbiom object, see \code{\link[=as_rbiom]{as_rbiom()}}.

Use \verb{$clone()} to create a copy of an rbiom object. This is necessary
because rbiom objects are \strong{passed by reference}. The usual \verb{<-} assignment
operator will simply create a second reference to the same object - it will
not create a second object. See \link[=speed]{speed ups} for more details.
}
\section{Readable Fields}{


Reading from fields will not change the rbiom object.\tabular{ll}{
   \strong{Accessor} \tab \strong{Content} \cr
   \verb{$counts} \tab Abundance of each OTU in each sample. \cr
   \verb{$metadata} \tab Sample mappings to metadata (treatment, patient, etc). \cr
   \verb{$taxonomy} \tab OTU mappings to taxonomic ranks (genus, phylum, etc). \cr
   \verb{$otus}, \verb{$n_otus} \tab OTU names. \cr
   \verb{$samples}, \verb{$n_samples} \tab Sample names. \cr
   \verb{$fields}, \verb{$n_fields} \tab Metadata field names. \cr
   \verb{$ranks}, \verb{$n_ranks} \tab Taxonomic rank names. \cr
   \verb{$tree}, \verb{$sequences} \tab Phylogenetic tree / sequences for the OTUs, or \code{NULL}. \cr
   \verb{$id}, \verb{$comment} \tab Arbitrary strings for describing the dataset. \cr
   \verb{$depth} \tab Rarefaction depth, or \code{NULL} if unrarefied. \cr
   \verb{$date} \tab Date from BIOM file. \cr
}
}

\section{Writable Fields}{


Assigning new values to these components will trigger
validation checks and inter-component synchronization.\tabular{ll}{
   \strong{Component} \tab \strong{What can be assigned.} \cr
   \verb{$counts} \tab Matrix of abundances; OTUs (rows) by samples (columns). \cr
   \verb{$metadata} \tab Data.frame with \code{'.sample'} column, or a file name. \cr
   \verb{$taxonomy} \tab Data.frame with \code{'.otu'} as the first column. \cr
   \verb{$otus} \tab Character vector with new names for the OTUs. \cr
   \verb{$samples} \tab Character vector with new names for the samples. \cr
   \verb{$tree} \tab Phylo object with the phylogenetic tree for the OTUs. \cr
   \verb{$sequences} \tab Named character vector of OTU reference sequences. \cr
   \verb{$id}, \verb{$comment} \tab String with dataset's title or comment. \cr
   \verb{$date} \tab Date-like object, or \code{"\%Y-\%m-\%dT\%H:\%M:\%SZ"} string. \cr
}
}

\section{Transformations}{


All functions return an rbiom object.\tabular{ll}{
   \strong{Function} \tab \strong{Transformation} \cr
   \code{<rbiom>$clone()} \tab Safely duplicate an rbiom object. \cr
   \code{\link[=subset]{<rbiom>[}} \tab Subset to a specific set of sample names. \cr
   \code{\link[=subset]{subset()}} \tab Subset samples according to metadata properties. \cr
   \code{\link[=slice]{slice()}} \tab Subset to a specific number of samples. \cr
   \code{\link[=mutate]{mutate()}} \tab Create, modify, and delete metadata fields. \cr
   \code{\link[=rarefy]{rarefy()}} \tab Sub-sample OTU counts to an even sampling depth. \cr
}
}

\examples{
    library(rbiom)
    
    # Duplicate the HMP50 example dataset.
    biom <- hmp50$clone()
    
    
    # Display an overall summary of the rbiom object.
    biom
    
    
    # Markdown syntax for comments is recommended.
    biom$comment \%>\% cli::cli_text()
    
    
    # Demonstrate a few accessors.
    biom$n_samples
    biom$fields
    biom$metadata
    
    
    # Edit the metadata table.
    biom$metadata$rand <- sample(1:50)
    biom \%<>\% mutate(Obese = BMI >= 30, Sex = NULL)
    biom \%<>\% rename('Years Old' = "Age")
    biom$metadata
    
    
    # Subset the rbiom object
    biom \%<>\% subset(`Body Site` == "Saliva" & !Obese)
    biom$metadata
    
    
    # Rarefy to an even sampling depth
    sample_sums(biom)
    
    biom \%<>\% rarefy()
    sample_sums(biom)

}
\keyword{internal}
