% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R, R/tape_bdryw.R
\name{tape_bdryw}
\alias{tape_bdryw}
\alias{tape_bdryw_inbuilt}
\title{Generate a tape of a custom boundary weight function}
\usage{
tape_bdryw_inbuilt(name, x, acut)

tape_bdryw(fileORcode = "", x, Cppopt = NULL)
}
\arguments{
\item{name}{Name of an inbuilt function. See details.}

\item{x}{Vector giving location in the manifold.}

\item{acut}{The \eqn{a_c} threshold used by the "minsq" and "prodsq" built-in functions. See details.}

\item{fileORcode}{A character string giving the path name of a file containing the unnormalised log-density definition \emph{OR} code. \code{fileORcode} will be treated as a file name if \code{fileORcode} contains no new line characters ('\\n' or '\\r\\n') and has a file extension detected by \code{\link[tools:fileutils]{tools::file_ext()}}.}

\item{Cppopt}{List of named options passed to \code{Rcpp::sourceCpp()}}
}
\value{
A list of three objects
\itemize{
\item \code{fun} a function that evaluates the function directly
\item \code{tape} a tape of the function
\item \code{file} the temporary file storing the final source code passed to \code{\link[Rcpp:sourceCpp]{Rcpp::sourceCpp()}}
}
}
\description{
Generate a tape of a boundary weight function, which is useful for specifying the boundary of manifolds in score matching.
Use \code{tape_bdryw()} to specify a custom function using \verb{C++} code much like \code{TMB::compile()}.
Use \code{tape_bdryw_inbuilt()} for tapes of inbuilt functions implemented in this package.
}
\details{
For \code{tape_bdryw_inbuilt()}, currently available functions are:
\itemize{
\item The function "ones" applies no weights and should be used whenever the manifold does not have a boundary.
\deqn{h(x)^2 = 1.}
\item The function "minsq" is the minima-based boundary weight function \insertCite{@Equation 12, @scealy2023sc}{scorematchingad}
\deqn{h(x)^2 = \min(x_1^2, x_2^2, ..., x_p^2, a_c^2).}{h(x)^2 = min(x1^2, x2^2, ..., xp^2, a_c^2),}
where \eqn{x_j}{xj} is the jth component of x.
\item The function "prodsq" is the product-based \insertCite{@Equation 9, @scealy2023sc}{scorematchingad}
\deqn{h(x)^2 = \min(\prod_{j=1}^{p} x_j^2, a_c^2).}{h(x)^2 = min(x1^2 * x2^2 * ... * xp^2, a_c^2),}
where \eqn{x_j}{xj} is the jth component of x.
}

The "minsq" and "prodsq" functions are useful when the manifold is the positive orthant, the p-dimensional unit sphere restricted to the positive orthant, or the unit simplex.
\insertCite{@scealy2023sc}{scorematchingad} prove that both "minsq" and "prodsq" can be used for score matching the PPI model on the simplex or positive orthant of the sphere.

The function \code{tape_bdryw()} uses \code{\link[Rcpp:sourceCpp]{Rcpp::sourceCpp()}} to generate a tape of a function defined in C++.

The result result is NOT safe to save or pass to other CPUs in a parallel operation.
}
\section{Writing the \code{fileORcode} Argument}{
The code (possibly in the file pointed to by \code{fileORcode}) must be \verb{C++} that uses only \code{CppAD} and \code{Eigen}, which makes it very similar to the requirements of the input to \code{TMB::compile()} (which also uses \code{CppAD} and \code{Eigen}).

The start of \code{code} should always be "\verb{a1type fname(const veca1 &x)\{}" where \code{fname} is your chosen name of the function, \code{x} represents a point in the manifold. This specifies that the function will a single two vector argument (of type \code{veca1}) and will return a single numeric value (\code{a1type}).

The type \code{a1type} is a double with special ability for being taped by \code{CppAD}. The \code{veca1} type is a vector of \code{a1type} elements, with the vector wrapping supplied by the \code{Eigen} C++ package (that is an \code{Eigen} matrix with 1 column and dynamic number of rows).
In place operations like \verb{+=} on \code{a1type}, \code{veca1} and similar have not worked for me (compile errors); fortunately the efficiency of in place operations is irrelevant for taping operations.

The body of the function must use operations from Eigen and/or CppAD, prefixed by \verb{Eigen::} and \verb{CppAD::} respectively.
There are no easy instructions for writing these as it is genuine \verb{C++} code, which can be very opaque to those unfamiliar with \verb{C++}.
However, recently ChatGPT and claude.ai have been able to very quickly translating \code{R} functions to \verb{C++} functions (KLH has been telling these A.I. to use Eigen and CppAD, and giving the definitions of \code{a1type} and \code{veca1}).
I've found the quick reference pages for for \href{https://libeigen.gitlab.io/eigen/docs-nightly/group__QuickRefPage.html}{\code{Eigen}} useful. Limited unary and binary operations are available directly from \href{https://cppad.readthedocs.io}{\code{CppAD}} without \code{Eigen}.

Non-smooth functions should be used with care, but can be specified using \code{CppAD}'s CondExp functions.
}

\examples{
\dontrun{
out <- tape_bdrw(
"a1type myminsq(const veca1 &x){
veca1 xsq = x.array().square();
a1type minval(0.1 * 0.1);
for(size_t i=0;i<x.size();i++){
  minval = CppAD::CondExpLe(xsq[i], minval, xsq[i], minval);
}
return(minval);
}", 
   rep(0.2, 5))
out$fun(c(0.01, 0.2, 0.2, 0.2, 0.2))
out$tape$forward(0, c(0.1, 0.2, 0.2, 0.2, 0.2))
out$tape$Jacobian(c(0.1, 0.2, 0.2, 0.2, 0.2))
out$tape$name
}
}
\seealso{
Other tape builders: 
\code{\link{avgrange}()},
\code{\link{fixdynamic}()},
\code{\link{fixindependent}()},
\code{\link{keeprange}()},
\code{\link{tape_Hessian}()},
\code{\link{tape_Jacobian}()},
\code{\link{tape_gradoffset}()},
\code{\link{tape_logJacdet}()},
\code{\link{tape_smd}()},
\code{\link{tape_swap}()},
\code{\link{tape_uld}()}
}
\concept{tape builders}
