% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ovb_bounds.R
\name{ovb_bounds}
\alias{ovb_bounds}
\alias{ovb_bounds.lm}
\alias{ovb_partial_r2_bound}
\alias{ovb_partial_r2_bound.numeric}
\title{Bounds on the strength of unobserved confounders using observed covariates}
\usage{
ovb_bounds(...)

\method{ovb_bounds}{lm}(
  model,
  treatment,
  benchmark_covariates = NULL,
  kd = 1,
  ky = kd,
  reduce = TRUE,
  bound = c("partial r2", "partial r2 no D", "total r2"),
  adjusted_estimates = TRUE,
  alpha = 0.05,
  h0 = 0,
  ...
)

ovb_partial_r2_bound(...)

\method{ovb_partial_r2_bound}{numeric}(
  r2dxj.x,
  r2yxj.dx,
  kd = 1,
  ky = kd,
  bound_label = "manual",
  ...
)
}
\arguments{
\item{...}{arguments passed to other methods. First argument should either be an \code{\link{lm}} model with the
outcome regression, or a \code{\link{formula}} describing the model along
with the \code{\link{data.frame}} containing the variables of the model.}

\item{model}{An \code{lm} object with the outcome regression.}

\item{treatment}{A character vector with the name of the treatment variable
of the model.}

\item{benchmark_covariates}{The user has two options: (i)  character vector of the names of covariates that will be used to bound the plausible strength of the unobserved confounders. Each variable will be considered separately; (ii) a named list with character vector names of covariates that will be used, \emph{as a group}, to bound the plausible strength of the unobserved confounders. The names of the list will be used for the benchmark labels. Note: for factor variables with more than two levels, you need to provide the name of each level as encoded in the \code{lm} model (the columns of \code{model.matrix}).}

\item{kd}{numeric vector. Parameterizes how many times stronger the confounder is related to the treatment in comparison to the observed benchmark covariate.
Default value is \code{1} (confounder is as strong as benchmark covariate).}

\item{ky}{numeric vector. Parameterizes how many times stronger the confounder is related to the outcome in comparison to the observed benchmark covariate.
Default value is the same as \code{kd}.}

\item{reduce}{Should the bias adjustment reduce or increase the
absolute value of the estimated coefficient? Default is \code{TRUE}.}

\item{bound}{type of bounding procedure. Currently only \code{"partial r2"} is implemented.}

\item{adjusted_estimates}{should the bounder also compute the adjusted estimates? Default is \code{TRUE}.}

\item{alpha}{significance level for computing the adjusted confidence intervals. Default is 0.05.}

\item{h0}{Null hypothesis for computation of the t-value. Default is zero.}

\item{r2dxj.x}{partial R2 of covariate Xj with the treatment D (after partialling out the effect of the remaining covariates X, excluding Xj).}

\item{r2yxj.dx}{partial R2 of covariate Xj with the outcome Y (after partialling out the effect of the remaining covariates X, excluding Xj).}

\item{bound_label}{label to bounds provided manually in \code{r2dz.x} and \code{r2yz.dx}.}
}
\value{
The function \code{ovb_bounds} returns a \code{\link{data.frame}} with the bounds on the strength of the unobserved confounder
as well with the adjusted point estimates, standard errors and t-values (optional, controlled by argument \code{adjusted_estimates = TRUE}).

The function `ovb_partial_r2_bound()` returns only \code{\link{data.frame}} with the bounds on the strength of the unobserved confounder. Adjusted estimates, standard
errors and t-values (among other quantities) need to be computed manually by the user using those bounds with the functions \code{\link{adjusted_estimate}}, \code{\link{adjusted_se}} and \code{\link{adjusted_t}}.
}
\description{
Bounds on the strength of unobserved confounders using observed covariates, as in Cinelli and Hazlett (2020). The main generic function is \code{ovb_bounds}, which can compute both the bounds on the strength of confounding as well as the adjusted estimates, standard errors, t-values and confidence intervals.

Other functions that compute only the bounds on the strength of confounding are also provided. These functions may be useful when computing benchmarks for using only summary statistics from papers you see in print.
}
\details{
Currently it implements only the bounds based on partial R2. Other bounds will be implemented soon.
}
\examples{

# runs regression model
model <- lm(peacefactor ~ directlyharmed + age + farmer_dar + herder_dar +
             pastvoted + hhsize_darfur + female + village, data = darfur)

# bounds on the strength of confounders 1, 2, or 3 times as strong as female
# and 1,2, or 3 times as strong as pastvoted
ovb_bounds(model, treatment = "directlyharmed",
          benchmark_covariates = c("female", "pastvoted"),
          kd = 1:3)


#########################################################
## Let's construct bounds from summary statistics only ##
#########################################################
# Suppose you didn't have access to the data, but only to
# the treatment and outcome regression tables.
# You can still compute the bounds.

# Use the t statistic of female in the outcome regression
# to compute the partial R2 of female with the outcome.
r2yxj.dx <- partial_r2(t_statistic = -9.789, dof = 783)

# Use the t-value of female in the *treatment* regression
# to compute the partial R2 of female with the treatment
r2dxj.x <- partial_r2(t_statistic = -2.680, dof = 783)

# Compute manually bounds on the strength of confounders 1, 2, or 3
# times as strong as female
bounds <- ovb_partial_r2_bound(r2dxj.x = r2dxj.x,
                              r2yxj.dx = r2yxj.dx,
                              kd = 1:3,
                              ky = 1:3,
                              bound_label = paste(1:3, "x", "female"))
# Compute manually adjusted estimates
bound.values <- adjusted_estimate(estimate = 0.0973,
                                 se = 0.0232,
                                 dof = 783,
                                 r2dz.x = bounds$r2dz.x,
                                 r2yz.dx = bounds$r2yz.dx)

# Plot contours and bounds
ovb_contour_plot(estimate = 0.0973,
                se = 0.0232,
                dof = 783)
add_bound_to_contour(bounds, bound_value = bound.values)


}
\references{
Cinelli, C. and Hazlett, C. (2020), "Making Sense of Sensitivity: Extending Omitted Variable Bias." Journal of the Royal Statistical Society, Series B (Statistical Methodology).

Cinelli, C. and Hazlett, C. (2020), "Making Sense of Sensitivity: Extending Omitted Variable Bias." Journal of the Royal Statistical Society, Series B (Statistical Methodology).
}
