% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/taxmap--parsers.R
\name{extract_tax_data}
\alias{extract_tax_data}
\title{Extracts taxonomy info from vectors with regex}
\usage{
extract_tax_data(tax_data, key, regex, class_key = "taxon_name",
  class_regex = "(.*)", class_sep = NULL, sep_is_regex = FALSE,
  class_rev = FALSE, database = "ncbi", include_match = FALSE,
  include_tax_data = TRUE)
}
\arguments{
\item{tax_data}{A vector from which to extract taxonomy information.}

\item{key}{(\code{character}) The identity of the capturing groups defined using
\code{regex}. The length of \code{key} must be equal to the number of capturing
groups specified in \code{regex}. Any names added to the terms will be used as
column names in the output. Only \code{"info"} can be used multiple times. Each
term must be one of those described below:
\itemize{
\item \code{taxon_id}: A unique numeric id for a taxon for a particular \code{database}
(e.g. ncbi accession number). Requires an internet connection.
\item \code{taxon_name}: The name of a taxon (e.g. "Mammalia" or "Homo sapiens").
Not necessarily unique, but interpretable by a particular \code{database}.
Requires an internet connection.
\item \code{fuzzy_name}: The name of a taxon, but check for misspellings first.
Only use if you think there are misspellings. Using \code{"taxon_name"} is
faster.
\item \code{class}: A list of taxon information that constitutes the full taxonomic
classification (e.g. "K_Mammalia;P_Carnivora;C_Felidae"). Individual
taxa are separated by the \code{class_sep} argument and the information is
parsed by the \code{class_regex} and \code{class_key} arguments.
\item \code{seq_id}: Sequence ID for a particular database that is associated with a
taxonomic classification. Currently only works with the "ncbi" database.
\item \code{info}: Arbitrary taxon info you want included in the output. Can be used
more than once.
}}

\item{regex}{(\code{character} of length 1) A regular expression with capturing
groups indicating the locations of relevant information. The identity of
the information must be specified using the \code{key} argument.}

\item{class_key}{(\code{character} of length 1) The identity of the capturing
groups defined using \code{class_regex}. The length of \code{class_key} must be equal
to the number of capturing groups specified in \code{class_regex}. Any names
added to the terms will be used as column names in the output. Only
\code{"info"} can be used multiple times. Each term must be one of those
described below:
\itemize{
\item \code{taxon_name}: The name of a taxon. Not necessarily unique.
\item \code{taxon_rank}: The rank of the taxon. This will be used to add rank info
into the output object that can be accessed by \code{out$taxon_ranks()}.
\item \code{info}: Arbitrary taxon info you want included in the output. Can be used
more than once.
}}

\item{class_regex}{(\code{character} of length 1)
A regular expression with capturing groups indicating the locations of data
for each taxon in the \code{class} term in the \code{key} argument. The identity of
the information must be specified using the \code{class_key} argument. The
\code{class_sep} option can be used to split the classification into data for
each taxon before matching. If \code{class_sep} is \code{NULL}, each match of
\code{class_regex} defines a taxon in the classification.}

\item{class_sep}{(\code{character} of length 1)
Used with the \code{class} term in the \code{key} argument. The character(s) used to
separate individual taxa within a classification. After the string defined
by the \code{class} capture group in \code{regex} is split by \code{class_sep}, its
capture groups are extracted by \code{class_regex} and defined by \code{class_key}.
If \code{NULL}, every match of \code{class_regex} is used instead with first
splitting by \code{class_sep}.}

\item{sep_is_regex}{(\code{TRUE}/\code{FALSE}) Whether or not \code{class_sep} should be
used as a \href{https://en.wikipedia.org/wiki/Regular_expression}{regular expression}.}

\item{class_rev}{(\code{logical} of length 1)
Used with the \code{class} term in the \code{key} argument. If \code{TRUE}, the order of
taxon data in a classification is reversed to be specific to broad.}

\item{database}{(\code{character} of length 1) The name of the database that
patterns given in \code{parser} will apply to. Valid databases include "ncbi",
"itis", "eol", "col", "tropicos", "nbn", and "none". \code{"none"} will cause no
database to be queried; use this if you want to not use the internet. NOTE:
Only \code{"ncbi"} has been tested extensively so far.}

\item{include_match}{(\code{logical} of length 1) If \code{TRUE}, include the part of
the input matched by \code{regex} in the output object.}

\item{include_tax_data}{(\code{TRUE}/\code{FALSE}) Whether or not to include \code{tax_data}
as a dataset.}
}
\value{
Returns an object of type \code{\link[=taxmap]{taxmap()}}
}
\description{
Convert taxonomic information in a character vector into a \code{\link[=taxmap]{taxmap()}} object.
The location and identity of important information in the input is specified
using a \href{https://en.wikipedia.org/wiki/Regular_expression}{regular expression}
with capture groups and a corresponding key. An object of type \code{\link[=taxmap]{taxmap()}} is
returned containing the specified information. See the \code{key} option for
accepted sources of taxonomic information.
}
\section{Failed Downloads}{
 If you have invalid inputs or a download fails for
another reason, then there will be a "unknown" taxon ID as a placeholder
and failed inputs will be assigned to this ID. You can remove these using
\code{\link[=filter_taxa]{filter_taxa()}} like so: \code{filter_taxa(result, taxon_ids != "unknown")}. Add
\code{drop_obs = FALSE} if you want the input data, but want to remove the
taxon.
}

\examples{

\dontrun{

  # For demonstration purposes, the following example dataset has all the
  # types of data that can be used, but any one of them alone would work.
  raw_data <- c(
  ">id:AB548412-tid:9689-Panthera leo-tax:K_Mammalia;P_Carnivora;C_Felidae;G_Panthera;S_leo",
  ">id:FJ358423-tid:9694-Panthera tigris-tax:K_Mammalia;P_Carnivora;C_Felidae;G_Panthera;S_tigris",
  ">id:DQ334818-tid:9643-Ursus americanus-tax:K_Mammalia;P_Carnivora;C_Felidae;G_Ursus;S_americanus"
  )

  # Build a taxmap object from classifications
  extract_tax_data(raw_data,
                   key = c(my_seq = "info", my_tid = "info", org = "info", tax = "class"),
                   regex = "^>id:(.+)-tid:(.+)-(.+)-tax:(.+)$",
                   class_sep = ";", class_regex = "^(.+)_(.+)$",
                   class_key = c(my_rank = "info", tax_name = "taxon_name"))

  # Build a taxmap object from taxon ids
  # Note: this requires an internet connection
  extract_tax_data(raw_data,
                   key = c(my_seq = "info", my_tid = "taxon_id", org = "info", tax = "info"),
                   regex = "^>id:(.+)-tid:(.+)-(.+)-tax:(.+)$")

  # Build a taxmap object from ncbi sequence accession numbers
  # Note: this requires an internet connection
  extract_tax_data(raw_data,
                   key = c(my_seq = "seq_id", my_tid = "info", org = "info", tax = "info"),
                   regex = "^>id:(.+)-tid:(.+)-(.+)-tax:(.+)$")

  # Build a taxmap object from taxon names
  # Note: this requires an internet connection
  extract_tax_data(raw_data,
                   key = c(my_seq = "info", my_tid = "info", org = "taxon_name", tax = "info"),
                   regex = "^>id:(.+)-tid:(.+)-(.+)-tax:(.+)$")
}
}
\seealso{
Other parsers: \code{\link{lookup_tax_data}},
  \code{\link{parse_edge_list}},
  \code{\link{parse_tax_data}}
}
\concept{parsers}
